/*
 * Copyright (C) 2001, John Leuner.
 *
 * This file is part of the kissme project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2,
 * or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>


#include "classfil.h"
#include "cptuplelist.h"
#include "interp.h"

#include <jni.h>

#include "classfile_methods.h"
#include "interp_methods.h"
#include "newobject.h"
#include "../lib/indigenous/java.lang/Class.h"
#include "sys_linux_host/kissme_classload.h"

#include <assert.h>

#ifdef KISSME_LINUX_USER
#include <stdio.h>
#include <string.h>
#endif

tMutex* classTupleMutex;

//static pthread_mutex_t classtupleMutex =   {0, 0, 0, PTHREAD_MUTEX_RECURSIVE_NP, {0, 0}}; 
extern tClassLoaderTuple* pstClassType; // this can be globally used to get the type for class

tClassLoaderTuple* CLASSFILE_FindOrLoadWithError(JNIEnv* env, char* pszClassName, tClassLoaderTuple* loadingClass, uint32* errorCode)
{
  tClassLoaderTuple* pstClass;
  char* ptr;

  /* convert all dots to slashes first */
  
#ifdef SHOWLOADING
  eprintf("Find or loading %s\n", pszClassName);
#endif
  
  sys_mutex_lock(classTupleMutex);

  ptr = strchr(pszClassName, '.');
  while (ptr)
  {
    ptr[0] = '/';
    ptr = strchr(ptr, '.');
  }

  /* just return the pointer if it is already loaded */
  pstClass = CPTUPLELIST_Find(pszClassName, loadingClass == NULL ? NULL : loadingClass->classLoader);
  if (pstClass != NULL)
    {
      sys_mutex_unlock(classTupleMutex);
      return pstClass;
  }

  /*
  if((strcmp(pszClassName, "java/lang/Object") == 0) && (loadingClass == NULL))
    eprintf("******** Failed to find java/lang/Object ********\n");
  if(strcmp(pszClassName, "[Ljava/lang/Class;") == 0)
    eprintf("******** Loading [java/lang/Class, %p ********\n", (loadingClass == NULL) ? NULL : loadingClass->classLoader);
    */
  sys_mutex_unlock(classTupleMutex);
  return CLASSFILE_LoadAndLink(env, pszClassName, errorCode, loadingClass); 
}

#include "loading_errors.h"

tClassLoaderTuple* CLASSFILE_FindOrLoad(JNIEnv* env, char* pszClassName, tClassLoaderTuple* loadingClass)
{
  uint32 lostError = 0;
  tClassLoaderTuple* ret;
  ret = CLASSFILE_FindOrLoadWithError(env, pszClassName, loadingClass, &lostError);
  if(lostError != 0)
    {
      eprintf("debug: lost error code %i, (loading class %s from %s): ", (int) lostError, pszClassName, loadingClass == NULL ? "null" : loadingClass->uidName);
      printLoadingError(lostError);
      eprintf("\n");
    }
  return ret;
}

tClassLoaderTuple* getClassFromByteArray(JNIEnv* env, byte* pbData, char* pszFileName, uint32* errorCode, tMutex* classfilMutex)
{
  tClassLoaderTuple* pstClass;

  sys_mutex_lock( classfilMutex);
  //  fprintf(stderr, "Doing CLASS_Load for %s, data at %p\n", pszFileName, pbData);
  pstClass = CLASSFILE_ClassCreate(env, pbData, pszFileName, NULL, errorCode); //boot class loader
  if(pstClass == NULL)
    {
      eprintf("Error occurred creating class from array %s\n" ,pszFileName);
      //throw an exception XXX
    }
  sys_mutex_unlock( classfilMutex);
  //  fprintf(stderr, "Did CLASS_Create for %s\n", pszFileName);
  return pstClass;
}
static char* pszObjectString = "java/lang/Object";

tClassLoaderTuple* CLASSFILE_LoadAndLink(JNIEnv* env, char* pszClassName, uint32* errorCode, tClassLoaderTuple* loadingClass)
{
  tClassLoaderTuple* pstClass;
  char* pszClassFileName;
  int useJavaLangObject = 0;
  tOBREF pstExOb;

  /* to handle the array case */
  tClassLoaderTuple* baseComponentType = NULL; //The base type for object arrays
  int isPrimArray = 0; //indicates whether this is an array of primitives or not

  if (pszClassName[0] == '[')
    {
      tClassLoaderTuple* findArray;
      /* just use Object class for arrays */
      int biggest = strlen("java/lang/Object");
      int actual = strlen(pszClassName) > biggest ? strlen(pszClassName) : biggest ;
      int strsize =  actual + 10;
      useJavaLangObject = 1;

      findArray = CPTUPLELIST_Find(pszClassName, (loadingClass == NULL) ? NULL : loadingClass->classLoader);
      if( findArray != NULL)
	return findArray;
	
      pszClassFileName = (char*) sys_malloc(strsize);

      /* check that we can load the component type (if it is an array of objects)*/
      strcpy(pszClassFileName, pszClassName);
      if(strstr(pszClassFileName, "L") != NULL)
	{
	  isPrimArray = 0;
	  assert(pszClassFileName[ strlen(pszClassFileName) - 1] == ';');
	  pszClassFileName[ strlen(pszClassFileName) - 1] = '\0';
	  baseComponentType = CLASSFILE_FindOrLoad(env, strstr(pszClassFileName, "L") + 1, loadingClass);
	  if(baseComponentType == NULL)
	    {
	      //we can't fnid the component type XXX return an error code
	      return NULL;
	    }
	  else
	    {
		findArray = CPTUPLELIST_Find(pszClassName, baseComponentType->classLoader);
		if(findArray != NULL)
		 return findArray;
	    }
	}
      else
	{
	 isPrimArray = 1;
	}
      assert(pszClassFileName);
      strcpy(pszClassFileName, pszObjectString);
    }
else
  {
    pszClassFileName = pszClassName; 
  }
  
  if((loadingClass != NULL) && (pszClassName[0] != '[')) //we don't use the custom loader for array types
    {
      if(loadingClass->classLoader != NULL)
	{
	  /* Try use the class loader object to load this class */
	  jclass clazz;
	  jmethodID methodID;
	  jclass retclazz;
	  char* pszCopyString;
	  int i;

	  pszCopyString = sys_malloc( strlen(pszClassFileName) + 1 );
	  if(pszCopyString == NULL)
	    {
	      return NULL;
	    }
	  strcpy(pszCopyString, pszClassFileName);
	  for(i = 0; pszCopyString[i] != 0;i++)
	    if(pszCopyString[i] == '/')
	      pszCopyString[i] = '.';

	  clazz = (*env)->GetObjectClass(env, loadingClass->classLoader);
	  assert(clazz);
	  methodID = (*env)->GetMethodID(env, clazz, "loadClass", "(Ljava/lang/String;Z)Ljava/lang/Class;");
	  assert(methodID);
	  retclazz = (*env)->CallObjectMethod(env, loadingClass->classLoader, methodID, INTERP_NewStringFromAsciz(env, pszCopyString), JNI_FALSE);
	  sys_free(pszCopyString);
	  if(retclazz)
	    {
	      pstClass = CLASS_GetClassStruct(env, retclazz);
	      if (pszClassName[0] == '[')
		{
		  /* overwrite its name with the array signature */
		  pstClass->uidName = UID_GetUid(pszClassName);
		  pstClass->pstClass->pstSuperClass = CLASSFILE_FindOrLoad(env, pszObjectString, NULL);

		  sys_free(pszClassFileName);
		}
	      return pstClass;
	    }
	  else
	    {
	      return NULL;
	    }
	}
      else
	{
	    pstClass = getClassData(env, pszClassFileName, errorCode, classTupleMutex);
	}
    }
else
{
  pstClass = getClassData(env, pszClassFileName, errorCode, classTupleMutex);
}
  
  if(pstClass == NULL)
    {
//      eprintf("kissme: ClassCreate result is NULL (%s), reason: ", pszClassName);
//      printLoadingError(*errorCode);
//      eprintf("\n");

//      assert(1 == 2);
      return NULL;
    }
  

  

  if (pszClassName[0] == '[')
    {
      /* overwrite its name with the array signature */
      pstClass->uidName = UID_GetUid(pszClassName);
      pstClass->pstClass->pstSuperClass = CLASSFILE_FindOrLoad(env, pszObjectString, NULL); 
      sys_free(pszClassFileName);

      if(isPrimArray == 0)
	{
	 if(baseComponentType->classLoader)
	  pstClass->classLoader = baseComponentType->classLoader;
	}
    }

#ifdef SHOWLOADING
  eprintf("Adding %s %p (%p) to the CPLIST\n", pstClass->uidName, pstClass, pstClass->classLoader);
#endif
       CPTUPLELIST_Insert(pstClass);

       assert(CPTUPLELIST_Find(pstClass->uidName, pstClass->classLoader) != NULL);

       pstExOb = CLASSFILE_InitClass(env, pstClass);
       if(pstExOb != NULL)
	 {
	   //throw an exception

	   if(pstClassType == NULL)
	     {
	       eprintf("an error occurred initialising class %s, exception object %p, type %s (also, java/lang/Class has not been loaded yet)\n", pstClass->uidName, pstExOb, ODEREF(pstExOb)->pstType->uidName);
	     }
	   else
	     {
	       //throw a new ExceptionOccurredInInitializer
	       tOBREF wrapException;
	       jmethodID mid;
	       jclass clazz;

	       INTERP_printStackTrace(env, pstExOb);

	     eprintf("an error occurred initialising class %s, exception object %p, type %s _\n", pstClass->uidName, pstExOb, ODEREF(pstExOb)->pstType->uidName);

	       (*env)->ExceptionClear(env);

	       clazz = (*env)->FindClass(env, "java/lang/ExceptionInInitializerError");
	       mid = (*env)->GetMethodID(env, clazz, "<init>", "(Ljava/lang/Throwable;)V");
	       if(mid == NULL) {  (*env)->Throw( env, INTERP_ExceptionObjectFromNameAndMessage(env, "java/lang/InternalError", "failed to find constructor for ExceptionInInitializerError")); return NULL; }
	       wrapException = (*env)->NewObject(env, clazz, mid, pstExOb);
	       if(wrapException == NULL)
		 {
		   //Another exception may have been throwing while initialising this one
		   if((*env)->ExceptionOccurred(env))
		     {
		       return NULL;
		     }
		   eprintf("Failed to create a wrapper exception while handling an exception thrown in a class initializer\n");
		   assert(1 == 2);
		 }
	       (*env)->Throw(env, wrapException);
	     eprintf("an error occurred initialising class %s, exception object %p, type %s, wrapped exception %p\n", pstClass->uidName, pstExOb, ODEREF(pstExOb)->pstType->uidName, wrapException);
	     }
	   return NULL;
	 }


 //     eprintf("<1>Load and link return %s %p\n", pstClass->uidName, pstClass);
      return pstClass;
      
}

int CLASSLOADER_TUPLE_InitializeConstants(tClassLoaderTuple* tuple)
{
 int numStatics;
 int i;
 int var_index;
 int field_index;
 int offset;

 numStatics = tuple->pstClass->u16StatCount;

 if(tuple->pstClass->u16StatSize == 0)
   {
     tuple->pi32StatVars = NULL;
   }
 else
   {
     tuple->pi32StatVars = sys_malloc( sizeof(int32) * tuple->pstClass->u16StatSize);
     if(tuple->pi32StatVars == NULL)
       return -2;

     memset(tuple->pi32StatVars, 0, sizeof(int32) * tuple->pstClass->u16StatSize);

     for(i = 0, var_index = 0; i < numStatics;i++)
       {
	 byte tagtype;
	 
	 field_index = tuple->pstClass->pstStatOffsets[i].u16ConstValueIndex; // position in constant pool 
   if(field_index != 0)
     {
      offset = tuple->pstClass->pstStatOffsets[i].u16Offset;
      assert(var_index >= 0);
      assert(var_index < tuple->pstClass->u16StatSize);
      switch (tagtype = CONSTTAG(tuple->pstClass, field_index))
      {
        case CONSTANT_Integer:
        {
          tuple->pi32StatVars[var_index] = CONSTGET_Int(tuple->pstClass, field_index);
          break;
        }
        case CONSTANT_Float:
        {
          tuple->pi32StatVars[var_index] = CONSTGET_Float(tuple->pstClass, field_index);
          break;
        }
        case CONSTANT_Long:
        {
          tuple->pi32StatVars[var_index] = CONSTGET_Long_High(tuple->pstClass, field_index);
          tuple->pi32StatVars[var_index + 1] = CONSTGET_Long_Low(tuple->pstClass, field_index);
	  var_index++;
          break;
        }
        case CONSTANT_Double:
        {
          tuple->pi32StatVars[var_index] = CONSTGET_Double_High(tuple->pstClass, field_index);
          tuple->pi32StatVars[var_index + 1] = CONSTGET_Double_Low(tuple->pstClass, field_index);
	  var_index++;
          break;
        }
	/*      default:
	{
	  eprintf("Unknown constant value %i encountered in class %s\n", (int) tagtype, tuple->uidName);
	  assert(2 == 3);
	}*/
      }
	}
      var_index++;
   }
   }
  return 0;
}













