#include <config.h> 
/* 
 * @doc MODULE 
 * @module cplist.c | 
 * 
 * Implementation of class-pointer list.  This is actually a binary search 
 * tree with classes stored in alphabetical order. Functions are provided 
 * to insert into the tree and retrieve a class specified by its name. There 
 * are also functions provided to iterate through the list in alphabetical 
 * order. This is achieved by doing an incremental in-order traversal of 
 * the tree. 
 * 
 */ 
 
#include "wrappers.h"
#include "assert.h"

#ifdef KISSME_LINUX_USER 
#include <string.h> 
#include <stdlib.h> 
#endif 
 
 
#include "cplist.h" 
 
#include "global.h" 
  
#ifdef OBSOLETED

 
static tCPListElement* pstTreeTop = NULL; 
static tCPListElement* pstCurrPointer = NULL; 
static int iChanged; 
 
/* 
 * @doc FUNC 
 * @func 
 * Inserts a class into the tree in alphabetical order. 
 * 
 */ 
 
void CPLIST_Insert 
( 
  tClass* pstClass   /* @parm The class to add */ 
) 
{ 
  tCPListElement* pstNew; 
  tCPListElement* pstTemp; 
  int iDone = 0; 
  int iCompare; 
 
  iChanged = 1; 
 
  pstNew = (tCPListElement*) sys_malloc(sizeof(tCPListElement)); 

  assert(pstNew); 

 
  pstNew->pstClass = pstClass; 
			     
  pstNew->pstLeft = NULL; 
  pstNew->pstRight = NULL; 
 
  /* if list is empty, simple case */ 
  if (pstTreeTop == NULL) 
  { 
    pstTreeTop = pstNew; 
    pstNew->pstParent = NULL; 
  } 
  /* otherwise search for place to put it */ 
  else 
  { 
    pstTemp = pstTreeTop; 
    while (!iDone) 
    { 
      iCompare = strcmp(pstClass->uidName, pstTemp->pstClass->uidName); 
  
      /* class is already here */ 
      if (iCompare == 0) 
      { 
        iDone = 1; 
      } 
      /* need to go left */ 
      else if (iCompare < 0) 
      { 
        if (pstTemp->pstLeft) 
        { 
          pstTemp = pstTemp->pstLeft; 
        } 
        else 
        { 
          pstNew->pstParent = pstTemp; 
          pstTemp->pstLeft = pstNew; 
          iDone = 1; 
        } 
      } 
      /* need to go right */ 
      else 
      { 
        if (pstTemp->pstRight) 
        { 
          pstTemp = pstTemp->pstRight; 
        } 
        else 
        { 
          pstNew->pstParent = pstTemp; 
          pstTemp->pstRight = pstNew; 
          iDone = 1; 
        } 
      } 
    } 
  } 
} 
 
/* 
 * @doc FUNC 
 * @func 
 * Searches for an element in the tree, given the name of the class. 
 * 
 * @rdesc Returns a pointer to the class if it's found. NULL otherwise. 
 * 
 */ 
 
tClass* CPLIST_Find 
( 
  char* pszName 
) 
{ 
  Uid uidName = UID_GetUid(pszName); 
 
  #ifdef UID_WARNINGS 
  if (uidName == pszName) 
  { 
    printf("Alert - Uid passed to CPLIST_Find - %s\n", pszName); 
  } 
  #endif /*UID_WARNINGS*/ 
 
  return CPLIST_UidFind(uidName); 
} 
 
/* 
 * @doc FUNC 
 * @func 
 * Searches for an element in the tree, given the name of the class. The string 
 * pointer that is passed must be a Uid. 
 * 
 * @rdesc Returns a pointer to the class if it's found. NULL otherwise. 
 * 
 */ 
 
tClass* CPLIST_UidFind 
( 
  Uid uidName 
) 
{ 
  tCPListElement* pstTemp; 
  int iCompare; 
 
  if (pstTreeTop == NULL) 
  { 
    return NULL; 
  } 
 
  pstTemp = pstTreeTop; 
  while (1) 
  { 

    iCompare = (strcmp(uidName, pstTemp->pstClass->uidName)); 
 
    /* found it */ 
    if (iCompare == 0) 
    { 
      return pstTemp->pstClass; 
    } 
    /* else go left */ 
    else if (iCompare < 0) 
    { 
      if (pstTemp->pstLeft) 
      { 
        pstTemp = pstTemp->pstLeft; 
      } 
      else 
      { 
        return NULL; 
      } 
    } 
    /* else go right */ 
    else 
    { 
      if (pstTemp->pstRight) 
      { 
        pstTemp = pstTemp->pstRight; 
      } 
      else 
      { 
        return NULL; 
      } 
    } 
  } 
} 
 
/* 
 * @doc FUNC 
 * @func 
 * Checks whether or not the list is empty. 
 * 
 * @rdesc Returns one of the following: 
 * 
 * @flag 0 | Not empty 
 * @flag 1 | Empty 
 * 
 */ 
 
int CPLIST_IsEmpty 
( 
  void 
) 
{ 
  return (pstTreeTop == NULL); 
} 
 
/* 
 * @doc FUNC 
 * @func 
 * Sets the current element to the first in the list. 
 * 
 */ 
 
void CPLIST_GoTop 
( 
  void 
) 
{ 
  /* find leftmost element in tree */ 
  pstCurrPointer = pstTreeTop; 
  if(pstCurrPointer)
    {
      while (pstCurrPointer->pstLeft) 
	{ 
	  pstCurrPointer = pstCurrPointer->pstLeft; 
	} 
    }
} 
 
/* 
 * @doc FUNC 
 * @func 
 * Sets the current element to the alphabetically next element in the list. 
 * 
 */ 
 
void CPLIST_Skip 
( 
  void 
) 
{ 
  if (pstCurrPointer) 
  { 
    /* if we can go right, we go right and then as left as possible */ 
    if (pstCurrPointer->pstRight) 
    { 
      pstCurrPointer = pstCurrPointer->pstRight; 
      while (pstCurrPointer->pstLeft) 
      { 
        pstCurrPointer = pstCurrPointer->pstLeft; 
      } 
    } 
    /* else we have to move back up */ 
    else 
    { 
      /* if parent is NULL, we have finished */ 
      if (pstCurrPointer->pstParent == NULL) 
      { 
        pstCurrPointer = NULL; 
        return; 
      } 
 
      /* if we are the left child, we just go to the parent */ 
      if (pstCurrPointer == pstCurrPointer->pstParent->pstLeft) 
      { 
        pstCurrPointer = pstCurrPointer->pstParent; 
      } 
      /* if we are the right child, we go to the parent and then continue 
         going up until we go up a "left branch" */ 
      else 
      { 
        /* keep going while we are still a right child */ 
        while (pstCurrPointer == pstCurrPointer->pstParent->pstRight) 
        { 
          pstCurrPointer = pstCurrPointer->pstParent; 
          /* check if we've got to the top */ 
          if (pstCurrPointer->pstParent == NULL) 
          { 
            pstCurrPointer = NULL; 
            return; 
          } 
        } 
        /* go up the left branch - if parent is NULL then so be it */ 
        pstCurrPointer = pstCurrPointer->pstParent; 
      } 
    } 
  } 
} 
 
/* 
 * @doc FUNC 
 * @func 
 * Checks if the previous call to CPLIST_Skip reached the end of the list. 
 * 
 * @rdesc Returns one of the following: 
 * 
 * @flag 0 | Reached end 
 * @flag 1 | Not at end 
 * 
 */ 
 
int CPLIST_IsMore 
( 
  void 
) 
{ 
  return (pstCurrPointer != NULL); 
} 
 
/* 
 * @doc FUNC 
 * @func 
 * Returns the class pointer of the current element. 
 * 
 * @rdesc Returns the class pointer of the current element. 
 * 
 */ 
 
tClass* CPLIST_GetCurrentClass 
( 
  void 
) 
{ 
  if (pstCurrPointer != NULL) 
  { 
    return pstCurrPointer->pstClass; 
  } 
  else 
  { 
    return NULL; 
  } 
} 
 
/* 
 * @doc FUNC 
 * @func 
 * Returns the class name of the current element. 
 * 
 * @rdesc Returns the class name of the current element. 
 * 
 */ 
 
char* CPLIST_GetCurrentName 
( 
  void 
) 
{ 
  if (pstCurrPointer != NULL) 
  { 
    return pstCurrPointer->pstClass->uidName; 
  } 
  else 
  { 
    return NULL; 
  } 
} 
 
/* 
 * @doc FUNC 
 * @func 
 * Clears the "changed" flag. 
 * 
 */ 
 
void CPLIST_ClearChanged 
( 
  void 
) 
{ 
  iChanged = 0; 
} 
 
/* 
 * @doc FUNC 
 * @func 
 * This checks if the "changed" flag is set. This flag is set whenever a new 
 * element is added to the list. It can be reset with CPLIST_ClearChanged(). 
 * 
 * @rdesc Returns 0 if it's not set, non-0 if it is set. 
 * 
 */ 
 
int CPLIST_HasChanged 
( 
  void 
) 
{ 
  return (iChanged == 1); 
} 
 
static void TreeDelete 
( 
  tCPListElement* pstRoot 
) 
{ 
  if (pstRoot->pstLeft) 
  { 
    TreeDelete(pstRoot->pstLeft); 
  } 
  if (pstRoot->pstRight) 
  { 
    TreeDelete(pstRoot->pstRight); 
  } 
  sys_free(pstRoot); 
} 
 
void CPLIST_Empty 
( 
  void 
) 
{ 
  if(pstTreeTop)
    TreeDelete(pstTreeTop); 
} 
 
 
 
 
 
void CPLIST_Dump() 
{ 
    fprintf(stderr, "CPLIST tree\n"); 
    CPLIST_DumpNode(pstTreeTop, 0); 
    fprintf(stderr, "end CPLIST tree\n"); 
} 
 
void CPLIST_DumpNode 
( 
 tCPListElement* root, 
 int indent 
)  
{ 
  if (root == NULL) 
  { 
      return; 
  } 
  else 
      { 
	  int i; 
	for(i = 0; i < indent;i++) 
	    fprintf(stderr, " "); 
 
	if(root->pstParent) 
	    { 
	    if(root->pstParent->pstLeft == root) 
		fprintf(stderr, "L"); 
	    else 
		fprintf(stderr, "R"); 
	    } 
	fprintf(stderr, "* %s\n", root->pstClass->uidName); 
 
	if(root->pstLeft) 
	    CPLIST_DumpNode( root->pstLeft, indent + 1); 
	if(root->pstRight) 
	    CPLIST_DumpNode( root->pstRight, indent + 1); 
 
	return; 
    } 
} 
/* end of cplist.c */ 
#endif
