#ifdef USE_DISTRIBUTED_STORE


/* alpha.c by John Leuner
   990906

   This file implements the function calls needed to send and receive data from the Virtual Disk Server (ALPHA).

   See alpha.h for a list for functions and their descriptions
 */


#include <pthread.h> //We need threads
#include <stdlib.h>
#include "storemanager.h" //All the socket stuff gets pulled in here
#include "alpha.h"
#include <assert.h>


/* The send queue */

struct send_queue_entry* ALPHA_send_queue = NULL;

/* The read queue */

struct listentry* ALPHA_read_queue = NULL;

/* We have mutexes around each queue to stop threads from mucking them up */

pthread_mutex_t ALPHA_read_queue_mutex = PTHREAD_MUTEX_INITIALIZER;
pthread_mutex_t ALPHA_write_queue_mutex = PTHREAD_MUTEX_INITIALIZER;

/* Just a variable that helps us wait until our threads are up */

int  listen_thread_is_ready = 0;

/* To generate unique sequences for requests to ALPHA.*/

int ALPHA_sequence = 0;

ALPHA_listen_for_read_socket = -1;
ALPHA_listen_for_write_socket = -1;
int ALPHA_is_finished = 0;

void ALPHA_shutdown()
{
    printf("ALPHA shutting down\n");
    if(ALPHA_listen_for_read_socket > 0)
	{
	    close(ALPHA_listen_for_read_socket);
	}
    if(ALPHA_listen_for_write_socket > 0)
	{
	    close(ALPHA_listen_for_write_socket);
	}
    ALPHA_is_finished = 1;
}

void ALPHA_initialize()
{
  
    pthread_t listen_read_t;
    pthread_t listen_write_t;
    pthread_t timeout_t;

    printf("ALPHA_initialize called\n");

    /* Set up mutexes */
    pthread_mutex_init(&ALPHA_read_queue_mutex, NULL);
    pthread_mutex_init(&ALPHA_write_queue_mutex, NULL);

    /* The sequences */

    ALPHA_sequence = 1001 + rand();

    /* The queues */

    ALPHA_read_queue = NULL;
    ALPHA_send_queue = NULL;

    /* Our socket for talking to the Virtual Disk Server */
    initialize_remote_socket();

    /* Start listen threads */

  if(pthread_create(&timeout_t, NULL,  & (ALPHA_send_timeout), NULL) == -1)
    {
      perror("ALPHA_initialize create_thread for send_timeout");
    }
  if(pthread_create(&listen_write_t, NULL,  & (listen_for_alpha_write_replies), NULL) == -1)
    {
      perror("ALPHA_initialize create_thread for listening to alpha write replies");
    }
  if(pthread_create(&listen_read_t, NULL,  & (listen_for_alpha_read_replies), NULL) == -1)
    {
      perror("ALPHA_initialize create_thread for listening to alpha read replies");
    }

  while( listen_thread_is_ready != 1)
    {
      /* Spin while thread starts 
	 And then we're ready to start servicing requests
       */
    }
  printf("ALPHA is ready\n");
}

/* Stuff for socket comms with the VDS */
int remote_socket = -1;
struct sockaddr_in to;
int tolen = sizeof(to);

/* Method which sets up the socket */

int initialize_remote_socket()
{
    struct in_addr remoteAddr;
    struct hostent* host;


 if(remote_socket == -1)
     {
       int the_port;
   /*Initialize address info*/

    to.sin_family = AF_INET;

    if(inet_aton("1.0.0.2", &to.sin_addr) == 0)
	{
	    perror("initialize_remote_socket");
	}

    the_port = ALPHA_PORT;
    to.sin_port =  htons( the_port );

    printf("Creating socket for %s\n", inet_ntoa(to.sin_addr));

    /*Create the socket*/

    remote_socket = socket(PF_INET, SOCK_DGRAM, IPPROTO_UDP);
    if(remote_socket == -1)
      {
	perror("write_object_entry: connecting to alpha");
	exit(3);
      }
     }
     return remote_socket;
}


int ALPHA_write_non_blocking(struct page_entry* stpePage)
{
  int s; // The socket
  int sequence, sent, willadd; //The sequence of this request, the amount of bytes sent and a boolean variable
  struct send_queue_entry* queue_entry = (struct send_queue_entry*) malloc(sizeof(struct send_queue_entry)), *iterator; //Two variables for playing with the send queue
  int* sendbuf = (int*) malloc( WRITE_HEADER_SIZE + BLOCK_SIZE); //Our datagram packet buffer
    assert(sendbuf != NULL);

    /* We lock the mutex for the write queue to make sure the threads behave */
    pthread_mutex_lock(&ALPHA_write_queue_mutex);
    sequence = ALPHA_sequence++; //Increment the sequence

    sendbuf[0] = htonl( ALPHA_listen_port + 1); //Tell the VDS what port they should reply to
    sendbuf[1] = htonl( ALPHA_WRITE );          //Tell them we want to write a page
    sendbuf[2] = htonl( WRITE_HEADER_SIZE + BLOCK_SIZE); //Tell them the size of this whole packet
    sendbuf[3] = htonl( 0 ); //Unused
    sendbuf[4] = htonl( sequence ); //Give them the request sequence

    sendbuf[5] = htonl( (int) (stpePage->address >> 32) ); //Give them the address we want to write to
    sendbuf[6] = htonl( (int) ((stpePage->address << 32) >> 32 ));

    sendbuf[7] = htonl( BLOCK_SIZE ); //How much we want to write

    memcpy(sendbuf + 8, stpePage->page_data, BLOCK_SIZE); //Copy the page data from the page structure

    /* Add an entry for this page to the write queue (only if it isn't there already)*/

    queue_entry->timeout = 5; //Set up the timeout
    queue_entry->sequence_number = sequence;
    queue_entry->stpePage = stpePage;

    willadd = 1; //We will add the entry, we'll set this to 0 if it's there already

    /* Look through the queue for this page */

    iterator = ALPHA_send_queue;
    while(iterator != NULL)
      {
	if(iterator->stpePage == queue_entry->stpePage)
	  {
	    /* Found it */
	    willadd = 0; //We won't add the entry
	    queue_entry->sequence_number = iterator->sequence_number; //we reuse the old sequence
	    sendbuf[4] = htonl(queue_entry->sequence_number); //and make sure we reflect that in the packet
	    break;
	  }
	iterator = iterator->next;
      }


    if(willadd == 1)
      {
	//We are going to add a new item, just do it at the head
	if(ALPHA_send_queue == NULL)
	  {
	    ALPHA_send_queue = queue_entry;
	    queue_entry->next = NULL;
	  }
	else
	  {
	    queue_entry->next = ALPHA_send_queue;
	    ALPHA_send_queue = queue_entry;
	  }
      }

    /*Send the datagram*/

    s = initialize_remote_socket();

    if((sent = sendto(s, sendbuf, WRITE_HEADER_SIZE + BLOCK_SIZE, 0, &to, tolen)) == -1)
      {
	perror("ALPHA_write_blocking sending to alpha");
	exit(3);
      }
    else
	{
	  assert(sent == (WRITE_HEADER_SIZE + BLOCK_SIZE)); //make sure we sent everything !!!
	    //	    printf("ALPHA_write_non_blocking sent %i bytes, sequence %i\n", sent, sequence);
	}

    free(sendbuf); //clean up buffer
    pthread_mutex_unlock(&ALPHA_write_queue_mutex); //unlock the mutex
    return 0; //successful
}

int ALPHA_read_blocking(long long address, void* buf, int buf_len)
{
/*Do a read from alpha for that address */ 

    int s; //socket
    int timeout = ALPHA_READ_TIMEOUT;
    int sequence;//The unique sequence of this request
    int* sendbuf = (int*) malloc( READ_HEADER_SIZE + buf_len); //Our request datagram buffer
    assert(sendbuf != NULL);
    sequence = ALPHA_sequence++;  

//    printf("Alpha read blocking, port %i, (size %i), address %d, length %i, sequence %i\n", remotePort, sizeof(address), address, buf_len, sequence);

    sendbuf[0] = htonl( ALPHA_listen_port ); //The port to which the reply should be sent
    sendbuf[1] = htonl( ALPHA_READ ); //We want to do a read
    sendbuf[2] = htonl( READ_HEADER_SIZE ); //How much header info is in this request
    sendbuf[3] = htonl( 0 ); //unused
    sendbuf[4] = htonl( sequence ); 

    sendbuf[5] = htonl( (int) (address >> 32) ); //the address
    sendbuf[6] = htonl( (int) ((address << 32) >> 32 ));

    sendbuf[7] = htonl( buf_len ); //how much data we want to read

    s = initialize_remote_socket();

    /*Send the datagram*/

    if(sendto(s, sendbuf, buf_len + READ_HEADER_SIZE, 0, &to, tolen) == -1)
      {
	printf("Sending to host %s\n",   to.sin_addr);
	perror("ALPHA_read_blocking : sending to alpha");
	exit(3);
      }



/*Now wait for the acknowledgement from the server */

while(1)
{
  int retval;
  int temp_len = buf_len;
  pthread_mutex_lock(&ALPHA_read_queue_mutex); //First lock the read queue

  if(timeout == 0) //If we have time out, resend the request
    {
    if(sendto(s, sendbuf, buf_len + READ_HEADER_SIZE, 0, &to, tolen) == -1)
      {
	printf("Sending to host %s\n",   to.sin_addr);
	perror("ALPHA_read_blocking : sending to alpha");
	exit(3);
      }
    timeout = ALPHA_READ_TIMEOUT; //reset the timeout
    }
  else
    {
      timeout--; //decrement the timeout
      retval = ALPHA_remove_item_from_read_queue(sequence, buf , &temp_len); //attempt to see if there was a reply
  if(retval == ALPHA_ACK)
    {
      //There was an ACK, this should never happen for reads
      assert(0);
#ifdef DEBUG_ALPHA
      printf("alpha_read_bl read ACK\n");
#endif
      pthread_mutex_unlock(&ALPHA_read_queue_mutex);
      break;
    }
  else if(retval == ALPHA_DATA)
    {
      //The server responded with the data we want, so unlock the queue 
#ifdef DEBUG_ALPHA
      printf("alpha_read_bl read DATA\n");
#endif
      pthread_mutex_unlock(&ALPHA_read_queue_mutex);
      break;
    }
  else if(retval == -2)
    {
      //There was a negative response, so this method fails (-1)
      //   printf("alpha_read_bl read NACK (-2)\n");
	free(sendbuf);
        pthread_mutex_unlock(&ALPHA_read_queue_mutex);
	return -1;
    }
   else
    {
	//	printf("Waiting for read ack %i\n", sequence);
    }
    }
  pthread_mutex_unlock(&ALPHA_read_queue_mutex); //unlock the read queue to give the other thread a chance to put something on it
  usleep( 5000 ); //sleep a bit
}
  free(sendbuf);
  return 0; //Success
}






/* This thread listens for replies from alpha. These replies are usually acks of writes or data for reads*/



void listen_for_alpha_read_replies(char** arg)
{
  int s;
  struct sockaddr_in sin;
  struct sockaddr_in from;
  int fromlen;
  char* buf = (char*) malloc( 8192 );

  printf("** Listen for alpha reads started, listening on port %i \n", ALPHA_listen_port);
  sin.sin_family = AF_INET;
  sin.sin_addr.s_addr = INADDR_ANY;
  sin.sin_port = htons(ALPHA_listen_port);

  ALPHA_listen_for_read_socket = s = socket(PF_INET, SOCK_DGRAM, IPPROTO_UDP);
  if(s == -1)
  {
    perror("listen_for_alpha create socket");
  }
 if( bind(s, &sin, sizeof(sin)) == -1)
   {
     perror("listen_for_alpha bind");
     exit(44);
   } 

 listen_thread_is_ready = 1;
 /*Loop continuously listening for acks*/
 while(1)
   {
     int* int_p;
     int actual  = 0;
     fromlen = sizeof(from);
     actual = recvfrom(s, buf, 8192, 0, &from, &fromlen); 
     if((actual == -1) || (actual < 5*4))
       {
	 perror("listen_for_alpha");
       }
     int_p = (int*) buf;

     #ifdef DEBUG_PROXY
     printf("listen_for_alpha: got reply sequence %i (read %i bytes)\n", ntohl(int_p[4]), actual);
     #endif

     ALPHA_put_item_on_read_queue( (int) ntohl(int_p[4]), buf, actual);
   
   }

}

/* and for write replies */



void listen_for_alpha_write_replies(char** arg)
{
  int s;
  struct sockaddr_in sin;
  struct sockaddr_in from;
  int fromlen;
  char* buf = (char*) malloc( 8192 );

  printf("** Listen for alpha writes started, listening on port %i \n", ALPHA_listen_port + 1);
  sin.sin_family = AF_INET;
  sin.sin_addr.s_addr = INADDR_ANY;
  sin.sin_port = htons(ALPHA_listen_port + 1);

ALPHA_listen_for_write_socket =  s = socket(PF_INET, SOCK_DGRAM, IPPROTO_UDP);
  if(s == -1)
  {
    perror("listen_for_alpha create socket");
  }
 if( bind(s, &sin, sizeof(sin)) == -1)
   {
     perror("listen_for_alpha bind");
     exit(44);
   } 

 /*Loop continuously listening for acks*/
 while(1)
   {
     int* int_p;
     struct send_queue_entry* iter, *prev;
   int sequence = 0;
     int actual  = 0;
     fromlen = sizeof(from);
     actual = recvfrom(s, buf, 8192, 0, &from, &fromlen); 
     if((actual == -1) || (actual < 5*4))
       {
	 perror("listen_for_alpha write acks");
       }
     int_p = (int*) buf;

     //     #ifdef DEBUG_PROXY
     printf("listen_for_alpha_write_replies: got reply sequence %i (read %i bytes)\n", ntohl(int_p[4]), actual);
     // #endif

     sequence = ntohl(int_p[4]);
    pthread_mutex_lock(&ALPHA_write_queue_mutex); 
     iter = prev = ALPHA_send_queue;
     while(iter != NULL)
       {
	 if(iter->sequence_number == sequence)
	   {
	     if(iter == ALPHA_send_queue)
	       {
		 ALPHA_send_queue = iter->next;
		 free(iter);
		 break;
	       }
	     else
	       {
		 prev->next = iter->next;
		 free(iter);
	       }
	   }
	 prev = iter;
	 iter = iter->next;
       }
    pthread_mutex_unlock(&ALPHA_write_queue_mutex);
   }

}

void ALPHA_send_timeout(char** arg)
{
    struct send_queue_entry* iter;
    int success, dosleep;
    
    while(1)
      {
        pthread_mutex_lock(&ALPHA_write_queue_mutex);
	dosleep = 1;
	if(ALPHA_send_queue != NULL)
	    printf("ALPHA_send_timeout looking to send\n");
     iter = ALPHA_send_queue;
     while(iter != NULL)
       {
	 if(iter->timeout > 0)
	   iter->timeout--;
	 else
	   {
	     dosleep = 0;
	     iter->timeout = ALPHA_SEND_TIMEOUT;
             printf("ALPHA_send_timeout sending, address %x %x\n", (int) (iter->stpePage->address >> 32), (int)((iter->stpePage->address << 32) >> 32));
       pthread_mutex_unlock(&ALPHA_write_queue_mutex);
	     success = ALPHA_write_non_blocking(iter->stpePage);
        pthread_mutex_lock(&ALPHA_write_queue_mutex);
	     assert(success == 0);
	   }
	 iter = iter->next;
       }
       pthread_mutex_unlock(&ALPHA_write_queue_mutex);
       if(dosleep == 1)
	   {
	       if(ALPHA_is_finished == 1)
		   return;
	       sleep(2);

	   }
      }

}



int entries = 0;

int ALPHA_put_item_on_read_queue(int sequence_number, char* buf, int buf_len)
{
  struct listentry* read_list = ALPHA_read_queue;
  if(read_list == NULL)
    {
      ALPHA_read_queue = (struct listentry*) malloc(sizeof(struct listentry));
      read_list = ALPHA_read_queue;
      assert(read_list != NULL);
      read_list->next = NULL;
      read_list->sequence_number = sequence_number;
      read_list->data = buf;
      read_list->data_length = buf_len;
    }
  else
   {
     struct listentry* prev = read_list;
     struct listentry* iter = read_list;
     while(iter != NULL)
       {
	 prev = iter;
	 iter = iter->next;
       }
     prev->next = (struct listentry*) malloc(sizeof(struct listentry));
     assert(prev->next != NULL);
     prev->next->next = NULL;
     prev->next->sequence_number = sequence_number;
     prev->next->data = buf;
     prev->next->data_length = buf_len;
   }
  entries++;
  return 0;
}

int ALPHA_remove_item_from_read_queue(int sequence_number, char* buf, int* buf_len)
/*int get_read_reply(int sequence_number, char* buf, int* buf_len)*/
{
    /*This method looks for a spec. seq. If it finds it, it removes it from the list */

  struct listentry* prev = ALPHA_read_queue;
  struct listentry* iter = prev;

  if(prev == NULL)
    return -1;

   while((iter != NULL) && (iter->sequence_number != sequence_number))
       {
	 prev = iter;
	 iter = iter->next;
       }

   if(iter == NULL)
     return -1;
   else
     {
       if(iter == ALPHA_read_queue)
	 {
	   int retval;
	   ALPHA_read_queue = iter->next;
	   retval =  processIter(iter, buf, buf_len);
	   entries--;
	   return retval;
	 }
       else
	 {
	   struct listentry* old = prev->next;       	
	   prev->next = old->next;
	   entries--;
	   return processIter(old, buf, buf_len);
	 }
       return 0;
     }
}

/* This is a helper method for ALPHA_remove_item_from_read_queue, it checks the header of the reply and returns an appropriate integer. It also copies the data if need be.
 */

int processIter(struct listentry *iter, void* buf, int *buf_len)
{
   int* int_p;

	   //Examine the reply command
	    int_p = (int*) iter->data;
	   if(htonl(int_p[1]) == ALPHA_ACK)
	     {
	       //If it is an acknowledgement
	   if(*buf_len < (iter->data_length - ACK_HEADER_SIZE))
	     {
	       printf("Buffer too small %i < %i \n", *buf_len, (iter->data_length - ACK_HEADER_SIZE));
	       exit(4);
	     }

	       free(iter);
	       return ALPHA_ACK;
	     }
	   else if(htonl(int_p[1]) == ALPHA_NACK)
	     {
	       //If it is a negative response
		 //	       printf("get_read_reply read a NACK do free\n");
	       free(iter);
	       return -2;
	     }
	   else if(htonl(int_p[1]) == ALPHA_DATA)
	     {
	       //if it is the data we want
	   if(*buf_len < (iter->data_length - DATA_HEADER_SIZE))
	     {
	       printf("Buffer too small %i < %i \n", *buf_len, (iter->data_length - DATA_HEADER_SIZE));
	       exit(4);
	     }

#ifdef DEBUG_ALPHA
	       printf("get_read_reply read a DATA from alpha\n");
	       printf("Do mem cpy total %i, copying %i, buf_len %i\n", iter->data_length, iter->data_length - DATA_HEADER_SIZE, *buf_len);
#endif

	       //Copy that data
	       memcpy(buf, iter->data + DATA_HEADER_SIZE, (iter->data_length - DATA_HEADER_SIZE));
	       //Indicate how much we copied
	       *buf_len = iter->data_length;
	       free(iter);
	       return ALPHA_DATA;
	     }
	   else
	       {
	       printf("alpha.c Unexpected return code   from alpha\n");
	       exit(6);
	     }
}






#endif
