#ifdef USE_DISTRIBUTED_STORE


/* eta.c 

   by John Leuner

   This file implements the methods required by the ETA module. See eta.h for a overview of the module.

 */
#include "stdtypes.h"
#include "eta.h"
#include <pthread.h> //We need to run listen threads in this module
#include <assert.h>
#include <stdlib.h>
#include "storemanager.h" //Includes a whole lot of socket stuff for us

#include "hash.h" //We need this for the hashtable

/* The implementation of ETA is based on Li and Hudak's paper on "Memory Coherence in Shared Virtual Memory Systems". (from page 235, section 4.5)

   The idea is to implement a distributed lock service, mainly aiming to minimize the number of messages needed to lock a page, and ensuring quick access once a lock has been obtained on a page. 

I describe here the algorithms used for various parts of the process:

p refers to the page being sought, ptable is a table of pages with various information for each (struct page_entry)

The "read fault handler" is what gets executed when an instance of ETA doesn't have read access to a page. It then contacts another instance of ETA which will run its "read server" to satisfy the request for the page. Similarly the "write fault handler" and "write server" operate in client / server fashion. 

The algorithm that keeps the pages coherent is best decsribed by the pseudo-code below. Essentially we are trying to solve the problem of having at most one writer per page, and keeping any amount of readers of that page aware of changes to it. This means that before any host can write to a page, it first has to tell all the hosts reading that page that the old copy is now invalid. It does this via broadcasting the Invalidate message to all those that have read copies. When a host requests a copy of a page for reading it gets added to the copy set for that page. When a host wants to write to a page it takes 'ownership' of the page (from the old owner) and then has the exclusive right to write to the page (until ownership is transferred). 

 
Read fault handler:

lock( ptable[p].lock )
ask ptable[p].prob_owner for read access to p
ptable[p].prob_owner := reply_node (not necessarily the one we contacted)
ptable[p].access := read;
unlock( ptable[p].lock )

Read Server:
if ptable[p].access != nil THEN BEGIN
lock(ptable[p].lock)
ptable[p].copy_set := ptable[p].copy_set union { request_node }
ptable[p].access = read;
send p 
unlock( ptable[p].lock );
END
ELSE BEGIN

forward request to ptable[p].prob_owner;
ptable[p].prob_owner := request_node;
END

Write fault handler:

lock (ptable[p].lock);
ask ptable[p].prob_owner for write access to p;
invalidate(ptable[p].copy_set);
ptable[p].prob_owner = self;
ptable[p].access = write;
ptable[p].copy_set = {};
unlock( ptable[p].lock);

Write server

If I am owner THEN
lock (ptable[p].lock)
ptable[p].access := nil;
send p, ptable[p].copy`set 
ptable[p].prob_owner = request_node;
unlock ( ptable[p].lock );
END
ELSE BEGIN
forward request to ptable[p].prob_owner;
ptable[p].prob_owner := request_node;
END;

Invalidate server 

if( ptable[p].access != nil ) THEN BEGIN
 Invalidate(p, ptable[p].copy_set)
 ptable[p].access = nil
 ptable[p].prob_owner = request_node
 ptable[p].copy_set = {}
END







 */
int eta_sequence = 0xb00b00b;

pthread_mutex_t ETA_read_mutex = PTHREAD_MUTEX_INITIALIZER;
pthread_mutex_t ETA_write_mutex = PTHREAD_MUTEX_INITIALIZER;

struct listentry {
  int sequence_number;
  char* data;
  int data_length;
  struct listentry* next;
};

int ETA_entries = 0;

struct listentry* ETA_outstanding_list = NULL;

int our_id_host = 0;
int our_id_port = 0;

int default_id_host = 0;
int default_id_port = 0;
void ETA_listen(void* arg);

int send_socket = -1;

void ETA_send_identification(int id, int port, int host)
{
    struct sockaddr_in to;
    int tolen = sizeof(to);
    int remote_socket;
    int* sendbuf = (int*) malloc( 4 * 4);

    sendbuf[0] =  ETA_IDENTIFY ;
    sendbuf[1] = id;
    sendbuf[2] = host;

   /*Initialize address info*/

    to.sin_family = AF_INET;
    to.sin_addr.s_addr = host;
    to.sin_port =  htons(port);

    #ifdef ETA_DEBUG
    printf("ETA_send_identification Creating socket for %s\n", inet_ntoa(to.sin_addr));
#endif
    /*Create the socket*/

    if(send_socket == -1)
      remote_socket = send_socket = socket(PF_INET, SOCK_DGRAM, IPPROTO_UDP);
    else
      remote_socket = send_socket;
    if(remote_socket == -1)
      {
	perror("ETA_send_identification: connecting to hosts");
	exit(247);
      }

   if(sendto(remote_socket, sendbuf, 4 * 4, 0, &to, tolen) == -1)
      {
	perror("send_id");
	exit(248);
      }

   free(sendbuf);
}


struct HASH_tstHashTable alt_hash_table;

int ETA_listen_socket = -1;

void ETA_shutdown()
{
    printf("ETA shutting down\n");
    //We need to kill the listen thread 
    if(ETA_listen_socket > 0)
	{
	    close(ETA_listen_socket);
	}

}

void ETA_initialize()
{
  int success;
  struct hostent* host_info;
  struct in_addr ina;
  struct in_addr in_our;
  struct in_addr in_default;
  FILE* file;
  pthread_t listen_tid;
  int host_id;


  HASH_InitHashTable(&alt_hash_table, 2); 


  eta_sequence += rand();
  pthread_mutex_init(&ETA_read_mutex, NULL);
  pthread_mutex_init(&ETA_write_mutex, NULL);

  our_id_port = 20001;

  if(pthread_create(&listen_tid, NULL, & (ETA_listen), NULL) == -1)
    {
      perror("ETA create_thread for listen");
    }

  //Sleep while the listen thread starts up

  sleep(1);
  host_id = gethostid();
  printf("Looking for eta.rc\n");
while(our_id_host == 0)
  {
  printf("Starting identify sequence\n");
  file = fopen("eta.rc", "r");
  if(file != NULL)
    {
      char buffer[256];
      char* res;
      int port;
      while((res = fgets(buffer, 256, file)) != NULL)
	{

	  //Parse the eta.rc file
	  //First entry is the ip address

	  char* next = (char*) strtok(buffer, " ");
	  if(inet_aton(next, &in_our) == 0)
	    {
	      printf("Host %s\n", next);
	      perror("ETA_initialize");
	      exit(7);
	    }

	  //Next entry is the port
    next = (char*) strtok(NULL, " ");
    port = atoi(next);

    //And then the optional "default" flag

    next = (char*) strtok(NULL, " ");
    if(next != NULL)
      {
	if(strncmp(next, "default", strlen("default")) == 0)
	  {
	    default_id_host = in_our.s_addr;
	    default_id_port = port;
	 }
      }

    //Now we say hello to all these hosts
      ETA_send_identification(host_id, port, in_our.s_addr);
    }
    fclose(file);
  }
 usleep(500);
}
  in_our.s_addr = our_id_host;
  in_default.s_addr = default_id_host;
  printf("Our host %s\n",  inet_ntoa(in_our));
  printf("Default host %s\n", inet_ntoa(in_default));

}

/*This method checks whether we have a local copy of this page, if there is no entry in the hashtable we return NULL. If we are the owner then it is returned, otherwise we block until we have a copy.



*/

int ETA_ready_read_copy(struct page_entry* mpage)
{
  int sequence;
  int success;
  //Check whether this is a read fault
  if( (mpage->access == 1) || (mpage->access == 2)) //Access is read or write
    {
      assert(mpage->prob_owner_host != 0);
      #ifdef ETA_DEBUG
            printf("ETA_ready_read_copy %x %x OURS\n", (int) (mpage->address >> 32) , (int)((mpage->address << 32) >> 32 ));
    #endif
      return 0;
    }
  else
    {
      //do the read fault
	int timeout = 2;
	#ifdef ETA_DEBUG
            printf("ETA_ready_read_copy %x %x THEIRS, LOCKING, owner is ", (int) (mpage->address >> 32) ,(int) ((mpage->address << 32) >> 32 ));
	#endif
      ETA_lock_page_entry(mpage);
#ifdef ETA_DEBUG
      print_host(mpage->prob_owner_host, mpage->prob_owner_port);
      printf("\n");
#endif
      sequence = ETA_request_read_access(mpage->address, mpage->prob_owner_host, mpage->prob_owner_port, our_id_host, our_id_port, -1);

      assert(sequence > 0);
      while( 1 )
	{
	    
	  int temp_len = sizeof(struct page_entry) + BLOCK_SIZE;
	  if(timeout == 0)
	      {
		  printf("Ready read copy re-requesting page\n");
		  timeout = 2;
ETA_request_read_access(mpage->address, mpage->prob_owner_host, mpage->prob_owner_port, our_id_host, our_id_port, sequence);
 sleep(1); 
	      }
	  else
	      {
	  char* temp = (char*) malloc( temp_len );
	  assert(temp != NULL);
	  timeout--;
	  success = ETA_get_read_reply(sequence, temp, &temp_len);
	  if(success == 0)
	    {

             printf("got page %i\n", sequence);

             memcpy(mpage->page_data, temp + sizeof(struct page_entry), BLOCK_SIZE);
	     mpage->access = 1; //read
	     mpage->prob_owner_host = ((struct page_entry*) temp)->prob_owner_host;
	     mpage->prob_owner_port = ((struct page_entry*) temp)->prob_owner_port;
	     break;
	    }
	  else
	    printf("Waiting for page %i\n", sequence);
	      }
	  usleep(5000);
	}
    #ifdef ETA_DEBUG
      printf("ETA_ready_read_copy %x %x fin\n", (int) (mpage->address >> 32) , (int) ((mpage->address << 32) >> 32 ));
#endif
      ETA_unlock_page_entry(mpage);
    #ifdef ETA_DEBUG
       printf("-Read ready copy - UNLOCKING %x\n", (int) mpage);
#endif
      return 0;
    }
}

int invalidate_socket = -1;

int ETA_invalidate(struct page_entry* mpage, int request_node_host, int request_node_port)
{
  int i;
  for( i = 0; i < 32;i++)
    {
      int entry = 1L << i;
      if((mpage->copy_set & entry) == entry)
	{
	  //send invalidate to each ip
    struct sockaddr_in to;
    int tolen = sizeof(to);
    int remote_socket;
    char address[256];
    int* sendbuf = (int*) malloc( 4 * 4);



    sendbuf[0] =  ETA_INVALIDATE ;
    sendbuf[1] = 0;
    sendbuf[2] = request_node_host;
    sendbuf[3] = request_node_port;

   /*Initialize address info*/

   
    sprintf(address, "1.0.0.%i", i);
    if(inet_aton(address, &to.sin_addr) == 0)
      {
	perror("Bad address");
	exit(11);
      }
    to.sin_family = AF_INET;
    //    to.sin_addr.s_addr = host;
    to.sin_port =  htons(our_id_port);
 printf("/\\/\\/\\/\\Invalidating page %s:%i %x:%x\n",  inet_ntoa(to.sin_addr), request_node_port, (int) (mpage->address >> 32), (int) ((mpage->address << 32) >> 32));
if(to.sin_addr.s_addr != our_id_host)
  {
		 printf("/\\/\\/\\/\\ 3 Invalidating page\n");
    printf("ETA_invalidate Creating socket for %s\n", inet_ntoa(to.sin_addr));

    /*Create the socket*/

    if(invalidate_socket == -1)
      remote_socket = invalidate_socket = socket(PF_INET, SOCK_DGRAM, IPPROTO_UDP);
    else
    remote_socket = invalidate_socket;
    if(remote_socket == -1)
      {
	perror("ETA_send_identification: connecting to hosts");
	exit(239);
      }

   if(sendto(remote_socket, sendbuf, 4 * 4, 0, &to, tolen) == -1)
      {
	perror("send_id");
	exit(240);
      }
  }
   free(sendbuf);
  
	}
    }
  return 0;
}

void print_host(int h, int p)
{
  struct in_addr in;
  in.s_addr = h;
  printf("%s:%i", inet_ntoa(in), p);
}

int ETA_ready_write_copy(struct page_entry* mpage)
{
  int sequence;
  int success;
  //Check whether this is a write fault
  if(mpage->access == 2) //Access is write
    {
    #ifdef ETA_DEBUG
            printf("ETA_ready_write_copy %x %x OURS, owner is ", (int) (mpage->address >> 32) , ((mpage->address << 32) >> 32 ));
      print_host(mpage->prob_owner_host, mpage->prob_owner_port);
      printf("\n");
#endif
      return 0;
    }
  else
    {
	int timeout = 7;
    #ifdef ETA_DEBUG
            printf("ETA_ready_write_copy %x %x THEIRS, LOCKING, owner is ", (int) (mpage->address >> 32) , ((mpage->address << 32) >> 32 ));
#endif
      ETA_lock_page_entry(mpage);
    #ifdef ETA_DEBUG
      print_host(mpage->prob_owner_host, mpage->prob_owner_port);
      printf("\n");
#endif
      sequence = ETA_request_write_access(mpage->address, mpage->prob_owner_host, mpage->prob_owner_port, our_id_host, our_id_port, -1);


      while( 1 )
	{
	  int temp_len = sizeof(struct page_entry) + BLOCK_SIZE;
	  char* temp = (char*) malloc( temp_len );
	  assert(temp != NULL);
	 if(timeout == 0)
	     {
	       exit(241);
      sequence = ETA_request_write_access(mpage->address, mpage->prob_owner_host, mpage->prob_owner_port, our_id_host, our_id_port, sequence);
sleep(3);
		 timeout = 20;
	     }
	 else
	     {    
	    timeout--;
	  success = ETA_get_read_reply(sequence, temp, &temp_len);
	  if(success == 0)
	    {
	      printf("success 17\n");
             memcpy(mpage->page_data, temp + sizeof(struct page_entry), BLOCK_SIZE);

	     mpage->prob_owner_host = our_id_host;
	     mpage->prob_owner_port = our_id_port;
	     mpage->copy_set = ((struct page_entry*) temp)->copy_set;
	    break;
	    }
	  else if(success == ETA_NACK)
	    {
	      printf("Hey jesus, stop that!\n");
	    exit(5);
	    }
	     }
	  usleep(5000);
	}
      success = ETA_invalidate(mpage, our_id_host, our_id_port);
      assert(success == 0);
      mpage->access = 2; //write
      mpage->copy_set = 0;
    #ifdef ETA_DEBUG
      printf("ETA_ready_write_copy %x %x fin\n", (int) (mpage->address >> 32) , ((mpage->address << 32) >> 32 ));
#endif
      ETA_unlock_page_entry(mpage);
    #ifdef ETA_DEBUG
       printf("-Read write copy - UNLOCKING %x\n", (int) mpage);
#endif
      return 0;
    }
}

int request_socket = -1;

/*
readwrite is 1 if this is going to be locked for writing, 0 for reading
*/
int request_access(long long address, int prob_owner_host, int prob_owner_port, int requesting_host, int requesting_port, int readwrite, int oldsequence)
{
  //readwrite is 0 for read, 1 for write
   struct hostent *remoteHost;
    struct sockaddr_in to;
    struct sockaddr_in to_temp;
    int tolen = sizeof(to);
    int remote_socket;
    int sequence;
    int* sendbuf = (int*) malloc( ETA_REQUEST_SIZE);

    if(oldsequence == -1)
	sequence = eta_sequence++;
    else
	sequence = oldsequence;

    assert(sequence > 0);
if(readwrite == 0)
    sendbuf[0] = ETA_READ_REQUEST;
else
    sendbuf[0] = ETA_WRITE_REQUEST;
    sendbuf[1] = sequence;
    sendbuf[2] = requesting_host ;
    sendbuf[3] = requesting_port ;
    *((long long*) (sendbuf + 4)) = address;

   /*Initialize address info*/

    to.sin_family = AF_INET;
    to.sin_addr.s_addr = prob_owner_host;
    to.sin_port =  htons( prob_owner_port);

    #ifdef ETA_DEBUG
    to_temp.sin_family = AF_INET;
    to_temp.sin_addr.s_addr = our_id_host;
    printf("request_access seq %i, %x %x\n", sequence, (int) (address >> 32) , (int) ((address << 32) >> 32 ), prob_owner_port);
    printf("Dest %s:%i\n", inet_ntoa(to.sin_addr), prob_owner_port);
    printf("Us%s:%i\n", inet_ntoa(to_temp.sin_addr), our_id_port);
#endif
    assert(prob_owner_port < 50000);
    /*    assert(strcmp(inet_ntoa(to.sin_addr),"null") != 0);
	  assert(inet_ntoa(to.sin_addr) != NULL);*/
    /*Create the socket*/

    if(request_socket == -1)
      remote_socket = request_socket = socket(PF_INET, SOCK_DGRAM, IPPROTO_UDP);
    else 
      remote_socket = request_socket;
    if(remote_socket == -1)
      {
	perror("request_access: connecting to eta");
	exit(242);
      }

   if(sendto(remote_socket, sendbuf, ETA_REQUEST_SIZE, 0, &to, tolen) == -1)
      {
	perror("request_access, sending");
	exit(243);
      }
   //   close(remote_socket);
   free(sendbuf);
    #ifdef ETA_DEBUG
   printf("eta request_access finished %i\n", sequence);
#endif
   return sequence;
}

int ETA_request_read_access(long long address, int prob_owner_host, int prob_owner_port, int requesting_host, int requesting_port, int sequence)
{
 return request_access(address, prob_owner_host, prob_owner_port, requesting_host, requesting_port, 0, sequence);
}

int ETA_request_write_access(long long address, int prob_owner_host, int prob_owner_port, int requesting_host, int requesting_port, int sequence)
{
 return request_access(address, prob_owner_host, prob_owner_port, requesting_host, requesting_port, 1, sequence);
}

struct page_entry* ETA_lookup_page_for_address(long long address)
{
  int key[2];
  HASH_tstHashEntry* t;
  key[0] = (int) (address >> 32);
  key[1] = (int) ((address << 32) >> 32);
  t = HASH_FindHashEntry(&alt_hash_table, (char*) &key);
  if(t != NULL)
    return HASH_GetHashValue(t);
}

/* Get page for address is the most commonly-called routine in ETA

   The caller supplies an address, and a boolean indicating whether she wants to read or write to that page

   The first thing is to look in the Hashtable for a page entry. If it isn't there, we return NULL and let the caller create the page.

   Then depending on whether it's a read or a write, we call ready_read_copy or ready_write_copy
 */
struct page_entry* ETA_get_page_for_address(long long address, int readwrite)
{
  int success;
  struct page_entry* mpage;

  //  pthread_mutex_lock(&ETA_read_mutex);
  mpage = ETA_lookup_page_for_address(address);

  assert((readwrite == ETA_GET_PAGE_FOR_READ) || (readwrite == ETA_GET_PAGE_FOR_WRITE));
  #ifdef ETA_DEBUG
  printf("ETA_get_page_for_address %x %x,\n",  (int) (address >> 32), (int)((address << 32) >> 32));
  #endif
    if(mpage == NULL)
      {
	//     pthread_mutex_unlock(&ETA_read_mutex);
      return NULL;
      }
    else
      {
      if(readwrite == ETA_GET_PAGE_FOR_READ)
	{
	 success =  ETA_ready_read_copy(mpage);
	 assert(success == 0);
	}
      else //WRITE
	{
	  success =  ETA_ready_write_copy(mpage);
  	 assert(success == 0);
	}
      //     pthread_mutex_unlock(&ETA_read_mutex);
      return mpage;
      }
}

int ETA_create_page_for_address(long long address)
{
    struct page_entry* newentry = (struct page_entry*) malloc(sizeof(struct page_entry));
    struct HASH_tstHashEntry* hash_e;
    int success;
    int i;

    assert(our_id_host > 0);
    assert(default_id_host > 0);
    assert(newentry != NULL);

    newentry->address = address;
    newentry->lock = 0;
    newentry->counter = 0;
    if((our_id_host == default_id_host) && (our_id_port == default_id_port))
      newentry->access = 2;//write      
    else
      newentry->access = 0;//nil
    newentry->copy_set = 0;
    newentry->prob_owner_host = default_id_host; 
    newentry->prob_owner_port = default_id_port; 
    newentry->next = NULL;
    newentry->flushed = 0;
 
    newentry->page_data = (void*) malloc(BLOCK_SIZE);
    assert(newentry->page_data != NULL);

    for( i = 0; i < BLOCK_SIZE;i++)
      {
	((char*)newentry->page_data)[i] = 0x99;
      }

    {
    int* key = (int*) malloc(8);
    assert(key != NULL);
    key[0] = (int) (address >> 32);
    key[1] = (int) ((address << 32) >> 32);
    hash_e = HASH_CreateHashEntry(&alt_hash_table, (char*) key, (int32*) &success);
    assert(hash_e != NULL);
    HASH_SetHashValue(hash_e, newentry);
    }
    return 0;
}

int ETA_lock_page_entry(struct page_entry* entry)
{
  assert(entry != NULL);
  //XXX
  assert(entry->prob_owner_host > 0);
  while( entry->lock == 1)
    {
      //Spin while entry is locked
	usleep(1000);
	printf("urrm where is the entry? \n");

    }
  entry->lock = 1;
  return 0;
}

int ETA_unlock_page_entry(struct page_entry* entry)
{
  assert(entry != NULL);
  assert(entry->prob_owner_host > 0);
 if(entry->lock == 0)
   return 1;
  entry->lock = 0;
  return 0;
}



/* Thread to listen for read/write requests for pages*/


void ETA_listen(void* arg)
{
  int s;
  struct sockaddr_in sin;
  struct sockaddr_in from;
  int fromlen;


  printf("** Listen for ETA started, listening on port %i \n", our_id_port);
  sin.sin_family = AF_INET;
  sin.sin_addr.s_addr = INADDR_ANY;
  sin.sin_port = htons(our_id_port);

  ETA_listen_socket = s = socket(PF_INET, SOCK_DGRAM, IPPROTO_UDP);
  if(s == -1)
  {
    perror("ETA_listen create socket");
  }
 if( bind(s, &sin, sizeof(sin)) == -1)
   {
     perror("ETA_listen bind");
     exit(44);
   } 

 /*Loop continuously listening for requests*/
 while(1)
   {
  char* buf = (char*) malloc( 8192 );
  
     int* int_p;
     long long address;
 int sequence;
     int actual  = 0;
     assert(buf != NULL);
     memset(buf, 0, 8192);
    fromlen = sizeof(from);
     actual = recvfrom(s, buf, 8192, 0, &from, &fromlen); 
     if((actual == -1) || (actual < 4*4))
       {
	 perror("ETA_listen");
	 exit(77);
       }
     int_p = (int*) buf;
     
     //lock the mutex
     //     pthread_mutex_lock(&ETA_read_mutex);
     if(int_p[0] == ETA_READ_REQUEST)
       {
	 int success;
	 sequence = int_p[1];
 	 address = *((long long*)&int_p[4]);
#ifdef ETA_DEBUG
	 printf("-//Read server servicing request \t%x %x, %i act %i\n", (int) (address >> 32), (int) ((address << 32) >> 32), sequence, actual);
#endif
	 if(ETA_lookup_page_for_address(address) != NULL)
	   {
#ifdef ETA_DEBUG
	     printf("r server got page\n");
#endif
	     success = service_request(address, int_p[2], int_p[3], ETA_GET_PAGE_FOR_READ, sequence); //read
	     assert(success == 0);
	   }
	 else
	     {
	    struct page_entry* mpage = NULL;
#ifdef ETA_DEBUG
	     printf("r server Creating page for this entry\n");
#endif
             success = ETA_create_page_for_address(address);
	     assert(success == 0);
	     mpage = ETA_get_page_for_address(address, ETA_GET_PAGE_FOR_READ); 
	     success = service_request(address, int_p[2], int_p[3], ETA_GET_PAGE_FOR_READ, sequence);//read
	     assert(success == 0);
	     }
#ifdef ETA_DEBUG
	 printf("-\\\\Read server servicing request \t%x %x, %i \n", (int) (address >> 32), (int) ((address << 32) >> 32), sequence);
#endif

       }
     else if(int_p[0] == ETA_WRITE_REQUEST)
       {

	 int success;
	 sequence = int_p[1];
 	 address = *((long long*)&int_p[4]);
#ifdef ETA_DEBUG
	 printf(" ETA_WRITE_REQUEST\n");
#endif
	 if(ETA_lookup_page_for_address(address) != NULL)
	   {
#ifdef ETA_DEBUG
	     	 printf(" ETA_WRITE_REQUEST servicing request\n");
#endif
	     success = service_request(address, int_p[2], int_p[3], ETA_GET_PAGE_FOR_WRITE, sequence); //write
	     assert(success == 0);
	   }
	 else
	   {
	    struct page_entry* mpage = NULL;
#ifdef ETA_DEBUG
	     printf(" ETA_WRITE_REQUEST Creating page for this entry before servicing request\n");
#endif
             success = ETA_create_page_for_address(  address);
	     assert(success == 0);
	     mpage = ETA_get_page_for_address(address, ETA_GET_PAGE_FOR_READ); //We first read the page from wherever, it may or may not be in distributed memory. This way we can be sure that we identify the owner for the next call
	     success = service_request(address, int_p[2], int_p[3], ETA_GET_PAGE_FOR_WRITE, sequence); //write
	     assert(success == 0);
	   }
#ifdef ETA_DEBUG
	 printf("-\\\\Write server fin servicing request \t%x %x, %i \n", (int) (address >> 32), (int) ((address << 32) >> 32), sequence);
#endif

       }
     else if(int_p[0] == ETA_IDENTIFY)
       {
	 if(int_p[1] == gethostid())
	   {
	     printf("Got IDENTIFY\n");
	     our_id_host = int_p[2];
	   }
       }
    else if(int_p[0] == ETA_PAGE_FOR_READ)
      {
	int success;
	sequence = int_p[1];
	#ifdef ETA_DEBUG
	printf("Got read reply %i\n", sequence);
	#endif
	success = ETA_add_read_reply(sequence, buf, sizeof(struct page_entry) + BLOCK_SIZE);

      }
     else if(int_p[0] == ETA_PAGE_FOR_WRITE)
      {
	int success;
	sequence = int_p[1];
#ifdef ETA_DEBUG
	printf("Got write reply %i\n", sequence);
#endif
	success = ETA_add_read_reply(sequence, buf, sizeof(struct page_entry) + BLOCK_SIZE);
      }
     else if(int_p[0] == ETA_INVALIDATE)
      {
	  struct page_entry* mpage = NULL;
	 unsigned int CurrentPos;

	  printf("/\\/\\/\\/\\ 1 Invalidating page %i %i %x:%x\n", int_p[2], int_p[3], (int) (mpage->address >> 32), (int) ((mpage->address << 32) >> 32));
 	 address = *((long long*)&int_p[4]);
	 mpage = ETA_lookup_page_for_address(address);
	 if(mpage != NULL)	  ///XXX != ?
	   {
	     if(mpage->access != 0)
	       {
		 assert(2 == 3);
		  printf("/\\/\\/\\/\\ 2 Invalidating page %i:%i %x:%x\n", int_p[2], int_p[3], (int) (mpage->address >> 32), (int) ((mpage->address << 32) >> 32));
		 ETA_invalidate(mpage, int_p[2], int_p[3]);
		 mpage->access = 0; //nil
		 mpage->prob_owner_host = int_p[2];
		 mpage->prob_owner_port = int_p[3];
		 mpage->copy_set = 0;
	       }
	   }

      } 
     else 
       {
	 printf("Received bad packet\n");
	 exit(885);
       }
     //Free the mutex
     //     pthread_mutex_unlock(&ETA_read_mutex);
     free(buf);
   }

}



int ETA_add_read_reply(int sequence_number , char* buf, int buf_len)
{
  struct listentry* read_list;
#ifdef ETA_DEBUG
  printf("add_read locking mutex\n");
#endif
  pthread_mutex_lock(&ETA_read_mutex);
  read_list = ETA_outstanding_list;
  if(read_list == NULL)
    {
      ETA_outstanding_list = (struct listentry*) malloc(sizeof(struct listentry));
      read_list = ETA_outstanding_list;

      assert(read_list != NULL);

      read_list->next = NULL;
      read_list->sequence_number = sequence_number;
      read_list->data = (void*) malloc(buf_len);
      assert(read_list->data != NULL);

      read_list->data_length = buf_len;
      memcpy(read_list->data, buf, buf_len);
    }
  else
   {
     struct listentry* prev = read_list;
     struct listentry* iter = read_list;
     while(iter != NULL)
       {
	 prev = iter;
	 iter = iter->next;
       }
     prev->next = (struct listentry*) malloc(sizeof(struct listentry));
     assert(prev->next != NULL);
     prev->next->next = NULL;
     prev->next->sequence_number = sequence_number;
     prev->next->data = (void*) malloc(buf_len);
     assert(prev->next->data != NULL);
     memcpy(prev->next->data, buf, buf_len);
     prev->next->data_length = buf_len;
   }
  ETA_entries++;
#ifdef ETA_DEBUG
  printf("add_read unlocking mutex\n");
#endif
  pthread_mutex_unlock(&ETA_read_mutex);
  return 0;
}






int ETA_get_read_reply(int sequence_number, char* buf, int* buf_len)
{
    /*This method looks for a spec. seq. If it finds it, it removes it from the list */

  struct listentry* prev; 
  struct listentry* iter;
#ifdef ETA_DEBUG
  printf("get_read locking mutex\n");
#endif
  pthread_mutex_lock(&ETA_read_mutex);
  prev = iter = ETA_outstanding_list;
  assert(sequence_number > 0);
  if(prev == NULL)
    {
#ifdef ETA_DEBUG
  printf("get_read unlocking mutex\n");
#endif
  pthread_mutex_unlock(&ETA_read_mutex);
    return -1;
    }
   while((iter != NULL) && (iter->sequence_number != sequence_number))
       {
	 prev = iter;
	 iter = iter->next;
       }

   if(iter == NULL)
     {
#ifdef ETA_DEBUG
  printf("get_read unlocking mutex\n");
#endif
  pthread_mutex_unlock(&ETA_read_mutex);
     return -1;
     }
   else
     {
       if(iter == ETA_outstanding_list)
	 {
	   int retval;
	   ETA_outstanding_list = iter->next;
	   retval =  ETA_processIter(iter, buf, buf_len);
	   ETA_entries--;
#ifdef ETA_DEBUG
  printf("get_read unlocking mutex\n");
#endif
  pthread_mutex_unlock(&ETA_read_mutex);
	   return retval;
	 }
       else
	 {
	   struct listentry* old = prev->next;       	
	   prev->next = old->next;
	   ETA_entries--;
#ifdef ETA_DEBUG
  printf("get_read unlocking mutex\n");
#endif
  pthread_mutex_unlock(&ETA_read_mutex);
	   return ETA_processIter(old, buf, buf_len);
	 }
#ifdef ETA_DEBUG
  printf("get_read unlocking mutex\n");
#endif
  pthread_mutex_unlock(&ETA_read_mutex);
       return 0;
     }
}

int ETA_processIter(struct listentry *iter, void* buf, int *buf_len)
{
   int* int_p;

	   //Examine the reply command
	    int_p = (int*) iter->data;
	   if(int_p[0] == ETA_PAGE_FOR_READ)
	     {
	       memcpy(buf, iter->data + 8, *buf_len);
	       free(iter);
	       return 0;
	     }
	   else if(int_p[0] == ETA_PAGE_FOR_WRITE)
	     {
	       memcpy(buf, iter->data + 8, *buf_len);
	       free(iter);
	       return 0;
	     }
	   else
	       {
	       printf("ETA Unexpected return code %x  from eta %i\n", int_p[0], iter->sequence_number);
	       exit(6);
	     }
}








#endif
