/*
 * Copyright (C) 2001, John Leuner.
 *
 * This file is part of the kissme project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2,
 * or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>

#ifdef KISSME_LINUX_USER
#include <stdio.h>
#include <assert.h>
#endif

#include "classfil.h"
#include "jni.h"
#include "thread.h"

#include "interp.h"
#include "methodstacks.h"

extern int INTERP_InitialStackSize;

/* NMS: The functions below handle method invocation and the creation and deletion of stack frames */

tStackFrame* InitialStackFrame
(
  tMethod* pstNewMethod,  /* @parm Pointer to the method to use */
  tOBREF pstNewObject,  /* @parm Pointer to the current object */
  int32* pi32Args,         /* @parm Pointer to method arguments */
  tAllocatorHeap* pstHeap
)
{
    return InitialStackFrameHelper(pstNewMethod, pstNewObject, pi32Args, pstHeap, 0); //Not an orphan frame
}

/*
 * @doc FUNC
 * @func
 * This function is used to create the first environment stack frame of a
 * thread.  It is passed the method to use, the "current object" and any
 * parameters that the method takes.  If this function is ever called from
 * inside Interpret(), LOADENV() should be called afterwards to load the
 * virtual machine registers from the new environment frame.  Frames created
 * by InitialStackFrame() should be disposed of only by a call to
 * PopStackFrame().  Otherwise, untold mayhem will likely ensue.
 *
 * @rdesc Returns a pointer to the new environment frame.
 *
 * @ex Example of use of InitialStackFrame(). In the example, the method
 *     being used is a static method, so the "current object" pointer is
 *     NULL.  pi32Args is assumed to point to the method's arguments. |
 *
 * tClass*      pstClass;
 * tMethod*     pstMethod;
 * tStackFrame* pstFrame;
 *
 * pstClass = CPLIST_Find("MyClass");
 * pstMethod = CLASSFILE_FindMethod(pstClass, "main", "([Ljava/lang/String;)V");
 * pstFrame = InitialStackFrame(pstMethod, NULL, pi32Args);
 *
 *
 *
 * Note that is has the bSuperFrame flag as true. This means it gets "free'ed" when it is popped from a frame stack
 *
 *
 * Done with NMS
 */

tStackFrame* InitialStackFrameHelper
(
  tMethod* pstNewMethod,  /* @parm Pointer to the method to use */
  tOBREF pstNewObject,  /* @parm Pointer to the current object */
  int32* pi32Args,         /* @parm Pointer to method arguments */
  tAllocatorHeap* pstHeap,
  byte isOrphan
)
{
  tStackFrame* pstFrame;
  uint16       u16VarSize;
  uint16       u16OpSize;
  int          i;
  tClassLoaderTuple*      pstNewClass = pstNewMethod->pstClass;

#ifdef DEBUG_OBJECT_FOR_GC
  assert(pstNewClass->magic1 == MAGIC1);
  assert(pstNewClass->magic2 == MAGIC2);
  assert(pstNewClass->magic3 == MAGIC3);
#endif

  //  fprintf(stderr, "-*- Creating new stack frame, method %s, class %s -*-\n", pstNewMethod->uidName, pstNewMethod->pstClass->uidName);
  if (pstNewMethod->pstCode)
  {
    u16VarSize = pstNewMethod->pstCode->u16MaxLocals + pstNewMethod->u16ArgSize + 1 ; //+1 for rounding
    u16OpSize = pstNewMethod->pstCode->u16MaxStack + 1;
  }
  else
  {
    if(!((pstNewMethod->u16AccessFlags & ACC_NATIVE) == ACC_NATIVE))
      eprintf( "%s.%s has no code, is native %i\n", pstNewMethod->pstClass->uidName, pstNewMethod->uidName, pstNewMethod->u16AccessFlags & ACC_NATIVE);
    u16VarSize = pstNewMethod->u16ArgSize + 1; 
    u16OpSize = 0;
  }


  //Now make the frame data
   
  //  fprintf(stderr, "Making new initial stack frame with size: %i (%i)\n", INTERP_InitialStackSize, u16FrameSize); 
  pstFrame = (tStackFrame*) sys_malloc(INTERP_InitialStackSize);
	if (pstFrame == NULL)
	{
		eprintf( "Error allocating frame space in InitialStackFrame, size was %i\n", INTERP_InitialStackSize);
		exit(1);
	}

  pstFrame->pstPrevFrame = NULL;
  pstFrame->pstChildFrame = NULL;
  pstFrame->pstCurrClass = NULL;
  pstFrame->bFirstFrame = 1;
  pstFrame->bSuperFrame = 1;

  if((pstNewMethod->u16AccessFlags & ACC_NATIVE) == ACC_NATIVE)
    pstFrame->bIsNativeFrame = 1;
  else
    pstFrame->bIsNativeFrame = 0;

  pstFrame->bIsOrphan = isOrphan;
  //Now make the variable data 

  pstFrame->pi32Vars = (int32*) sys_malloc( 2048 ); //Let's start with 2K
  assert( pstFrame->pi32Vars );

  //Now set the operand stack to point past the local vars
  pstFrame->pi32OpBot = pstFrame->pi32Vars + u16VarSize - 1; //-1 because it points behind the stack!
  pstFrame->pi32OpTop = pstFrame->pi32OpBot;

  if (u16VarSize == 0)
  {
    pstFrame->pi32Vars = NULL;
  }
  else
  {
      memset(pstFrame->pi32Vars, 0, u16VarSize * sizeof(int32));//Watch out, is this bytes?
  }

  pstFrame->iCumVarData = u16OpSize + u16VarSize;
  /* set the new environment */
  pstFrame->pstHeap = pstHeap;
  pstFrame->pstPrevFrame = NULL;
  pstFrame->pstCurrClass = pstNewClass;
  pstFrame->pstCurrMethod = pstNewMethod;
  pstFrame->pstCurrObject = pstNewObject;

  if (pstNewMethod->pstCode)
  {
    pstFrame->pbPC = pstFrame->pbCode = pstNewMethod->pstCode->pbCode;
  }
  else
  {
    pstFrame->pbPC = pstFrame->pbCode = NULL;
  }
  for (i = 0; i < pstNewMethod->u16ArgSize; i++)
  {
    pstFrame->pi32Vars[i] = pi32Args[i];
  }
  
  pstFrame->pstException = NULL;
  pstFrame->iNumStackFrames = 1;
  return pstFrame;
}

/*
 * @doc FUNC
 * @func
 * This function is used to create a new environment stack frame and push
 * it onto the environment stack.  It is passed the new method to use, the
 * "current object" and the current enironment frame. The method's parameters
 * are taken from the operand stack in the current environment frame.  Before
 * this function is called, the virtual machine registers have to be saved
 * in the current stack frame using SAVEENV().  After it is called, they have
 * to be loaded from the new stack frame using LOADENV().  Frames created
 * by PushStackFrame() should be disposed of only by a call to
 * PopStackFrame().  Otherwise, untold mayhem will likely ensue.
 *
 * @rdesc Returns a pointer to the new environment frame.
 *
 * @ex Example of use of PushStackFrame(). optop is a pointer to the top of
 *     the operand stack. |
 *
 * tClass*      pstClass;
 * tMethod*     pstMethod;
 *
 * pstClass = CPLIST_Find("MyClass");
 * pstMethod = CLASSFILE_FindMethod(pstClass, "MyMethod", "(I)I");
 * SAVEENV();
 * pstCurrFrame = PushStackFrame(pstMethod, (tOBREF) *optop, pstCurrFrame);
 * LOADENV();
 *
 */

tStackFrame* PushStackFrame
(
  tMethod* pstNewMethod,    /* @parm Pointer to new method */
  tOBREF pstNewObject,      /* @parm Pointer to new "current object" */
  tStackFrame* pstOldFrame  /* @parm Pointer to current stack frame */
)
{
  tStackFrame* pstFrame;
  uint16       u16VarSize;
  uint16       u16OpSize;
  int          i;
  tClassLoaderTuple*      pstNewClass = pstNewMethod->pstClass;
  uint16       u16AdditionalVarSlots;

  /* find number of local variable slots */

  assert((((pstNewClass->uidName[0] >= 'a') && (pstNewClass->uidName[0] <= 'z')) || ((pstNewClass->uidName[0] >= 'A') && (pstNewClass->uidName[0] <= 'Z'))  || (pstNewClass->uidName[0] == '[')));

  // x  fprintf(stderr, "pushing stack frame with %s.%s\n", pstNewMethod->pstClass->uidName, pstNewMethod->uidName);
  /*
	fprintf(stderr, "Creating frame for themethod\n");
	fprintf(stderr, "Max stack (Op) size is %i\n", pstNewMethod->pstCode->u16MaxStack);
	fprintf(stderr, "Max locals size is %i\n", pstNewMethod->pstCode->u16MaxLocals);
	fprintf(stderr, "Arg size is %i\n", pstNewMethod->u16ArgSize);
  */

  u16VarSize = pstNewMethod->pstCode->u16MaxLocals; // + pstNewMethod->u16ArgSize; //no plus 1's here

  if(pstNewMethod->u16ArgSize > u16VarSize)
      u16VarSize = pstNewMethod->u16ArgSize;

  //fprintf(stderr, "stack for %s.%s has %i local vars\n", pstNewMethod->pstClass->uidName, pstNewMethod->uidName, u16VarSize);
  assert(u16VarSize >= pstNewMethod->u16ArgSize);

  /* find number of operand stack slots */
  u16OpSize = pstNewMethod->pstCode->u16MaxStack + 1; //Jewel

if( u16VarSize > (pstOldFrame->pstCurrMethod->pstCode->u16MaxStack + 1))
    {
	u16AdditionalVarSlots = u16VarSize - (pstOldFrame->pstCurrMethod->pstCode->u16MaxStack + 1);
	//  fprintf(stderr, "Old: %i\n", u16AdditionalVarSlots);

	u16AdditionalVarSlots = u16VarSize - ((pstOldFrame->pstCurrMethod->pstCode->u16MaxStack + 1) - (pstOldFrame->pi32OpTop - pstOldFrame->pi32OpBot)); 

	//  fprintf(stderr, "new frame has opsize %i, var size %i, args %i, additional %i\n", u16OpSize, u16VarSize, pstNewMethod->u16ArgSize, u16AdditionalVarSlots);

    }
else
    u16AdditionalVarSlots = 0;


//fprintf(stderr, "new frame is using %i additional slots\n", u16AdditionalVarSlots);
 
  //Now allocate the frame data
  //First check if we have space
  if(((pstOldFrame->iNumStackFrames + 1) * sizeof(tStackFrame)) > INTERP_InitialStackSize)
   {

       pstOldFrame->pi32OpTop -= ( pstNewMethod->u16ArgSize ); //Pull it back to *before* the args

       pstFrame = InitialStackFrame(pstNewMethod, pstNewObject, pstOldFrame->pi32OpTop + 1, pstOldFrame->pstHeap);
       pstFrame->pstHeap = pstOldFrame->pstHeap;
       pstOldFrame->pstChildFrame = pstFrame;
       //       fprintf(stderr, "Creating new superframe %x, prev %i\n", pstFrame, pstOldFrame->iNumStackFrames);
       pstFrame->pstPrevFrame = pstOldFrame;
       pstFrame->bFirstFrame = 0;
       pstFrame->bIsOrphan = 0;

       return pstFrame;
   }
  else
      {
	  pstFrame = (tStackFrame*) ((byte*) pstOldFrame + sizeof(tStackFrame));
	  pstOldFrame->pstChildFrame = pstFrame;
	  pstFrame->iNumStackFrames = pstOldFrame->iNumStackFrames + 1;
	  pstFrame->bSuperFrame = 0;
          pstFrame->bFirstFrame = 0;
	  pstFrame->bIsNativeFrame = 0;
	  pstFrame->bIsOrphan = 0;
	  pstFrame->pstHeap = pstOldFrame->pstHeap;
          pstFrame->pstCurrClass = NULL;
      }

  /* move old frame's operand stack pointer to before the arguments */
  pstOldFrame->pi32OpTop -= ( pstNewMethod->u16ArgSize ); //Pull it back to *before* the args

  if(pstOldFrame->bIsNativeFrame)
      {
	  assert(pstOldFrame->pi32OpTop >= pstOldFrame->pi32OpBot);
	  assert(pstOldFrame->pi32OpTop >= pstOldFrame->pi32Vars);
      }

  pstFrame->pi32Vars = pstOldFrame->pi32OpTop + 1; //At first arg

  //Now make the variable data
  if(pstOldFrame->iCumVarData > 2048)
      {
	  assert(2 == 3); //Need to grow
      }
  else
      {
	  pstFrame->pi32OpBot = pstOldFrame->pi32OpTop + u16VarSize; 
	  pstFrame->pi32OpTop = pstFrame->pi32OpBot;
	  pstFrame->iCumVarData = pstOldFrame->iCumVarData + u16OpSize + u16AdditionalVarSlots;
      }

  if (u16VarSize == 0)
  {
    pstFrame->pi32Vars = NULL;
  }
  else
  {
    /* initialise all local variables to zero */
      //    memset(pstFrame->pi32Vars, 0, u16VarSize * sizeof(int32));
  }

  /* set the new environment */
  /* set previous frame pointer to point to old frame */
 pstFrame->pstPrevFrame = pstOldFrame;
  /* point current class pointer to class of the new method */
 pstFrame->pstCurrClass = pstNewClass;
  /* point current method pointer to new method */
 pstFrame->pstCurrMethod = pstNewMethod;
  /* point current object pointer to new object */
 pstFrame->pstCurrObject = pstNewObject;
  /* set program counter to beginning of this method's code */
 pstFrame->pbPC = pstFrame->pbCode = pstNewMethod->pstCode->pbCode;


 //We don't need to do this anymore, they are already there
 /* copy the method's parameters from the old frame's operand stack
    into the first few local variables of the new frame */

 //Actually we must zero those excepting the args
	for (i = 0; i < u16VarSize - pstNewMethod->u16ArgSize; i++)
	{
		pstFrame->pi32Vars[i + pstNewMethod->u16ArgSize] = 0;
	}
 

  /* set exception to NULL */
  pstFrame->pstException = NULL;

  pstFrame->pstChildFrame = NULL;

  /* return pointer to the new frame */

  return pstFrame;
}

/* This method pushes a 'native' stack frame on the Java stack. This represents the invocation of an optimised method. We need the frame entry to be able to determine what message was being invoked we have to handle an exception.

 */

tStackFrame* PushNativeStackFrame
(
  tMethod* pstNewMethod,    /* @parm Pointer to new method */
  tOBREF pstNewObject,      /* @parm Pointer to new "current object" */
  tStackFrame* pstOldFrame  /* @parm Pointer to current stack frame */
)
{
  tStackFrame* pstFrame;
  tClassLoaderTuple*      pstNewClass = pstNewMethod->pstClass;

  /*
	fprintf(stderr, "Creating native stack frame for %s\n", pstNewMethod->uidName);
	fprintf(stderr, "Max stack (Op) size is %i\n", pstNewMethod->pstCode->u16MaxStack);
	fprintf(stderr, "Max locals size is %i\n", pstNewMethod->pstCode->u16MaxLocals);
	fprintf(stderr, "Arg size is %i\n", pstNewMethod->u16ArgSize);
  */

  //Now allocate the frame data
  //First check if we have space
  if(((pstOldFrame->iNumStackFrames + 1) * sizeof(tStackFrame)) > INTERP_InitialStackSize)
   {
       //We don't touch the old frame here
       //       pstOldFrame->pi32OpTop -= ( pstNewMethod->u16ArgSize ); //Pull it back to *before* the args
       assert(1 == 2); //Need to create an initial native frame
       //       pstFrame = InitialStackFrame(pstNewMethod, pstNewObject, pstOldFrame->pi32OpTop + 1);
       //       fprintf(stderr, "Creating new superframe %x, prev %i\n", pstFrame, pstOldFrame->iNumStackFrames);
       //       pstFrame->pstPrevFrame = pstOldFrame;
	  pstFrame->bIsOrphan = 0;
	  pstFrame->pstHeap = pstOldFrame->pstHeap;
       return pstFrame;
   }
  else
      {
	  pstFrame = (tStackFrame*) ((byte*) pstOldFrame + sizeof(tStackFrame));
	  pstFrame->iNumStackFrames = pstOldFrame->iNumStackFrames + 1;
          pstFrame->bSuperFrame = 0;
	  pstFrame->pstCurrClass = NULL;
          pstFrame->bFirstFrame = 0;
	  pstFrame->bIsOrphan = 0;
	  pstFrame->bIsNativeFrame = 1;
      }

  /* move old frame's operand stack pointer to before the arguments */
  //  pstOldFrame->pi32OpTop -= ( pstNewMethod->u16ArgSize ); //Pull it back to *before* the args

  pstFrame->pi32Vars = NULL;
  pstFrame->pstPrevFrame = pstOldFrame;
pstFrame->pstHeap = pstOldFrame->pstHeap;
 pstFrame->pstCurrClass = pstNewClass;
  /* point current method pointer to new method */
 pstFrame->pstCurrMethod = pstNewMethod;
  /* point current object pointer to new object */
 pstFrame->pstCurrObject = pstNewObject;
  /* set program counter to beginning of this method's code */
 pstFrame->pbPC = pstFrame->pbCode = 0; //pstNewMethod->pstCode->pbCode;

  /* set exception to NULL */
  pstFrame->pstException = NULL;

  pstFrame->pstChildFrame = NULL;

  /* return pointer to the new frame */
  return pstFrame;

}


/*
 * @doc FUNC
 * @func
 * This function is used to dispose of environment stack frames and pop the
 * previous one off the environment stack.  It is passed the current enironment
 * frame and returns the previous one.  It is used when methods return to
 * their calling method.  After this function is called, the virtual machine
 * registers must be loaded with LOADENV().
 *
 * @rdesc Returns a pointer to the new environment frame.
 *
 * @ex Example of use of PopStackFrame(). |
 *
 * pstCurrFrame = PopStackFrame(pstCurrFrame);
 * LOADENV();
 *
 * NMS
 */

tStackFrame* PopStackFrame
(
  tStackFrame* pstOldFrame  /* @parm Pointer to current stack frame */
)
{
  tStackFrame* pstTempFrame;

#ifdef DEBUG
  if (!pstOldFrame)
    {
      eprintf( "Tried to delete a NULL stack frame.\n");
      exit(1);
    }
#endif
  

  /* check if this is the first frame in the superframe */
  if (pstOldFrame->bSuperFrame)
    {
      pstTempFrame = pstOldFrame->pstPrevFrame;
      
       if(pstOldFrame->bIsNativeFrame == 0)
	  {
	      //Also free the variable data
	      sys_free(pstOldFrame->pi32Vars);
	      /* if so, delete this superframe and return pointer to previous frame */
	      sys_free(pstOldFrame);
	  }
       if(pstTempFrame)
	 pstTempFrame->pstChildFrame = NULL;
    return pstTempFrame;
   }
  else
  {
    pstOldFrame->pstPrevFrame->pstChildFrame = NULL;
    /* otherwise just return pointer to previous frame */
    return pstOldFrame->pstPrevFrame;
  }
}

/* A debugging method to show the Java stack frames */

void INTERP_ShowStackTrace(tStackFrame* pstFrame){

  int count = 0;
  while(pstFrame != NULL)
    {
      eprintf("%i %s.%s(%s)\n", count++, pstFrame->pstCurrMethod->pstClass->uidName, pstFrame->pstCurrMethod->uidName, pstFrame->pstCurrMethod->uidSignature);
      pstFrame = pstFrame->pstPrevFrame;
    }
}





