/*
 * Copyright (C) 2001, John Leuner.
 *
 * This file is part of the kissme/teaseme project, which in turn is part of the JOS project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2,
 * or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>

#include <assert.h>

#ifdef KISSME_LINUX_USER
#include <stdio.h>
#include <string.h>
#include <pthread.h>
#endif


#include "lib/indigenous/java.lang/java_lang_Class.h"  
#include "lib/indigenous/java.lang/Class_Reflection.h"
#include "lib/indigenous/java.lang/VMClassLoader.h" 
#include "garbage.h"
#include "cplist.h"
#include "jni.h"
#include "interp_methods.h"
#include "classfile_methods.h"
#include "newobject.h"
#include "methodstacks.h"
#include "locks.h" 


extern tClassLoaderTuple* pstClassType; // this can be globally used to get the type for class

#include "wrappers.h"
#ifdef KISSME_LINUX_USER
//for Dynamic loading
#ifdef DLOPEN
 #include <dlfcn.h>

int JNI_NumberLoadedLibraries = 0;
void* JNI_DynLoadedLibraries[20];
#endif
#include <stdio.h>
#include <string.h> //strchr
#endif



/* macros to simplify things */

#define SETCLASSLOADER(obj, i) ((DEREF(obj)->pi32Vars[u16LoaderOff]) = (int32) (i))
#define SETCLASSSTRUCT(obj, i) ((DEREF(obj)->pi32Vars[u16StructOff]) = (int32) (i))

#define GETCLASSLOADER(obj)    ((tOBREF) (DEREF(obj)->pi32Vars[u16LoaderOff]))
#define GETCLASSSTRUCT(obj)    ((tClassLoaderTuple*) (DEREF(obj)->pi32Vars[u16StructOff]))

/* global variables */

//hmmm, threading?

uint16 JNI_u16NumGlobalRefs;
uint16 JNI_u16NumGlobalRefsUsed;
tOBREF* JNI_ppstGlobalRefs;

extern tAllocatorHeap* system_heap;

/* local prototypes */

static int32* GetArgsFromVaList(tMethod* pstMethod, va_list args, int iStart, int iMinSize);
static int32* GetArgsFromJValues(tMethod* pstMethod, jvalue* args, int iStart, int iMinSize);
static void CreateLocalRef(tJNIData* pstJNIData, jobject obj);

/* local global variables */

int init2 = 0;
uint16 u16StructOff;    /* offset of _classStruct field in Class class */
uint16 u16LoaderOff;    /* offset of _classLoader field in Class class */

tJNIData* nativeJNITable[65536];


tJNIData* JNI_getJNIData(int pid)
{
  if((pid > 65535) || (pid < 0))
    {
      eprintf("---------------- PID OUT OF RANGE in JNI_getJNIData (%i)---------------\n", pid);
    }
  else
    {
      //      eprintf("Getting JNIData at %i as %p\n", pid, nativeJNITable[pid]);
      return nativeJNITable[pid];
    }
  return NULL;
}


void JNI_setJNIData(tJNIData* data, int pid)
{
  if((pid > 65535) || (pid < 0))
    {
      eprintf("---------------- PID OUT OF RANGE in JNI_setJNIData (%i)---------------\n", pid);
    }
  else
    {
      //      eprintf("Setting JNIData at %i to %p\n", pid, data);
      nativeJNITable[pid] = data;
    }
}

void JNI_zeroJNITable()
{
  int i;
  for(i = 0; i < 65536;i++)
    {
      nativeJNITable[i] = NULL;
    }
}


/* Version Information */
jint GetVersion (JNIEnv* env)
{
  return 0x00010001L;
}

/* Class Operations */

static jclass DefineClass (JNIEnv* env, jobject loader, const jbyte* buf, jsize bufLen)

{
  const byte* pbData = (const jbyte*) buf;
  tClassLoaderTuple*  pstClass;
  tOBREF   pstClassObject;
  uint32 errorCode;

  pstClass = CLASSFILE_ClassCreate(env,(byte*) pbData, NULL, loader, &errorCode); 
  pstClassObject = INTERP_NewObject(env, pstClassType); //must use system class?
  /* set _classStruct field in new class object */
  SETCLASSSTRUCT(pstClassObject, pstClass);
  /* set _classLoader field in new class object */
  SETCLASSLOADER(pstClassObject, loader);
  /* set class struct to point to class object */
  pstClass->classObject = pstClassObject;
  CreateLocalRef( JNI_getJNIData(sys_current_thread()), pstClassObject);
  return pstClassObject;
}


//Jewel Oct 99

/* Class Operations */

static jclass DefinePrimitiveClass (JNIEnv* env, jobject loader, tClassLoaderTuple* class)

{
  tClassLoaderTuple*  pstClass;
  tOBREF   pstClassObject;

  pstClass = class;

  pstClassObject = INTERP_NewObject(env, pstClassType);
  // set _classStruct field in new class object 
  SETCLASSSTRUCT(pstClassObject, pstClass);
  // set _classLoader field in new class object 
  SETCLASSLOADER(pstClassObject, loader);
  // set class struct to point to class object 
  pstClass->classObject = pstClassObject;


  CreateLocalRef(JNI_getJNIData(  sys_current_thread()), pstClassObject);
  return pstClassObject;
}


static jclass FindClass (JNIEnv* env, const char* name)
/* throws things */
{
  tClassLoaderTuple*  pstClass;
  tOBREF   pstClassObject;

  pstClass = CLASSFILE_FindOrLoad(env, (char*) name, NULL) ;// JNI_getJNIData(sys_current_thread())->pstTopFrame->pstCurrClass);

  if(pstClassType == NULL)
    pstClassType = CLASSFILE_FindOrLoad(env, "java/lang/Class", NULL);

  if(pstClass == NULL)
  {
    return java_lang_VMClassLoader_getPrimitiveClassFromAsciz(env, name);
  }
  if ((pstClass->classObject == NULL))
    {
      pstClassObject = INTERP_NewObject(env, pstClassType);
      /* set _classStruct field in new class object */
      
      if(u16StructOff != (uint16) -1)
	SETCLASSSTRUCT(pstClassObject, pstClass);

      /* set _classLoader field in new class object to NULL */
      if(u16LoaderOff != (uint16) -1)
	SETCLASSLOADER(pstClassObject, NULL);

      /* set class struct to point to class object */
      pstClass->classObject = pstClassObject;
      CreateLocalRef( JNI_getJNIData(sys_current_thread()), pstClassObject);
    }
 else
     {
	 assert( GETCLASSSTRUCT(pstClass->classObject) == pstClass);
     }
  return pstClass->classObject;
}

static jclass GetSuperClass (JNIEnv* env, jclass clazz)
{
  tOBREF   pstClassObject = clazz;
  tOBREF   pstSuperObject;
  tClassLoaderTuple*  pstSuperClass;

  pstSuperClass = GETCLASSSTRUCT(pstClassObject)->pstClass->pstSuperClass;
  if (pstSuperClass->classObject)
  {
    CreateLocalRef( JNI_getJNIData(sys_current_thread()), pstSuperClass->classObject);
    return pstSuperClass->classObject;
  }
  else
  {
    pstSuperObject = INTERP_NewObjectFromName(env, "java/lang/Class");
    SETCLASSSTRUCT(pstSuperObject, pstSuperClass);
    pstSuperClass->classObject = pstSuperObject;
    CreateLocalRef( JNI_getJNIData(sys_current_thread()), pstSuperClass->classObject);
    return pstSuperObject;
  }
}

static jboolean IsAssignableFrom (JNIEnv* env, jclass clazz1, jclass clazz2)
{
  tClassLoaderTuple* pstClass1;
  tClassLoaderTuple* pstClass2;

  if(clazz2 == NULL)
    {
      (*env)->ThrowNew(env, (*env)->FindClass(env, "java/lang/NullPointerException"), "null class passed to IsAssignableFrom");
      return JNI_TRUE;
    }


  pstClass1 = GETCLASSSTRUCT(clazz1);
  pstClass2 = GETCLASSSTRUCT(clazz2);

  if (INTERP_CheckCast(env, pstClass1, pstClass2) == 0)
  {
    return JNI_FALSE;
  }
  else
  {
    return JNI_TRUE;
  }
}

/* Exceptions */
static jint Throw (JNIEnv* env, jthrowable obj)
{
  JNI_getJNIData( sys_current_thread())->pstException = obj;
  return 0;
}

static jint ThrowNew (JNIEnv* env, jclass clazz, const char* message)
{
  tOBREF   pstException;
  tClassLoaderTuple*  pstExClass;
  uint16   u16Offset;

  pstExClass = GETCLASSSTRUCT(clazz);
  pstException = INTERP_NewObject(env, pstExClass);
  //Jewel, Feb 00, I think this should be message for classpath
  u16Offset = CLASSFILE_NativeInstOffsetPtr(env, pstExClass, "message");
  if (message)
  {
    DEREF(pstException)->pi32Vars[u16Offset] = (int32) INTERP_NewStringFromAsciz(env, (char*) message);
  }
  else
  {
    DEREF(pstException)->pi32Vars[u16Offset] = (int32) NULL;
  }
  JNI_getJNIData(  sys_current_thread())->pstException = pstException;
  CreateLocalRef( JNI_getJNIData(  sys_current_thread()), pstException);
  return 0;
}

static jthrowable ExceptionOccurred (JNIEnv* env)
{
  return (jthrowable) JNI_getJNIData( sys_current_thread())->pstException;
}

static void ExceptionDescribe (JNIEnv* env)
{
  tOBREF pstExOb = JNI_getJNIData( sys_current_thread() )->pstException;
  if ( pstExOb )
  {
    eprintf( "Exception waiting at ExceptionDescribe call: %s\n", ODEREF(pstExOb)->pstType->uidName);
  }
  else
  {
    eprintf( "No exception waiting when ExceptionDescribe was called.\n");
  }
}

static void ExceptionClear (JNIEnv* env)
{
  JNI_getJNIData( sys_current_thread())->pstException = NULL;
}

static void FatalError (JNIEnv* env, const char* msg)
{
  eprintf( "FatalError called from native code: (%s) Virtual machine exitting.\n", msg);
#ifdef KISSME_LINUX_USER
  exit(-1);
#endif
}

static jboolean IsSameObject (JNIEnv* env, jobject ref1, jobject ref2)
{
  if (ref1 == ref2)
  {
    return JNI_TRUE;
  }
  else
  {
    return JNI_FALSE;
  }
}


jint EnsureLocalCapacity (JNIEnv *env, jint capacity)
{
  return 0;
}



/* Global and Local References */
static jobject NewGlobalRef (JNIEnv* env, jobject obj)
{
  int i;
  int iDone = 0;

  THREAD_SynchroniseEnter( pstClassType->classObject );

  /* stick it on the end or in a NULL space if there is no space at the end */
  if (JNI_u16NumGlobalRefsUsed < JNI_u16NumGlobalRefs)
  {
    for(i = 0; i < JNI_u16NumGlobalRefsUsed;i++)
      {
	if(JNI_ppstGlobalRefs[i] == obj)
	  {
	    return obj;
	  }
      }
    JNI_ppstGlobalRefs[JNI_u16NumGlobalRefsUsed++] = obj;
    iDone = 1;
  }
  else
  {
    for (i = 0; i < JNI_u16NumGlobalRefs; i++)
    {
      if (JNI_ppstGlobalRefs[i] == NULL)
      {
        JNI_ppstGlobalRefs[i] = obj;
        iDone = 1;
      }
    }
  }

  /* if there is no space then create more */
  if (iDone == 0)
  {
    tOBREF* oldrefs = JNI_ppstGlobalRefs;
    JNI_ppstGlobalRefs = sys_malloc( JNI_u16NumGlobalRefs * 2);
    memset(JNI_ppstGlobalRefs, 0, JNI_u16NumGlobalRefs * 2 * sizeof(tOBREF));
    for(i = 0; i < JNI_u16NumGlobalRefs;i++)
      {
	JNI_ppstGlobalRefs[i] = oldrefs[i];
      }
    JNI_u16NumGlobalRefs *= 2;
    JNI_ppstGlobalRefs[JNI_u16NumGlobalRefsUsed++] = obj;
    sys_free(oldrefs);
  }

  THREAD_SynchroniseExit( pstClassType->classObject );
  return obj;
}

static void DeleteGlobalRef (JNIEnv* env, jobject globalRef)
{

  int i;
  int position = -1;

  THREAD_SynchroniseEnter( pstClassType->classObject );

  for (i = 0; i < JNI_u16NumGlobalRefsUsed; i++)
  {
    if (JNI_ppstGlobalRefs[i] == globalRef)
    {
      position = i;
      break;
    }
  }

  if(position == -1)
    {
      //Ref not found
    }
  else
    {
      for(i = position; i < JNI_u16NumGlobalRefsUsed;i++)
	{
	  JNI_ppstGlobalRefs[i] = JNI_ppstGlobalRefs[i + 1];
	}
      JNI_u16NumGlobalRefsUsed--;
    }
  THREAD_SynchroniseExit( pstClassType->classObject );
  return;
}



/* Object Operations */
static jobject AllocObject (JNIEnv* env, jclass clazz)
{
  tClassLoaderTuple* pstClass;
  tOBREF pstObject;

  pstClass = GETCLASSSTRUCT(clazz);
  if ((pstClass->pstClass->u16AccessFlags & ACC_ABSTRACT) ||
      (pstClass->pstClass->u16AccessFlags & ACC_INTERFACE))
  {
    (*env)->Throw(env, INTERP_NewObjectFromName(env, "java/lang/InstantiationError"));
    return NULL;
  }
  pstObject = INTERP_NewObject(env, pstClass);
  if (pstObject == NULL)
  {
    (*env)->Throw(env, INTERP_NewObjectFromName(env, "java/lang/OutOfMemoryError"));
    return NULL;
  }
  CreateLocalRef( JNI_getJNIData(  sys_current_thread()), pstObject);
  return pstObject;
}

static jobject NewObject (JNIEnv* env, jclass clazz, jmethodID methodID, ...)
{
  tClassLoaderTuple* pstClass;
  tOBREF pstObject;
  tMethod* pstMethod = methodID;
  tOBREF pstExOb;
  va_list args;
  int32*  pi32Args;

  pstClass = GETCLASSSTRUCT(clazz);
  if ((pstClass->pstClass->u16AccessFlags & ACC_ABSTRACT) ||
      (pstClass->pstClass->u16AccessFlags & ACC_INTERFACE))
  {
    (*env)->Throw(env, INTERP_NewObjectFromName(env,"java/lang/InstantiationError"));
    return NULL;
  }

  pstObject = INTERP_NewObject(env, pstClass);
  if (pstObject == NULL)
  {
    (*env)->Throw(env, INTERP_NewObjectFromName(env, "java/lang/OutOfMemoryError"));
    return NULL;
  }

  va_start(args, methodID);
  pi32Args = GetArgsFromVaList(pstMethod, args, 1, 0);
  va_end(args);
  pi32Args[0] = (int32) pstObject;
  pstExOb = INTERP_RunSpecialMethodFromPtr(env, pstMethod, pi32Args);
  sys_free(pi32Args);
  CreateLocalRef(  JNI_getJNIData( sys_current_thread()), pstObject);
  if(pstExOb)
    {
      (*env)->Throw(env, pstExOb);
      return NULL;
    }
  return pstObject;
}

static jobject NewObjectA (JNIEnv* env, jclass clazz, jmethodID methodID, jvalue* args)
{
  tClassLoaderTuple* pstClass;
  tOBREF pstObject;
  int32*  pi32Args;
  tOBREF pstExOb;

  pstClass = GETCLASSSTRUCT(clazz);
  if ((pstClass->pstClass->u16AccessFlags & ACC_ABSTRACT) ||
      (pstClass->pstClass->u16AccessFlags & ACC_INTERFACE))
  {
    (*env)->Throw(env, INTERP_NewObjectFromName(env, "java/lang/InstantiationError"));
    return NULL;
  }

  pstObject = INTERP_NewObject(env, pstClass);
  if (pstObject == NULL)
  {
    (*env)->Throw(env, INTERP_NewObjectFromName(env, "java/lang/OutOfMemoryError"));
    return NULL;
  }

  pi32Args = GetArgsFromJValues(methodID, args, 1, 1 + methodID->u16ArgSize);
  pi32Args[0] = (int32) pstObject;
  pstExOb = INTERP_RunSpecialMethodFromPtr(env, methodID, pi32Args);
  sys_free(pi32Args);

  CreateLocalRef(   JNI_getJNIData( sys_current_thread()), pstObject);
  if(pstExOb)
    {
      (*env)->Throw(env, pstExOb);
      return NULL;
    }
  return pstObject;
}

static jobject NewObjectV (JNIEnv* env, jclass clazz, jmethodID methodID, va_list args)
{
  tClassLoaderTuple* pstClass;
  tOBREF pstObject;
  int32*  pi32Args;
  tOBREF pstExOb;

  pstClass = GETCLASSSTRUCT(clazz);
  if ((pstClass->pstClass->u16AccessFlags & ACC_ABSTRACT) ||
      (pstClass->pstClass->u16AccessFlags & ACC_INTERFACE))
  {
    (*env)->Throw(env, INTERP_NewObjectFromName(env, "java/lang/InstantiationError"));
    return NULL;
  }

  pstObject = INTERP_NewObject(env, pstClass);
  if (pstObject == NULL)
  {
    (*env)->Throw(env, INTERP_NewObjectFromName(env, "java/lang/OutOfMemoryError"));
    return NULL;
  }

  pi32Args = GetArgsFromVaList(methodID, args, 1, 0);
  pi32Args[0] = (int32) pstObject;
  pstExOb = INTERP_RunSpecialMethodFromPtr(env, methodID, pi32Args);
  sys_free(pi32Args);

  CreateLocalRef(  JNI_getJNIData( sys_current_thread()), pstObject);
  if(pstExOb)
    {
      (*env)->Throw(env, pstExOb);
      return NULL;
    }
  return pstObject;
}

static jclass GetObjectClass (JNIEnv* env, jobject obj)
{
  jclass clazz = CLASS_GetClass(env,obj);
  CreateLocalRef(  JNI_getJNIData( sys_current_thread()), clazz);
  return clazz;
}

static jboolean InstanceOf (JNIEnv* env, jobject obj, jclass clazz)
{
  tClassLoaderTuple* pstClassTo;

  pstClassTo = GETCLASSSTRUCT(clazz);
  if (INTERP_CheckCast(env, pstClassTo, DEREF(obj)->pstType) == 0)
  {
    return JNI_FALSE;
  }
  else
  {
    return JNI_TRUE;
  }
}

/* Calling Instance Methods */
static jmethodID GetMethodID (JNIEnv* env, jclass clazz, const char* name, const char* sig)
{
  tClassLoaderTuple* pstClass;
  tMethod* pstMethod;

  pstClass = GETCLASSSTRUCT(clazz);
  pstMethod = CLASSFILE_FindMethodInVT(env, pstClass, (char*) name, (char*) sig);

  if (pstMethod)
  {
    return pstMethod;
  }
  else
  {
    (*env)->Throw(env, INTERP_NewObjectFromName(env, "java/lang/NoSuchMethodError"));
    return NULL;
  }
}

static jobject CallObjectMethod (JNIEnv* env, jobject obj, jmethodID methodID, ...)
{
  va_list  args;
  int32*   pi32Args;
  jobject  ret;
  jobject  ex;

  va_start(args, methodID);
  pi32Args = GetArgsFromVaList(methodID, args, 1, 1);
  va_end(args);
  pi32Args[0] = (int32) obj;

  assert((*env)->getHeap());
  ex = INTERP_RunVirtualMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  ret = (jobject) pi32Args[0];
  sys_free(pi32Args);
  if (!ex)
  {
    CreateLocalRef( JNI_getJNIData(  sys_current_thread()), ret);
  }
  return ret;
}

static jobject CallObjectMethodA (JNIEnv* env, jobject obj, jmethodID methodID, jvalue* args)
{
  int32*   pi32Args;
  jobject  ret;
  jobject  ex;

  assert(!(methodID->u16AccessFlags & ACC_STATIC));
  pi32Args = GetArgsFromJValues(methodID, args, 1, 1);
  pi32Args[0] = (int32) obj;
  ex = INTERP_RunVirtualMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  ret = (jobject) pi32Args[0];
  sys_free(pi32Args);
  if (!ex)
  {
    CreateLocalRef(  JNI_getJNIData( sys_current_thread()), ret);
  }
  return ret;
}

static jobject CallObjectMethodV (JNIEnv* env, jobject obj, jmethodID methodID, va_list args)
{
  int32*   pi32Args;
  jobject  ret;
  jobject  ex;

  pi32Args = GetArgsFromVaList(methodID, args, 1, 1);
  pi32Args[0] = (int32) obj;
  ex = INTERP_RunVirtualMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  ret = (jobject) pi32Args[0];
  sys_free(pi32Args);
  if (!ex)
  {
    CreateLocalRef(JNI_getJNIData(  sys_current_thread()), ret);
  }
  return ret;
}

static jboolean CallBooleanMethod (JNIEnv* env, jobject obj, jmethodID methodID, ...)
{
  va_list  args;
  int32*   pi32Args;
  jboolean ret;
  jobject  ex;

  va_start(args, methodID);
  pi32Args = GetArgsFromVaList(methodID, args, 1, 1);
  va_end(args);
  pi32Args[0] = (int32) obj;
  ex = INTERP_RunVirtualMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  ret = (jboolean) pi32Args[0];
  sys_free(pi32Args);
  return ret;
}

static jboolean CallBooleanMethodA (JNIEnv* env, jobject obj, jmethodID methodID, jvalue* args)
{
  int32*   pi32Args;
  jboolean ret;
  jobject  ex;

  pi32Args = GetArgsFromJValues(methodID, args, 1, 1);
  pi32Args[0] = (int32) obj;
  ex = INTERP_RunVirtualMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  ret = (jboolean) pi32Args[0];
  sys_free(pi32Args);
  return ret;
}

static jboolean CallBooleanMethodV (JNIEnv* env, jobject obj, jmethodID methodID, va_list args)
{
  int32*   pi32Args;
  jboolean ret;
  jobject  ex;

  pi32Args = GetArgsFromVaList(methodID, args, 1, 1);
  pi32Args[0] = (int32) obj;
  ex = INTERP_RunVirtualMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  ret = (jboolean) pi32Args[0];
  sys_free(pi32Args);
  return ret;
}

static jbyte CallByteMethod (JNIEnv* env, jobject obj, jmethodID methodID, ...)
{
  va_list  args;
  int32*   pi32Args;
  jbyte    ret;
  jobject  ex;

  va_start(args, methodID);
  pi32Args = GetArgsFromVaList(methodID, args, 1, 1);
  va_end(args);
  pi32Args[0] = (int32) obj;
  ex = INTERP_RunVirtualMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  ret = (jbyte) pi32Args[0];
  sys_free(pi32Args);
  return ret;
}

static jbyte CallByteMethodA (JNIEnv* env, jobject obj, jmethodID methodID, jvalue* args)
{
  int32*   pi32Args;
  jbyte    ret;
  jobject  ex;

  pi32Args = GetArgsFromJValues(methodID, args, 1, 1);
  pi32Args[0] = (int32) obj;
  ex = INTERP_RunVirtualMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  ret = (jbyte) pi32Args[0];
  sys_free(pi32Args);
  return ret;
}

static jbyte CallByteMethodV (JNIEnv* env, jobject obj, jmethodID methodID, va_list args)
{
  int32*   pi32Args;
  jbyte    ret;
  jobject  ex;

  pi32Args = GetArgsFromVaList(methodID, args, 1, 1);
  pi32Args[0] = (int32) obj;
  ex = INTERP_RunVirtualMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  ret = (jbyte) pi32Args[0];
  sys_free(pi32Args);
  return ret;
}

static jchar CallCharMethod (JNIEnv* env, jobject obj, jmethodID methodID, ...)
{
  va_list  args;
  int32*   pi32Args;
  jchar    ret;
  jobject  ex;

  va_start(args, methodID);
  pi32Args = GetArgsFromVaList(methodID, args, 1, 1);
  va_end(args);
  pi32Args[0] = (int32) obj;
  ex = INTERP_RunVirtualMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  ret = (jchar) pi32Args[0];
  sys_free(pi32Args);
  return ret;
}

static jchar CallCharMethodA (JNIEnv* env, jobject obj, jmethodID methodID, jvalue* args)
{
  int32*   pi32Args;
  jchar    ret;
  jobject  ex;

  pi32Args = GetArgsFromJValues(methodID, args, 1, 1);
  pi32Args[0] = (int32) obj;
  ex = INTERP_RunVirtualMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  ret = (jchar) pi32Args[0];
  sys_free(pi32Args);
  return ret;
}

static jchar CallCharMethodV (JNIEnv* env, jobject obj, jmethodID methodID, va_list args)
{
  int32*   pi32Args;
  jchar    ret;
  jobject  ex;

  pi32Args = GetArgsFromVaList(methodID, args, 1, 1);
  pi32Args[0] = (int32) obj;
  ex = INTERP_RunVirtualMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  ret = (jchar) pi32Args[0];
  sys_free(pi32Args);
  return ret;
}

static jshort CallShortMethod (JNIEnv* env, jobject obj, jmethodID methodID, ...)
{
  va_list  args;
  int32*   pi32Args;
  jshort   ret;
  jobject  ex;

  va_start(args, methodID);
  pi32Args = GetArgsFromVaList(methodID, args, 1, 1);
  va_end(args);
  pi32Args[0] = (int32) obj;
  ex = INTERP_RunVirtualMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  ret = (jshort) pi32Args[0];
  sys_free(pi32Args);
  return ret;
}

static jshort CallShortMethodA (JNIEnv* env, jobject obj, jmethodID methodID, jvalue* args)
{
  int32*   pi32Args;
  jshort   ret;
  jobject  ex;

  pi32Args = GetArgsFromJValues(methodID, args, 1, 1);
  pi32Args[0] = (int32) obj;
  ex = INTERP_RunVirtualMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  ret = (jshort) pi32Args[0];
  sys_free(pi32Args);
  return ret;
}

static jshort CallShortMethodV (JNIEnv* env, jobject obj, jmethodID methodID, va_list args)
{
  int32*   pi32Args;
  jshort   ret;
  jobject  ex;

  pi32Args = GetArgsFromVaList(methodID, args, 1, 1);
  pi32Args[0] = (int32) obj;
  ex = INTERP_RunVirtualMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  ret = (jshort) pi32Args[0];
  sys_free(pi32Args);
  return ret;
}

static jint CallIntMethod (JNIEnv* env, jobject obj, jmethodID methodID, ...)
{
  va_list  args;
  int32*   pi32Args;
  jint     ret;
  jobject  ex;

  va_start(args, methodID);
  pi32Args = GetArgsFromVaList(methodID, args, 1, 1);
  va_end(args);
  pi32Args[0] = (int32) obj;
  ex = INTERP_RunVirtualMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  ret = (jint) pi32Args[0];
  sys_free(pi32Args);
  return ret;
}

static jint CallIntMethodA (JNIEnv* env, jobject obj, jmethodID methodID, jvalue* args)
{
  int32*   pi32Args;
  jint     ret;
  jobject  ex;

  pi32Args = GetArgsFromJValues(methodID, args, 1, 1);
  pi32Args[0] = (int32) obj;
  ex = INTERP_RunVirtualMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  ret = (jint) pi32Args[0];
  sys_free(pi32Args);
  return ret;
}

static jint CallIntMethodV (JNIEnv* env, jobject obj, jmethodID methodID, va_list args)
{
  int32*   pi32Args;
  jint     ret;
  jobject  ex;

  pi32Args = GetArgsFromVaList(methodID, args, 1, 1);
  pi32Args[0] = (int32) obj;
  ex = INTERP_RunVirtualMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  ret = (jint) pi32Args[0];
  sys_free(pi32Args);
  return ret;
}

static jlong CallLongMethod (JNIEnv* env, jobject obj, jmethodID methodID, ...)
{
  va_list  args;
  int32*   pi32Args;
  jlong    ret;
  jobject  ex;

  va_start(args, methodID);
  pi32Args = GetArgsFromVaList(methodID, args, 1, 2);
  va_end(args);
  pi32Args[0] = (int32) obj;
  ex = INTERP_RunVirtualMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  ret=pi32Args[0];
  ret<<=32;
  ret|=pi32Args[1];

  sys_free(pi32Args);
  return ret;
}

static jlong CallLongMethodA (JNIEnv* env, jobject obj, jmethodID methodID, jvalue* args)
{
  int32*   pi32Args;
  jlong    ret;
  jobject  ex;

  pi32Args = GetArgsFromJValues(methodID, args, 1, 2);
  pi32Args[0] = (int32) obj;
  ex = INTERP_RunVirtualMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  ret=pi32Args[0];
  ret<<=32;
  ret|=pi32Args[1];
  sys_free(pi32Args);
  return ret;
}

static jlong CallLongMethodV (JNIEnv* env, jobject obj, jmethodID methodID, va_list args)
{
  int32*   pi32Args;
  jlong    ret;
  jobject  ex;

  pi32Args = GetArgsFromVaList(methodID, args, 1, 2);
  pi32Args[0] = (int32) obj;
  ex = INTERP_RunVirtualMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  ret=pi32Args[0];
  ret<<=32;
  ret|=pi32Args[1];
  sys_free(pi32Args);
  return ret;
}

static jfloat CallFloatMethod (JNIEnv* env, jobject obj, jmethodID methodID, ...)
{
  va_list  args;
  int32*   pi32Args;
  jfloat   ret;
  jobject  ex;

  va_start(args, methodID);
  pi32Args = GetArgsFromVaList(methodID, args, 1, 1);
  va_end(args);
  pi32Args[0] = (int32) obj;
  ex = INTERP_RunVirtualMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  ret = ((jfloat*) pi32Args)[0];
  sys_free(pi32Args);
  return ret;
}

static jfloat CallFloatMethodA (JNIEnv* env, jobject obj, jmethodID methodID, jvalue* args)
{
  int32*   pi32Args;
  jfloat   ret;
  jobject  ex;

  pi32Args = GetArgsFromJValues(methodID, args, 1, 1);
  pi32Args[0] = (int32) obj;
  ex = INTERP_RunVirtualMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  ret = ((jfloat*) pi32Args)[0];
  sys_free(pi32Args);
  return ret;
}

static jfloat CallFloatMethodV (JNIEnv* env, jobject obj, jmethodID methodID, va_list args)
{
  int32*   pi32Args;
  jfloat   ret;
  jobject  ex;

  pi32Args = GetArgsFromVaList(methodID, args, 1, 1);
  pi32Args[0] = (int32) obj;
  ex = INTERP_RunVirtualMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  ret = ((jfloat*) pi32Args)[0];
  sys_free(pi32Args);
  return ret;
}

static jdouble CallDoubleMethod (JNIEnv* env, jobject obj, jmethodID methodID, ...)
{
  va_list  args;
  int32*   pi32Args;
  tDConv   ret;
  jobject  ex;

  va_start(args, methodID);
  pi32Args = GetArgsFromVaList(methodID, args, 1, 2);
  va_end(args);
  pi32Args[0] = (int32) obj;
  ex = INTERP_RunVirtualMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  ret.i.hi = pi32Args[0];
  ret.i.lo = pi32Args[1];
  sys_free(pi32Args);
  return ret.d;
}

static jdouble CallDoubleMethodA (JNIEnv* env, jobject obj, jmethodID methodID, jvalue* args)
{
  int32*   pi32Args;
  tDConv   ret;
  jobject  ex;

  pi32Args = GetArgsFromJValues(methodID, args, 1, 2);
  pi32Args[0] = (int32) obj;
  ex = INTERP_RunVirtualMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  ret.i.hi = pi32Args[0];
  ret.i.lo = pi32Args[1];
  sys_free(pi32Args);
  return ret.d;
}

static jdouble CallDoubleMethodV (JNIEnv* env, jobject obj, jmethodID methodID, va_list args)
{
  int32*   pi32Args;
  tDConv   ret;
  jobject  ex;

  pi32Args = GetArgsFromVaList(methodID, args, 1, 2);
  pi32Args[0] = (int32) obj;
  ex = INTERP_RunVirtualMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  ret.i.hi = pi32Args[0];
  ret.i.lo = pi32Args[1];
  sys_free(pi32Args);
  return ret.d;
}

static void CallVoidMethod (JNIEnv* env, jobject obj, jmethodID methodID, ...)
{
  va_list  args;
  int32*   pi32Args;
  jobject  ex;

  va_start(args, methodID);
  pi32Args = GetArgsFromVaList(methodID, args, 1, 0);
  va_end(args);
  pi32Args[0] = (int32) obj;
  ex = INTERP_RunVirtualMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  sys_free(pi32Args);
  return;
}

static void CallVoidMethodA (JNIEnv* env, jobject obj, jmethodID methodID, jvalue* args)
{
  int32*   pi32Args;
  jobject  ex;

  pi32Args = GetArgsFromJValues(methodID, args, 1, 0);
  pi32Args[0] = (int32) obj;
  ex = INTERP_RunVirtualMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  sys_free(pi32Args);
  return;
}

static void CallVoidMethodV (JNIEnv* env, jobject obj, jmethodID methodID, va_list args)
{
  int32*   pi32Args;
  jobject  ex;

  pi32Args = GetArgsFromVaList(methodID, args, 1, 0);
  pi32Args[0] = (int32) obj;
  ex = INTERP_RunVirtualMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  sys_free(pi32Args);
  return;
}

static jobject CallSpecialObjectMethod (JNIEnv* env, jobject obj, jclass clazz, jmethodID methodID, ...)
{
  va_list  args;
  int32*   pi32Args;
  jobject  ret;
  jobject  ex;

  va_start(args, methodID);
  pi32Args = GetArgsFromVaList(methodID, args, 1, 1);
  va_end(args);
  pi32Args[0] = (int32) obj;
  ex = INTERP_RunSpecialMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  ret = (jobject) pi32Args[0];
  sys_free(pi32Args);
  if (!ex)
  {
    CreateLocalRef(JNI_getJNIData(  sys_current_thread()), ret);
  }
  return ret;
}

static jobject CallSpecialObjectMethodA (JNIEnv* env, jobject obj, jclass clazz, jmethodID methodID, jvalue* args)
{
  int32*   pi32Args;
  jobject  ret;
  jobject  ex;

  pi32Args = GetArgsFromJValues(methodID, args, 1, 1);
  pi32Args[0] = (int32) obj;
  ex = INTERP_RunSpecialMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  ret = (jobject) pi32Args[0];
  sys_free(pi32Args);
  if (!ex)
  {
    CreateLocalRef(JNI_getJNIData(sys_current_thread()), ret);
  }
  return ret;
}

static jobject CallSpecialObjectMethodV (JNIEnv* env, jobject obj, jclass clazz, jmethodID methodID, va_list args)
{
  int32*   pi32Args;
  jobject  ret;
  jobject  ex;

  pi32Args = GetArgsFromVaList(methodID, args, 1, 1);
  pi32Args[0] = (int32) obj;
  ex = INTERP_RunSpecialMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  ret = (jobject) pi32Args[0];
  sys_free(pi32Args);
  if (!ex)
  {
    CreateLocalRef(JNI_getJNIData(  sys_current_thread()), ret);
  }
  return ret;
}

static jboolean CallSpecialBooleanMethod (JNIEnv* env, jobject obj, jclass clazz, jmethodID methodID, ...)
{
  va_list  args;
  int32*   pi32Args;
  jboolean ret;
  jobject  ex;

  va_start(args, methodID);
  pi32Args = GetArgsFromVaList(methodID, args, 1, 1);
  va_end(args);
  pi32Args[0] = (int32) obj;
  ex = INTERP_RunSpecialMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  ret = (jboolean) pi32Args[0];
  sys_free(pi32Args);
  return ret;
}

static jboolean CallSpecialBooleanMethodA (JNIEnv* env, jobject obj, jclass clazz, jmethodID methodID, jvalue* args)
{
  int32*   pi32Args;
  jboolean ret;
  jobject  ex;

  pi32Args = GetArgsFromJValues(methodID, args, 1, 1);
  pi32Args[0] = (int32) obj;
  ex = INTERP_RunSpecialMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  ret = (jboolean) pi32Args[0];
  sys_free(pi32Args);
  return ret;
}

static jboolean CallSpecialBooleanMethodV (JNIEnv* env, jobject obj, jclass clazz, jmethodID methodID, va_list args)
{
  int32*   pi32Args;
  jboolean ret;
  jobject  ex;

  pi32Args = GetArgsFromVaList(methodID, args, 1, 1);
  pi32Args[0] = (int32) obj;
  ex = INTERP_RunSpecialMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  ret = (jboolean) pi32Args[0];
  sys_free(pi32Args);
  return ret;
}

static jbyte CallSpecialByteMethod (JNIEnv* env, jobject obj, jclass clazz, jmethodID methodID, ...)
{
  va_list  args;
  int32*   pi32Args;
  jbyte    ret;
  jobject  ex;

  va_start(args, methodID);
  pi32Args = GetArgsFromVaList(methodID, args, 1, 1);
  va_end(args);
  pi32Args[0] = (int32) obj;
  ex = INTERP_RunSpecialMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  ret = (jbyte) pi32Args[0];
  sys_free(pi32Args);
  return ret;
}

static jbyte CallSpecialByteMethodA (JNIEnv* env, jobject obj, jclass clazz, jmethodID methodID, jvalue* args)
{
  int32*   pi32Args;
  jbyte    ret;
  jobject  ex;

  pi32Args = GetArgsFromJValues(methodID, args, 1, 1);
  pi32Args[0] = (int32) obj;
  ex = INTERP_RunSpecialMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  ret = (jbyte) pi32Args[0];
  sys_free(pi32Args);
  return ret;
}

static jbyte CallSpecialByteMethodV (JNIEnv* env, jobject obj, jclass clazz, jmethodID methodID, va_list args)
{
  int32*   pi32Args;
  jbyte    ret;
  jobject  ex;

  pi32Args = GetArgsFromVaList(methodID, args, 1, 1);
  pi32Args[0] = (int32) obj;
  ex = INTERP_RunSpecialMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  ret = (jbyte) pi32Args[0];
  sys_free(pi32Args);
  return ret;
}

static jchar CallSpecialCharMethod (JNIEnv* env, jobject obj, jclass clazz, jmethodID methodID, ...)
{
  va_list  args;
  int32*   pi32Args;
  jchar    ret;
  jobject  ex;

  va_start(args, methodID);
  pi32Args = GetArgsFromVaList(methodID, args, 1, 1);
  va_end(args);
  pi32Args[0] = (int32) obj;
  ex = INTERP_RunSpecialMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  ret = (jchar) pi32Args[0];
  sys_free(pi32Args);
  return ret;
}

static jchar CallSpecialCharMethodA (JNIEnv* env, jobject obj, jclass clazz, jmethodID methodID, jvalue* args)
{
  int32*   pi32Args;
  jchar    ret;
  jobject  ex;

  pi32Args = GetArgsFromJValues(methodID, args, 1, 1);
  pi32Args[0] = (int32) obj;
  ex = INTERP_RunSpecialMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  ret = (jchar) pi32Args[0];
  sys_free(pi32Args);
  return ret;
}

static jchar CallSpecialCharMethodV (JNIEnv* env, jobject obj, jclass clazz, jmethodID methodID, va_list args)
{
  int32*   pi32Args;
  jchar    ret;
  jobject  ex;

  pi32Args = GetArgsFromVaList(methodID, args, 1, 1);
  pi32Args[0] = (int32) obj;
  ex = INTERP_RunSpecialMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  ret = (jchar) pi32Args[0];
  sys_free(pi32Args);
  return ret;
}

static jshort CallSpecialShortMethod (JNIEnv* env, jobject obj, jclass clazz, jmethodID methodID, ...)
{
  va_list  args;
  int32*   pi32Args;
  jshort   ret;
  jobject  ex;

  va_start(args, methodID);
  pi32Args = GetArgsFromVaList(methodID, args, 1, 1);
  va_end(args);
  pi32Args[0] = (int32) obj;
  ex = INTERP_RunSpecialMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  ret = (jshort) pi32Args[0];
  sys_free(pi32Args);
  return ret;
}

static jshort CallSpecialShortMethodA (JNIEnv* env, jobject obj, jclass clazz, jmethodID methodID, jvalue* args)
{
  int32*   pi32Args;
  jshort   ret;
  jobject  ex;

  pi32Args = GetArgsFromJValues(methodID, args, 1, 1);
  pi32Args[0] = (int32) obj;
  ex = INTERP_RunSpecialMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  ret = (jshort) pi32Args[0];
  sys_free(pi32Args);
  return ret;
}

static jshort CallSpecialShortMethodV (JNIEnv* env, jobject obj, jclass clazz, jmethodID methodID, va_list args)
{
  int32*   pi32Args;
  jshort   ret;
  jobject  ex;

  pi32Args = GetArgsFromVaList(methodID, args, 1, 1);
  pi32Args[0] = (int32) obj;
  ex = INTERP_RunSpecialMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  ret = (jshort) pi32Args[0];
  sys_free(pi32Args);
  return ret;
}

static jint CallSpecialIntMethod (JNIEnv* env, jobject obj, jclass clazz, jmethodID methodID, ...)
{
  va_list  args;
  int32*   pi32Args;
  jint     ret;
  jobject  ex;

  va_start(args, methodID);
  pi32Args = GetArgsFromVaList(methodID, args, 1, 1);
  va_end(args);
  pi32Args[0] = (int32) obj;
  ex = INTERP_RunSpecialMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  ret = (jint) pi32Args[0];
  sys_free(pi32Args);
  return ret;
}

static jint CallSpecialIntMethodA (JNIEnv* env, jobject obj, jclass clazz, jmethodID methodID, jvalue* args)
{
  int32*   pi32Args;
  jint     ret;
  jobject  ex;

  pi32Args = GetArgsFromJValues(methodID, args, 1, 1);
  pi32Args[0] = (int32) obj;
  ex = INTERP_RunSpecialMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  ret = (jint) pi32Args[0];
  sys_free(pi32Args);
  return ret;
}

static jint CallSpecialIntMethodV (JNIEnv* env, jobject obj, jclass clazz, jmethodID methodID, va_list args)
{
  int32*   pi32Args;
  jint     ret;
  jobject  ex;

  pi32Args = GetArgsFromVaList(methodID, args, 1, 1);
  pi32Args[0] = (int32) obj;
  ex = INTERP_RunSpecialMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  ret = (jint) pi32Args[0];
  sys_free(pi32Args);
  return ret;
}

static jlong CallSpecialLongMethod (JNIEnv* env, jobject obj, jclass clazz, jmethodID methodID, ...)
{
  va_list  args;
  int32*   pi32Args;
  jlong    ret;
  jobject  ex;

  va_start(args, methodID);
  pi32Args = GetArgsFromVaList(methodID, args, 1, 2);
  va_end(args);
  pi32Args[0] = (int32) obj;
  ex = INTERP_RunSpecialMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }

  ret=pi32Args[0];
  ret<<=32;
  ret|=pi32Args[1];
  sys_free(pi32Args);
  return ret;
}

static jlong CallSpecialLongMethodA (JNIEnv* env, jobject obj, jclass clazz, jmethodID methodID, jvalue* args)
{
  int32*   pi32Args;
  jlong    ret;
  jobject  ex;

  pi32Args = GetArgsFromJValues(methodID, args, 1, 2);
  pi32Args[0] = (int32) obj;
  ex = INTERP_RunSpecialMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  ret=pi32Args[0];
  ret<<=32;
  ret|=pi32Args[1];
  sys_free(pi32Args);
  return ret;
}

static jlong CallSpecialLongMethodV (JNIEnv* env, jobject obj, jclass clazz, jmethodID methodID, va_list args)
{
  int32*   pi32Args;
  jlong    ret;
  jobject  ex;

  pi32Args = GetArgsFromVaList(methodID, args, 1, 2);
  pi32Args[0] = (int32) obj;
  ex = INTERP_RunSpecialMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  ret = pi32Args[0];
  ret <<= 32;
  ret |= pi32Args[1];
  sys_free(pi32Args);
  return ret;
}

static jfloat CallSpecialFloatMethod (JNIEnv* env, jobject obj, jclass clazz, jmethodID methodID, ...)
{
  va_list  args;
  int32*   pi32Args;
  jfloat   ret;
  jobject  ex;

  va_start(args, methodID);
  pi32Args = GetArgsFromVaList(methodID, args, 1, 1);
  va_end(args);
  pi32Args[0] = (int32) obj;
  ex = INTERP_RunSpecialMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  ret = ((jfloat*) pi32Args)[0];
  sys_free(pi32Args);
  return ret;
}

static jfloat CallSpecialFloatMethodA (JNIEnv* env, jobject obj, jclass clazz, jmethodID methodID, jvalue* args)
{
  int32*   pi32Args;
  jfloat   ret;
  jobject  ex;

  pi32Args = GetArgsFromJValues(methodID, args, 1, 1);
  pi32Args[0] = (int32) obj;
  ex = INTERP_RunSpecialMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  ret = ((jfloat*) pi32Args)[0];
  sys_free(pi32Args);
  return ret;
}

static jfloat CallSpecialFloatMethodV (JNIEnv* env, jobject obj, jclass clazz, jmethodID methodID, va_list args)
{
  int32*   pi32Args;
  jfloat   ret;
  jobject  ex;

  pi32Args = GetArgsFromVaList(methodID, args, 1, 1);
  pi32Args[0] = (int32) obj;
  ex = INTERP_RunSpecialMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  ret = ((jfloat*) pi32Args)[0];
  sys_free(pi32Args);
  return ret;
}

static jdouble CallSpecialDoubleMethod (JNIEnv* env, jobject obj, jclass clazz, jmethodID methodID, ...)
{
  va_list  args;
  int32*   pi32Args;
  tDConv   ret;
  jobject  ex;

  va_start(args, methodID);
  pi32Args = GetArgsFromVaList(methodID, args, 1, 2);
  va_end(args);
  pi32Args[0] = (int32) obj;
  ex = INTERP_RunSpecialMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  ret.i.hi = pi32Args[0];
  ret.i.lo = pi32Args[1];
  sys_free(pi32Args);
  return ret.d;
}

static jdouble CallSpecialDoubleMethodA (JNIEnv* env, jobject obj, jclass clazz, jmethodID methodID, jvalue* args)
{
  int32*   pi32Args;
  tDConv   ret;
  jobject  ex;

  pi32Args = GetArgsFromJValues(methodID, args, 1, 2);
  pi32Args[0] = (int32) obj;
  ex = INTERP_RunSpecialMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  ret.i.hi = pi32Args[0];
  ret.i.lo = pi32Args[1];
  sys_free(pi32Args);
  return ret.d;
}

static jdouble CallSpecialDoubleMethodV (JNIEnv* env, jobject obj, jclass clazz, jmethodID methodID, va_list args)
{
  int32*   pi32Args;
  tDConv   ret;
  jobject  ex;

  pi32Args = GetArgsFromVaList(methodID, args, 1, 2);
  pi32Args[0] = (int32) obj;
  ex = INTERP_RunSpecialMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  ret.i.hi = pi32Args[0];
  ret.i.lo = pi32Args[1];
  sys_free(pi32Args);
  return ret.d;
}

static void CallNonvirtualVoidMethod (JNIEnv* env, jobject obj, jclass clazz, jmethodID methodID, ...)
{
  va_list  args;
  int32*   pi32Args;
  jobject  ex;

  va_start(args, methodID);
  pi32Args = GetArgsFromVaList(methodID, args, 1, 0);
  va_end(args);
  pi32Args[0] = (int32) obj;
  ex = INTERP_RunSpecialMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  sys_free(pi32Args);
  return;
}

static void CallNonvirtualVoidMethodA (JNIEnv* env, jobject obj, jclass clazz, jmethodID methodID, jvalue* args)
{
  int32*   pi32Args;
  jobject  ex;

  pi32Args = GetArgsFromJValues(methodID, args, 1, 0);
  pi32Args[0] = (int32) obj;
  ex = INTERP_RunSpecialMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  sys_free(pi32Args);
  return;
}

static void CallNonvirtualVoidMethodV (JNIEnv* env, jobject obj, jclass clazz, jmethodID methodID, va_list args)
{
  int32*   pi32Args;
  jobject  ex;

  pi32Args = GetArgsFromVaList(methodID, args, 1, 0);
  pi32Args[0] = (int32) obj;
  ex = INTERP_RunSpecialMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  sys_free(pi32Args);
  return;
}


/* Accessing Fields of Objects */
static jfieldID GetFieldID (JNIEnv* env, jclass clazz, const char* name, const char* sig)
{
  tClassLoaderTuple* pstClass;
  tField* pstField;

  pstClass = GETCLASSSTRUCT(clazz);
  pstField = CLASSFILE_InstFieldOffset(env, pstClass, (char*) name);
  if (pstField == NULL)
  {
    (*env)->Throw(env, INTERP_NewObjectFromName(env, "java/lang/NoSuchFieldException"));
    return NULL;
  }
  if(sig != NULL)
    {
      if(strcmp(sig, "") != 0)
	{
	  if(strcmp(sig, pstField->uidFieldSig) != 0)
	    {
	      return NULL; //doesn't match the signature
	    }
	}
    }
  return pstField;
}

static jobject GetObjectField (JNIEnv* env, jobject obj, jfieldID fieldID)
{
  jobject obj1 = (jobject) DEREF(obj)->pi32Vars[fieldID->u16Offset];
  if(obj1 != NULL)
    CreateLocalRef(  JNI_getJNIData( sys_current_thread()), obj1);
  return obj1;
}

static jboolean GetBooleanField (JNIEnv* env, jobject obj, jfieldID fieldID)
{
  return (jboolean) DEREF(obj)->pi32Vars[fieldID->u16Offset];
}

static jbyte GetByteField (JNIEnv* env, jobject obj, jfieldID fieldID)
{
  return (jbyte) DEREF(obj)->pi32Vars[fieldID->u16Offset];
}

static jchar GetCharField (JNIEnv* env, jobject obj, jfieldID fieldID)
{
  return (jchar) DEREF(obj)->pi32Vars[fieldID->u16Offset];
}

static jshort GetShortField (JNIEnv* env, jobject obj, jfieldID fieldID)
{
  return (jshort) DEREF(obj)->pi32Vars[fieldID->u16Offset];
}

static jint GetIntField (JNIEnv* env, jobject obj, jfieldID fieldID)
{
  return (jint) DEREF(obj)->pi32Vars[fieldID->u16Offset];
}

static jlong GetLongField (JNIEnv* env, jobject obj, jfieldID fieldID)
{
  jlong jlTemp;
  jlTemp = ((long long) DEREF(obj)->pi32Vars[fieldID->u16Offset]) << 32;
  jlTemp |= DEREF(obj)->pi32Vars[fieldID->u16Offset + 1];
  return jlTemp;
}

static jfloat GetFloatField (JNIEnv* env, jobject obj, jfieldID fieldID)
{
  return ((jfloat*) DEREF(obj)->pi32Vars)[fieldID->u16Offset];
}

static jdouble GetDoubleField (JNIEnv* env, jobject obj, jfieldID fieldID)
{
  tDConv dconv;
  dconv.i.hi = DEREF(obj)->pi32Vars[fieldID->u16Offset];
  dconv.i.lo = DEREF(obj)->pi32Vars[fieldID->u16Offset + 1];
  return dconv.d;
}

static void SetObjectField (JNIEnv* env, jobject obj, jfieldID fieldID, jobject value)
{
  DEREF(obj)->pi32Vars[fieldID->u16Offset] = (int32) value;
}

static void SetBooleanField (JNIEnv* env, jobject obj, jfieldID fieldID, jboolean value)
{
  DEREF(obj)->pi32Vars[fieldID->u16Offset] = (int32) value;
}

static void SetByteField (JNIEnv* env, jobject obj, jfieldID fieldID, jbyte value)
{
  DEREF(obj)->pi32Vars[fieldID->u16Offset] = (int32) value;
}

static void SetCharField (JNIEnv* env, jobject obj, jfieldID fieldID, jchar value)
{
  DEREF(obj)->pi32Vars[fieldID->u16Offset] = (int32) value;
}

static void SetShortField (JNIEnv* env, jobject obj, jfieldID fieldID, jshort value)
{
  DEREF(obj)->pi32Vars[fieldID->u16Offset] = (int32) value;
}

static void SetIntField (JNIEnv* env, jobject obj, jfieldID fieldID, jint value)
{
  DEREF(obj)->pi32Vars[fieldID->u16Offset] = (int32) value;
}

static void SetLongField (JNIEnv* env, jobject obj, jfieldID fieldID, jlong value)
{
  DEREF(obj)->pi32Vars[fieldID->u16Offset] = value>>32;
  DEREF(obj)->pi32Vars[fieldID->u16Offset + 1] = value&0xFFFFFFFF;
}

static void SetFloatField (JNIEnv* env, jobject obj, jfieldID fieldID, jfloat value)
{
  DEREF(obj)->pi32Vars[fieldID->u16Offset] = *((int32*) &value);
}

static void SetDoubleField (JNIEnv* env, jobject obj, jfieldID fieldID, jdouble value)
{
  tDConv dconv;
  dconv.d = value;
  DEREF(obj)->pi32Vars[fieldID->u16Offset] = dconv.i.hi;
  DEREF(obj)->pi32Vars[fieldID->u16Offset + 1] = dconv.i.lo;
}


/* Calling Static Methods */
static jmethodID GetStaticMethodID (JNIEnv* env, jclass clazz, const char* name, const char* sig)
{
  tClassLoaderTuple* pstClass;
  tMethod* pstMethod;

  pstClass = GETCLASSSTRUCT(clazz);
  pstMethod = CLASSFILE_FindMethod(env, pstClass, (char*) name, (char*) sig);

  if (pstMethod)
  {
    return pstMethod;
  }
  else
  {
    (*env)->Throw(env, INTERP_NewObjectFromName(env, "java/lang/NoSuchMethodError"));
    return NULL;
  }
}

static jobject CallStaticObjectMethod (JNIEnv* env, jclass clazz, jmethodID methodID, ...)

{
  va_list  args;
  int32*   pi32Args;
  jobject  ret;
  jobject  ex;

  assert((methodID->u16AccessFlags & ACC_STATIC));
  va_start(args, methodID);
  pi32Args = GetArgsFromVaList(methodID, args, 0, 1);
  va_end(args);
  ex = INTERP_RunStaticMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  ret = (jobject) pi32Args[0];
  sys_free(pi32Args);
  if (!ex)
  {
    CreateLocalRef(JNI_getJNIData(  sys_current_thread()), ret);
  }
  return ret;
}

static jobject CallStaticObjectMethodA (JNIEnv* env, jclass clazz, jmethodID methodID, jvalue* args)

{
  int32*   pi32Args;
  jobject  ret;
  jobject  ex;

  pi32Args = GetArgsFromJValues(methodID, args, 0, 1);
  ex = INTERP_RunStaticMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  ret = (jobject) pi32Args[0];
  sys_free(pi32Args);
  if (!ex)
  {
    CreateLocalRef(JNI_getJNIData(  sys_current_thread()), ret);
  }
  return ret;
}

static jobject CallStaticObjectMethodV (JNIEnv* env, jclass clazz, jmethodID methodID, va_list args)

{
  int32*   pi32Args;
  jobject  ret;
  jobject  ex;

  pi32Args = GetArgsFromVaList(methodID, args, 0, 1);
  ex = INTERP_RunStaticMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  ret = (jobject) pi32Args[0];
  sys_free(pi32Args);
  if (!ex)
  {
    CreateLocalRef(JNI_getJNIData(  sys_current_thread()), ret);
  }
  return ret;
}

static jboolean CallStaticBooleanMethod (JNIEnv* env, jclass clazz, jmethodID methodID, ...)

{
  va_list  args;
  int32*   pi32Args;
  jboolean ret;
  jobject  ex;

  va_start(args, methodID);
  pi32Args = GetArgsFromVaList(methodID, args, 0, 1);
  va_end(args);
  ex = INTERP_RunStaticMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  ret = (jboolean) pi32Args[0];
  sys_free(pi32Args);
  return ret;
}

static jboolean CallStaticBooleanMethodA (JNIEnv* env, jclass clazz, jmethodID methodID, jvalue* args)

{
  int32*   pi32Args;
  jboolean ret;
  jobject  ex;

  pi32Args = GetArgsFromJValues(methodID, args, 0, 1);
  ex = INTERP_RunStaticMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  ret = (jboolean) pi32Args[0];
  sys_free(pi32Args);
  return ret;
}

static jboolean CallStaticBooleanMethodV (JNIEnv* env, jclass clazz, jmethodID methodID, va_list args)

{
  int32*   pi32Args;
  jboolean ret;
  jobject  ex;

  pi32Args = GetArgsFromVaList(methodID, args, 0, 1);
  ex = INTERP_RunStaticMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  ret = (jboolean) pi32Args[0];
  sys_free(pi32Args);
  return ret;
}

static jbyte CallStaticByteMethod (JNIEnv* env, jclass clazz, jmethodID methodID, ...)

{
  va_list  args;
  int32*   pi32Args;
  jbyte    ret;
  jobject  ex;

  va_start(args, methodID);
  pi32Args = GetArgsFromVaList(methodID, args, 0, 1);
  va_end(args);
  ex = INTERP_RunStaticMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  ret = (jbyte) pi32Args[0];
  sys_free(pi32Args);
  return ret;
}

static jbyte CallStaticByteMethodA (JNIEnv* env, jclass clazz, jmethodID methodID, jvalue* args)

{
  int32*   pi32Args;
  jbyte    ret;
  jobject  ex;

  pi32Args = GetArgsFromJValues(methodID, args, 0, 1);
  ex = INTERP_RunStaticMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  ret = (jbyte) pi32Args[0];
  sys_free(pi32Args);
  return ret;
}

static jbyte CallStaticByteMethodV (JNIEnv* env, jclass clazz, jmethodID methodID, va_list args)

{
  int32*   pi32Args;
  jbyte    ret;
  jobject  ex;

  pi32Args = GetArgsFromVaList(methodID, args, 0, 1);
  ex = INTERP_RunStaticMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  ret = (jbyte) pi32Args[0];
  sys_free(pi32Args);
  return ret;
}

static jchar CallStaticCharMethod (JNIEnv* env, jclass clazz, jmethodID methodID, ...)

{
  va_list  args;
  int32*   pi32Args;
  jchar    ret;
  jobject  ex;

  va_start(args, methodID);
  pi32Args = GetArgsFromVaList(methodID, args, 0, 1);
  va_end(args);
  ex = INTERP_RunStaticMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  ret = (jchar) pi32Args[0];
  sys_free(pi32Args);
  return ret;
}

static jchar CallStaticCharMethodA (JNIEnv* env, jclass clazz, jmethodID methodID, jvalue* args)

{
  int32*   pi32Args;
  jchar    ret;
  jobject  ex;

  pi32Args = GetArgsFromJValues(methodID, args, 0, 1);
  ex = INTERP_RunStaticMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  ret = (jchar) pi32Args[0];
  sys_free(pi32Args);
  return ret;
}

static jchar CallStaticCharMethodV (JNIEnv* env, jclass clazz, jmethodID methodID, va_list args)

{
  int32*   pi32Args;
  jchar    ret;
  jobject  ex;

  pi32Args = GetArgsFromVaList(methodID, args, 0, 1);
  ex = INTERP_RunStaticMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  ret = (jchar) pi32Args[0];
  sys_free(pi32Args);
  return ret;
}

static jshort CallStaticShortMethod (JNIEnv* env, jclass clazz, jmethodID methodID, ...)

{
  va_list  args;
  int32*   pi32Args;
  jshort   ret;
  jobject  ex;

  va_start(args, methodID);
  pi32Args = GetArgsFromVaList(methodID, args, 0, 1);
  va_end(args);
  ex = INTERP_RunStaticMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  ret = (jshort) pi32Args[0];
  sys_free(pi32Args);
  return ret;
}

static jshort CallStaticShortMethodA (JNIEnv* env, jclass clazz, jmethodID methodID, jvalue* args)

{
  int32*   pi32Args;
  jshort   ret;
  jobject  ex;

  pi32Args = GetArgsFromJValues(methodID, args, 0, 1);
  ex = INTERP_RunStaticMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  ret = (jshort) pi32Args[0];
  sys_free(pi32Args);
  return ret;
}

static jshort CallStaticShortMethodV (JNIEnv* env, jclass clazz, jmethodID methodID, va_list args)

{
  int32*   pi32Args;
  jshort   ret;
  jobject  ex;

  pi32Args = GetArgsFromVaList(methodID, args, 0, 1);
  ex = INTERP_RunStaticMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  ret = (jshort) pi32Args[0];
  sys_free(pi32Args);
  return ret;
}

static jint CallStaticIntMethod (JNIEnv* env, jclass clazz, jmethodID methodID, ...)

{
  va_list  args;
  int32*   pi32Args;
  jint  ret;
  jobject  ex;

  va_start(args, methodID);
  pi32Args = GetArgsFromVaList(methodID, args, 0, 1);
  va_end(args);
  ex = INTERP_RunStaticMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  ret = (jint) pi32Args[0];
  sys_free(pi32Args);
  return ret;
}

static jint CallStaticIntMethodA (JNIEnv* env, jclass clazz, jmethodID methodID, jvalue* args)

{
  int32*   pi32Args;
  jint     ret;
  jobject  ex;

  pi32Args = GetArgsFromJValues(methodID, args, 0, 1);
  ex = INTERP_RunStaticMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  ret = (jint) pi32Args[0];
  sys_free(pi32Args);
  return ret;
}

static jint CallStaticIntMethodV (JNIEnv* env, jclass clazz, jmethodID methodID, va_list args)

{
  int32*   pi32Args;
  jint     ret;
  jobject  ex;

  pi32Args = GetArgsFromVaList(methodID, args, 0, 1);
  ex = INTERP_RunStaticMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  ret = (jint) pi32Args[0];
  sys_free(pi32Args);
  return ret;
}

static jlong CallStaticLongMethod (JNIEnv* env, jclass clazz, jmethodID methodID, ...)

{
  va_list  args;
  int32*   pi32Args;
  jlong    ret;
  jobject  ex;

  va_start(args, methodID);
  pi32Args = GetArgsFromVaList(methodID, args, 0, 2);
  va_end(args);
  ex = INTERP_RunStaticMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  ret = pi32Args[0];
  ret <<= 32;
  ret |= pi32Args[1];
  sys_free(pi32Args);
  return ret;
}

static jlong CallStaticLongMethodA (JNIEnv* env, jclass clazz, jmethodID methodID, jvalue* args)

{
  int32*   pi32Args;
  jlong    ret;
  jobject  ex;

  pi32Args = GetArgsFromJValues(methodID, args, 0, 2);
  ex = INTERP_RunStaticMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  ret = pi32Args[0];
  ret <<= 32;
  ret |= pi32Args[1];
  sys_free(pi32Args);
  return ret;
}

static jlong CallStaticLongMethodV (JNIEnv* env, jclass clazz, jmethodID methodID, va_list args)

{
  int32*   pi32Args;
  jlong    ret;
  jobject  ex;

  pi32Args = GetArgsFromVaList(methodID, args, 0, 2);
  ex = INTERP_RunStaticMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  ret = pi32Args[0];
  ret <<= 32;
  ret |= pi32Args[1];
  sys_free(pi32Args);
  return ret;
}

static jfloat CallStaticFloatMethod (JNIEnv* env, jclass clazz, jmethodID methodID, ...)

{
  va_list  args;
  int32*   pi32Args;
  jfloat  ret;
  jobject  ex;

  va_start(args, methodID);
  pi32Args = GetArgsFromVaList(methodID, args, 0, 1);
  va_end(args);
  ex = INTERP_RunStaticMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  ret = ((jfloat*) pi32Args)[0];
  sys_free(pi32Args);
  return ret;
}

static jfloat CallStaticFloatMethodA (JNIEnv* env, jclass clazz, jmethodID methodID, jvalue* args)

{
  int32*   pi32Args;
  jfloat   ret;
  jobject  ex;

  pi32Args = GetArgsFromJValues(methodID, args, 0, 1);
  ex = INTERP_RunStaticMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  ret = ((jfloat*) pi32Args)[0];
  sys_free(pi32Args);
  return ret;
}

static jfloat CallStaticFloatMethodV (JNIEnv* env, jclass clazz, jmethodID methodID, va_list args)

{
  int32*   pi32Args;
  jfloat   ret;
  jobject  ex;

  pi32Args = GetArgsFromVaList(methodID, args, 0, 1);
  ex = INTERP_RunStaticMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  ret = ((jfloat*) pi32Args)[0];
  sys_free(pi32Args);
  return ret;
}

static jdouble CallStaticDoubleMethod (JNIEnv* env, jclass clazz, jmethodID methodID, ...)

{
  va_list  args;
  int32*   pi32Args;
  tDConv   ret;
  jobject  ex;

  va_start(args, methodID);
  pi32Args = GetArgsFromVaList(methodID, args, 0, 2);
  va_end(args);
  ex = INTERP_RunStaticMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  ret.i.hi = pi32Args[0];
  ret.i.lo = pi32Args[1];
  sys_free(pi32Args);
  return ret.d;
}

static jdouble CallStaticDoubleMethodA (JNIEnv* env, jclass clazz, jmethodID methodID, jvalue* args)

{
  int32*   pi32Args;
  tDConv   ret;
  jobject  ex;

  pi32Args = GetArgsFromJValues(methodID, args, 0, 2);
  ex = INTERP_RunStaticMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  ret.i.hi = pi32Args[0];
  ret.i.lo = pi32Args[1];
  sys_free(pi32Args);
  return ret.d;
}

static jdouble CallStaticDoubleMethodV (JNIEnv* env, jclass clazz, jmethodID methodID, va_list args)

{
  int32*   pi32Args;
  tDConv   ret;
  jobject  ex;

  pi32Args = GetArgsFromVaList(methodID, args, 0, 2);
  ex = INTERP_RunStaticMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  ret.i.hi = pi32Args[0];
  ret.i.lo = pi32Args[1];
  sys_free(pi32Args);
  return ret.d;
}

static void CallStaticVoidMethod (JNIEnv* env, jclass clazz, jmethodID methodID, ...)

{
  va_list  args;
  int32*   pi32Args;
  jobject  ex;

  va_start(args, methodID);

  pi32Args = GetArgsFromVaList(methodID, args, 0, 0);
  va_end(args);
  ex = INTERP_RunStaticMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  sys_free(pi32Args);
  return;
}

static void CallStaticVoidMethodA (JNIEnv* env, jclass clazz, jmethodID methodID, jvalue* args)

{
  int32*   pi32Args;
  jobject  ex;

//iStart is set to 0, because that seems to suit the Void and static case

  pi32Args = GetArgsFromJValues(methodID, args, 0, 0);
  ex = INTERP_RunStaticMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  sys_free(pi32Args);
  return;
}

static void CallStaticVoidMethodV (JNIEnv* env, jclass clazz, jmethodID methodID, va_list args)

{
  int32*   pi32Args;
  jobject  ex;

  //iStart should be 1 !!!

  pi32Args = GetArgsFromVaList(methodID, args, 0, 0);
  ex = INTERP_RunStaticMethodFromPtr(env, methodID, pi32Args);
  if (ex)
  {
    (*env)->Throw(env, ex);
  }
  sys_free(pi32Args);
  return;
}


/* Accessing Static Fields */
static jfieldID GetStaticFieldID (JNIEnv* env, jclass clazz, const char* name, const char* sig)
{
  tClassLoaderTuple* pstClass;

  pstClass = GETCLASSSTRUCT(clazz);
  return CLASSFILE_StatFieldOffset(env, pstClass, (char*) name);
}

static jobject GetStaticObjectField (JNIEnv* env, jclass clazz, jfieldID fieldID)
{
  jobject obj = (jobject) GETCLASSSTRUCT(clazz)->pi32StatVars[fieldID->u16Offset];
  CreateLocalRef(JNI_getJNIData(  sys_current_thread()), obj);
  return obj;
}

static jboolean GetStaticBooleanField (JNIEnv* env, jclass clazz, jfieldID fieldID)
{
  return (jboolean) GETCLASSSTRUCT(clazz)->pi32StatVars[fieldID->u16Offset];
}

static jbyte GetStaticByteField (JNIEnv* env, jclass clazz, jfieldID fieldID)
{
  return (jbyte) GETCLASSSTRUCT(clazz)->pi32StatVars[fieldID->u16Offset];
}

static jchar GetStaticCharField (JNIEnv* env, jclass clazz, jfieldID fieldID)
{
  return (jchar) GETCLASSSTRUCT(clazz)->pi32StatVars[fieldID->u16Offset];
}

static jshort GetStaticShortField (JNIEnv* env, jclass clazz, jfieldID fieldID)
{
  return (jshort) GETCLASSSTRUCT(clazz)->pi32StatVars[fieldID->u16Offset];
}

static jint GetStaticIntField (JNIEnv* env, jclass clazz, jfieldID fieldID)
{
  return (jint) GETCLASSSTRUCT(clazz)->pi32StatVars[fieldID->u16Offset];
}

static jlong GetStaticLongField (JNIEnv* env, jclass clazz, jfieldID fieldID)
{
  jlong jlTemp;
  jlTemp = GETCLASSSTRUCT(clazz)->pi32StatVars[fieldID->u16Offset];
  jlTemp <<= 32;
  jlTemp |= GETCLASSSTRUCT(clazz)->pi32StatVars[fieldID->u16Offset + 1];
  return jlTemp;
}

static jfloat GetStaticFloatField (JNIEnv* env, jclass clazz, jfieldID fieldID)
{
  return ((jfloat*) GETCLASSSTRUCT(clazz)->pi32StatVars)[fieldID->u16Offset];
}

static jdouble GetStaticDoubleField (JNIEnv* env, jclass clazz, jfieldID fieldID)
{
  tDConv dconv;
  dconv.i.hi = GETCLASSSTRUCT(clazz)->pi32StatVars[fieldID->u16Offset];
  dconv.i.lo = GETCLASSSTRUCT(clazz)->pi32StatVars[fieldID->u16Offset + 1];
  return dconv.d;
}

static void SetStaticObjectField (JNIEnv* env, jclass clazz, jfieldID fieldID, jobject value)
{
  GETCLASSSTRUCT(clazz)->pi32StatVars[fieldID->u16Offset] = (int32) value;
}

static void SetStaticBooleanField (JNIEnv* env, jclass clazz, jfieldID fieldID, jboolean value)
{
  GETCLASSSTRUCT(clazz)->pi32StatVars[fieldID->u16Offset] = (int32) value;
}

static void SetStaticByteField (JNIEnv* env, jclass clazz, jfieldID fieldID, jbyte value)
{
  GETCLASSSTRUCT(clazz)->pi32StatVars[fieldID->u16Offset] = (int32) value;
}

static void SetStaticCharField (JNIEnv* env, jclass clazz, jfieldID fieldID, jchar value)
{
  GETCLASSSTRUCT(clazz)->pi32StatVars[fieldID->u16Offset] = (int32) value;
}

static void SetStaticShortField (JNIEnv* env, jclass clazz, jfieldID fieldID, jshort value)
{
  GETCLASSSTRUCT(clazz)->pi32StatVars[fieldID->u16Offset] = (int32) value;
}

static void SetStaticIntField (JNIEnv* env, jclass clazz, jfieldID fieldID, jint value)
{
  GETCLASSSTRUCT(clazz)->pi32StatVars[fieldID->u16Offset] = (int32) value;
}

static void SetStaticLongField (JNIEnv* env, jclass clazz, jfieldID fieldID, jlong value)
{
  GETCLASSSTRUCT(clazz)->pi32StatVars[fieldID->u16Offset] = value>>32;
  GETCLASSSTRUCT(clazz)->pi32StatVars[fieldID->u16Offset] = value&0xFFFFFFFF;
}

static void SetStaticFloatField (JNIEnv* env, jclass clazz, jfieldID fieldID, jfloat value)
{
  GETCLASSSTRUCT(clazz)->pi32StatVars[fieldID->u16Offset] = *((int32*) &value);
}

static void SetStaticDoubleField (JNIEnv* env, jclass clazz, jfieldID fieldID, jdouble value)
{
  tDConv dconv;
  dconv.d = value;
  GETCLASSSTRUCT(clazz)->pi32StatVars[fieldID->u16Offset] = dconv.i.hi;
  GETCLASSSTRUCT(clazz)->pi32StatVars[fieldID->u16Offset] = dconv.i.lo;
}

/* String Operations */
static jstring NewString (JNIEnv* env, const jchar* unicodeChars, jsize len)
/* throws things */
{
  jstring string;
  tARREF pstArrayTemp;
  jsize i;
  uint16 u16Temp;

  pstArrayTemp = INTERP_NewArray(env, T_CHAR, len, NULL, pstClassType ); //XXX System class?
  for (i = 0; i < len; i++)
  {
    ((jchar*) ADEREF(pstArrayTemp)->pvElements)[i] = unicodeChars[i];
  }

  string = INTERP_NewObjectFromName(env, "java/lang/String");

  u16Temp = CLASSFILE_NativeInstOffset(env, pstClassType, "java/lang/String", "value");
  DEREF(string)->pi32Vars[u16Temp] = (int32) pstArrayTemp;
  u16Temp = CLASSFILE_NativeInstOffset(env, pstClassType, "java/lang/String", "offset");
  DEREF(string)->pi32Vars[u16Temp] = 0;
  u16Temp = CLASSFILE_NativeInstOffset(env, pstClassType, "java/lang/String", "count");
  DEREF(string)->pi32Vars[u16Temp] = len;
  CreateLocalRef(JNI_getJNIData(  sys_current_thread()), string);
  return string;
}

static jsize GetStringLength (JNIEnv* env, jstring string)
{
  uint16 u16Temp;

  u16Temp = CLASSFILE_NativeInstOffset(env, pstClassType, "java/lang/String", "count");
  return DEREF(string)->pi32Vars[u16Temp];
}

static const jchar* GetStringChars (JNIEnv* env, jstring string, jboolean* isCopy)
{
  uint16 u16Temp;
  int32 i32Offset;
  tArray* pstArray;

  u16Temp = CLASSFILE_NativeInstOffset(env, pstClassType, "java/lang/String", "offset");
  i32Offset = DEREF(string)->pi32Vars[u16Temp];
  u16Temp = CLASSFILE_NativeInstOffset(env, pstClassType, "java/lang/String", "value");
  pstArray = (tArray*) DEREF(string)->pi32Vars[u16Temp];

  if (isCopy)
  {
    *isCopy = JNI_FALSE;
  }

  return ((jchar*) pstArray->pvElements) + i32Offset;
}

static void ReleaseStringChars (JNIEnv* env, jstring string, const jchar* chars)
{
  sys_free( (char*)chars);
  return;
}

static jstring NewStringUTF (JNIEnv* env, const char* bytes)
/* throws things */
{
  jstring string = INTERP_NewStringFromAsciz(env, (char*) bytes);
  CreateLocalRef(  JNI_getJNIData( sys_current_thread()), string);
  return string;
}

static jsize GetStringUTFLength (JNIEnv* env, jstring string)
{
  char* pszTempString;
  jsize len;

  pszTempString = INTERP_AscizFromString(env, string);
  len = strlen(pszTempString);
  sys_free(pszTempString);
  return len;
}

static const jbyte* GetStringUTFChars (JNIEnv* env, jstring string, jboolean* isCopy)
{
  if (isCopy)
  {
    *isCopy = JNI_TRUE;
  }
  return INTERP_AscizFromString(env, string);
}

static void ReleaseStringUTFChars (JNIEnv* env, jstring string, const char* chars)
{
  sys_free( (char*) chars);
  return;
}

/* Array Operations */
static jsize GetArrayLength (JNIEnv* env, jarray array)
{
  return ADEREF(array)->i32Number;
}

static jarray NewObjectArray (JNIEnv* env, jsize length, jclass elementClass, jobject initialElement)
/* throws things */
{
  jarray pstArray;
  tClassLoaderTuple* pstClass;
  jsize i;
  
  if(length < 0)
    {
      (*env)->ThrowNew(env, (*env)->FindClass(env, "java/lang/NegativeArraySizeException"), "negative array size passed to NewObjectArray");
      return NULL;
    }

  pstClass = GETCLASSSTRUCT(elementClass);
  //  fprintf(stderr, "NewObjectArray: length %i elementClass %x, type %s\n", length, elementClass, pstClass->uidName);
  if(pstClass->uidName[0] == '[')
    pstArray = (jarray) INTERP_NewArray(env, T_ARRAY, length, pstClass->uidName, pstClass);
  else
    pstArray = (jarray) INTERP_NewArray(env, T_OBJECT, length, pstClass->uidName, pstClass);

  if(pstArray == NULL)
    return NULL;

  for (i = 0; i < length; i++)
  {
    ((jobject*) ADEREF(pstArray)->pvElements)[i] = initialElement;
  }

  CreateLocalRef(  JNI_getJNIData( sys_current_thread()), (jobject) pstArray);
  return pstArray;
}

static jobject GetObjectArrayElement (JNIEnv* env, jobjectArray array, jsize index)
{
  jobject obj = ((jobject*) ADEREF(array)->pvElements)[index];
  if(obj)
    CreateLocalRef(JNI_getJNIData(sys_current_thread()), obj);
  return obj;
}

static void SetObjectArrayElement (JNIEnv* env, jobjectArray array, jsize index, jobject value)
/* throws things */
{
  if( (index < 0) || (index >= ADEREF(array)->i32Number))
    {
      (*env)->Throw(env, INTERP_NewObjectFromName(env, "java/lang/ArrayIndexOutOfBoundsException"));
      return;
    }
  else
    {
      if(value != NULL)
	{
	  if(!(GARBAGE_InHeap(env, value)))
	    {
	      (*env)->Throw(env, INTERP_NewObjectFromName(env, "java/lang/InternalError"));
	      return;
	    }
	}
    ((jobject*) ADEREF(array)->pvElements)[index] = value;
    }
}

static jbooleanArray NewBooleanArray (JNIEnv* env, jsize length)
{
  jbooleanArray a = (jbooleanArray) INTERP_NewArray(env, T_BOOLEAN, length, NULL, pstClassType);
  CreateLocalRef(  JNI_getJNIData( sys_current_thread()), (jobject) a);
  return a;
}

static jbyteArray NewByteArray (JNIEnv* env, jsize length)
{
  jbyteArray a = (jbyteArray) INTERP_NewArray(env, T_BYTE, length, NULL, pstClassType);
  CreateLocalRef(  JNI_getJNIData( sys_current_thread()), (jobject) a);
  return a;
}

static jcharArray NewCharArray (JNIEnv* env, jsize length)
{
  jcharArray a = (jcharArray) INTERP_NewArray(env, T_CHAR, length, NULL, pstClassType);
  CreateLocalRef(   JNI_getJNIData( sys_current_thread()), (jobject) a);
  return a;
}

static jshortArray NewShortArray (JNIEnv* env, jsize length)
{
  jshortArray a = (jshortArray) INTERP_NewArray(env, T_SHORT, length, NULL, pstClassType);
  CreateLocalRef(  JNI_getJNIData( sys_current_thread()), (jobject) a);
  return a;
}

static jintArray NewIntArray (JNIEnv* env, jsize length)
{
  jintArray a = (jintArray) INTERP_NewArray(env, T_INT, length, NULL, pstClassType);
  CreateLocalRef(  JNI_getJNIData( sys_current_thread()), (jobject) a);
  return a;
}

static jlongArray NewLongArray (JNIEnv* env, jsize length)
{
  jlongArray a = (jlongArray) INTERP_NewArray(env, T_LONG, length, NULL, pstClassType);
  CreateLocalRef(  JNI_getJNIData( sys_current_thread()), (jobject) a);
  return a;
}

static jfloatArray NewFloatArray (JNIEnv* env, jsize length)
{
  jfloatArray a = (jfloatArray) INTERP_NewArray(env, T_FLOAT, length, NULL, pstClassType);
  CreateLocalRef(  JNI_getJNIData( sys_current_thread()), (jobject) a);
  return a;
}

static jdoubleArray NewDoubleArray (JNIEnv* env, jsize length)
{
  jdoubleArray a = (jdoubleArray) INTERP_NewArray(env, T_DOUBLE, length, NULL, pstClassType);
  CreateLocalRef(  JNI_getJNIData( sys_current_thread()), (jobject) a);
  return a;
}


static jboolean* GetBooleanArrayElements (JNIEnv* env, jbooleanArray array, jboolean* isCopy)
{
  if (isCopy)
  {
    *isCopy = JNI_FALSE;
  }
  return (jboolean*) ADEREF(array)->pvElements;
}

static jbyte* GetByteArrayElements (JNIEnv* env, jbyteArray array, jboolean* isCopy)
{
  if (isCopy)
  {
    *isCopy = JNI_FALSE;
  }
  if(ADEREF(array)->pvElements == NULL)
    {
      eprintf("Internal error in GetByteArrayElements, pointer is NULL for %p\n", array);
    }
  return (jbyte*) ADEREF(array)->pvElements;
}

static jchar* GetCharArrayElements (JNIEnv* env, jcharArray array, jboolean* isCopy)
{
  if (isCopy)
  {
    *isCopy = JNI_FALSE;
  }
  return (jchar*) ADEREF(array)->pvElements;
}

static jshort* GetShortArrayElements (JNIEnv* env, jshortArray array, jboolean* isCopy)
{
  if (isCopy)
  {
    *isCopy = JNI_FALSE;
  }
  return (jshort*) ADEREF(array)->pvElements;
}

static jint* GetIntArrayElements (JNIEnv* env, jintArray array, jboolean* isCopy)
{
  if (isCopy)
  {
    *isCopy = JNI_FALSE;
  }
  return (jint*) ADEREF(array)->pvElements;
}

static jlong* GetLongArrayElements (JNIEnv* env, jlongArray array, jboolean* isCopy)
{
  if (isCopy)
  {
    *isCopy = JNI_FALSE;
  }
  return (jlong*) ADEREF(array)->pvElements;
}

static jfloat* GetFloatArrayElements (JNIEnv* env, jfloatArray array, jboolean* isCopy)
{
  if (isCopy)
  {
    *isCopy = JNI_FALSE;
  }
  return (jfloat*) ADEREF(array)->pvElements;
}

static jdouble* GetDoubleArrayElements (JNIEnv* env, jdoubleArray array, jboolean* isCopy)
{
  if (isCopy)
  {
    *isCopy = JNI_FALSE;
  }
  return (jdouble*) ADEREF(array)->pvElements;
}

static void ReleaseBooleanArrayElements (JNIEnv* env, jbooleanArray array, jboolean* elems, jint mode)
{
  return;
}

static void ReleaseByteArrayElements (JNIEnv* env, jbyteArray array, jbyte* elems, jint mode)
{
  return;
}

static void ReleaseCharArrayElements (JNIEnv* env, jcharArray array, jchar* elems, jint mode)
{
  return;
}

static void ReleaseShortArrayElements (JNIEnv* env, jshortArray array, jshort* elems, jint mode)
{
  return;
}

static void ReleaseIntArrayElements (JNIEnv* env, jintArray array, jint* elems, jint mode)
{
  return;
}

static void ReleaseLongArrayElements (JNIEnv* env, jlongArray array, jlong* elems, jint mode)
{
  return;
}

static void ReleaseFloatArrayElements (JNIEnv* env, jfloatArray array, jfloat* elems, jint mode)
{
  return;
}

static void ReleaseDoubleArrayElements (JNIEnv* env, jdoubleArray array, jdouble* elems, jint mode)
{
  return;
}


static void GetBooleanArrayRegion (JNIEnv* env, jbooleanArray array, jsize start, jsize len, jboolean* buf)
{
  memcpy(buf, ((jboolean*) ADEREF(array)->pvElements) + start, sizeof(jboolean) * len);
}

static void GetByteArrayRegion (JNIEnv* env, jbyteArray array, jsize start, jsize len, jbyte* buf)
{
  memcpy(buf, ((jbyte*) ADEREF(array)->pvElements) + start, sizeof(jbyte) * len);
}

static void GetCharArrayRegion (JNIEnv* env, jcharArray array, jsize start, jsize len, jchar* buf)
{
  memcpy(buf, ((jchar*) ADEREF(array)->pvElements) + start, sizeof(jchar) * len);
}

static void GetShortArrayRegion (JNIEnv* env, jshortArray array, jsize start, jsize len, jshort* buf)
{
  memcpy(buf, ((jshort*) ADEREF(array)->pvElements) + start, sizeof(jshort) * len);
}

static void GetIntArrayRegion (JNIEnv* env, jintArray array, jsize start, jsize len, jint* buf)
{
  memcpy(buf, ((jint*) ADEREF(array)->pvElements) + start, sizeof(jint) * len);
}

static void GetLongArrayRegion (JNIEnv* env, jlongArray array, jsize start, jsize len, jlong* buf)
{
  memcpy(buf, ((jlong*) ADEREF(array)->pvElements) + start, sizeof(jlong) * len);
}

static void GetFloatArrayRegion (JNIEnv* env, jfloatArray array, jsize start, jsize len, jfloat* buf)
{
  memcpy(buf, ((jfloat*) ADEREF(array)->pvElements) + start, sizeof(jfloat) * len);
}

static void GetDoubleArrayRegion (JNIEnv* env, jdoubleArray array, jsize start, jsize len, jdouble* buf)
{
  memcpy(buf, ((jdouble*) ADEREF(array)->pvElements) + start, sizeof(jdouble) * len);
}


static void SetBooleanArrayRegion (JNIEnv* env, jbooleanArray array, jsize start, jsize len, jboolean* buf)
{
  memcpy(((jboolean*) ADEREF(array)->pvElements) + start, buf, sizeof(jboolean) * len);
}

static void SetByteArrayRegion (JNIEnv* env, jbyteArray array, jsize start, jsize len, jbyte* buf)
{
  memcpy(((jbyte*) ADEREF(array)->pvElements) + start, buf, sizeof(jbyte) * len);
}

static void SetCharArrayRegion (JNIEnv* env, jcharArray array, jsize start, jsize len, jchar* buf)
{
  memcpy(((jchar*) ADEREF(array)->pvElements) + start, buf, sizeof(jchar) * len);
}

static void SetShortArrayRegion (JNIEnv* env, jshortArray array, jsize start, jsize len, jshort* buf)
{
  memcpy(((jshort*) ADEREF(array)->pvElements) + start, buf, sizeof(jshort) * len);
}

static void SetIntArrayRegion (JNIEnv* env, jintArray array, jsize start, jsize len, jint* buf)
{
  memcpy(((jint*) ADEREF(array)->pvElements) + start, buf, sizeof(jint) * len);
}

static void SetLongArrayRegion (JNIEnv* env, jlongArray array, jsize start, jsize len, jlong* buf)
{
  memcpy(((jlong*) ADEREF(array)->pvElements) + start, buf, sizeof(jlong) * len);
}

static void SetFloatArrayRegion (JNIEnv* env, jfloatArray array, jsize start, jsize len, jfloat* buf)
{
  memcpy(((jfloat*) ADEREF(array)->pvElements) + start, buf, sizeof(jfloat) * len);
}

static void SetDoubleArrayRegion (JNIEnv* env, jdoubleArray array, jsize start, jsize len, jdouble* buf)
{
  memcpy(((jdouble*) ADEREF(array)->pvElements) + start, buf, sizeof(jdouble) * len);
}

#ifndef GARBAGE_GENERATIONAL
#ifdef PERSIST
static tOBREF SwizzleObjectField (JNIEnv* env, jobject obj, jfieldID field)
{
  /* only do something if it's a reference field */
  switch (field->uidFieldSig[0])
  {
    case 'L':
    case '[':
    {
      if (ISPID(DEREF(obj)->pi32Vars[field->u16Offset]))
      {
        DEREF(obj)->pi32Vars[field->u16Offset] = (int32) RPOT_FaultObject(DEREF(obj)->pi32Vars[field->u16Offset]);
      }
      return (tOBREF) DEREF(obj)->pi32Vars[field->u16Offset];
    }
    default:
    {
      return NULL;
    }
  }
}

static tOBREF SwizzleArrayElement (JNIEnv* env, jobjectArray array, jint element)
{
  /* only do something if it's a reference array */
  switch (ADEREF(array)->enType)
  {
    case T_ARRAY:
    case T_OBJECT:
    {
      if (ISPID(((PID*) ADEREF(array)->pvElements)[element]))
      {
        ((PID*) ADEREF(array)->pvElements)[element] = (int32) RPOT_FaultObject(((PID*) ADEREF(array)->pvElements)[element]);
      }
      return (tOBREF) ((PID*) ADEREF(array)->pvElements)[element];
    }
    default:
    {
      return NULL;
    }
  }
}

static void MarkUpdated (JNIEnv* env, jobject obj)
{
  DEREF(obj)->i32Flags |= PERSIST_Dirty;
}

static void MarkRefsUpdated (JNIEnv* env, jobject obj)
{
  DEREF(obj)->i32Flags |= PERSIST_RefsDirty;
}
#endif
#endif //GC GEN

/* Regsitering Native Methods */
static jint RegisterNatives (JNIEnv* env, jclass clazz, const JNINativeMethod* methods, jint nMethods)
{
  int32 i, j;
  tClassLoaderTuple* pstClass = GETCLASSSTRUCT(clazz);
  tMethod* pstMethod;

  for (i = 0, j = 0; j < nMethods; i++)
  {
    if (pstClass->pstClass->pstNativeMethods[i].fnPtr == NULL)
    {
      /* must check each of the methods in the new array - put it here
         so that it happens once for each j */
      pstMethod = CLASSFILE_FindMethod(env, pstClass, methods[j].name, methods[j].signature);
      if ((pstMethod == NULL) || ((pstMethod->u16AccessFlags & ACC_NATIVE) == 0))
      {
        (*env)->Throw(env, INTERP_NewObjectFromName(env, "java/lang/NoSuchMethodError"));
        return -1;
      }

      #ifdef PRELOAD
      if (pstClass->bPreloaded)
      {
        INTERP_pstGlobalNativeMethods[pstClass->i32NativeMethodsOffset + j].name = UID_GetUid(methods[j].name);
        INTERP_pstGlobalNativeMethods[pstClass->i32NativeMethodsOffset + j].signature = UID_GetUid(methods[j].signature);
        INTERP_pstGlobalNativeMethods[pstClass->i32NativeMethodsOffset + j].fnPtr = methods[j].fnPtr;
      }
      else
      {
        pstClass->pstNativeMethods[j].name = UID_GetUid(methods[j].name);
        pstClass->pstNativeMethods[j].signature = UID_GetUid(methods[j].signature);
        pstClass->pstNativeMethods[j].fnPtr = methods[j].fnPtr;
      }
      #else
      pstClass->pstClass->pstNativeMethods[j].name = UID_GetUid(methods[j].name);
      pstClass->pstClass->pstNativeMethods[j].signature = UID_GetUid(methods[j].signature);
      pstClass->pstClass->pstNativeMethods[j].fnPtr = methods[j].fnPtr;
      #endif
      j += 1;
    }
  }

  return 0;
}

static jint UnregisterNatives (JNIEnv* env, jclass clazz, const JNINativeMethod* methods, jint nMethods)
{
  int32 i, j;
  tClassLoaderTuple* pstClass = GETCLASSSTRUCT(clazz);

  for (i = 0; i < nMethods; i++)
  {
    for (j = 0; j < pstClass->pstClass->u16NumNatives; j++)
    {
      #ifdef PRELOAD
      if (pstClass->bPreloaded)
      {
        if (pstClass->pstNativeMethods[j].fnPtr)
        {
          if ((strcmp(pstClass->pstNativeMethods[j].name, methods[i].name) == 0) &&
              (strcmp(pstClass->pstNativeMethods[j].signature, methods[i].signature) == 0))
          {
            pstClass->pstNativeMethods[j].name = NULL;
            pstClass->pstNativeMethods[j].signature = NULL;
            pstClass->pstNativeMethods[j].fnPtr = NULL;
            break;
          }
        }
      }
      else
      {
        if (INTERP_pstGlobalNativeMethods[pstClass->i32NativeMethodsOffset + j].fnPtr)
        {
          if ((strcmp(INTERP_pstGlobalNativeMethods[pstClass->i32NativeMethodsOffset + j].name, methods[i].name) == 0) &&
              (strcmp(INTERP_pstGlobalNativeMethods[pstClass->i32NativeMethodsOffset + j].signature, methods[i].signature) == 0))
          {
            INTERP_pstGlobalNativeMethods[pstClass->i32NativeMethodsOffset + j].name = NULL;
            INTERP_pstGlobalNativeMethods[pstClass->i32NativeMethodsOffset + j].signature = NULL;
            INTERP_pstGlobalNativeMethods[pstClass->i32NativeMethodsOffset + j].fnPtr = NULL;
            break;
          }
        }
      }
      #else
      if (pstClass->pstClass->pstNativeMethods[j].fnPtr)
      {
        if ((strcmp(pstClass->pstClass->pstNativeMethods[j].name, methods[i].name) == 0) &&
            (strcmp(pstClass->pstClass->pstNativeMethods[j].signature, methods[i].signature) == 0))
        {
          pstClass->pstClass->pstNativeMethods[j].name = NULL;
          pstClass->pstClass->pstNativeMethods[j].signature = NULL;
          pstClass->pstClass->pstNativeMethods[j].fnPtr = NULL;
          break;
        }
      }
      #endif
    }
  }
  return 0;
}

/* Monitor Operations */
static jint MonitorEnter (JNIEnv* env, jobject obj)
{
  THREAD_SynchroniseEnter(obj);

//  eprintf( "MonitorEnter not supported from native code. Enter the monitor in the calling Java method.\n");
//  exit(-1);
  return 0; /* to keep compiler happy */
}

static jint MonitorExit (JNIEnv* env, jobject obj)
{
  THREAD_SynchroniseExit(obj);
//  eprintf( "MonitorExit not supported from native code. Enter the monitor in the calling Java method.\n");
//  exit(-1);
  return 0; /* to keep compiler happy */
}

/* Java VM Interface */
static jint GetJavaVM (JNIEnv* env, JavaVM** vm)
{
  return 0; /* do nothing for the moment */
}

void GetStringRegion(JNIEnv *env, jstring str, jsize start, jsize len, jchar *buf)
{
  assert(1 == 7);
  return ;
}
void GetStringUTFRegion(JNIEnv *env, jstring str, jsize start, jsize len, char *buf)
{
  return;
}

void* GetPrimitiveArrayCritical(JNIEnv *env, jarray array, jboolean *isCopy)
{
  assert(1 == 7);
  return NULL;
}
void ReleasePrimitiveArrayCritical(JNIEnv *env, jarray array, void *carray, jint mode)
{
  assert(1 == 7);
}
 
const jchar* GetStringCritical(JNIEnv *env, jstring string, jboolean *isCopy)
{
  assert(1 == 7);
  return NULL;
}

void ReleaseStringCritical(JNIEnv *env, jstring string, const jchar *cstring)
{
  return ;
}
jweak NewWeakGlobalRef(JNIEnv *env, jobject obj)
{
  assert(1 == 7);
  return NULL;
}
void DeleteWeakGlobalRef(JNIEnv *env, jweak ref)
{
  assert(1 == 7);
  return;
}
jboolean ExceptionCheck(JNIEnv *env)
{
  assert(1 == 2);
  return JNI_FALSE;
}


/*
 * @doc FUNC
 * @func
 * This function creates an array of int32s holding the parameters for a
 * Java method. The parameters are taken from a va_list and put into
 * the int32 array. The iStart parameter specifies which slot in the int32
 * array to use first (non-static methods need to put the "this" pointer in
 * slot 0 so it must be kept free). The minimum size of the array is also
 * specified. This is needed because return values are put back into the array
 * after the invocation of the method and there needs to be enough space for
 * them - so void functions pass 0 in iMinSize, long and double functions pass
 * 2 and the rest pass 1.
 *
 */

static int32* GetArgsFromVaList
(
  tMethod* pstMethod, /* @parm The method for which the arguments are being created */
  va_list args,       /* @parm The va_list from which to take the arguments */
  int iStart,         /* @parm Which slot in the int32 array to start in */
  int iMinSize        /* @parm The minimum size of the int32 array */
)
{
  int32* pi32Args;
  int i;
  int j;
  tDConv dconv;
  jlong jlTemp;

  /* argument array must be at least iMinSize int32s big */
  if (pstMethod->u16ArgSize + iStart < iMinSize)
  {
    pi32Args = (int32*) sys_malloc(sizeof(int32) * iMinSize);
  }
  else
  {
    pi32Args = (int32*) sys_malloc(sizeof(int32) * (pstMethod->u16ArgSize + iStart));
  }

  for (i = 1, j = iStart; pstMethod->uidSignature[i] != ')'; i++, j++)
  {
    switch (pstMethod->uidSignature[i])
    {
      case 'Z': /* these all get promoted to int (I hope) */
      case 'B':
      case 'C':
      case 'S':
      {
        pi32Args[j] = va_arg(args, int);
        break;
      }
      case 'I':
      {
        pi32Args[j] = va_arg(args, int32);
        break;
      }
      case 'F':
      {
        /* float gets promoted to double */
        ((float*) pi32Args)[j] = (float) va_arg(args, double);
        break;
      }
      case 'D':
      {
        dconv.d = va_arg(args, double);
        pi32Args[j++] = dconv.i.hi;
        pi32Args[j] = dconv.i.lo;
        break;
      }
      case 'J':
      {
        jlTemp = va_arg(args, jlong);
        pi32Args[j++] = jlTemp>>32;
        pi32Args[j] = jlTemp&0xFFFFFFFF;
        break;
      }
      case 'L':
      {
        pi32Args[j] = (int32) va_arg(args, tOBREF);
        for (; pstMethod->uidSignature[i] != ';'; i++);
        break;
      }
      case '[':
      {
        pi32Args[j] = (int32) va_arg(args, tOBREF);
        while ((pstMethod->uidSignature[i] == '[' || (pstMethod->uidSignature[i] >= '0' && pstMethod->uidSignature[i] <= '9')))
        {
          i++;
        }
        if (pstMethod->uidSignature[i] == 'L')
        {
          while (pstMethod->uidSignature[i] != ';')
          {
            i++;
          }
        }
        break;
      }
    }
  }

  return pi32Args;
}

/*
 * @doc FUNC
 * @func
 * This function creates an array of int32s holding the parameters for a
 * Java method. The parameters are taken from an array of jvalues and put into
 * the int32 array. The iStart parameter specifies which slot in the int32
 * array to use first (non-static methods need to put the "this" pointer in
 * slot 0 so it must be kept free). The minimum size of the array is also
 * specified. This is needed because return values are put back into the array
 * after the invocation of the method and there needs to be enough space for
 * them - so void functions pass 0 in iMinSize, long and double functions pass
 * 2 and the rest pass 1.
 *
 */

static int32* GetArgsFromJValues
(
  tMethod* pstMethod, /* @parm The method for which the arguments are being created */
  jvalue* args,       /* @parm Pointer to the jvalue array */
  int iStart,         /* @parm Which slot in the int32 array to start in */
  int iMinSize        /* @parm The minimum size of the int32 array */
)
{
  int32* pi32Args;
  int i;
  int j;
  int k;
  tDConv dconv;

  /* argument array must be at least iMinSize int32s big */
  if (pstMethod->u16ArgSize + iStart < iMinSize)
  {
    pi32Args = (int32*) sys_malloc(sizeof(int32) * iMinSize);
  }
  else
  {
    pi32Args = (int32*) sys_malloc(sizeof(int32) * (pstMethod->u16ArgSize + iStart));
  }

  assert(pi32Args);
  if(iMinSize > 0)
    pi32Args[0] = 0;

  for (i = 1, j = iStart, k = 0; pstMethod->uidSignature[i] != ')'; i++, j++, k++)
  {
    switch (pstMethod->uidSignature[i])
    {
      case 'Z':
      {
        pi32Args[j] = (int32) args[k].z;
        break;
      }
      case 'B':
      {
        pi32Args[j] = (int32) args[k].b;
        break;
      }
      case 'C':
      {
        pi32Args[j] = (int32) args[k].c;
        break;
      }
      case 'S':
      {
        pi32Args[j] = (int32) args[k].s;
        break;
      }
      case 'I':
      {
        pi32Args[j] = (int32) args[k].i;
        break;
      }
      case 'F':
      {
        pi32Args[j] = (int32) args[k].f;
        break;
      }
      case 'D':
      {
        dconv.d = args[k].d;
        pi32Args[j++] = (int32) dconv.i.hi;
        pi32Args[j] = (int32) dconv.i.lo;
        break;
      }
      case 'J':
      {
        pi32Args[j++] = (int32) (args[k].j >> 32);
        pi32Args[j] = (int32) args[k].j & 0xFFFFFFFF;
        break;
      }
      case 'L':
      {
        pi32Args[j] = (int32) args[k].l;
        for (; pstMethod->uidSignature[i] != ';'; i++);
        break;
      }
      case '[':
      {
        pi32Args[j] = (int32) args[k].l;
        while ((pstMethod->uidSignature[i] == '[' || (pstMethod->uidSignature[i] >= '0' && pstMethod->uidSignature[i] <= '9')))
        {
          i++;
        }
        if (pstMethod->uidSignature[i] == 'L')
        {
          while (pstMethod->uidSignature[i] != ';')
          {
            i++;
          }
        }
        break;
      }
    }
  }
  return pi32Args;
}

//XXX Mem leak here?
static void DeleteLocalRef (JNIEnv* env, jobject localRef)
{
  int i;

  tJNIData* pstJNIData = JNI_getJNIData( sys_current_thread());

  if(pstJNIData->ppstLocalRefs)
    {
      for (i = 0; i < pstJNIData->u16NumLocalRefs; i++)
	{
	  if (pstJNIData->ppstLocalRefs[i] == localRef)
	    {
	      pstJNIData->ppstLocalRefs[i] = NULL;
	    }
	}
    }
}

static jobject NewLocalRef (JNIEnv *env, jobject ref)
{
  CreateLocalRef( JNI_getJNIData( sys_current_thread()), ref);
  return ref;
}


/*
 * @doc FUNC
 * @func
 * This function is called to create local references. It is passed the
 * JNIData for the current native call and the object that needs a local ref
 * made for it.
 *
 */

static void CreateLocalRef
(
 tJNIData* pstJNIData,  /* @parm JNIData for current native call */
 jobject   obj          /* @parm Object which needs a local ref */
 )
{
  int i = 0;
  int found = 0;

  assert(pstJNIData->u16NumLocalRefsUsed <= pstJNIData->u16NumLocalRefs);

  if(obj == NULL)
    return;

  //See	if there is an empty slot

  if(pstJNIData->ppstLocalRefs == NULL)
    {
      found = 0;
    }
  else
    {
      for(i = 0; i < pstJNIData->u16NumLocalRefsUsed;i++)
	if( pstJNIData->ppstLocalRefs[i] == NULL) 
	  {
	    found = 1;
	    break;
	  }
    }


  if(found == 0)
    {
      /* If there are no local ref spaces then make some */
      if (pstJNIData->u16NumLocalRefs == 0)
	{
	  pstJNIData->ppstLocalRefs = (jobject*) sys_malloc(sizeof(jobject) * 5);
	  assert(pstJNIData->ppstLocalRefs);
	  memset(pstJNIData->ppstLocalRefs, 0, 5 * sizeof(jobject)); //Jewel
	  pstJNIData->u16NumLocalRefs = 5;
	  pstJNIData->u16NumLocalRefsUsed = 0;
	}
      /* if there is no space then expand the space */
      else if (pstJNIData->u16NumLocalRefsUsed == pstJNIData->u16NumLocalRefs)
	{
	  tOBREF* oldrefs = pstJNIData->ppstLocalRefs;
	  pstJNIData->ppstLocalRefs = (jobject*) sys_malloc(sizeof(jobject) * pstJNIData->u16NumLocalRefs * 2 * 5);
	  assert(pstJNIData->ppstLocalRefs);
	  memset( pstJNIData->ppstLocalRefs, 0, sizeof(jobject) * pstJNIData->u16NumLocalRefs * 2 * 5);


	  for(i = 0; i < pstJNIData->u16NumLocalRefsUsed; i++)
	    pstJNIData->ppstLocalRefs[i] = oldrefs[i];

	  pstJNIData->u16NumLocalRefs *= 2;
		 
	  //		 fprintf(stderr, "There are now %i local references\n", pstJNIData->u16NumLocalRefs);
	}

      /* Stick The new one on the end */
      pstJNIData->ppstLocalRefs[pstJNIData->u16NumLocalRefsUsed++] = obj;
      assert(GARBAGE_InHeap(&pstJNIData->functions, pstJNIData->ppstLocalRefs[pstJNIData->u16NumLocalRefsUsed - 1]));
    }
  else
    {
      pstJNIData->ppstLocalRefs[i] = obj;
      //	 pstJNIData->u16NumLocalRefsUsed++; //We don't increment this, because we didn't use a slot
    }


  for(i = 0; i < pstJNIData->u16NumLocalRefsUsed;i++)
    if(pstJNIData->ppstLocalRefs[i] != NULL)
      {
	assert( GARBAGE_InHeap(&pstJNIData->functions, pstJNIData->ppstLocalRefs[i]));
      }

  /*    for(i = 0; i < pstJNIData->u16NumLocalRefsUsed;i++)
	if(pstJNIData->ppstLocalRefs[i] != NULL)
	assert( ((tObject*) (*( (jobject*) pstJNIData->ppstLocalRefs[i])))->pstType );
  */
  assert(pstJNIData->u16NumLocalRefsUsed <= pstJNIData->u16NumLocalRefs);
} 

extern tClassLoaderTuple* pstClassType; // this can be globally used to get the type for class

/*
 * @doc FUNC
 * @func
 * This function initialises the internal data needed by the JNI code. It must
 * be called before any JNI operations are attemped.
 *
 */

void JNI_Init(JNIEnv* env)
{
  u16StructOff = -1;
  u16LoaderOff = -1;
  JNI_u16NumGlobalRefs = 20;
  JNI_u16NumGlobalRefsUsed = 0;
  JNI_ppstGlobalRefs = (tOBREF*) sys_malloc(sizeof(tOBREF) * JNI_u16NumGlobalRefs);

  assert(JNI_ppstGlobalRefs);	
  memset(JNI_ppstGlobalRefs, 0, sizeof(tOBREF) * JNI_u16NumGlobalRefs);

#ifdef DLOPEN
  JNI_DynLoadedLibraries[ JNI_NumberLoadedLibraries ] = dlopen( NULL, RTLD_LAZY);
  JNI_NumberLoadedLibraries++;
#endif

//  eprintf("JNI Init\n");
  //  printf("********** JNI_Init ***********\n");
  u16StructOff = CLASSFILE_InstFieldOffset(env, pstClassType, "_classStruct")->u16Offset;
  u16LoaderOff = CLASSFILE_InstFieldOffset(env, pstClassType, "_classLoader")->u16Offset;

  //Construct the class object for java/lang/Class
    {
      jclass pstClassObject = INTERP_NewObject(env, pstClassType);
      SETCLASSSTRUCT(pstClassObject, pstClassType);
      assert( GETCLASSSTRUCT(pstClassObject));
      /* set _classLoader field in new class object to NULL */
      SETCLASSLOADER(pstClassObject, NULL);
      pstClassType->classObject = pstClassObject;
      CreateLocalRef( JNI_getJNIData(sys_current_thread()), pstClassObject);
    }

}

void JNI_Finish
(
  void
)
{
  sys_free(JNI_ppstGlobalRefs);
  JNI_ppstGlobalRefs = NULL;
}

tAllocatorHeap* getHeap()
{
  return JNI_getJNIData(sys_current_thread())->pstHeap;
}

static const struct JNINativeInterface functiontable =
{
  /* Version Information */
  NULL,
  NULL,
  NULL,
  NULL,
  GetVersion,

  /* Class Operations */
  DefineClass,
  FindClass,
  NULL,
  NULL,
  NULL,
  GetSuperClass,
  IsAssignableFrom,
  NULL,

  /* Exceptions */
  Throw,
  ThrowNew,
  ExceptionOccurred,
  ExceptionDescribe,
  ExceptionClear,
  FatalError,
  NULL,
  NULL,

  /* Global and Local References */
  NewGlobalRef,
  DeleteGlobalRef,
  DeleteLocalRef,

  IsSameObject,
  NewLocalRef,
  EnsureLocalCapacity,

  /* Object Operations */
  AllocObject,
  NewObject,
  NewObjectA,
  NewObjectV,

  GetObjectClass,
  InstanceOf,

  /* Calling Instance Methods */
  GetMethodID,

  CallObjectMethod,
  CallObjectMethodV,
  CallObjectMethodA,
  CallBooleanMethod,
  CallBooleanMethodV,
  CallBooleanMethodA,
  CallByteMethod,
  CallByteMethodV,
  CallByteMethodA,
  CallCharMethod,
  CallCharMethodV,
  CallCharMethodA,
  CallShortMethod,
  CallShortMethodV,
  CallShortMethodA,
  CallIntMethod,
  CallIntMethodV,
  CallIntMethodA,
  CallLongMethod,
  CallLongMethodV,
  CallLongMethodA,
  CallFloatMethod,
  CallFloatMethodV,
  CallFloatMethodA,
  CallDoubleMethod,
  CallDoubleMethodV,
  CallDoubleMethodA,
  CallVoidMethod,
  CallVoidMethodV, 
  CallVoidMethodA,

  CallSpecialObjectMethod,
  CallSpecialObjectMethodV,
  CallSpecialObjectMethodA,
  CallSpecialBooleanMethod,
  CallSpecialBooleanMethodV,
  CallSpecialBooleanMethodA,
  CallSpecialByteMethod,
  CallSpecialByteMethodV,
  CallSpecialByteMethodA,
  CallSpecialCharMethod,
  CallSpecialCharMethodV,
  CallSpecialCharMethodA,
  CallSpecialShortMethod,
  CallSpecialShortMethodV,
  CallSpecialShortMethodA,
  CallSpecialIntMethod,
  CallSpecialIntMethodV,
  CallSpecialIntMethodA,
  CallSpecialLongMethod,
  CallSpecialLongMethodV,
  CallSpecialLongMethodA,
  CallSpecialFloatMethod,
  CallSpecialFloatMethodV,
  CallSpecialFloatMethodA,
  CallSpecialDoubleMethod,
  CallSpecialDoubleMethodV,
  CallSpecialDoubleMethodA,
  CallNonvirtualVoidMethod,
  CallNonvirtualVoidMethodV,
  CallNonvirtualVoidMethodA,

  /* Accessing Fields of Objects */
  GetFieldID,

  GetObjectField,
  GetBooleanField,
  GetByteField,
  GetCharField,
  GetShortField,
  GetIntField,
  GetLongField,
  GetFloatField,
  GetDoubleField,

  SetObjectField,
  SetBooleanField,
  SetByteField,
  SetCharField,
  SetShortField,
  SetIntField,
  SetLongField,
  SetFloatField,
  SetDoubleField,

  /* Calling Static Methods */
  GetStaticMethodID,

  CallStaticObjectMethod,
  CallStaticObjectMethodV,
  CallStaticObjectMethodA,
  CallStaticBooleanMethod,
  CallStaticBooleanMethodV,
  CallStaticBooleanMethodA,
  CallStaticByteMethod,
  CallStaticByteMethodV,
  CallStaticByteMethodA,
  CallStaticCharMethod,
  CallStaticCharMethodV,
  CallStaticCharMethodA,
  CallStaticShortMethod,
  CallStaticShortMethodV,
  CallStaticShortMethodA,
  CallStaticIntMethod,
  CallStaticIntMethodV,
  CallStaticIntMethodA,
  CallStaticLongMethod,
  CallStaticLongMethodV,
  CallStaticLongMethodA,
  CallStaticFloatMethod,
  CallStaticFloatMethodV,
  CallStaticFloatMethodA,
  CallStaticDoubleMethod,
  CallStaticDoubleMethodV,
  CallStaticDoubleMethodA,
  CallStaticVoidMethod,
  CallStaticVoidMethodV,
  CallStaticVoidMethodA,

  /* Accessing Static Fields */
  GetStaticFieldID,

  GetStaticObjectField,
  GetStaticBooleanField,
  GetStaticByteField,
  GetStaticCharField,
  GetStaticShortField,
  GetStaticIntField,
  GetStaticLongField,
  GetStaticFloatField,
  GetStaticDoubleField,

  SetStaticObjectField,
  SetStaticBooleanField,
  SetStaticByteField,

  SetStaticCharField,
  SetStaticShortField,
  SetStaticIntField,
  SetStaticLongField,
  SetStaticFloatField,
  SetStaticDoubleField,

  /* String Operations */
  NewString,
  GetStringLength,
  GetStringChars,
  ReleaseStringChars,

  NewStringUTF,
  GetStringUTFLength,
  GetStringUTFChars,
  ReleaseStringUTFChars,

  /* Array Operations */
  GetArrayLength,

  NewObjectArray,
  GetObjectArrayElement,
  SetObjectArrayElement,

  NewBooleanArray,
  NewByteArray,
  NewCharArray,
  NewShortArray,
  NewIntArray,
  NewLongArray,
  NewFloatArray,
  NewDoubleArray,

  GetBooleanArrayElements,
  GetByteArrayElements,
  GetCharArrayElements,
  GetShortArrayElements,
  GetIntArrayElements,
  GetLongArrayElements,
  GetFloatArrayElements,
  GetDoubleArrayElements,

  ReleaseBooleanArrayElements,
  ReleaseByteArrayElements,
  ReleaseCharArrayElements,
  ReleaseShortArrayElements,
  ReleaseIntArrayElements,
  ReleaseLongArrayElements,
  ReleaseFloatArrayElements,
  ReleaseDoubleArrayElements,

  GetBooleanArrayRegion,
  GetByteArrayRegion,
  GetCharArrayRegion,
  GetShortArrayRegion,
  GetIntArrayRegion,
  GetLongArrayRegion,
  GetFloatArrayRegion,
  GetDoubleArrayRegion,

  SetBooleanArrayRegion,
  SetByteArrayRegion,
  SetCharArrayRegion,
  SetShortArrayRegion,
  SetIntArrayRegion,
  SetLongArrayRegion,
  SetFloatArrayRegion,
  SetDoubleArrayRegion,

  /* Regsitering Native Methods */
  RegisterNatives,
  UnregisterNatives,

  /* Monitor Operations */
  MonitorEnter,
  MonitorExit,

  /* Java VM Interface */
  GetJavaVM,

  GetStringRegion,
  GetStringUTFRegion,
  GetPrimitiveArrayCritical,
  ReleasePrimitiveArrayCritical,
  GetStringCritical,
  ReleaseStringCritical,
  NewWeakGlobalRef,
  DeleteWeakGlobalRef,
  ExceptionCheck,

  DefinePrimitiveClass,
#ifndef GARBAGE_GENERATIONAL
  #ifdef PERSIST
  /* stuff for persistence */
  SwizzleObjectField,
  SwizzleArrayElement,
  MarkUpdated,
  MarkRefsUpdated,
  #endif
#endif
  getHeap
};

JNIEnv JNI_pstFunctionTable = &functiontable;
