/*
 * Copyright (C) 2001, John Leuner.
 *
 * This file is part of the kissme/teaseme project, which in turn is part of the JOS project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2,
 * or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
 * @doc MODULE
 * @module jni.h |
 *
 * This file contains the declarations for the native method interface.
 *
 */

#include <config.h>

#ifndef __JNI_H__
#define __JNI_H__

#include "interp.h"
#include "classfil.h"
#include "stdtypes.h"

#include "jni_md.h"

#ifdef KISSME_LINUX_USER
#include <stdarg.h>
#include <malloc.h>
#endif

#ifdef KISSME_RJK_KERNEL
#include <stdarg.h>
#endif

#ifndef KISSME_LINUX_KERNEL
#include <stdarg.h>
#endif

/* These are used in javah -jni headers */

#define JNIEXPORT

// extern ?????
#define JNICALL  


/* true and false for convenience */
#define JNI_FALSE 0
#define JNI_TRUE  1

/* commit and abort for array element releasing */
#define JNI_COMMIT 1
#define JNI_ABORT  2

/* primitive types */
typedef unsigned char    jboolean;
typedef int8     jbyte;
typedef uint16   jchar;
typedef int16    jshort;
typedef int32    jint;
typedef long long jlong;
typedef float    jfloat;
typedef double   jdouble;

/* for sizes and indices */
typedef jint     jsize;

/* object and array types */
typedef tOBREF jobject;
typedef jobject  jclass;
typedef jobject  jstring;
typedef jobject  jthrowable;
typedef jobject  jarray;
typedef jarray   jobjectArray;
typedef jarray   jbooleanArray;
typedef jarray   jbyteArray;
typedef jarray   jcharArray;
typedef jarray   jshortArray;
typedef jarray   jintArray;
typedef jarray   jlongArray;
typedef jarray   jfloatArray;
typedef jarray   jdoubleArray;

#ifndef _JNINATIVEMETHOD
#define _JNINATIVEMETHOD
typedef struct jninativemethod {char* name; char* signature; void* fnPtr;} JNINativeMethod;
#endif
typedef void     JavaVM;

typedef jobject jweak;

/* union for representing any primitive type */
typedef union jvalue
{
  jboolean z;
  jbyte    b;
  jchar    c;
  jshort   s;
  jint     i;
  jlong    j;
  jfloat   f;
  jdouble  d;
  jobject  l;
} jvalue;


/* definition for converting doubles to two int32s */
typedef union
{
  double d;
  struct
  {
    int32 hi;
    int32 lo;
  } i;
} tDConv;

typedef const struct JNINativeInterface* JNIEnv;

//kissme specific types 
/*
struct tclassLoadertuple;
typedef struct tclassloadertuple tClassLoaderTuple;

struct _tAllocatorHeap;
typedef struct _tAllocatorHeap* tAllocatorHeap;
struct _tStackFrame;
typedef struct _tStackFrame* tStackFrame;
*/



typedef tMethod* jmethodID;
typedef tField* jfieldID;

#ifndef PERSIST
#include "classfil.h"
typedef void* tClassLoaderTuple;
#endif 

//typedef struct _tClassLoaderTuple* tClassLoaderTuple;
//typedef struct _tAllocatorHeap* tClassLoaderTuple;

//----

/*
 * @doc STRUCT
 * @struct JNINativeInterface |
 * This structure contains pointers to all the JNI functions so that they
 * can be called from native methods.
 *
 */

struct JNINativeInterface
{
  /* Version Information */
  void* blank0;
  void* blank1;
  void* blank2;
  void* blank3;
  jint (*GetVersion) (JNIEnv* env);

  /* Class Operations */
  jclass (*DefineClass) (JNIEnv* env, jobject loader, const jbyte* buf, jsize bufLen);
  jclass (*FindClass) (JNIEnv* env, const char* name);
  void* blank4;
  void* blank5;
  void* blank6;
  jclass (*GetSuperClass) (JNIEnv* env, jclass clazz);
  jboolean (*IsAssignableFrom) (JNIEnv* env, jclass clazz1, jclass clazz2);
  void* blank7;

  /* Exceptions */
  jint (*Throw) (JNIEnv* env, jthrowable obj);
  jint (*ThrowNew) (JNIEnv* env, jclass clazz, const char* message);
  jthrowable (*ExceptionOccurred) (JNIEnv* env);
  void (*ExceptionDescribe) (JNIEnv* env);
  void (*ExceptionClear) (JNIEnv* env);
  void (*FatalError) (JNIEnv* env, const char* msg);

  void* blank8;
  void* blank9;

  /* Global and Local References */
  jobject (*NewGlobalRef) (JNIEnv* env, jobject obj);
  void (*DeleteGlobalRef) (JNIEnv* env, jobject globalRef);
  void (*DeleteLocalRef) (JNIEnv* env, jobject localRef);

  jboolean (*IsSameObject) (JNIEnv* env, jobject ref1, jobject ref2);
  jobject (*NewLocalRef) (JNIEnv *env, jobject ref);
  jint (*EnsureLocalCapacity) (JNIEnv *env, jint capacity);

  /* Object Operations */
  jobject (*AllocObject) (JNIEnv* env, jclass clazz);
  jobject (*NewObject) (JNIEnv* env, jclass clazz, jmethodID methodID, ...);
  jobject (*NewObjectA) (JNIEnv* env, jclass clazz, jmethodID methodID, jvalue* args);
  jobject (*NewObjectV) (JNIEnv* env, jclass clazz, jmethodID methodID, va_list args);

  jclass (*GetObjectClass) (JNIEnv* env, jobject obj);
  jboolean (*IsInstanceOf) (JNIEnv* env, jobject obj, jclass clazz);

  /* Calling Instance Methods */
  jmethodID (*GetMethodID) (JNIEnv* env, jclass clazz, const char* name, const char* sig);

  jobject (*CallObjectMethod) (JNIEnv* env, jobject obj, jmethodID methodID, ...);
  jobject (*CallObjectMethodV) (JNIEnv* env, jobject obj, jmethodID methodID, va_list args);
  jobject (*CallObjectMethodA) (JNIEnv* env, jobject obj, jmethodID methodID, jvalue* args);

  jboolean (*CallBooleanMethod) (JNIEnv* env, jobject obj, jmethodID methodID, ...);
  jboolean (*CallBooleanMethodV) (JNIEnv* env, jobject obj, jmethodID methodID, va_list args);
  jboolean (*CallBooleanMethodA) (JNIEnv* env, jobject obj, jmethodID methodID, jvalue* args);

  jbyte (*CallByteMethod) (JNIEnv* env, jobject obj, jmethodID methodID, ...);
  jbyte (*CallByteMethodV) (JNIEnv* env, jobject obj, jmethodID methodID, va_list args);
  jbyte (*CallByteMethodA) (JNIEnv* env, jobject obj, jmethodID methodID, jvalue* args);

  jchar (*CallCharMethod) (JNIEnv* env, jobject obj, jmethodID methodID, ...);
  jchar (*CallCharMethodV) (JNIEnv* env, jobject obj, jmethodID methodID, va_list args);
  jchar (*CallCharMethodA) (JNIEnv* env, jobject obj, jmethodID methodID, jvalue* args);

  jshort (*CallShortMethod) (JNIEnv* env, jobject obj, jmethodID methodID, ...);
  jshort (*CallShortMethodV) (JNIEnv* env, jobject obj, jmethodID methodID, va_list args);
  jshort (*CallShortMethodA) (JNIEnv* env, jobject obj, jmethodID methodID, jvalue* args);

  jint (*CallIntMethod) (JNIEnv* env, jobject obj, jmethodID methodID, ...);
  jint (*CallIntMethodV) (JNIEnv* env, jobject obj, jmethodID methodID, va_list args);
  jint (*CallIntMethodA) (JNIEnv* env, jobject obj, jmethodID methodID, jvalue* args);

  jlong (*CallLongMethod) (JNIEnv* env, jobject obj, jmethodID methodID, ...);
  jlong (*CallLongMethodV) (JNIEnv* env, jobject obj, jmethodID methodID, va_list args);
  jlong (*CallLongMethodA) (JNIEnv* env, jobject obj, jmethodID methodID, jvalue* args);

  jfloat (*CallFloatMethod) (JNIEnv* env, jobject obj, jmethodID methodID, ...);
  jfloat (*CallFloatMethodV) (JNIEnv* env, jobject obj, jmethodID methodID, va_list args);
  jfloat (*CallFloatMethodA) (JNIEnv* env, jobject obj, jmethodID methodID, jvalue* args);

  jdouble (*CallDoubleMethod) (JNIEnv* env, jobject obj, jmethodID methodID, ...);
  jdouble (*CallDoubleMethodV) (JNIEnv* env, jobject obj, jmethodID methodID, va_list args);
  jdouble (*CallDoubleMethodA) (JNIEnv* env, jobject obj, jmethodID methodID, jvalue* args);

  void (*CallVoidMethod) (JNIEnv* env, jobject obj, jmethodID methodID, ...);
  void (*CallVoidMethodV) (JNIEnv* env, jobject obj, jmethodID methodID, va_list args);
  void (*CallVoidMethodA) (JNIEnv* env, jobject obj, jmethodID methodID, jvalue* args);


  jobject (*CallNonvirtualObjectMethod) (JNIEnv* env, jobject obj, jclass clazz, jmethodID methodID, ...);
  jobject (*CallNonvirtualObjectMethodV) (JNIEnv* env, jobject obj, jclass clazz, jmethodID methodID, va_list args);
  jobject (*CallNonvirtualObjectMethodA) (JNIEnv* env, jobject obj, jclass clazz, jmethodID methodID, jvalue* args);

  jboolean (*CallNonvirtualBooleanMethod) (JNIEnv* env, jobject obj, jclass clazz, jmethodID methodID, ...);
  jboolean (*CallNonvirtualBooleanMethodV) (JNIEnv* env, jobject obj, jclass clazz, jmethodID methodID, va_list args);
  jboolean (*CallNonvirtualBooleanMethodA) (JNIEnv* env, jobject obj, jclass clazz, jmethodID methodID, jvalue* args);

  jbyte (*CallNonvirtualByteMethod) (JNIEnv* env, jobject obj, jclass clazz, jmethodID methodID, ...);
  jbyte (*CallNonvirtualByteMethodV) (JNIEnv* env, jobject obj, jclass clazz, jmethodID methodID, va_list args);
  jbyte (*CallNonvirtualByteMethodA) (JNIEnv* env, jobject obj, jclass clazz, jmethodID methodID, jvalue* args);

  jchar (*CallNonvirtualCharMethod) (JNIEnv* env, jobject obj, jclass clazz, jmethodID methodID, ...);
  jchar (*CallNonvirtualCharMethodV) (JNIEnv* env, jobject obj, jclass clazz, jmethodID methodID, va_list args);
  jchar (*CallNonvirtualCharMethodA) (JNIEnv* env, jobject obj, jclass clazz, jmethodID methodID, jvalue* args);

  jshort (*CallNonvirtualShortMethod) (JNIEnv* env, jobject obj, jclass clazz, jmethodID methodID, ...);
  jshort (*CallNonvirtualShortMethodV) (JNIEnv* env, jobject obj, jclass clazz, jmethodID methodID, va_list args);
  jshort (*CallNonvirtualShortMethodA) (JNIEnv* env, jobject obj, jclass clazz, jmethodID methodID, jvalue* args);

  jint (*CallNonvirtualIntMethod) (JNIEnv* env, jobject obj, jclass clazz, jmethodID methodID, ...);
  jint (*CallNonvirtualIntMethodV) (JNIEnv* env, jobject obj, jclass clazz, jmethodID methodID, va_list args);
  jint (*CallNonvirtualIntMethodA) (JNIEnv* env, jobject obj, jclass clazz, jmethodID methodID, jvalue* args);

  jlong (*CallNonvirtualLongMethod) (JNIEnv* env, jobject obj, jclass clazz, jmethodID methodID, ...);
  jlong (*CallNonvirtualLongMethodV) (JNIEnv* env, jobject obj, jclass clazz, jmethodID methodID, va_list args);
  jlong (*CallNonvirtualLongMethodA) (JNIEnv* env, jobject obj, jclass clazz, jmethodID methodID, jvalue* args);

  jfloat (*CallNonvirtualFloatMethod) (JNIEnv* env, jobject obj, jclass clazz, jmethodID methodID, ...);
  jfloat (*CallNonvirtualFloatMethodV) (JNIEnv* env, jobject obj, jclass clazz, jmethodID methodID, va_list args);
  jfloat (*CallNonvirtualFloatMethodA) (JNIEnv* env, jobject obj, jclass clazz, jmethodID methodID, jvalue* args);

  jdouble (*CallNonvirtualDoubleMethod) (JNIEnv* env, jobject obj, jclass clazz, jmethodID methodID, ...);
  jdouble (*CallNonvirtualDoubleMethodV) (JNIEnv* env, jobject obj, jclass clazz, jmethodID methodID, va_list args);
  jdouble (*CallNonvirtualDoubleMethodA) (JNIEnv* env, jobject obj, jclass clazz, jmethodID methodID, jvalue* args);

  void (*CallNonvirtualVoidMethod) (JNIEnv* env, jobject obj, jclass clazz, jmethodID methodID, ...);
  void (*CallNonvirtualVoidMethodV) (JNIEnv* env, jobject obj, jclass clazz, jmethodID methodID, va_list args);
  void (*CallNonvirtualVoidMethodA) (JNIEnv* env, jobject obj, jclass clazz, jmethodID methodID, jvalue* args);


  /* Accessing Fields of Objects */
  jfieldID (*GetFieldID) (JNIEnv* env, jclass clazz, const char* name, const char* sig);

  jobject (*GetObjectField) (JNIEnv* env, jobject obj, jfieldID fieldID);
  jboolean (*GetBooleanField) (JNIEnv* env, jobject obj, jfieldID fieldID);
  jbyte (*GetByteField) (JNIEnv* env, jobject obj, jfieldID fieldID);
  jchar (*GetCharField) (JNIEnv* env, jobject obj, jfieldID fieldID);
  jshort (*GetShortField) (JNIEnv* env, jobject obj, jfieldID fieldID);
  jint (*GetIntField) (JNIEnv* env, jobject obj, jfieldID fieldID);
  jlong (*GetLongField) (JNIEnv* env, jobject obj, jfieldID fieldID);
  jfloat (*GetFloatField) (JNIEnv* env, jobject obj, jfieldID fieldID);
  jdouble (*GetDoubleField) (JNIEnv* env, jobject obj, jfieldID fieldID);

  void (*SetObjectField) (JNIEnv* env, jobject obj, jfieldID fieldID, jobject value);
  void (*SetBooleanField) (JNIEnv* env, jobject obj, jfieldID fieldID, jboolean value);
  void (*SetByteField) (JNIEnv* env, jobject obj, jfieldID fieldID, jbyte value);
  void (*SetCharField) (JNIEnv* env, jobject obj, jfieldID fieldID, jchar value);
  void (*SetShortField) (JNIEnv* env, jobject obj, jfieldID fieldID, jshort value);
  void (*SetIntField) (JNIEnv* env, jobject obj, jfieldID fieldID, jint value);
  void (*SetLongField) (JNIEnv* env, jobject obj, jfieldID fieldID, jlong value);
  void (*SetFloatField) (JNIEnv* env, jobject obj, jfieldID fieldID, jfloat value);
  void (*SetDoubleField) (JNIEnv* env, jobject obj, jfieldID fieldID, jdouble value);

  /* Calling Static Methods */
  jmethodID (*GetStaticMethodID) (JNIEnv* env, jclass clazz, const char* name, const char* sig);

  jobject (*CallStaticObjectMethod) (JNIEnv* env, jobject obj, jmethodID methodID, ...);
  jobject (*CallStaticObjectMethodV) (JNIEnv* env, jobject obj, jmethodID methodID, va_list args);
  jobject (*CallStaticObjectMethodA) (JNIEnv* env, jobject obj, jmethodID methodID, jvalue* args);

  jboolean (*CallStaticBooleanMethod) (JNIEnv* env, jobject obj, jmethodID methodID, ...);
  jboolean (*CallStaticBooleanMethodV) (JNIEnv* env, jobject obj, jmethodID methodID, va_list args);
  jboolean (*CallStaticBooleanMethodA) (JNIEnv* env, jobject obj, jmethodID methodID, jvalue* args);

  jbyte (*CallStaticByteMethod) (JNIEnv* env, jobject obj, jmethodID methodID, ...);
  jbyte (*CallStaticByteMethodV) (JNIEnv* env, jobject obj, jmethodID methodID, va_list args);
  jbyte (*CallStaticByteMethodA) (JNIEnv* env, jobject obj, jmethodID methodID, jvalue* args);

  jchar (*CallStaticCharMethod) (JNIEnv* env, jobject obj, jmethodID methodID, ...);
  jchar (*CallStaticCharMethodV) (JNIEnv* env, jobject obj, jmethodID methodID, va_list args);
  jchar (*CallStaticCharMethodA) (JNIEnv* env, jobject obj, jmethodID methodID, jvalue* args);

  jshort (*CallStaticShortMethod) (JNIEnv* env, jobject obj, jmethodID methodID, ...);
  jshort (*CallStaticShortMethodV) (JNIEnv* env, jobject obj, jmethodID methodID, va_list args);
  jshort (*CallStaticShortMethodA) (JNIEnv* env, jobject obj, jmethodID methodID, jvalue* args);

  jint (*CallStaticIntMethod) (JNIEnv* env, jobject obj, jmethodID methodID, ...);
  jint (*CallStaticIntMethodV) (JNIEnv* env, jobject obj, jmethodID methodID, va_list args);
  jint (*CallStaticIntMethodA) (JNIEnv* env, jobject obj, jmethodID methodID, jvalue* args);

  jlong (*CallStaticLongMethod) (JNIEnv* env, jobject obj, jmethodID methodID, ...);
  jlong (*CallStaticLongMethodV) (JNIEnv* env, jobject obj, jmethodID methodID, va_list args);
  jlong (*CallStaticLongMethodA) (JNIEnv* env, jobject obj, jmethodID methodID, jvalue* args);

  jfloat (*CallStaticFloatMethod) (JNIEnv* env, jobject obj, jmethodID methodID, ...);
  jfloat (*CallStaticFloatMethodV) (JNIEnv* env, jobject obj, jmethodID methodID, va_list args);
  jfloat (*CallStaticFloatMethodA) (JNIEnv* env, jobject obj, jmethodID methodID, jvalue* args);

  jdouble (*CallStaticDoubleMethod) (JNIEnv* env, jobject obj, jmethodID methodID, ...);
  jdouble (*CallStaticDoubleMethodV) (JNIEnv* env, jobject obj, jmethodID methodID, va_list args);
  jdouble (*CallStaticDoubleMethodA) (JNIEnv* env, jobject obj, jmethodID methodID, jvalue* args);

  void (*CallStaticVoidMethod) (JNIEnv* env, jobject obj, jmethodID methodID, ...);
  void (*CallStaticVoidMethodV) (JNIEnv* env, jobject obj, jmethodID methodID, va_list args);
  void (*CallStaticVoidMethodA) (JNIEnv* env, jobject obj, jmethodID methodID, jvalue* args);


  /* Accessing Static Fields */
  jfieldID (*GetStaticFieldID) (JNIEnv* env, jclass clazz, const char* name, const char* sig);

  jobject (*GetStaticObjectField) (JNIEnv* env, jclass clazz, jfieldID fieldID);
  jboolean (*GetStaticBooleanField) (JNIEnv* env, jclass clazz, jfieldID fieldID);
  jbyte (*GetStaticByteField) (JNIEnv* env, jclass clazz, jfieldID fieldID);
  jchar (*GetStaticCharField) (JNIEnv* env, jclass clazz, jfieldID fieldID);
  jshort (*GetStaticShortField) (JNIEnv* env, jclass clazz, jfieldID fieldID);
  jint (*GetStaticIntField) (JNIEnv* env, jclass clazz, jfieldID fieldID);
  jlong (*GetStaticLongField) (JNIEnv* env, jclass clazz, jfieldID fieldID);
  jfloat (*GetStaticFloatField) (JNIEnv* env, jclass clazz, jfieldID fieldID);
  jdouble (*GetStaticDoubleField) (JNIEnv* env, jclass clazz, jfieldID fieldID);

  void (*SetStaticObjectField) (JNIEnv* env, jclass clazz, jfieldID fieldID, jobject value);
  void (*SetStaticBooleanField) (JNIEnv* env, jclass clazz, jfieldID fieldID, jboolean value);
  void (*SetStaticByteField) (JNIEnv* env, jclass clazz, jfieldID fieldID, jbyte value);
  void (*SetStaticCharField) (JNIEnv* env, jclass clazz, jfieldID fieldID, jchar value);
  void (*SetStaticShortField) (JNIEnv* env, jclass clazz, jfieldID fieldID, jshort value);
  void (*SetStaticIntField) (JNIEnv* env, jclass clazz, jfieldID fieldID, jint value);
  void (*SetStaticLongField) (JNIEnv* env, jclass clazz, jfieldID fieldID, jlong value);
  void (*SetStaticFloatField) (JNIEnv* env, jclass clazz, jfieldID fieldID, jfloat value);
  void (*SetStaticDoubleField) (JNIEnv* env, jclass clazz, jfieldID fieldID, jdouble value);

  /* String Operations */
  jstring (*NewString) (JNIEnv* env, const jchar* unicodeChars, jsize len);
  jsize (*GetStringLength) (JNIEnv* env, jstring string);
  const jchar* (*GetStringChars) (JNIEnv* env, jstring string, jboolean* isCopy);
  void (*ReleaseStringChars) (JNIEnv* env, jstring string, const jchar* chars);

  jstring (*NewStringUTF) (JNIEnv* env, const char* bytes);
  jsize (*GetStringUTFLength) (JNIEnv* env, jstring string);
  const jbyte* (*GetStringUTFChars) (JNIEnv* env, jstring string, jboolean* isCopy);
  void (*ReleaseStringUTFChars) (JNIEnv* env, jstring string, const char* chars);

  /* Array Operations */
  jsize (*GetArrayLength) (JNIEnv* env, jarray array);

  jarray (*NewObjectArray) (JNIEnv* env, jsize length, jclass elementClass, jobject initialElement);
  jobject (*GetObjectArrayElement) (JNIEnv* env, jobjectArray array, jsize index);
  void (*SetObjectArrayElement) (JNIEnv* env, jobjectArray array, jsize index, jobject value);

  jbooleanArray (*NewBooleanArray) (JNIEnv* env, jsize length);
  jbyteArray (*NewByteArray) (JNIEnv* env, jsize length);
  jcharArray (*NewCharArray) (JNIEnv* env, jsize length);
  jshortArray (*NewShortArray) (JNIEnv* env, jsize length);
  jintArray (*NewIntArray) (JNIEnv* env, jsize length);
  jlongArray (*NewLongArray) (JNIEnv* env, jsize length);
  jfloatArray (*NewFloatArray) (JNIEnv* env, jsize length);
  jdoubleArray (*NewDoubleArray) (JNIEnv* env, jsize length);

  jboolean* (*GetBooleanArrayElements) (JNIEnv* env, jbooleanArray array, jboolean* isCopy);
  jbyte* (*GetByteArrayElements) (JNIEnv* env, jbyteArray array, jboolean* isCopy);
  jchar* (*GetCharArrayElements) (JNIEnv* env, jcharArray array, jboolean* isCopy);
  jshort* (*GetShortArrayElements) (JNIEnv* env, jshortArray array, jboolean* isCopy);
  jint* (*GetIntArrayElements) (JNIEnv* env, jintArray array, jboolean* isCopy);
  jlong* (*GetLongArrayElements) (JNIEnv* env, jlongArray array, jboolean* isCopy);
  jfloat* (*GetFloatArrayElements) (JNIEnv* env, jfloatArray array, jboolean* isCopy);
  jdouble* (*GetDoubleArrayElements) (JNIEnv* env, jdoubleArray array, jboolean* isCopy);

  void (*ReleaseBooleanArrayElements) (JNIEnv* env, jbooleanArray array, jboolean* elems, jint mode);
  void (*ReleaseByteArrayElements) (JNIEnv* env, jbyteArray array, jbyte* elems, jint mode);
  void (*ReleaseCharArrayElements) (JNIEnv* env, jcharArray array, jchar* elems, jint mode);
  void (*ReleaseShortArrayElements) (JNIEnv* env, jshortArray array, jshort* elems, jint mode);
  void (*ReleaseIntArrayElements) (JNIEnv* env, jintArray array, jint* elems, jint mode);
  void (*ReleaseLongArrayElements) (JNIEnv* env, jlongArray array, jlong* elems, jint mode);
  void (*ReleaseFloatArrayElements) (JNIEnv* env, jfloatArray array, jfloat* elems, jint mode);
  void (*ReleaseDoubleArrayElements) (JNIEnv* env, jdoubleArray array, jdouble* elems, jint mode);

  void (*GetBooleanArrayRegion) (JNIEnv* env, jbooleanArray array, jsize start, jsize len, jboolean* buf);
  void (*GetByteArrayRegion) (JNIEnv* env, jbyteArray array, jsize start, jsize len, jbyte* buf);
  void (*GetCharArrayRegion) (JNIEnv* env, jcharArray array, jsize start, jsize len, jchar* buf);
  void (*GetShortArrayRegion) (JNIEnv* env, jshortArray array, jsize start, jsize len, jshort* buf);
  void (*GetIntArrayRegion) (JNIEnv* env, jintArray array, jsize start, jsize len, jint* buf);
  void (*GetLongArrayRegion) (JNIEnv* env, jlongArray array, jsize start, jsize len, jlong* buf);
  void (*GetFloatArrayRegion) (JNIEnv* env, jfloatArray array, jsize start, jsize len, jfloat* buf);
  void (*GetDoubleArrayRegion) (JNIEnv* env, jdoubleArray array, jsize start, jsize len, jdouble* buf);

  void (*SetBooleanArrayRegion) (JNIEnv* env, jbooleanArray array, jsize start, jsize len, jboolean* buf);
  void (*SetByteArrayRegion) (JNIEnv* env, jbyteArray array, jsize start, jsize len, jbyte* buf);
  void (*SetCharArrayRegion) (JNIEnv* env, jcharArray array, jsize start, jsize len, jchar* buf);
  void (*SetShortArrayRegion) (JNIEnv* env, jshortArray array, jsize start, jsize len, jshort* buf);
  void (*SetIntArrayRegion) (JNIEnv* env, jintArray array, jsize start, jsize len, jint* buf);
  void (*SetLongArrayRegion) (JNIEnv* env, jlongArray array, jsize start, jsize len, jlong* buf);
  void (*SetFloatArrayRegion) (JNIEnv* env, jfloatArray array, jsize start, jsize len, jfloat* buf);
  void (*SetDoubleArrayRegion) (JNIEnv* env, jdoubleArray array, jsize start, jsize len, jdouble* buf);

  /* Regsitering Native Methods */
  jint (*RegisterNatives) (JNIEnv* env, jclass clazz, const JNINativeMethod* methods, jint nMethods);
  jint (*UnregisterNatives) (JNIEnv* env, jclass clazz, const JNINativeMethod* methods, jint nMethods);

  /* Monitor Operations */
  jint (*MonitorEnter) (JNIEnv* env, jobject obj);
  jint (*MonitorExit) (JNIEnv* env, jobject obj);

  /* Java VM Interface */
  jint (*GetJavaVM) (JNIEnv* env, JavaVM** vm);
  void ( *GetStringRegion)(JNIEnv *env, jstring str, jsize start, jsize len, jchar *buf);
  void ( *GetStringUTFRegion)(JNIEnv *env, jstring str, jsize start, jsize len, char *buf);
  void *( *GetPrimitiveArrayCritical)(JNIEnv *env, jarray array, jboolean *isCopy);
  void ( *ReleasePrimitiveArrayCritical)(JNIEnv *env, jarray array, void *carray, jint mode);
  const jchar * (JNICALL *GetStringCritical)(JNIEnv *env, jstring string, jboolean *isCopy);
  void ( *ReleaseStringCritical)(JNIEnv *env, jstring string, const jchar *cstring);
  jweak ( *NewWeakGlobalRef)(JNIEnv *env, jobject obj);
  void ( *DeleteWeakGlobalRef)(JNIEnv *env, jweak ref);
  jboolean ( *ExceptionCheck) (JNIEnv *env);

  //--------------------------------------------------
  //kissme-specific methods
  //--------------------------------------------------

  jclass (*DefinePrimitiveClass) (JNIEnv* env, jobject loader, tClassLoaderTuple* class);

  #ifdef PERSIST
  /* for supporting persistence */
  jobject (*SwizzleObjectField) (JNIEnv* env, jobject obj, jfieldID field);
  jobject (*SwizzleArrayElement) (JNIEnv* env, jobjectArray array, jint element);
  void (*MarkUpdated) (JNIEnv* env, jobject obj);
  void (*MarkRefsUpdated) (JNIEnv* env, jobject obj);
  #endif PERSIST

  tAllocatorHeap* (*getHeap)(); //Gets the heap for the current process 
};

/*
 * @doc STRUCT
 * @struct tJNIData |
 * This structure stores data for JNI calls.
 *
 */

typedef struct tjnidata
{
  JNIEnv       functions;      /* @field Pointer to the JNINativeInterface structure */
  jobject       pstException;   /* @field Pointer to an exception if one has is being thrown */
  uint16       u16NumLocalRefs;/* @field Number of local references for current call */
  uint16       u16NumLocalRefsUsed; /* @field Number of local references that have been used */
  jobject*      ppstLocalRefs;  /* @field Pointer to array of local references for current call */
  tStackFrame* pstTopFrame;    /* @field Top stack frame before native call */

  tAllocatorHeap* pstHeap; /* Pointer to the heap associated with this JOSProcess */
} tJNIData;

extern uint16 JNI_u16NumGlobalRefs;
extern uint16 JNI_u16NumGlobalRefsUsed;
extern jobject* JNI_ppstGlobalRefs;
extern JNIEnv JNI_pstFunctionTable;

void JNI_zeroJNITable();
void JNI_Init(JNIEnv* env);
void JNI_Finish(void);
int JNI_CallNativeMethod(tStackFrame*, jmethodID);
tJNIData* JNI_getJNIData(int pid);
void JNI_setJNIData(tJNIData* data, int pid);

#ifdef PERSIST
#include "rpot.h"
/*#define JNI_SWIZZLE(o,f) (ISPID((PID)((o)->pi32Vars[(f)->u16Offset])) ? RPOT_SwizzleObjectField((o),(f)) : ((void*) ((o)->pi32Vars[(f)->u16Offset])))
#define JNI_WANGLE(x) (ISPID(x) ? RPOT_FaultObject(x) : (void*) (x))*/
#endif

#endif


//Variables for dynamic linking

#ifdef DLOPEN

extern int JNI_NumberLoadedLibraries;
extern void* JNI_DynLoadedLibraries[20];

#endif
