/*
 * Copyright (C) 2001, John Leuner.
 *
 * This file is part of the teaseme project, which in turn is part of the JOS project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2,
 * or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "jni.h"
#include "interp.h"

typedef jobject jthread;
typedef jobject jthreadGroup;

typedef unsigned long long jlocation;
typedef jint jvmdiError;
typedef tStackFrame* jframeID;


typedef jvmdiError (*JVMDI_AllocHook)(jint size, jbyte** memPtr);
typedef jvmdiError (*JVMDI_DeallocHook)(jbyte* buffer);

typedef tObject** JVMDI_RawMonitor;

typedef struct {
  jlocation start_location;
  jint line_number;
} JVMDI_line_number_entry;

typedef struct {
  jlocation start_location;   /* variable valid start_location */
  jint length;                /* upto start_location+length */ 
  char *name;                 /* name in UTF-8 */
  char *signature;            /* type signature in UTF-8 */
  jint slot;                  /* variable slot, see JVMDI_GetLocal*()  */
} JVMDI_local_variable_entry;

typedef struct {
  jlocation start_location;
  jlocation end_location;
  jlocation handler_location;
  jclass exception;           /* if null, all exceptions */
} JVMDI_exception_handler_entry;


typedef struct {
  unsigned int can_watch_field_modification      : 1;
  unsigned int can_watch_field_access            : 1;
  unsigned int can_get_bytecodes                 : 1;
  unsigned int can_get_synthetic_attribute       : 1;
  unsigned int can_get_owned_monitor_info        : 1;
  unsigned int can_get_current_contended_monitor : 1;
  unsigned int can_get_monitor_info              : 1;
} JVMDI_capabilities;

typedef struct {
  char *name;             /* Name in UTF-8 */
  jint priority;
  jboolean is_daemon;
  jthreadGroup thread_group;
  jobject context_class_loader;
} JVMDI_thread_info;

typedef struct {
  jthread owner;
  jint entry_count;
  jint waiter_count;
  jthread *waiters;
} JVMDI_monitor_info;

typedef struct {
  jint owned_monitor_count;
  jobject *owned_monitors;
} JVMDI_owned_monitor_info;

typedef void (*JVMDI_StartFunction)(void *);

typedef struct {
  jthreadGroup parent;
  char *name;             /* Name in UTF-8 */
  jint  max_priority;
  jboolean is_daemon;
} JVMDI_thread_group_info;

//Events

//Single-step

typedef struct { 
  jthread thread;
  jclass clazz;
  jmethodID method;
  jlocation location;
} JVMDI_single_step_event_data;

//Breakpoint

typedef struct { 
  jthread thread;
  jclass clazz;
  jmethodID method;
  jlocation location;
} JVMDI_breakpoint_event_data;

//Field events

/* kind = JVMDI_EVENT_FIELD_ACCESS */
typedef struct { 
  jthread thread;
  jclass clazz;
  jmethodID method;
  jlocation location;
  jclass field_clazz;
  jobject object;
  jfieldID field;
} JVMDI_field_access_event_data;
    
/* kind = JVMDI_EVENT_FIELD_MODIFICATION */
typedef struct { 
  jthread thread;
  jclass clazz;
  jmethodID method;
  jlocation location;
  jclass field_clazz;
  jobject object;
  jfieldID field;
  char signature_type;
  jvalue new_value;
} JVMDI_field_modification_event_data;

//Frame events

typedef struct { 
  jthread thread;
  jclass clazz;
  jmethodID method;
  jframeID frame;
} JVMDI_frame_event_data;

//Exception events

typedef struct { 
  jthread thread;
  jclass clazz;
  jmethodID method;
  jlocation location;
  jobject exception;
  jclass catch_clazz;
  jmethodID catch_method;
  jlocation catch_location;
} JVMDI_exception_event_data;

//Exception catch events

typedef struct { 
  jthread thread;
  jclass clazz;
  jmethodID method;
  jlocation location;
  jobject exception;
} JVMDI_exception_catch_event_data;

//User defined

typedef struct { 
  jobject object;
  jint key;
} JVMDI_user_event_data;

//Thread change

typedef struct { 
  jthread thread;
} JVMDI_thread_change_event_data;

//Class events
typedef struct { 
  jthread thread;
  jclass clazz;
} JVMDI_class_event_data;


typedef struct {
  jint kind;/* the discriminant */

  union {
    /* kind = JVMDI_EVENT_SINGLE_STEP */
    JVMDI_single_step_event_data single_step;
    
    /* kind = JVMDI_EVENT_BREAKPOINT */
    JVMDI_breakpoint_event_data breakpoint;

    /* kind = JVMDI_EVENT_FRAME_POP */
    /* kind = JVMDI_EVENT_METHOD_ENTRY */
    /* kind = JVMDI_EVENT_METHOD_EXIT */
    JVMDI_frame_event_data frame;

    /* kind = JVMDI_EVENT_FIELD_ACCESS */
    JVMDI_field_access_event_data field_access;

    /* kind = JVMDI_EVENT_FIELD_MODIFICATION */
    JVMDI_field_modification_event_data field_modification;

    /* kind = JVMDI_EVENT_EXCEPTION */
    JVMDI_exception_event_data exception;

    /* kind = JVMDI_EVENT_EXCEPTION_CATCH */
    JVMDI_exception_catch_event_data exception_catch;

    /* kind = JVMDI_EVENT_USER_DEFINED */
    JVMDI_user_event_data user;
    
    /* kind = JVMDI_EVENT_THREAD_END or */
    /* JVMDI_EVENT_THREAD_START */
    JVMDI_thread_change_event_data thread_change;
    
    /* kind = JVMDI_EVENT_CLASS_LOAD, */
    /* JVMDI_EVENT_CLASS_UNLOAD, or */
    /* JVMDI_EVENT_CLASS_PREPARE */
    JVMDI_class_event_data class_event;
    
    /* kind = JVMDI_EVENT_VM_DEATH, JVMDI_EVENT_VM_INIT */
    /* no additional fields */
  } u;
} JVMDI_Event;


//JVMDI_EventHook

typedef void (*JVMDI_EventHook)(JNIEnv *env , JVMDI_Event *event);



/* General return codes for JVMDI methods */


#define JVMDI_ERROR_NONE 0
//No error has occurred. This is the error code that is returned on successful completion of the function.

#define JVMDI_ERROR_OUT_OF_MEMORY 1
//The function needed to allocate memory and no more memory was available for allocation.

#define JVMDI_ERROR_ACCESS_DENIED 2
//Debugging has not been enabled in this virtual machine. JVMDI cannot be used.

#define JVMDI_ERROR_UNATTACHED_THREAD 3
//The thread being used to call this function is not attached to the virtual machine. Calls must be made from attached threads. See AttachCurrentThread() in the JNI invocation API.

#define JVMDI_ERROR_VM_DEAD 4
//The virtual machine is not running.

#define JVMDI_ERROR_INTERNAL 5
//An unexpected internal error has occurred.

/* Return codes for specific methods */

#define JVMDI_ERROR_INVALID_THREAD 6
//passed thread is not a valid thread or has exited

#define JVMDI_ERROR_INVALID_FIELDID 7
//invalid field

#define JVMDI_ERROR_INVALID_METHODID 8
//invalid method

#define JVMDI_ERROR_INVALID_LOCATION 9
//invalid location

#define JVMDI_ERROR_INVALID_FRAMEID 10
//invalid jframeID

#define JVMDI_ERROR_NO_MORE_FRAMES 11
//There are no more Java or JNI frames on the call stack.

#define JVMDI_ERROR_OPAQUE_FRAME 12
//information about the frame is not available (e.g. for native frames)

#define JVMDI_ERROR_NOT_CURRENT_FRAME 13
//operation can only be performed on current frame

#define JVMDI_ERROR_TYPE_MISMATCH 14
//the variable is not an appropriate type for the function used

#define JVMDI_ERROR_INVALID_SLOT 15
//invalid slot

#define JVMDI_ERROR_DUPLICATE 16
//item already set

#define JVMDI_ERROR_THREAD_NOT_SUSPENDED 17
//thread was not suspended

#define JVMDI_ERROR_THREAD_SUSPENDED 18
//thread already suspended

#define JVMDI_ERROR_INVALID_OBJECT 19
//invalid object

#define JVMDI_ERROR_INVALID_CLASS 20
//invalid class

#define JVMDI_ERROR_CLASS_NOT_PREPARED 21
//class has been loaded but not yet prepared

#define JVMDI_ERROR_NULL_POINTER 22
//invalid pointer

#define JVMDI_ERROR_ABSENT_INFORMATION 23
//The requested information is not available.

#define JVMDI_ERROR_INVALID_EVENT_TYPE 24
//The specified event type id is not recognized.

#define JVMDI_ERROR_NOT_IMPLEMENTED 25
//The functionality is not implemented in this virtual machine

#define JVMDI_ERROR_INVALID_THREAD_GROUP 26
//thread group invalid

#define JVMDI_ERROR_INVALID_PRIORITY 27
//invalid priority

#define JVMDI_ERROR_NOT_FOUND 28
//Desired element (e.g. field or breakpoint) not found

#define JVMDI_ERROR_INVALID_MONITOR 29
//invalid monitor

#define JVMDI_ERROR_ILLEGAL_ARGUMENT 30
//illegal argument

#define JVMDI_ERROR_NOT_MONITOR_OWNER 31
//This thread doesn't own the monitor.

//also defined as 23
//#define JVMDI_ERROR_ABSENT_INFORMATION 32
//Desired information is not available.

#define JVMDI_ERROR_INTERRUPT 33
//The call has been interrupted before completion.

#define JVMDI_ERROR_INVALID_TYPESTATE 33
//The state of the thread has been modified, and is now inconsistent.

