/*
 * Copyright (C) 2001, John Leuner.
 *
 * This file is part of the teaseme project, which in turn is part of the JOS project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2,
 * or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* JVMDI support for manipulating raw monitors 
 *
 * Lets use the mechanisms in place for Java monitors, we'll just create fake objects (not on the GC heap)
 *
 */

#include <config.h>

#ifdef HAVE_JVMDI

#include "jvmdi.h"
#include "jvmdi_methods.h"
#include "thread.h"
#include "locks.h"

jvmdiError
CreateRawMonitor(char *name, JVMDI_RawMonitor *monitorPtr)
{
  tObject* rawObject;
  tObject** handle;
  jvmdiError tmpError;

  tmpError = JVMDI_Allocate(sizeof(tObject*) + sizeof(tObject), (jbyte**) &handle);
  if(tmpError != JVMDI_ERROR_NONE)
    {
      return tmpError;
    }

  *handle = ((tObject*) handle) + 1;
  rawObject = *handle;

  rawObject->pid = NULL;
  rawObject->pi32Vars = NULL;
  rawObject->lock_word = 0;
  rawObject->condVariable = sys_condition_create();

  *monitorPtr = handle;
  return JVMDI_ERROR_NONE;
}

jvmdiError
DestroyRawMonitor(JVMDI_RawMonitor monitor)
{
  tObject* rawObject;

  rawObject = *monitor;
  sys_condition_destroy(rawObject->condVariable);
  JVMDI_Deallocate( (jbyte*) monitor);
  return JVMDI_ERROR_NONE;
}

jvmdiError
RawMonitorEnter(JVMDI_RawMonitor monitor)
{
  THREAD_SynchroniseEnter(monitor);
  return JVMDI_ERROR_NONE;
}


jvmdiError
RawMonitorExit(JVMDI_RawMonitor monitor)
{
  THREAD_SynchroniseExit(monitor);
  return JVMDI_ERROR_NONE;
}

jvmdiError
RawMonitorWait(JVMDI_RawMonitor monitor, jlong millis)
{
  THREAD_WaitOnObject(NULL, monitor, millis);
  return JVMDI_ERROR_NONE;
}

jvmdiError
RawMonitorNotify(JVMDI_RawMonitor monitor)
{
  THREAD_NotifyRandom(NULL, monitor);
  return JVMDI_ERROR_NONE;
}

jvmdiError
RawMonitorNotifyAll(JVMDI_RawMonitor monitor)
{
  THREAD_NotifyAll(NULL, monitor);
  return JVMDI_ERROR_NONE;
}
#endif
