 
#include <config.h>

#ifndef GARBAGE_GENERATIONAL
#ifndef GARBAGE_COPYING
/*
 * RPOT.c
 *
 * Contains the Resident Persistent Object Table.
 *
 */

#ifndef GARBAGE
#error GARBAGE must be defined for RPOT.c
#endif

#ifdef PERSIST
/*#ifndef PERSIST
#error PERSIST must be defined for RPOT.c
#endif*/

#include <assert.h>

#ifdef KISSME_LINUX_USER
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#endif

#include <global.h>
#include "garbage.h"
#include "store.h"
#include "rpot.h"
#include "jni.h"
#include "classfile_methods.h"

#ifdef PJSLREAL
#include "pjsl_kind_to_regime.h"
#endif

#ifdef REVRPOT
tRPOTElement** RPOT_table = NULL;
tRPOTElement** RPOT_revtable = NULL;
#else
tRPOTElement* RPOT_table = NULL;
#endif
static int iNumItems;


#ifdef NEWRPOT

/*
   New RPOT:

   Instead of allocating a main table and then a separate block for each
   overflow entry, we allocate a large bucket to accomodate a number of
   overflow entries. We keep track of all these bucket so that we can
   find easily if an arbitrary address is within the RPOT.

   The buckets we create a PHYSICAL buckets, not LOGICAL buckets in the
   way that they are usually associated with hash tables. The logical
   buckets are created by the linked lists that are kept, but these lists
   can include elements from any of the physical buckets.

   The entries have the same format as before - with a "next" pointer
   pointing to overflows. This means that searching can be done the
   same as it was previously.

*/

typedef struct rpotoverflowbucket
{
  tRPOTElement* pstFirstFreeElement;
  struct rpotoverflowbucket* pstNext;
  struct rpotoverflowbucket* pstNextFree;
  tRPOTElement  astElements[1]; /* must be last member */
} tRPOTOverflowBucket;

tRPOTOverflowBucket* pstBucketList = NULL;
tRPOTOverflowBucket* pstFreeBucketList = NULL;

static tRPOTOverflowBucket* WhichBucket
(
  tRPOTElement* elem
)
{
  tRPOTOverflowBucket* temp;

  temp = pstBucketList;
  while (temp)
  {
    /* if the element is within the memory range of this bucket, we;ve found it */
    if (((void*) elem) > ((void*) temp) &&
        ((void*) elem) <= ((void*) &(temp->astElements[RPOT_OVERFLOWSIZE - 1])))
    {
      return temp;
    }
    temp = temp->pstNext;
  }
  return NULL;
}

int RPOT_InRPOT
(
  void* ref
)
{
  /* just pass ref to WhichBucket - it doesn't have to be a real tRPOTElement
     pointer - will work if it points to the la field of the tRPOTElement and
     not to the front */
  return (WhichBucket(ref) != NULL);
}


void RPOT_Init
(
  void
)
{
  int i;

  /* only need main table */

  RPOT_table = (tRPOTElement*) sys_malloc(sizeof(tRPOTElement) * RPOT_TABLESIZE);
  for (i = 0; i < RPOT_TABLESIZE; i++)
  {
    RPOT_table[i].pid = (PID) NULL;
    RPOT_table[i].la = (PID) NULL;
    RPOT_table[i].next = NULL;
  }
}

void RPOT_Kill
(
  void
)
{
  tRPOTOverflowBucket* temp;
  tRPOTOverflowBucket* next;

  sys_free(RPOT_table);
  temp = pstBucketList;
  while (temp)
  {
    next = temp->pstNext;
    sys_free(temp);
    temp = next;
  }
}

tObject* RPOT_GetLA
(
  PID pid
)
{
  tRPOTElement* temp;

/*  printf("g");*/
/*  printf("RPOT find: %d", pid);*/

  if ((temp = &(RPOT_table[HASHFUNC(pid)]))->pid == pid)
  {
/*    printf(" %p\n", temp->la);*/
    return temp->la;
  }
  temp = temp->next;
  while (temp)
  {
    if (temp->pid == pid)
    {
/*      printf(" %p\n", temp->la);*/
      return temp->la;
    }
    temp = temp->next;
  }
/*  printf(" NULL\n");*/
  return NULL;
}

/* Adding is a simple process if there is nothing in the required slot
   in the main table. If this slot is occupied, however, we must find a
   free slot in an overflow bucket. If none is available, we must create
   a bucket into which to put it. */

tRPOTElement* RPOT_Add
(
  PID pid,
  void* la
)
{
  tRPOTElement* temp;
  tRPOTElement* newtemp;
  tRPOTElement* retval;
  tRPOTOverflowBucket* pstNewBucket;
  int i;

/*  printf("a");*/
/*  printf("RPOT add: %d (%d)\n", pid, HASHFUNC(pid));*/

  temp = &(RPOT_table[HASHFUNC(pid)]);
  if (temp->pid)
  {
    /* this space is used, so go to the end of the overflow chain
       (will be 0 steps if there is no overflow */
    while (temp->next)
    {
      temp = temp->next;
    }
    /* now we need to add a new entry, so look to see if there is space
       in an overflow bucket (if there is an overflow bucket) */
    if (pstFreeBucketList)
    {
      /* use the first available space in first available bucket */
      newtemp = pstFreeBucketList->pstFirstFreeElement;
      pstFreeBucketList->pstFirstFreeElement = newtemp->next;
      /* if that was the last free space in that bucket then we need to remove
         the bucket from the list of free buckets */
      if (pstFreeBucketList->pstFirstFreeElement == NULL)
      {
        pstFreeBucketList = pstFreeBucketList->pstNextFree;
      }
      newtemp->pid = pid;
      newtemp->la = la;
      newtemp->next = NULL;
      temp->next = newtemp;
      retval = newtemp;
    }
    else
    {
      /* allocate new bucket */
      /* put in -1 because the bucket definition contains 1 */
      pstNewBucket = (tRPOTOverflowBucket*) sys_malloc(sizeof(tRPOTOverflowBucket) + (sizeof(tRPOTElement) * (RPOT_OVERFLOWSIZE - 1)));
      #ifdef DEBUG
      assert(pstNewBucket);
      #endif

      /* add bucket to bucket list */
      pstNewBucket->pstNext = pstBucketList;
      pstBucketList = pstNewBucket;
      /* add bucket to free bucket list */
      pstNewBucket->pstNextFree = pstFreeBucketList;
      pstFreeBucketList = pstNewBucket;

      /* go through all elements and initialise them... use their next pointers
         to point to the next free element - set them all up beforehand */
      for (i = 0; i < (RPOT_OVERFLOWSIZE - 2); i++)
      {
        pstNewBucket->astElements[i].pid = (PID) NULL;
        pstNewBucket->astElements[i].la = (PID) NULL;
        pstNewBucket->astElements[i].next = &(pstNewBucket->astElements[i + 1]);
      }
      pstNewBucket->astElements[i].pid = (PID) NULL;
      pstNewBucket->astElements[i].la = (PID) NULL;
      pstNewBucket->astElements[i].next = NULL;

      /* set first free to second slot coz we're about to use the first
         slot right now */
      pstNewBucket->pstFirstFreeElement = &(pstNewBucket->astElements[1]);

      /* put info into first slot */
      pstNewBucket->astElements[0].pid = pid;
      pstNewBucket->astElements[0].la = la;
      pstNewBucket->astElements[0].next = NULL;
      temp->next = &(pstNewBucket->astElements[0]);
      retval = &(pstNewBucket->astElements[0]);
    }
  }
  else
  {
    /* this space is empty, so just set the values */
    temp->pid = pid;
    temp->la = la;
    retval = temp;
  }

  iNumItems += 1;

  return retval;
}

/* Removal can be complicated. If an overflow element is freed, it must
   be added to its bucket's free list. If the bucket previously had no
   free elements, it must be added to the free bucket list. */

void RPOT_Remove
(
  PID pid
)
{
  tRPOTElement* temp;
  tRPOTElement* next;
  tRPOTElement* prev;
  tRPOTOverflowBucket* pstTempBucket;

/*  printf("r");*/

  temp = &(RPOT_table[HASHFUNC(pid)]);
  if (temp->pid == pid)
  {
    /* it's in the main table... but we might have to move the info from
       the next overflow element into here if there is an overflow */
    if (temp->next)
    {
      /* put next item's info into main table */
      next = temp->next;
      temp->pid = next->pid;
      temp->la = next->la;
      temp->next = next->next;

      /* free up the overflow item */
      pstTempBucket = WhichBucket(next);
      next->next = pstTempBucket->pstFirstFreeElement;
      pstTempBucket->pstFirstFreeElement = next;
      /* check if this bucket has to be put onto the free bucket list -
         will be true if it previously had no free elements (ie if the
         newly freed element's next pointer is NULL */
      if (next->next == NULL)
      {
        pstTempBucket->pstNextFree = pstFreeBucketList;
        pstFreeBucketList = pstTempBucket;
      }
    }
    else
    {
      temp->pid = (PID) NULL;
      temp->la = (PID) NULL;
    }
  }
  else
  {
    prev = temp;
    temp = temp->next;
    while (temp)
    {
      if (temp->pid == pid)
      {
        /* link previous one to next one, skipping this one */
        prev->next = temp->next;
        /* deal with the free list in the bucket */
        pstTempBucket = WhichBucket(temp);
        temp->next = pstTempBucket->pstFirstFreeElement;
        pstTempBucket->pstFirstFreeElement = temp;
        if (temp->next == NULL)
        {
          pstTempBucket->pstNextFree = pstFreeBucketList;
          pstFreeBucketList = pstTempBucket;
        }
        temp = NULL; /* to make the loop end without using "break" */
      }
      else
      {
        prev = temp;
        temp = temp->next;
      }
    }
  }
  iNumItems -= 1;
}

tOBREF RPOT_GetObref
(
  PID pid
)
{
  tRPOTElement* temp;

  if ((temp = &(RPOT_table[HASHFUNC(pid)]))->pid == pid)
  {
    return &(temp->la);
  }
  temp = temp->next;
  while (temp)
  {
    if (temp->pid == pid)
    {
      return &(temp->la);
    }
    temp = temp->next;
  }

  return NULL;
}

#else

int RPOT_InRPOT
(
  void* ref
)
{
  return 1;
}

/*
 * Initialises the RPOT_table.
 *
 */

void RPOT_Init
(
  void
)
{
  int i;

  #ifdef REVRPOT
  RPOT_revtable = (tRPOTElement**) sys_malloc(sizeof(tRPOTElement*) * RPOT_TABLESIZE);
  RPOT_table = (tRPOTElement**) sys_malloc(sizeof(tRPOTElement*) * RPOT_TABLESIZE);
  memset(RPOT_revtable, 0, sizeof(tRPOTElement*) * RPOT_TABLESIZE);
  memset(RPOT_table, 0, sizeof(tRPOTElement*) * RPOT_TABLESIZE);
  #else
  RPOT_table = (tRPOTElement*) sys_malloc(sizeof(tRPOTElement) * RPOT_TABLESIZE);
  for (i = 0; i < RPOT_TABLESIZE; i++)
  {
    RPOT_table[i].pid = NULL;
    RPOT_table[i].la = NULL;
    RPOT_table[i].next = NULL;
  }
  #endif

  iNumItems = 0;
}

/*
 * Reclaims all space used by the RPOT_table.
 *
 */

void RPOT_Kill
(
  void
)
{
  int i;
  tRPOTElement* temp;
  tRPOTElement* next;

  #ifdef REVRPOT
  /* go through forward RPOT_table and delete all elements */
  /* we won't need to do this with the reverse RPOT_table
     because all the elements will have been killed
     by going through the forward RPOT_table */
  for (i = 0; i < RPOT_TABLESIZE; i++)
  {
    if (RPOT_table[i])
    {
      temp = RPOT_table[i]->next;
      sys_free(RPOT_table[i]);
      while (temp)
      {
        next = temp->next;
        sys_free(temp);
        temp = next;
      }
    }
  }
  sys_free(RPOT_table);
  sys_free(RPOT_revtable);
  #else
  for (i = 0; i < RPOT_TABLESIZE; i++)
  {
    temp = RPOT_table[i].next;
    while (temp)
    {
      next = temp->next;
      sys_free(temp);
      temp = next;
    }
  }
  sys_free(RPOT_table);
  #endif
}

/*
 * Returns the LA of the object if it is in the RPOT_table.
 *
 */

tObject* RPOT_GetLA
(
  PID pid
)
{
  tRPOTElement* temp;

  #ifdef REVRPOT
  if (RPOT_table[HASHFUNC(pid)] == NULL)
  {
    return NULL;
  }
  if ((temp = RPOT_table[HASHFUNC(pid)])->pid == pid)
  {
    return temp->la;
  }
  #else
  if ((temp = &(RPOT_table[HASHFUNC(pid)]))->pid == pid)
  {
    return temp->la;
  }
  #endif

  temp = temp->next;
  while (temp)
  {
    if (temp->pid == pid)
    {
      return temp->la;
    }
    temp = temp->next;
  }

  return NULL;
}

#ifdef REVRPOT
/*
 * Returns the LA of the object if it is in the RPOT_table.
 *
 */

PID RPOT_GetPID
(
  void* la
)
{
  tRPOTElement* temp;

  if (RPOT_revtable[REVHASHFUNC(la)])
  {
    if ((temp = RPOT_revtable[REVHASHFUNC(la)])->la == la)
    {
      return temp->pid;
    }

    temp = temp->revnext;
    while (temp)
    {
      if (temp->la == la)
      {
        return temp->pid;
      }
      temp = temp->revnext;
    }
  }

  return NULL;
}
#endif

/*
 * Adds a PID/LA pair to the RPOT_table. Does not check for duplicates because
 * this should only be called when a search for the entry has failed.
 *
 */

tRPOTElement* RPOT_Add
(
  PID pid,
  void* la
)
{
  tRPOTElement* temp;

  #ifdef REVRPOT
  int index;
  tRPOTElement* revtemp;

  index = HASHFUNC(pid);
  if (temp = RPOT_table[index])
  {
    while (temp->next)
    {
      temp = temp->next;
    }
    temp->next = (tRPOTElement*) sys_malloc(sizeof(tRPOTElement));
    temp = temp->next;
    temp->pid = pid;
    temp->la = la;
    temp->next = NULL;
    temp->revnext = NULL;
  }
  else
  {
    RPOT_table[index] = (tRPOTElement*) sys_malloc(sizeof(tRPOTElement));
    temp = RPOT_table[index];
    temp->pid = pid;
    temp->la = la;
    temp->next = NULL;
    temp->revnext = NULL;
  }

  index = REVHASHFUNC(temp->la);
  if (revtemp = RPOT_revtable[index])
  {
    while (revtemp->revnext)
    {
      revtemp = revtemp->revnext;
    }
    /* just point the next pointer to the existing element */
    revtemp->revnext = temp;
  }
  else
  {
    /* point straight to existing element */
    RPOT_revtable[index] = temp;
  }
  #else
  if ((temp = &(RPOT_table[HASHFUNC(pid)]))->pid)
  {
    while (temp->next)
    {
      temp = temp->next;
    }
    temp->next = (tRPOTElement*) sys_malloc(sizeof(tRPOTElement));
    temp = temp->next;
    temp->pid = pid;
    temp->la = la;
    temp->next = NULL;
  }
  else
  {
    temp->pid = pid;
    temp->la = la;
  }
  #endif

  iNumItems += 1;

  return temp;
}

/*
 * Removes an entry from the RPOT_table. Indexed by PID.
 *
 */

void RPOT_Remove
(
  PID pid
)
{
  tRPOTElement* temp;
  tRPOTElement* prev;

  #ifdef REVRPOT
  int index;
  void* la;

  index = HASHFUNC(pid);
  /* find the element */
  for (prev = NULL, temp = RPOT_table[index]; temp->pid != pid; prev = temp, temp = temp->next);
  /* remember la for later */
  la = temp->la;
  if (prev)
  {
    prev->next = temp->next;
  }
  else
  {
    RPOT_table[index] = temp->next;
  }
  /* don't free it yet - have to do the reverse version first */

  index = REVHASHFUNC(la);
  for (prev = NULL, temp = RPOT_revtable[index]; temp->pid != pid; prev = temp, temp = temp->revnext);
  if (prev)
  {
    prev->revnext = temp->revnext;
  }
  else
  {
    RPOT_revtable[index] = temp->revnext;
  }
  /* now free it */
  sys_free(temp);

  #else
  tRPOTElement* next;

  if ((temp = &(RPOT_table[HASHFUNC(pid)]))->pid == pid)
  {
    if (temp->next)
    {
      temp->pid = temp->next->pid;
      temp->la = temp->next->la;
      next = temp->next;
      temp->next = temp->next->next;
      sys_free(next);
    }
    else
    {
      temp->pid = NULL;
      temp->la = NULL;
    }
  }
  else
  {
    prev = temp;
    temp = temp->next;
    #ifdef DEBUG
    assert(temp);
    #endif
    while (temp->pid != pid)
    {
      prev = temp;
      temp = temp->next;
      #ifdef DEBUG
      assert(temp);
      #endif
    }
    prev->next = temp->next;
    sys_free(temp);
  }

  #endif

  iNumItems -= 1;
}

/*
 * Returns the address of the object's la field of its RPOT entry if it is
 * in the table.
 *
 */

tOBREF RPOT_GetObref
(
  PID pid
)
{
  tRPOTElement* temp;

  #ifdef REVRPOT
  if (RPOT_table[HASHFUNC(pid)] == NULL)
  {
    return NULL;
  }
  if ((temp = RPOT_table[HASHFUNC(pid)])->pid == pid)
  {
    return &(temp->la);
  }
  #else
  if ((temp = &(RPOT_table[HASHFUNC(pid)]))->pid == pid)
  {
    return &(temp->la);
  }
  #endif

  temp = temp->next;
  while (temp)
  {
    if (temp->pid == pid)
    {
      return &(temp->la);
    }
    temp = temp->next;
  }

  return NULL;
}

#endif

/*
 * Faults in an object, returning its handle.
 *
 */

tOBREF RPOT_FaultObject
(
  PID pid
)
{
  tObject* la;
  tObject* obj;
  tArray* arr;
  char* className;
  PID classPID;
  #ifdef PJSLCOMPAT
	pjsl_obj_info_t info;
	pjsl_size_t objSize;
  pjsl_kind_t kind;
  pjsl_regime_t regime;
  pjsl_hid_t histID = PJSL_HISTMAN_ILLEGAL_HID;
  pjsl_err_t err;
  #else
	  int objSize;
  #endif

  #ifdef RPOTREFS
	tRPOTElement* rpotelem;
  #else
	  tOBREF objref;
  #endif
  #ifdef EAGERINDIRECT
  PID refPID;
  tClass* pstCurrType;
  int iFinished = 0;
  int i;
  tField* field;
  tObject* ref;
  tOBREF refH;
  tRPOTElement* refrpotelem;
  #endif

  #ifdef DEBUG
  if (!ISPID(pid))
  {
    eprintf("meow");
  }
  assert(ISPID(pid));
  #endif



  /* residency check first */
  la = RPOT_GetLA(pid);
  if (la)
  {
    #ifdef EVICTIONS
    if (ISPID(la))
    {
      /* the object has been faulted in but subsequently swopped out
         so....... we swop it back in here to make the interpreter
         happy */
      return RPOT_SwapIn(pid);
    }
    #endif
    #ifdef RPOTREFS
    /* it's already resident (false object fault) so just return the handle */
    if (la->hHandle)
    {
      return la->hHandle;
    }
    else
    {
      return RPOT_GetObref(pid);
    }
    #else
    return la->hHandle;
    #endif
  }

  //  printf("PID not resident %x\n", pid);

#ifdef DEBUG
	#ifndef USEOSKIT
  assert(pid < 0);
#endif
#endif
  #ifdef EVICTIONS
  la = TPOT_GetLA(pid);
  if (la)
  {
    if (ISPID(la))
    {
      /* the object has been faulted in but subsequently swopped out
         so....... we swop it back in here to make the interpreter
         happy */
      return RPOT_SwapIn(pid);
    }
    else
    {
      return la->hHandle;
    }
  }
  #endif


    /* get object from store */
    #ifdef PJSLCOMPAT

	err = pjsl_objInspect(STORE_Store, pid, histID, &kind, &regime, &objSize);
    #else
	    objSize = STORE_GetObjectSize(pid);
#ifdef DEBUG
	#ifndef USEOSKIT
	    assert(objSize >= 0);
	#endif
#endif
    #endif
    #ifdef RPOTREFS
	obj = NULL; //GARBAGE_MallocPtr(objSize); //XXX broken by new GC scheme
    #else
	objref = GARBAGE_Malloc(objSize);
	obj = PDEREF(objref);
    #endif
    #ifdef PJSLCOMPAT
	err = PJSL_KIND_OP(kind,regime,fetch)(STORE_Store, pid, histID, &info, (uchar*) obj);
    #ifdef RPOTREFS
	rpotelem = RPOT_Add(pid, obj);
    #else
	RPOT_Add(pid, obj);
    #endif
    #else
	STORE_LoadObject(pid, (void*) obj);
    #ifdef RPOTREFS
	rpotelem = RPOT_Add(pid, obj);
    #else
	RPOT_Add(pid, obj);
    #endif
    #endif

    #ifndef REVRPOT
	obj->pid = pid;
    #endif

    #ifdef RPOTREFS
    /* its handle field must be NULL because it's using its RPOT entry as
       its handle */
    obj->hHandle = NULL;
    #else
    obj->hHandle = objref;
    #endif


    /* point field pointer to end of object struct data */
    if ((obj->i32Flags & GARBAGE_TYPEMASK) == GARBAGE_ARRAY)
    {
      arr = (tArray*) obj;
      arr->pvElements = (((byte*) obj) + ROUND32BIT(sizeof(tArray)));
    }
    else
    {
      obj->pi32Vars = (int32*) (((byte*) obj) + ROUND32BIT(sizeof(tObject)));
    }

    /* ### deal with locks, etc */

    /* get the object's class */
    #ifdef PJSLCOMPAT
    err = pjsl_objInspect(STORE_Store, (PID) obj->pstType, histID, &kind, &regime, &objSize);
    #else

    objSize = STORE_GetObjectSize((PID) obj->pstType);
#ifdef DEBUG
	#ifndef USEOSKIT
    assert(objSize >= 0);
      #endif
    #endif
#endif

    if(objSize < 0)
      {
	     eprintf("RPOT_FaultObject, obj size is %i, class pid is %p\n", objSize, obj->pstType);
	     exit(33);
      }

    #ifdef SHOWLOADING
     eprintf("RPOT_FaultObject, obj size is %i, class pid is %x\n", objSize, obj->pstType);
    #endif
    className = (char*) sys_malloc(sizeof(char) * (objSize + 1));
    classPID = (PID) obj->pstType;
    #ifdef PJSLCOMPAT
    err = PJSL_KIND_OP(kind,regime,fetch)(STORE_Store, classPID, histID, &info, (uchar*) className);
    #else
    STORE_LoadObject(classPID, className);
    #endif
    className[objSize] = 0;
    assert( 1 == 2);
//    obj->pstType = CLASSFILE_FindOrLoad(className);
    if (obj->pstType->pstClass->persIndex == (PID) NULL)
    {
      obj->pstType->pstClass->persIndex = classPID;
    }
    sys_free(className);

    #ifdef EVICTIONS
    SETFLAG(obj,GARBAGE_PERSISTENT);
    #endif

  #ifdef EAGERINDIRECT
  /* make handles for all the object's references */
  /* for each reference field, if the object referred to by the PID is
     already loaded then we must use its handle, otherwise allocate a
     new handle and put the PID in it */
  if ((obj->i32Flags & GARBAGE_TYPEMASK) == GARBAGE_OBJECT)
  {
    if (obj->pi32Vars)
    {
      /* Check the fields (looking at parent classes too) */
      for (pstCurrType = obj->pstType; iFinished == 0; pstCurrType = pstCurrType->pstSuperClass)
      {
        for (i = 0; i < pstCurrType->u16InstCount; i++)
        {
          field = &(pstCurrType->pstInstOffsets[i]);
          switch (field->uidFieldSig[0])
          {
            case '[':
            case 'L':
            {
              refPID = obj->pi32Vars[field->u16Offset];
              if (refPID)
              {
                /* look in RPOT */
                ref = RPOT_GetLA(refPID);
                if (ref)
                {
                  #ifdef RPOTREFS
                  /* ### maybe check in case this object somehow has a handle
                         as well as being in the RPOT */
                  refH = RPOT_GetObref(refPID);
                  #else
                  refH = ref->hHandle;
                  #endif
                }
                else
                {
                  ref = TPOT_GetLA(refPID);
                  if (ref)
                  {
                    /* should have a handle */
                    refH = ref->hHandle;
                  }
                  else
                  {
                    /* object not resident so allocate a handle / RPOT entry
                       for it in its absence */
                    #ifdef RPOTREFS
                    refrpotelem = RPOT_Add(refPID, (void*) refPID);
                    refH = &(refrpotelem->la);
                    #else
                    refH = GARBAGE_AllocHandle();
                    PDEREF(refH) = (tObject*) refPID;
                    #endif
                  }
                }

                /* overwrite reference with the handle */
                obj->pi32Vars[field->u16Offset] = (int32) refH;
              }

              break;
            }
          }
        }
        if ((pstCurrType->pstSuperClass == NULL) ||
            ((pstCurrType->pstInstOffsets) && (pstCurrType->pstInstOffsets[0].u16Offset == 0)))
        {
          iFinished = 1;
        }
      }

    }
  }
  else
  {
    arr = (tArray*) obj;

    if ((arr->enType == T_ARRAY) || (arr->enType == T_OBJECT))
    {
      for (i = 0; i < arr->i32Number; i++)
      {
        refPID = ((PID*) arr->pvElements)[i];

              if (refPID)
              {
                /* look in RPOT */
                ref = RPOT_GetLA(refPID);
                if (ref)
                {
                  #ifdef RPOTREFS
                  /* ### maybe check in case this object somehow has a handle
                         as well as being in the RPOT */
                  refH = RPOT_GetObref(refPID);
                  #else
                  refH = ref->hHandle;
                  #endif
                }
                else
                {
                  ref = TPOT_GetLA(refPID);
                  if (ref)
                  {
                    /* should have a handle */
                    refH = ref->hHandle;
                  }
                  else
                  {
                    #ifdef RPOTREFS
                    refrpotelem = RPOT_Add(refPID, (void*) refPID);
                    refH = &(refrpotelem->la);
                    #else
                    refH = GARBAGE_AllocHandle();
                    PDEREF(refH) = (tObject*) refPID;
                    #endif
                  }
                }

                /* overwrite reference with the handle */
                ((tOBREF*) arr->pvElements)[i] = refH;
              }
        }


    }




  }
  #endif


    #ifdef RPOTREFS
    /* return handle */
    return &(rpotelem->la);
    #else
    return obj->hHandle;
    #endif
}

/*
 * Handles everything concerned with promoting an object to persistence:
 *
 * 1) Stores its class's name in the store if it is not already there.
 * 2) Stores the object in the store.
 * 3) Puts the PID->LA mapping into the RPOT.
 *
 */

#ifndef EVICTIONS

int calcArrayElementSize(  tAType enType,     /* @parm Type of elements in array */ 
char* pszSig)
{
  int32 i32ElSize;
  char* pszThisSig;
  int iSigLen;

  switch (enType)
  {
    /* pointer */
    case T_ARRAY:
    {
      i32ElSize = sizeof(tARREF);
      pszThisSig = (char*) sys_malloc(strlen(pszSig) + 2);
      pszThisSig[0] = '[';
      strcpy(pszThisSig + 1, pszSig);
      break;
    }
    /* pointer... same as T_ARRAY but do it separately for fun */
    case T_OBJECT:
    {
      i32ElSize = sizeof(tOBREF);
      iSigLen = strlen(pszSig) + 4;
      pszThisSig = (char*) sys_malloc(iSigLen);
      pszThisSig[0] = '[';
      pszThisSig[1] = 'L';
      strcpy(pszThisSig + 2, pszSig);
      pszThisSig[iSigLen - 2] = ';';
      pszThisSig[iSigLen - 1] = 0;
      break;
    }
    /* 8 bits */
    case T_BOOLEAN:
    {
      i32ElSize = 1;
      pszThisSig = "[Z";
      break;
    }
    case T_BYTE:
    {
      i32ElSize = 1;
      pszThisSig = "[B";
      break;
    }
    /* 16 bits */
    case T_CHAR:
    {
      i32ElSize = 2;
      pszThisSig = "[C";
      break;
    }
    case T_SHORT:
    {
      i32ElSize = 2;
      pszThisSig = "[S";
      break;
    }
    /* 32 bits */
    case T_INT:
    {
      i32ElSize = 4;
      pszThisSig = "[I";
      break;
    }
		case T_FLOAT:
    {
      i32ElSize = 4;
      pszThisSig = "[F";
      break;
    }
    /* 64 bits */
    case T_LONG:
    {
      i32ElSize = 8;
      pszThisSig = "[J";
      break;
    }
    case T_DOUBLE:
    {
      i32ElSize = 8;
      pszThisSig = "[D";
      break;
    }
    default:
    {
      exit(1);
    }
  }
return i32ElSize;
}
PID RPOT_PromoteObject
(
  tObject* ref
)
{
  PID pid;
  tClassLoaderTuple* type;
  tClassLoaderTuple* origType;

#ifdef DEBUG
  assert(ISPID((PID) ref) == 0);
#endif
  //  fprintf(stderr, "//////////// promoting object %x\n", ref);

  /* sort out this object's class first */
  origType = type = ref->pstType;
  if (type->pstClass->persIndex == (PID) NULL)
  {
    /* not in store so we must add it */
      //      fprintf(stderr, "Adding type %s to store for %x\n", type->uidName, ref);
    type->pstClass->persIndex = STORE_NewObject(strlen(type->uidName));
    STORE_UpdateObject(type->pstClass->persIndex, type->uidName);
  }
  /* set type to class name's PID so that it gets stored */
  ref->pstType = (tClassLoaderTuple*) type->pstClass->persIndex;

  /* ### deal with locks, etc */
  #ifdef EVICTIONS
  ref->i32Flags |= GARBAGE_PERSISTENT;
  #endif

  /* put object into store */
  //What if it's an array?
if(ref->i32Flags & GARBAGE_OBJECT)
    {
	pid = STORE_NewObject(ROUND32BIT(sizeof(tObject)) + sizeof(int32) * origType->pstClass->u16InstSize);
    }
else
    pid = STORE_NewObject(ROUND32BIT(sizeof(tArray)) + calcArrayElementSize( ((tArray*) ref)->enType, origType->uidName) * ((tArray*) ref)->i32Number);
			//sizeof(int32) * ((int32*) ref)[-1]);
  STORE_UpdateObject(pid, ref);

  /* add PID->LA mapping to RPOT_table */
  RPOT_Add(pid, ref);

  /* record PID in object */
  #ifndef REVRPOT
  ref->pid = pid;
  #endif

  /* reset type to point to class */
  ref->pstType = type;

  // fprintf(stderr, "Promoted %s %i of type %s\n", ((ref->i32Flags & GARBAGE_TYPEMASK) == GARBAGE_ARRAY) ? "array" : "instance", pid, ref->pstType->uidName);

  //  fprintf(stderr, "\\\\\\\\\\\\ promoted object %x\n", ref);
  return pid;
}
#endif

/*
 * Handles everything concerned with updating a persistent object
 *
 * 1) Stores its class's name in the store if it is not already there.
 * 2) Updates the object in the store.
 *
 */

void RPOT_UpdateObject
(
  PID pid,
  tObject* ref
)
{
  tClassLoaderTuple* type;

  #ifdef DEBUG
  assert(ISPID((PID) ref) == 0);
  #endif

  type = ref->pstType;
  ref->pstType = (tClassLoaderTuple*) type->pstClass->persIndex;
  STORE_UpdateObject(pid, ref);
  ref->pstType = type;

/*  printf("Updated %s %i of type %s\n", ((ref->i32Flags & GARBAGE_TYPEMASK) == GARBAGE_ARRAY) ? "array" : "instance", pid, ref->pstType->uidName);*/

}



#ifdef EVICTIONS
tOBREF RPOT_SwapIn
(
  PID pid
)
{
  tArray* a;
  tObject* o;
  int32 iSize;
  char* className;
  PID classPID;
  #ifdef MEMSWAP
  tClass* pstCurrType;
  int i;
  int iFinished;
  tField* field;
  tOBREF  refH;
  tObject* ref;
  #endif
  #ifdef EAGERINDIRECT
  PID refPID;
  tClass* pstCurrType;
  int iFinished = 0;
  int i;
  tField* field;
  tObject* ref;
  tOBREF refH;
  tRPOTElement* refrpotelem;
  #endif
  #ifdef PJSLCOMPAT
  pjsl_obj_info_t info;
  pjsl_size_t objSize;
  pjsl_kind_t kind;
  pjsl_regime_t regime;
  pjsl_hid_t histID = PJSL_HISTMAN_ILLEGAL_HID;
  #endif

  o = RPOT_GetLA(pid);
  if (o && !ISPID(o))
  {
    #ifdef RPOTREFS
    if (o->hHandle)
    {
      return o->hHandle;
    }
    else
    {
      return RPOT_GetObref(pid);
    }
    #else
    return o->hHandle;
    #endif
  }
  o = TPOT_GetLA(pid);
  if (o && !ISPID(o))
  {
    return o->hHandle;
  }

  #ifdef PJSLCOMPAT
  pjsl_objInspect(STORE_Store, pid, histID, &kind, &regime, &objSize);
  iSize = objSize;
  #else
  iSize = STORE_GetObjectSize(pid);
  #endif
  o = GARBAGE_MallocPtr(iSize);
  #ifdef PJSLCOMPAT
  PJSL_KIND_OP(kind,regime,fetch)(STORE_Store, pid, histID, &info, (uchar*) o);
  #else
  STORE_LoadObject(pid, o);
  #endif
  /* do this coz they don't all have their PIDs done properly in the store */
  if (o->pid == NULL)
  {
    o->pid = pid;
  }

  /* get the object's class */
  classPID = (PID) o->pstType;
  if (ISPID(classPID))
  {
    #ifdef PJSLCOMPAT
    pjsl_objInspect(STORE_Store, pid, histID, &kind, &regime, &objSize);
    iSize = objSize;
    #else
    iSize = STORE_GetObjectSize(classPID);
    #endif
    className = (char*) sys_malloc(sizeof(char) * (iSize + 1));
    #ifdef PJSLCOMPAT
    PJSL_KIND_OP(kind,regime,fetch)(STORE_Store, classPID, histID, &info, (uchar*) className);
    #else
    STORE_LoadObject(classPID, className);
    #endif
    className[iSize] = 0;
    o->pstType = CLASSFILE_FindOrLoad(className);
    if (o->pstType->persIndex == NULL)
    {
      o->pstType->persIndex = classPID;
    }
    sys_free(className);
  }

  if ((o->i32Flags & GARBAGE_TYPEMASK) == GARBAGE_OBJECT)
  {
    if (o->pstType->u16InstSize)
    {
      o->pi32Vars = (int32*) (((byte*) o) + ROUND32BIT(sizeof(tObject)));
      #ifdef MEMSWAP
      if (!HASFLAG(o, GARBAGE_MEMSWOPPED))
      {
        iFinished = 0;
        for (pstCurrType = o->pstType; iFinished == 0; pstCurrType = pstCurrType->pstSuperClass)
        {
          for (i = 0; i < pstCurrType->u16InstCount; i++)
          {
            field = &(pstCurrType->pstInstOffsets[i]);
            switch (field->uidFieldSig[0])
            {
              case '[':
              case 'L':
              {
                refH = (tOBREF) o->pi32Vars[field->u16Offset];
                /* only bother if it's not null or a PID */
                if (refH && (!ISPID(refH)))
                {
                  ref = PDEREF(refH);
                  if (!ISPID(ref))
                  {
                    ref->i32ExternalRefCount -= 1;
                  }
                }
                break;
              }
            }
          }
          /* if we've got up to the Object class or we've got to a class that
             does not inherit anything then we have finished */
          if ((pstCurrType->pstSuperClass == NULL) ||
              ((pstCurrType->pstInstOffsets) && (pstCurrType->pstInstOffsets[0].u16Offset == 0)))
          {
            iFinished = 1;
          }
        }
      }
      #endif
    }
    else
    {
      o->pi32Vars = NULL;
    }
  }
  /* if object is an array */
  else
  {
    a = (tArray*) o;
    if (a->i32Number)
    {
      a->pvElements = (((byte*) a) + ROUND32BIT(sizeof(tArray)));
      #ifdef MEMSWAP
      /* if it's an array of references, fix the refs */
      if (a->enType == T_OBJECT)
      {
        for (i = 0; i < a->i32Number; i++)
        {
          refH = ((tOBREF*) a->pvElements)[i];
          /* if not a null reference */
          if (refH && (!ISPID(refH)))
          {
            ref = PDEREF(refH);
            if (!ISPID(ref))
            {
              /* only worry about ref count if it's not a PID */
              ref->i32ExternalRefCount -= 1;
            }
          }
        }
      }
      #endif
    }
    else
    {
      a->pvElements = NULL;
    }
  }

  #ifdef EAGERINDIRECT
  /* make handles for all the object's references */
  /* for each reference field, if the object referred to by the PID is
     already loaded then we must use its handle, otherwise allocate a
     new handle and put the PID in it */
  if ((o->i32Flags & GARBAGE_TYPEMASK) == GARBAGE_OBJECT)
  {
    if (o->pi32Vars)
    {
      /* Check the fields (looking at parent classes too) */
      for (pstCurrType = o->pstType; iFinished == 0; pstCurrType = pstCurrType->pstSuperClass)
      {
        for (i = 0; i < pstCurrType->u16InstCount; i++)
        {
          field = &(pstCurrType->pstInstOffsets[i]);
          switch (field->uidFieldSig[0])
          {
            case '[':
            case 'L':
            {
              refPID = o->pi32Vars[field->u16Offset];
              if (refPID)
              {
                /* look in RPOT */
                ref = RPOT_GetLA(refPID);
                if (ref)
                {
                  #ifdef RPOTREFS
                  /* ### maybe check in case this object somehow has a handle
                         as well as being in the RPOT */
                  refH = RPOT_GetObref(refPID);
                  #else
                  refH = ref->hHandle;
                  #endif
                }
                else
                {
                  ref = TPOT_GetLA(refPID);
                  if (ref)
                  {
                    /* should have a handle */
                    refH = ref->hHandle;
                  }
                  else
                  {
                    /* object not resident so allocate a handle / RPOT entry
                       for it in its absence */
                    #ifdef RPOTREFS
                    refrpotelem = RPOT_Add(refPID, (void*) refPID);
                    refH = &(refrpotelem->la);
                    #else
                    refH = GARBAGE_AllocHandle();
                    PDEREF(refH) = (tObject*) refPID;
                    #endif
                  }
                }

                /* overwrite reference with the handle */
                o->pi32Vars[field->u16Offset] = (int32) refH;
              }

              break;
            }
          }
        }
        if ((pstCurrType->pstSuperClass == NULL) ||
            ((pstCurrType->pstInstOffsets) && (pstCurrType->pstInstOffsets[0].u16Offset == 0)))
        {
          iFinished = 1;
        }
      }

    }
  }
  else
  {
    a = (tArray*) o;

    if ((a->enType == T_ARRAY) || (a->enType == T_OBJECT))
    {
      for (i = 0; i < a->i32Number; i++)
      {
        refPID = ((PID*) a->pvElements)[i];

              if (refPID)
              {
                /* look in RPOT */
                ref = RPOT_GetLA(refPID);
                if (ref)
                {
                  #ifdef RPOTREFS
                  /* ### maybe check in case this object somehow has a handle
                         as well as being in the RPOT */
                  refH = RPOT_GetObref(refPID);
                  #else
                  refH = ref->hHandle;
                  #endif
                }
                else
                {
                  ref = TPOT_GetLA(refPID);
                  if (ref)
                  {
                    /* should have a handle */
                    refH = ref->hHandle;
                  }
                  else
                  {
                    #ifdef RPOTREFS
                    refrpotelem = RPOT_Add(refPID, (void*) refPID);
                    refH = &(refrpotelem->la);
                    #else
                    refH = GARBAGE_AllocHandle();
                    PDEREF(refH) = (tObject*) refPID;
                    #endif
                  }
                }

                /* overwrite reference with the handle */
                ((tOBREF*) a->pvElements)[i] = refH;
              }
        }
    }
  }
  #endif


  /* check if this is a swopped-out object or a persistent one */
  if (HASFLAG(o, GARBAGE_PERSISTENT))
  {
    tOBREF refThing;

    #ifndef RPOTREFS
    /* the la field is set to the handle when it's swopped out, so
       write the correct handle pointer back now */
    o->hHandle = (tOBREF) TPOT_GetLA(o->pid);
    #endif

    /* persistent objects use RPOT entries as refs */
    refThing = RPOT_ChangeLA(o->pid, o);

    /* persistent objects that have been loaded from the store and were
       not actually created during this run won't have handles, so only
       change its handle if it actually has one */

    #ifdef RPOTREFS
    if (o->hHandle)
    {
      *(o->hHandle) = o;
      return o->hHandle;
    }
    else
    {
      return refThing;
    }
    #else
    *(o->hHandle) = o;
    return o->hHandle;
    #endif
  }
  else
  {
    #ifndef RPOTREFS
    o->hHandle = (tOBREF) TPOT_GetLA(o->pid);
    #endif

    /* swopped-out objects use handles as refs */
    *(o->hHandle) = o;
    /* change TPOT entry */
    TPOT_ChangeLA(o->pid, o);
    return o->hHandle;
  }
}

tOBREF RPOT_ChangeLA
(
  PID pid,
  void* newLA
)
{
  tRPOTElement* temp;

  #ifdef REVRPOT
  void* la;
  /* in reverse RPOT case we must remove and re-insert so that the reverse
     hashing can be fixed up */
  la = RPOT_Remove(pid);
  if (la == NULL)
  {

    exit(1);
  }
  RPOT_Add(pid, newLA);
  return;
  #else
  if ((temp = &(RPOT_table[HASHFUNC(pid)]))->pid == pid)
  {
    temp->la = newLA;
    return (tOBREF) &(temp->la);
  }
  #endif

  temp = temp->next;
  while (temp)
  {
    if (temp->pid == pid)
    {
      temp->la = newLA;
      return (tOBREF) &(temp->la);
    }
    temp = temp->next;
  }


  exit(1);
  return NULL; /* keep compiler happy */
}
#endif


/*
 * Handles updating an object, including all objects reachable from it
 * that also need updating (or promoting).
 *
 */

#ifndef EVICTIONS

void RPOT_Stabilise
(
  tObject* o
)
{
  int i;
  tField* field;
  tOBREF refH;
  tObject* ref;
  tArray* a;
  #ifdef RESWIZZLE
  void* pvStore = NULL;
  #endif
  int iFinished = 0;
  tClassLoaderTuple* pstCurrType;

  /* if this object has neither its PERSIST_Updated or PERSIST_RefsUpdated
     flags set and is already persistent then we can forget about it */
  #ifdef REVRPOT
  if (((o->i32Flags & PERSIST_DirtyMask) == 0) && RPOT_GetPID(o))
  {
    return;
  }
  #else

#ifdef DEBUG
  assert(o->pstType);
#endif
  if (((o->i32Flags & PERSIST_DirtyMask) == 0) && o->pid)
  {
    return;
  }
  #endif



  /* clear handle field */
/*  o->hHandle = NULL;*/

  /* set persistent flag */
  o->i32Flags |= GARBAGE_PERSISTENT;


    /* clear update flags
       we do this here instead of at the end so that this object will be
       skipped if it is reachable via one of its own references */
    o->i32Flags &= ~PERSIST_DirtyMask;

    /* if object is a class instance */
    if ((o->i32Flags & GARBAGE_TYPEMASK) == GARBAGE_OBJECT)
    {
      /* if this object doesn't have any variables then we don't need to
         do anything here */
      if (o->pi32Vars)
      {

      #ifdef RESWIZZLE
      /* store fields so that we can write them back afterwards */
      pvStore = sys_malloc(o->pstType->u16InstSize * sizeof(int32));
#ifdef DEBUG
      assert(pvStore);
#endif
      memcpy(pvStore, o->pi32Vars, o->pstType->u16InstSize * sizeof(int32));
      #endif

      /* Checking the fields is complicated because information about fields
         that are inherited from a parent class are not stored in the child
         class's class definition. We have to check the parent classes until
         we find one whose first field is stored in slot 0, i.e. one of its
         own fields is the first one which means it does not inherit any. */

      for (pstCurrType = o->pstType; iFinished == 0; pstCurrType = pstCurrType->pstClass->pstSuperClass)
      {

        for (i = 0; i < pstCurrType->pstClass->u16InstCount; i++)
        {
          field = &(pstCurrType->pstClass->pstInstOffsets[i]);
          switch (field->uidFieldSig[0])
          {
            case '[':
            case 'L':
            {
              refH = (tOBREF) o->pi32Vars[field->u16Offset];
              /* only bother if it's not null */
              if (refH)
              {
/*                if (ISPID((PID) refH))
                {
                  ref = RPOT_GetLA((PID) refH);
                  if (ref == NULL)
                  {
                    continue;
                  }
                }
                else
                {
                  ref = DEREF(refH);
                }*/
                if (ISPID(refH))
                {
                  continue;
                }
                ref = DEREF(refH);  /* keep this like this... hope for no swopped out objects */
                RPOT_Stabilise(ref);
#ifdef DEBUG
                assert(ref->pid);
#endif
                /* unswizzle the reference field */
                #ifdef REVRPOT
                o->pi32Vars[field->u16Offset] = RPOT_GetPID(ref);
                #else
                o->pi32Vars[field->u16Offset] = ref->pid;
                #endif
              }
              break;
            }
          }
        }

        /* if we've got up to the Object class or we've got to a class that
           does not inherit anything then we have finished */
        if ((pstCurrType->pstClass->pstSuperClass == NULL) ||
            ((pstCurrType->pstClass->pstInstOffsets) && (pstCurrType->pstClass->pstInstOffsets[0].u16Offset == 0)))
        {
          iFinished = 1;
        }
      }
      }
    }
    /* if object is an array */
    else
    {
      a = (tArray*) o;

      /* if the array is not an array of arrays or of objects then we don't
         have to do anything */
      if ((a->enType == T_ARRAY) || (a->enType == T_OBJECT))
      {
	  //	  fprintf(stderr, "Stabilising an array that needs stabilising %x, els %i\n", a, a->i32Number);
      #ifdef RESWIZZLE
      /* store fields so that we can write them back afterwards */
      pvStore = sys_malloc(a->i32Number * sizeof(int32));
#ifdef DEBUG
      assert(pvStore);
#endif
      memcpy(pvStore, a->pvElements, a->i32Number * sizeof(int32));
      #endif

      for (i = 0; i < a->i32Number; i++)
      {
        refH = ((tOBREF*) a->pvElements)[i];
        /* if not a null reference */
        if (refH)
        {
/*          if (ISPID((PID) refH))
          {
            ref = RPOT_GetLA((PID) refH);
            if (ref == NULL)
            {
              continue;
            }
          }
          else
          {
            ref = DEREF(refH);
          }*/
          /* stabilise object */
          /* if object is not already persistent then it'll be promoted during
             the call */
          if (ISPID(refH))
          {
            continue;
          }
          ref = DEREF(refH);
          RPOT_Stabilise(ref);

#ifdef DEBUG
	#ifndef USEOSKIT
          assert(ref->pid);
	  #endif
#endif
          /* unswizzle the slot so that the current object will
             be saved properly */
          #ifdef REVRPOT
          ((tObject**) a->pvElements)[i] = (tObject*) RPOT_GetPID(ref);
          #else
          ((tObject**) a->pvElements)[i] = (tObject*) ref->pid;
          #endif
        }
      }
     }
      else
	  {
	      //	      	  fprintf(stderr, "NOT Stabilising an array of primitives %x %i\n", a, a->i32Number);
	  }
    } //end else

  /* if this object is already persistent then update it otherwise promote it */
  #ifdef REVRPOT
  if (RPOT_GetPID(o))
  {
    RPOT_UpdateObject(RPOT_GetPID(o), o);
  }
  else
  {
    RPOT_PromoteObject(o);
  }
  #else

  //  fprintf(stderr, "%x 's pid is %x\n", o, o->pid);
  if (o->pid)
  {
    RPOT_UpdateObject(o->pid, o);
  }
  else
  {
    RPOT_PromoteObject(o);
  }
  #endif

  #ifdef RESWIZZLE
  /* don't attempt to do this if we didn't actually copy the fields - check
     by checking if pvStore was allocated */
  if (pvStore)
  {
    if ((o->i32Flags & GARBAGE_TYPEMASK) == GARBAGE_OBJECT)
    {
      /* write fields back */
      memcpy(o->pi32Vars, pvStore, o->pstType->u16InstSize * sizeof(int32));
    }
    else
    {
      memcpy(a->pvElements, pvStore, a->i32Number * sizeof(int32));
    }
    sys_free(pvStore);
  }
  #endif
}
#endif

/*
 * Updates all objects in the RPOT that have been changed.
 *
 */
void RPOT_StabiliseAll
(
  void
)
{
#if defined(EVICTIONS) && !defined(OLDSTABILISE)
/*
   What we want to do in the stabilise is record the state of all objects
   onto the store.

   We'll go through all objects in memory and write them to the store...
   that should sort it all out. Since we're only swopping in store format,
   any objects that have been swopped out and not swopped back in can be
   left as they are on the store.

*/
  #ifdef TIMING
  GLOBAL_PrintTime("Before stabilise");
  #endif
#ifdef DEBUG
  assert( 2 == 5); //bobo jewel
#endif
  GARBAGE_WriteEverything();
  #ifdef TIMING
  GLOBAL_PrintTime("After stabilise");
  #endif
#else
  int i;
  tRPOTElement* temp;

  #ifdef TIMING
  GLOBAL_PrintTime("Before stabilise");
  #endif
  for (i = 0; i < RPOT_TABLESIZE; i++)
  {
    #ifdef REVRPOT
    for (temp = RPOT_table[i]; temp; temp = temp->next)
    {
      RPOT_Stabilise(temp->la);
    }
    #else

    if (RPOT_table[i].pid)
    {
      RPOT_Stabilise(RPOT_table[i].la);
      temp = RPOT_table[i].next;
      while (temp)
      {
        RPOT_Stabilise(temp->la);
        temp = temp->next;
      }
    }
    #endif
  }
  #ifdef TIMING
  GLOBAL_PrintTime("After stabilise");
  #endif
/*  GARBAGE_gc();*/
#endif
}


/*
 * Prints out the RPOT_table for debugging.
 *
 */

void RPOT_PrintTable
(
  void
)
{
  int i;
  tRPOTElement* temp;
  //  fprintf(stderr, "Iterating through RPOT\n");
  for (i = 0; i < RPOT_TABLESIZE; i++)
  {
      //    printf("%02i ", i);
    if (RPOT_table[i].pid)
    {
      eprintf("pid %x ", RPOT_table[i].pid);
      temp = RPOT_table[i].next;
      while (temp)
      {

        temp = temp->next;
      }
    }
    //  fprintf(stderr, "\n");
  }
  //  fprintf(stderr, "fin Iterating through RPOT\n");
}

#endif


int asdfljkas;







#endif //GARBAGE_COPYING
#endif //GARBAGE_GENERATIONAL

