/*
 * Copyright (C) 2001, John Leuner.
 *
 * This file is part of the kissme project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2,
 * or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>

#include <stdio.h>
#include <pthread.h>
#include "stdtypes.h"
#include <string.h>

#include "classfil.h"
#include "jni.h"
#include "classfile_methods.h"
#include "interp_methods.h"
#include "newobject.h"

#include "wrappers.h"
#include <assert.h>

extern int showLoading; //defined in interp.c

#include "loading_errors.h"

static int iHaveBootstrapped = 0;

void setBootstrapped()
{
  iHaveBootstrapped = 1;
}

typedef enum tjavaloadertype {
  JAR_LOADER_TYPE = 1,
  ZIP_LOADER_TYPE,
  HTTP_LOADER_TYPE
} JavaLoaderType;

static tClassLoaderTuple* getClassFromByteArray(JNIEnv* env, byte* pbData, char* pszFileName, uint32* errorCode, pthread_mutex_t* classfilMutex);
static tClassLoaderTuple* getClassDataFromFileSystem
(
  JNIEnv* env,
  char* pszFileName,
  char* pszClassPath,
  uint16 u16ClassPathLen,
  uint32* errorCode,
  pthread_mutex_t* classfilMutex
);

static tClassLoaderTuple* getClassDataFromJavaLoader
(
  JNIEnv* env,
  JavaLoaderType type,
  char* pszFileName,
  char* pszClassPath,
  uint16 u16ClassPathLen,
  uint32* errorCode,
  pthread_mutex_t* classfilMutex
);

tClassLoaderTuple* getClassData
(
  JNIEnv* env,
  char* pszFileName,
  uint32* errorCode,
  pthread_mutex_t* classfilMutex
)
{
  tClassLoaderTuple*  pstClass;
  int u16PathLen;
  int i = 0;
  pstClass = NULL;


  if(CLASSFILE_u16NumClassPaths == 0)
    {
      *errorCode = LOADING_ERROR_NO_FILE;
      return NULL;
    }
 for (i = 0; i < CLASSFILE_u16NumClassPaths; i++)
  {
    int try_file_system = 1;

    u16PathLen = strlen(CLASSFILE_ppszClassPaths[i]);
    /* decide whether to use the local filesystem or a jar/zip file or http loader */
    if(iHaveBootstrapped != 0)
      {
    if(u16PathLen > 4) //must be greater than 4 for a .jar or .zip file
      {							
	if(strcmp(CLASSFILE_ppszClassPaths[i] + (u16PathLen - 4), ".jar") == 0)
	  {
	    /* it is a jar file */
	    try_file_system = 0;
	    pstClass = getClassDataFromJavaLoader(env, JAR_LOADER_TYPE, pszFileName, CLASSFILE_ppszClassPaths[i], u16PathLen, errorCode, classfilMutex);
	    if(pstClass != NULL)
	      {
		return pstClass;
	      }
	  }
	else if(strcmp(CLASSFILE_ppszClassPaths[i] + (u16PathLen - 4), ".zip") == 0)
	  {
	    /* it is a zip file */
	    try_file_system = 0;
	    pstClass = getClassDataFromJavaLoader(env, ZIP_LOADER_TYPE, pszFileName, CLASSFILE_ppszClassPaths[i], u16PathLen, errorCode, classfilMutex);
	    if(pstClass != NULL)
	      {
		return pstClass;
	      }
	  }
      }
    if(u16PathLen > 2) //must be greater than two, to have "//"
      {
	if(strncmp(CLASSFILE_ppszClassPaths[i], "//", 2) == 0)
	  {
	    /* use network loader */
	    try_file_system = 0;
	    pstClass = getClassDataFromJavaLoader(env, HTTP_LOADER_TYPE, pszFileName, CLASSFILE_ppszClassPaths[i], u16PathLen, errorCode, classfilMutex);
	    if(pstClass != NULL)
	      {
		return pstClass;
	      }
	  }
       }
      }
    if(try_file_system != 0)
      {
	pstClass = getClassDataFromFileSystem(env, pszFileName, CLASSFILE_ppszClassPaths[i], u16PathLen, errorCode, classfilMutex);
	if(pstClass != NULL)
	  return pstClass;
      }
  }

 if(*errorCode == 0)
  *errorCode = LOADING_ERROR_NO_FILE;
  return NULL;
}


static tClassLoaderTuple* getClassDataFromFileSystem
(
  JNIEnv* env,
  char* pszFileName,
  char* pszClassPath,
  uint16 u16ClassPathLen,
  uint32* errorCode,
  pthread_mutex_t* classfilMutex
)
{
  char* pszFullPath;
  FILE*    fp;
  int read ;
  long int iFileSize;
  byte*    pbData;
  tClassLoaderTuple*  pstClass;

  pszFullPath = sys_malloc( u16ClassPathLen + strlen(pszFileName) + 10);
  if(pszFullPath == NULL)
    {
      *errorCode = LOADING_ERROR_OUT_OF_MEMORY;
      return NULL;
    }
  strcpy(pszFullPath, pszClassPath);
  if( pszFullPath[ strlen(pszFullPath) - 1] != '/')
    strcat(pszFullPath, "/");
  strcat(pszFullPath, pszFileName);
  strcat(pszFullPath, ".class");
    
  fp = fopen( pszFullPath, "r");
    
  if(fp != NULL)
    {
      if(showLoading)
	eprintf("[Loading %s]\n", pszFullPath);
      sys_free(pszFullPath);

      /* We have found the binary data for the class, read it and create the internal representation */
      fseek(fp, 0, SEEK_END);
      iFileSize = ftell(fp);
      fseek(fp, 0, SEEK_SET);
      pbData = (byte*) sys_malloc(iFileSize);
      assert(pbData);
      read = fread(pbData, iFileSize, 1, fp);
      assert(read == 1);
      fclose(fp);

      pstClass = getClassFromByteArray(env, pbData, pszFileName, errorCode, classfilMutex);
      sys_free(pbData);
      return pstClass;
    }
  else
  {
//     eprintf("couldn't open %s\n", pszFullPath);
  }
  return NULL;
}

static tClassLoaderTuple* getClassFromByteArray(JNIEnv* env, byte* pbData, char* pszFileName, uint32* errorCode, pthread_mutex_t* classfilMutex)
{
  tClassLoaderTuple* pstClass;

  pthread_mutex_lock( classfilMutex);
  //  fprintf(stderr, "Doing CLASS_Load for %s, data at %p\n", pszFileName, pbData);
  pstClass = CLASSFILE_ClassCreate(env, pbData, pszFileName, NULL, errorCode); //boot class loader
  if(pstClass == NULL)
    {
      //throw an exception XXX
    }
  pthread_mutex_unlock( classfilMutex);
  //  fprintf(stderr, "Did CLASS_Create for %s\n", pszFileName);
  return pstClass;
}



static tClassLoaderTuple* getClassDataFromJavaLoader
(
  JNIEnv* env,
  JavaLoaderType type,
  char* pszFileName,
  char* pszClassPath,
  uint16 u16ClassPathLen,
  uint32* errorCode,
  pthread_mutex_t* classfilMutex
)
{
  jclass JavaLoaderClass;
  jmethodID mid;
  jbyteArray result;
  jbyte* pbData;
  jboolean isCopy = 0;
  tClassLoaderTuple* pstClass;

#ifdef DEBUG_JAVA_LOADER
  eprintf("Loading %s in JavaClassLoader\n", pszFileName);
#endif
     
  JavaLoaderClass = (*env)->FindClass(env, "kissme/vm/loaders/JavaClassLoader");
  if(JavaLoaderClass == NULL)
    return NULL;

#ifdef DEBUG_JAVA_LOADER
  eprintf("Loading %s in JavaClassLoader 1\n", pszFileName);
#endif

  mid = (*env)->GetStaticMethodID(env, JavaLoaderClass, "getClassData", "(Ljava/lang/String;Ljava/lang/String;I)[B");
  if(mid == NULL)
    {
      eprintf("Could not find method getClassData in kissme.vm.loaders.JavaClassLoader");
      return NULL;
    }
#ifdef DEBUG_JAVA_LOADER
  eprintf("Loading %s in JavaClassLoader 2\n", pszFileName);
#endif

  if(showLoading) //XXX this should only happen fi the file exists => move to java code
    eprintf("[Loading %s from %s]\n", pszFileName, pszClassPath);
  result = (jbyteArray) (*env)->CallStaticObjectMethod(env, JavaLoaderClass, mid, INTERP_NewStringFromAsciz(env, pszFileName), INTERP_NewStringFromAsciz(env, pszClassPath), type);

  if((*env)->ExceptionOccurred(env) != NULL)
    {
      jobject pstExOb = (*env)->ExceptionOccurred(env);
      eprintf("exception occurred while calling JavaClassLoader.getClassData, %s\n", ODEREF(pstExOb)->pstType->uidName);
    }
#ifdef DEBUG_JAVA_LOADER
  eprintf("Loading %s in JavaClassLoader 3, result %p\n", pszFileName, result);
#endif
  if(result == NULL)
    return NULL;

  pbData = (*env)->GetByteArrayElements(env, result, &isCopy);
  (*env)->NewGlobalRef(env, (tOBREF) result);
  pstClass = getClassFromByteArray(env, pbData, pszFileName, errorCode, classfilMutex);
  (*env)->DeleteGlobalRef(env, (tOBREF) result);

  if(pstClass != NULL)
    {
      if(showLoading)
	{
	  eprintf("[Loaded %s from %s]\n", pszFileName, pszClassPath);
	}
#ifdef DEBUG_JAVA_LOADER
      eprintf("Returning %p, %s in JavaClassLoader\n", pstClass,pstClass->uidName);
#endif
    }
  return pstClass;
}
