/***************************************************************************
                          account.h  -  description
                             -------------------
    begin                : Sun Jan 5 2003
    copyright            : (C) 2003 by Mike K. Bennett
    email                : mkb137b@hotmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef ACCOUNT_H
#define ACCOUNT_H

#include "contact/msnstatus.h"
#include "utils/richtextparser.h"

#include <QFont>

// Forward declarations
class KMessTest;
class MsnObject;



/**
 * @brief Data class for user account settings.
 *
 * The properties of this class correspond with the profile section in the <code>kmessrc</code> file.
 * The data is read with the readProperties() method and saved with the saveProperties() method.
 * When the user switches to another account, the data is copied to the CurrentAccount class with
 * the copyAccount() method. The <code>set*</code> methods are issued from the AccountSettingsDialog
 * code. When the account is disconnected, the data is copied back and saved to disk.
 *
 * @author Mike K. Bennett
 * @ingroup Root
 */
class Account : public QObject
{
  Q_OBJECT

  friend class KMessTest;

  public:  // Public enumerations
    // Directory structure constants
    enum ChatDirectoryStructure { SINGLEDIRECTORY = 0
                                , BYYEAR          = 1
                                , BYMONTH         = 2
                                , BYDAY           = 3
                                };
    /// Available chat exporting formats
    enum ChatExportFormat
    {
      EXPORT_XML    /// Export the chat to XML
    , EXPORT_HTML   /// Export the chat to HTML
    , EXPORT_TEXT   /// Export the chat to text
    };
    // Contact List view organization constants
    enum ContactListDisplayMode
    {
      VIEW_BYGROUP
    , VIEW_BYSTATUS
    , VIEW_MIXED
    };

  public:  // Public methods
    // The constructor
    Account();
    // The destructor
    virtual ~Account();
    // Copy an account
    virtual void                 copyAccount( const Account *account );
    // Set the account as deleted
    void                         setDeleted();
    // Read the auto-reply message.
    const QString&               getAutoreplyMessage() const;
    // Return the chat style name.
    const QString&               getChatStyle() const;
    // Read how contacts are shown, by group, by online/offline, or mixed.
    ContactListDisplayMode       getContactListDisplayMode() const;
    // Return the font used for contact messages, if forced to.
    const QFont&                 getContactFont() const;
    // Return the color of the forced contact font.
    const QString&               getContactFontColor() const;
    // Return whether email notifications are supported
    bool                         getEmailSupported() const;
    // Return the user message font.
    const QFont&                 getFont() const;
    // Return the color of the user message font.
    const QString&               getFontColor() const;
    // Return the user's friendlyName
    QString                      getFriendlyName( FormattingMode mode = STRING_CLEANED ) const;
    // Whether follow-up messages from the contact should be grouped.
    bool                         getGroupFollowupMessages() const;
    // Return the user's handle
    QString                      getHandle() const;
    // Return the length of time waited before the user goes idle.
    const int&                   getIdleTime() const;
    // Return the status to set upon connection
    Status                       getInitialStatus() const;
    // Returns display mode used to show pictures in contact list.
    int                          getListPictureSize() const;
    // Return the user's password
    QString                      getPassword() const;
    // Return the personal message
    QString                      getPersonalMessage( FormattingMode mode = STRING_CLEANED ) const;
   // Return whether or not to save chats
    bool                         getSaveChats() const;
   // Return whether or not to save chats to external files
    bool                         getSaveChatsToFile() const;
   // Return the format in which chats will be saved
    ChatExportFormat             getSaveChatsFormat() const;
    // Return the base directory to which chats are saved.
    const QString&               getSaveChatPath() const;
    // Return the code of the structure used to save chats.
    const int&                   getSavedChatDirectoryStructure() const;
    // Return the user's saved password
    QString                      getSavedPassword() const;
    // Read whether or not allowed contacts are shown.
    bool                         getShowAllowedContacts() const;
    // Read whether to show the user picture in chat windows
    bool                         getShowChatUserPicture() const;
    // Read whether the email information should be shown in the main view.
    bool                         getShowEmail() const;
    // Read whether to show the contact list background image
    bool                         getShowContactListBird() const;
    // Read the selected emoticon's list number
    const QString&               getEmoticonStyle() const;
    // Read whether to hide notifications when busy
    bool                         getHideNotificationsWhenBusy() const;
    // Read whether the password must be stored
    bool                         getSavePassword() const;
    // Read whether the chat window should shake when a nudge is received or sent.
    bool                         getShakeNudge() const;
    // Read whether or not to show the email of contacts in the contact list instead of their friendly name
    bool                         getShowContactEmail() const;
    // Read whether empty groups are shown
    bool                         getShowEmptyGroups() const;
    // Read whether or not to show the contact list history box
    bool                         getShowHistoryBox() const;
    // Read whether offline contacts should be shown.
    bool                         getShowOfflineContacts() const;
    // Read property of bool showMessageTime.
    bool                         getShowMessageTime() const;
    // Return whether now listening information should be displayed.
    bool                         getShowNowListening() const;
    // Read property of bool showOtherFolders.
    bool                         getShowOtherFolders() const;
    // Read whether or not removed (reverse) contacts are shown.
    bool                         getShowRemovedContacts() const;
    // Read whether or not we need to show a search bar in the contact list
    bool                         getShowSearchBar() const;
    // Read whether or not to show join/part messages in chat
    bool                         getShowSessionInfo() const;
    // Read whether to show winks
    bool                         getShowWinks() const;
    // Read the way the chats will be grouped together
    int                          getTabbedChatMode() const;
    // Return the a temporary password for the account for login attempts from the initialview dialog
    const QString &              getTemporaryPassword() const;
    // Returns whether to show seconds in message timestamp
    bool                         getTimestampShowSeconds() const;
    // Returns whether to show date in message timestamp
    bool                         getTimestampShowDate() const;
    // Return whether or not to autologin with this profile.
    bool                         getUseAutologin() const;
    // Return whether or not to enable MSN Plus! formatting in chat
    bool                         getUseChatFormatting() const;
    // Return whether or not to show contact messages in the stored font.
    bool                         getUseContactFont() const;
    // Return whether or not to show emoticons in chats.
    bool                         getUseEmoticons() const;
    // Return whether or not to use bold/italic/underline effects in chats.
    bool                         getUseFontEffects() const;
    // Return whether the idle timer should be used.
    bool                         getUseIdleTimer() const;
    // Return whether or not to enable MSN Plus! formatting in the contact list
    bool                         getUseListFormatting() const;
    // Returns whether or not to send our display picture
    bool                         getShowPicture() const;
    // Return the path to the *current* display picture
    const QString                getPicturePath() const;
    // Return location of the display picture original
    const QString                getOriginalPicturePath() const;
    // Returns the string representation of the MsnObject
    const QString                getMsnObjectString() const;
    // Whether the account is marked to be deleted
    bool                         isDeleted() const;
    // Whether the account has unsaved settings changes
    bool                         isDirty() const;
    // Whether the account is a guest account, not permanently saved on the computer
    bool                         isGuestAccount() const;
    // Return whether the account's passport is verified
    bool                         isVerified() const;
    // Read in account properties
    void                         readProperties( const QString &handle );
    // Save account properties
    void                         saveProperties();
    // The login succeeded with the new password. Save the temporary password as new account password.
    void                         saveTemporaryPassword();
    // Set the auto-reply message.
    void                         setAutoreplyMessage( const QString&  autoreplyMessage);
    // Set chat information
    void                         setChatInformation( bool useContactFont, bool useEmoticons, bool showWinks, bool useFontEffects, bool useChatFormatting, bool shakeNudge, bool showMessageTime, bool timestampShowDate, bool timestampShowSeconds, bool groupFollowupMessages, int tabbedChatMode, const QString& chatStyle );
    // Set chat logging information
    void                         setChatLoggingInformation( bool saveChats, bool saveChatsToFile, ChatExportFormat saveChatsFormat, const QString& saveChatPath, int directoryStructure );
    // Set how contacts are shown, by group, by online/offline, or mixed.
    void                         setContactListDisplayMode( ContactListDisplayMode displayMode );
    // Set contact list appearance options
    void                         setContactListOptions( bool showNowListening, bool showContactEmail, bool useListFormatting );
    // Set email information
    void                         setEmailInformation( bool showEmail, bool showOtherFolders );
    // Set the font
    void                         setFont( const QFont& font );
    // Set the font color
    void                         setFontColor( const QString&  fontColor );
    // Set the fonts and font colors
    void                         setFontInformation( const QFont& font, const QString  &fontColor, const QFont& contactFont, const QString  &contactFontColor);
    // Change the friendly name
    void                         setFriendlyName( const QString&  newName );
    // Set the emoticon folder
    void                         setEmoticonStyle( const QString& emoticonStyle );
    // Whether the account is a guest account, not permanently saved on the computer
    void                         setGuestAccount( bool guestAccount );
    // Change the status to set upon connection
    void                         setInitialStatus( Status status );
    // Set the mode of display list pictures
    void                         setListPictureSize( int mode );
    // change the personal message
    void                         setPersonalMessage( const QString& newMessage );
    // Set basic account data
    void                         setLoginInformation( QString handle, QString friendlyName, QString password = QString() );
    // Set the account's login password
    void                         setPassword( const QString &password );
    // Set the display picture file
    void                         setPicturePath( const QString& picturePath );
    // Save the location of the display picture original
    void                         setOriginalPicturePath( const QString& picturePath );
    // Set the account's saved password
    void                         setSavedPassword( const QString &password );
    // Set if the password must be stored
    void                         setSavePassword( bool savePassword );
    // Set whether or not allowed contacts are shown.
    void                         setShowAllowedContacts( bool showAllowedContacts );
    // Set whether to show the user picture in chat windows
    void                         setShowChatUserPicture( bool showChatUserPicture );
    // Set whether to show the contact list background image
    void                         setShowContactListBird( bool showBird );
    // Set whether or not to show the contact list history box
    void                         setShowHistoryBox( bool showHistoryBox );
    // Set whether empty groups are shown
    void                         setShowEmptyGroups( bool showEmptyGroups );
    // Set whether offline contacts should be shown.
    void                         setShowOfflineContacts( bool showOfflineContacts );
    // Set whether or not removed (reverse) contacts are shown.
    void                         setShowRemovedContacts( bool showRemovedContacts );
    // Set whether or not we need to show a search bar in the contact list at login
    void                         setShowSearchBar( bool showSearchBar );
    // Set the status options
    void                         setStatusOptions( bool useIdleTimer, int idleTime, bool hideNotificationsWhenBusy );
    // Set whether or not to show join/part messages in chat
    void                         setShowSessionInfo( bool showSessionInfo );
    // Set the way the chats will be grouped together
    void                         setTabbedChatMode( int tabbedChatMode );
    // Set a temporary password for the account for login attempts from the initialview dialog
    void                         setTemporaryPassword( const QString &temporaryPassword );
    // Set whether or not to login with this account.
    void                         setUseAutologin( bool useAutologin);
    // Set whether or not to show emoticons in chats.
    void                         setUseEmoticons( bool useEmoticons);
    // Set whether the account's passport is verified
    void                         setVerified( bool verified );
    // Set whether or not to send our display picture
    void                         setShowPicture( bool showPicture );
    // Creates the MsnObject only if the user wants to show a display picture
    void                         updateMsnObject();

  public: // Static public members
    // Validate a contact email
    static bool                  isValidEmail( QString email );

  protected: // Protected methods
    // Update the email-supported state (protected method)
    void                         setEmailSupported(bool emailSupported);

  private: // Private attributes
    // The message used to automatically reply to messages when the user is away.
    QString                      autoreplyMessage_;
    // The chat style
    QString                      chatStyle_;
    // The message font to be used by contact messages.
    QFont                        contactFont_;
    // The font color to be applied to contact messages.
    QString                      contactFontColor_;
    // Whether contacts should be shown by group, by online/offline status, or mixed.
    ContactListDisplayMode       contactListDisplayMode_;
    // Whether the account is marked to be deleted
    bool                         deleted_;
    // Whether the account has been changed at all
    bool                         dirty_;
    // Path of the display picture set by the user
    QString                      displayPicturePath_;
    // Path of the display picture original
    QString                      originalDisplayPicturePath_;
    // Whether the user can receive email notifications (hotmail.com or msn.com account)
    bool                         emailSupported_;
    // The directory of a selected emoticon
    QString                      emoticonStyle_;
    // The user's message font.
    QFont                        font_;
    // The user's message font color.
    QString                      fontColor_;
    // The user's friendly name, i.e. "Mike"
    FormattedString              friendlyName_;
    // Whether the account is a "guest account", not permanently saved on the computer
    bool                         guestAccount_;
    // Whether follow-up messages from the contact should be grouped.
    bool                         groupFollowupMessages_;
    // The user's handle, i.e. mkb137b@hotmail.com
    QString                      handle_;
    // Whether to hide popup notifications when busy.
    bool                         hideNotificationsWhenBusy_;
    // The number of minutes of inactivity before the user status is
    //  changed to idle.
    int                          idleTime_;
    // The status to connect as
    Status                       initialStatus_;
    // Display list pictures dimensions.
    int                          listPictureSize_;
    // The duration in seconds that a notification balloon is shown.
    int                          notificationDuration_;
    // The user's password
    QString                      password_;
    // The path to which chat files are automatically saved.
    QString                      saveChatPath_;
    // Whether or not to save chat files.
    bool                         saveChats_;
    // Whether or not to save chat files to external files
    bool                         saveChatsToFile_;
    // The format to use to save chat files
    ChatExportFormat             saveChatsFormat_;
    // Whether saved chats are organized by year, month, day, or all in a single directory
    int                          savedChatDirectoryStructure_;
    // If the password must be stored
    bool                         savePassword_;
    // The password intially read from the store
    QString                      savedPassword_;
    // Whether the chat window shakes for nudges
    bool                         shakeNudge_;
    // Whether or not allowed contacts should be visible.
    bool                         showAllowedContacts_;
    // Whether or not to show our display picture in chat
    bool                         showChatUserPicture_;
    // Whether or not to show the email of contacts in the contact list instead of their friendly name
    bool                         showContactEmail_;
    // Whether to show the contact list background image
    bool                         showContactListBird_;
    // Whether or not to show any email notifications (the user may not
    //  have a hotmail account).
    bool                         showEmail_;
    // Whether or not to display empty groups in the contact list.
    bool                         showEmptyGroups_;
    // Whether or not time information is shown for chat messages.
    bool                         showMessageTime_;
    // Whether now listening information should be shown or not.
    bool                         showNowListening_;
    // Whether or not offline contacts are shown.
    bool                         showOfflineContacts_;
    // Whether or not notifications should be shown for "other folder" email.
    bool                         showOtherFolders_;
    // Whether or not removed (or reverse) contacts should be visible.
    bool                         showRemovedContacts_;
    // Whether or not to show the history box
    bool                         showHistoryBox_;
    // Whether or not to show the search bar at login
    bool                         showSearchBar_;
    // Whether or not to show join/part messages in chat
    bool                         showSessionInfo_;
    // whether to show winks
    bool                         showWinks_;
    // Whether or not chats should be tabbed in one window
    int                          tabbedChatMode_;
    // A temporary password for the account for login attempts from the initialview dialog
    QString                      temporaryPassword_;
    // Whether to show seconds in message timestamps
    bool                         timestampShowSeconds_;
    // Whether to show date in message timestamps
    bool                         timestampShowDate_;
    // Whether or not to autologin with this user.
    bool                         useAutologin_;
    // Whether or not the contact's font should be forced to some value.
    bool                         useContactFont_;
    // Whether or not emoticons should be used.
    bool                         useEmoticons_;
    // Whether or not to use bold/italic/underline effects in chats.
    bool                         useFontEffects_;
    // Whether or not the user wants to go idle after a certain time.
    bool                         useIdleTimer_;
    // Whether or not to enable MSN Plus formatting in the contact list
    bool                         useListFormatting_;
    // Whether or not to enable MSN Plus formatting in chat
    bool                         useChatFormatting_;
    // Whether the account's passport is verified
    bool                         verified_;
    // the user's personal message
    FormattedString              personalMessage_;
    // Whether or not to send our display picture
    bool                         showPicture_;
    // MsnObject representing our display picture if we have one
    MsnObject                    *msnObject_;

  signals: // Public signals
    // Signal that the chat style settings have been changed
    void                         changedChatStyleSettings();
    // Signal that the contact list settings have been changed
    void                         changedContactListOptions();
    // Signal that group/contact display settings have been changed
    void                         changedDisplaySettings();
    // Signal that the user changed email display settings
    void                         changedEmailDisplaySettings();
    // Signal that the emoticon settings changed
    void                         changedEmoticonSettings();
    // Signal that font settings have changed
    void                         changedFontSettings();
    // Signal that the friendly name has changed
    void                         changedFriendlyName();
    // Signal that the msn object has changed.
    void                         changedMsnObject();
    // Signal that the now listening settings have changed
    void                         changedNowListeningSettings();
    // Signal that the personal message has changed
    void                         changedPersonalMessage();
    // Signal that timer settings were changed
    void                         changedTimerSettings();
    // Signal that the view mode has changed
    void                         changedViewMode();
};

#endif
