//
// C++ Interface: kmfnetzone
//
// Description:
//
//
// Author: Christian Hubinger <chubinger@irrsinnig.org>, (C) 2003
//
// Copyright: See COPYING file that comes with this distribution
//
//
/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/


#ifndef KMFNETZONE_H
#define KMFNETZONE_H

#include "netfilterobject.h"

// QT includes
#include <qptrlist.h>
#include <qguardedptr.h>
#include <qstring.h>
#include <quuid.h>
#include <qobject.h>

// KDE includes
#include <kdemacros.h>

// Project includes
#include "ipaddress.h"

namespace KMF {
class KMFGenericDoc;
class KMFProtocol;
class KMFProtocolUsage;
class KMFError;
class KMFCheckInput;
class IPAddress;
class KMFNetHost;
class KMFTarget;
class KMFNetwork;

//############ KMFNetZone #############
class KDE_EXPORT KMFNetZone : public NetfilterObject {
	Q_OBJECT
public:
	KMFNetZone( NetfilterObject *parent, const char* objectname, const QString& name );

	virtual ~KMFNetZone();
	
	virtual int type();
	virtual void clear();
	
	IPAddress* address() const {
		return m_address;
	};

	int operator==( const KMFNetZone& );
	
	bool isSameZone( KMFNetZone* );
	
	IPAddress* mask() const {
		IPAddress *addr = new IPAddress();
		addr->setAddress( IPAddress::calcNetworkMaskFromLength( m_maskLen ).toString() );
		return addr;
	};
	
	int maskLength() const {
		return m_maskLen;
	}
	
	KMFGenericDoc* doc() const;
	KMFNetwork* network() const;
	
	KMFNetZone* zone() const;
	KMFNetZone* rootZone();

	int getZoneType() const {
		return m_zoneType;
	};

	const QString& guiName() const {
		return m_guiName;
	};
	
	virtual const QString& name();
	const QString& generateName( const QString& );
	

	void setGuiName( const QString& );
	
	void setNetwork( KMFNetwork* );
	void setZone( const IPAddress& , int );
	void setParentZone( KMFNetZone* );
	bool isRootZone();
	
	void setMaskLength( int );
	
	

	const QDomDocument& getDOMTree();
	virtual void loadXML(const QDomDocument&, QStringList& errors );
	virtual void loadXML( QDomNode, QStringList& errors );
	
	KMFNetZone* addZone( const QString& , KMFError* );
	void delZone( KMFNetZone*, bool desructive = true );

	KMFProtocolUsage* addProtocolUsage( const QUuid& uuid, const QDomDocument& xml );
	
	void delProtocolUsage( KMFProtocolUsage*, bool desructive = true );

	KMFNetHost* addNetHost( const QString& name, const QDomDocument& xml );
	KMFTarget* addTarget( const QString& name, const QDomDocument& xml );

	KMFTarget* findTargetByName( const QString& name, bool fromRoot = true );
	KMFNetHost* findNetHostByName( const QString& name, bool fromRoot = true );
	KMFNetZone* findNetZoneByName( const QString& name, bool fromRoot = true );
	
	void delHost( KMFTarget*, bool desructive = true );

	QPtrList<KMFProtocolUsage>& protocols() const;
	QPtrList<KMFNetZone>& zones() const;
	QPtrList<KMFTarget>& hosts() const;

	KMFProtocolUsage* findProtocolUsageByProtocolUuid( const QUuid& uuid ) const;
	
	bool protocolInherited( const QUuid& uuid ) const;
	
	KMFTarget* placeHostInZone( KMFTarget* host );
	KMFNetZone* placeZoneInZone( KMFNetZone* zone );
	void refreshNetworkTree();
	
	bool readOnly() const {
		return m_readOnly;
	};
	void setReadOnly( bool );	
	
	QString toString();
	
	enum { ROOT, NODE } Type;
	
	void getAllTargets( KMFNetZone* zone, QPtrList<KMFTarget>* list );
	
protected slots:
	void slotOnProtocolUsageDeleted( QObject* protocol );
	
private: // Methods
	KMFTarget* findTargetByName_internal( const QString& name );
	KMFNetZone* findNetZoneByName_internal( const QString& name );
	QString zoneInfo();
	
	void getAllZones( KMFNetZone* zone, QPtrList<KMFNetZone>* list );
	void getAllHosts( KMFNetZone* zone, QPtrList<KMFTarget>* list );
	
	
	
	
private: // Data
	QGuardedPtr<KMFGenericDoc> m_doc;
	QGuardedPtr<KMFNetZone> m_zone;
	QGuardedPtr<KMFNetwork> m_network;
	IPAddress* m_address;
	
	QPtrList<KMFProtocolUsage> m_protocols;
	QPtrList<KMFNetZone> m_zones;
	QPtrList<KMFTarget> m_hosts;
	KMFError *m_err;
	int m_zoneType;
	int m_maskLen;
// 	bool m_deadEnd;
	QString m_guiName;
	bool m_readOnly;	
};
}


#endif
