/* @(#) .del-dbfsmount.c 1.8 @(#) */
/***************************************************************\
*	Copyright (c) 1999 First Step Internet Services, Inc.
*		All Rights Reserved
*	Distributed under the BSD Licenese
*
*	Module: DB
\***************************************************************/

#define _KOALAMUD_DBFSMOUNT_C "@(#) nitehawk@winghove.1ststep.net|BitKeeper/deleted/.del-dbfsmount.c|20000307194803|38498 @(#)"

#include "autoconf.h"
#include <sys/stat.h>

#include "version.h"
#include "koalatypes.h"
#include "log.h"
#include "conf.h"
#include "lib/dbinternal.h"
#include "db.h"
#include "memory.h"

/* This file contains mount and unmount routines.  Mounting the root file
 * system has the effect of creating a new database if one does not already
 * exist.  Mounting the root filesystem also does state initialization.
 */

/* Database code state */
dbstate_t dbstate =
{
	BLOCKSIZE,	/* If dbinitstate never gets called to get the real page size
				 *	we still have a reasonable default */
	NULL,		/* mountlist starts off null */
	NULL,		/* root vnode is null until we mount '/' */
	NULL,		/* To start off, there is no memory state. */
	0,			/* No mappings to start */
};

/* Initialize database state */
void dbinitstate(void)
{
	dbstate.pagesize = getpagesize();
}

/* Mount a database filesystem.
 *
 * 	A mountpoint of NULL has the special meaning that we are mounting the root
 *		database.
 */
koalaerror dbfsmount(char *devicepath, char *mountpoint, int mountflags)
{
	int openflags;
	pmountstat newmount;
	bool mountingroot=FALSE;

	/* First we initialize state */
	dbinitstate();

	/* Figure out what we flags we need to use while opening the database file
	 */
	if (mountflags == MF_NORMAL)
	{
		openflags = O_RDWR;
	}

	/* We need to allocate a new mountstat structure */
	if ((newmount = kmalloc(sizeof(mountstat), ALLOC_DATABASE)) == NULL)
	{
		logerr("Unable to allocate memory for mount struct.");
		return KENOMEM;
	}

	/* First try to open the database file */
	while ((newmount->dbfd = open(devicepath, openflags)) == -1)
	{
		switch(errno)
		{
			case ENOTDIR:
			case ENAMETOOLONG:
				kmfree(newmount, ALLOC_DATABASE);
				logerr("Invalid database path");
				return KEFOPEN;
			case EACCES:
				if (openflags == O_RDWR)
				{
					logerr("Unable to mount read/write - retrying read only");
					openflags = O_RDONLY;
				}
				else
				{
					kmfree(newmount, ALLOC_DATABASE);
					logerr("Permissions failure - check permissions on file");
					return KEFOPEN;
				}
				break;

			case EISDIR:
				kmfree(newmount, ALLOC_DATABASE);
				logerr("Must specify file, not directory for database");
				return KEFOPEN;
			case EROFS:
				kmfree(newmount, ALLOC_DATABASE);
				logerr("Disk filesystem mounted read only");
				return KEFOPEN;
			case ELOOP:
				kmfree(newmount, ALLOC_DATABASE);
				logerr("Symbolic link limit reached");
				return KEFOPEN;
			case ENXIO:
				kmfree(newmount, ALLOC_DATABASE);
				logerr("Device does not exist");
				return KEFOPEN;
			default:
				dbinitdb(devicepath);
		}
	}

	/* At this point, we should have an open database file */
	/* Copy state data into the mount stat */
	newmount->devicepath = kmalloc(strlen(devicepath), ALLOC_DATABASE);
	strcpy(newmount->devicepath, devicepath);

	/* Mount path is special - If it is null, we are mounting the root
	 * filesystem and there cannot be any existing mounts */
	if (mountpoint == NULL || (strcmp(mountpoint, "/") == 0))
	{
		mountingroot = TRUE;

		/* If root is already mounted... */
		if (dbstate.mountlist != NULL)
		{
			logerr("Root already mounted!");
			close(newmount->dbfd);
			kmfree(newmount->devicepath, ALLOC_DATABASE);
			kmfree(newmount, ALLOC_DATABASE);
			return KEDBMOUNTERROR;
		}

		/* allocate memory for the mount path to be 0 */
		newmount->mountpath = kmalloc(2, ALLOC_DATABASE);
		strcpy(newmount->mountpath, "/");
		newmount->next = NULL;

		/* Setup vnode and state pointers */
		if ((newmount->root = kmalloc(sizeof(kvnode), ALLOC_DATABASE)) == NULL)
		{
			logerr("Unable to allocate root vnode");
			close(newmount->dbfd);
			kmfree(newmount->devicepath, ALLOC_DATABASE);
			kmfree(newmount->mountpath, ALLOC_DATABASE);
			kmfree(newmount, ALLOC_DATABASE);
			return KENOMEM;
		}

		/* we now have the head of the mount list and root vnode */
		dbstate.mountlist = newmount;
		dbstate.rootvnode = newmount->root;

		/* Fixup root vnode */
		newmount->root->mount = newmount;
		newmount->root->parent = NULL;
		newmount->root->dnodenum = 0;
		newmount->root->superblock = 0;
	}
	else  // Not mounting root
	{
		/* otherwise, we need to search the directory tree to find the
		 * mountpoint and make sure it is empty */
	}

	/* At this point, we just need to read in the root directory tree.  We
	 * don't need to read the entire directory tree at this point */
	{
		//dnode *dn;
		kdb_dirent *dir;
		pkvdirent memdir;
		void *blk;

		/* Start by mapping the super block so we can find things */
		newmount->sb = mmapblockaligned(newmount->dbfd, 0, 1);
		
		/* Now that we know where the superblock is, we need to map the dnode
		 * table and block table */
		newmount->dnodetbl = mmapblockaligned(newmount->dbfd,
				newmount->sb->fs_dbno, newmount->sb->fs_dtsize);
		newmount->blktbl = mmapblockaligned(newmount->dbfd,
				newmount->sb->fs_btno, newmount->sb->fs_btsize);

		/* Read the root directory entry */
		newmount->root->dn = newmount->dnodetbl + sizeof(dnode) * DNODE_ROOT;
		blk = mmapblockaligned(newmount->dbfd, newmount->root->dn->d_db[0], 1);
		dir = blk;
		
		/* While there is a directory entry */
		while(dir->d_reclen)
		{
			/* First we need to allocate a virtual directory entry */
			if ((memdir = kmalloc(sizeof(kvdirent), ALLOC_DATABASE)) == NULL)
			{
				/* Out of memory.  Attempt to free the list before falling out
				 */
				memdir = newmount->root->dir;
				while (memdir)
				{
					pkvdirent tdir = memdir;
					memdir = memdir->next;

					kmfree(tdir->name, ALLOC_DATABASE);
					kmfree(tdir, ALLOC_DATABASE);
				}
				/* Free the rest of the memory used for this mount - state
				 * should be the same as it was at the start of this function */
				return KENOMEM;
			}  // No mem

			/* We have a new vdirent - link it in */
			memdir->next = newmount->root->dir;
			newmount->root->dir = memdir;

			/* Allocate room for the entry name and copy the name in */
			memdir->name = kmalloc(dir->d_namelen + 1, ALLOC_DATABASE);
			strcpy(memdir->name, dir->d_name);

			/* Copy the rest of the information in */
			memdir->mount = newmount;
			memdir->type = dir->d_type;
			memdir->dnodenum = dir->d_dnode;
			memdir->vn = NULL;

			/* Point special directories to the right place */
			if (strcmp(memdir->name, ".") == 0)
			{
				memdir->vn = newmount->root;
			}
			else
			if (strcmp(memdir->name, "..") == 0)
			{
				if (mountingroot)
				{
					memdir->vn = newmount->root;
				}
				else
				{
					/* Point to parent directory */
				}
			}

			/* get to the next directory entry */
			dir = (void*)dir + dir->d_reclen + 1;
		}  // While there are more directory entries
	} // Done reading root directory

	return KESUCCESS;
}
