/***************************************************************************
 *   Copyright (C) 2005-2008 by Georg Hennig                               *
 *   Email: georg.hennig@web.de                                            *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <qeventloop.h>

#include <klocale.h>

#include "kfileitemext.h"

#include "komparatoremptydirectoriesjob.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

KomparatorEmptyDirectoryJob::KomparatorEmptyDirectoryJob( QWidget *_parent )
	: KomparatorJob( _parent )
{
	m_running = 0;
	m_status = i18n( "Searching for empty directories..." );
	m_progress = 0;
	m_total = 0;

	m_is_running = false;
}

KomparatorEmptyDirectoryJob::~KomparatorEmptyDirectoryJob()
{
}

bool KomparatorEmptyDirectoryJob::initialize( KURL::List *url_list )
{
	m_mutex.lock();

	m_url_list.clear();
	m_search_list.clear();
	m_empty_directory_list.clear();

	m_is_running = false;

	m_progress = 0;

	m_url_list = *url_list;
	qHeapSort( m_url_list );

	m_total = m_url_list.count();

	m_initialized = true;
	m_canceled = false;

	m_mutex.unlock();

	return true;
}

void KomparatorEmptyDirectoryJob::run()
{
	if ( !m_initialized ) return;

	if ( m_url_list.count() == 0 )
	{
		emitProgress( m_status, -1 );

		if ( !isCanceled() )
		{
			QCustomEvent *event = new QCustomEvent( RESULT_EMPTY_FINISHED );
			QApplication::postEvent( m_parent, event );
		}
		return;
	}

	bool skip;
	KURL::List appended_urls;

	// Start from end of the list, because parent directories usually come first in the list
	// (starting from the beginning, we would have to search parent directories AND child directories).
	KURL::List::iterator it = m_url_list.end();
	for ( --it; ; --it )
	{
		if ( isCanceled() ) break;

		skip = false;

		// skip already appended URLs
		KURL::List::iterator it2;
		for ( it2=appended_urls.begin(); it2 != appended_urls.end(); ++it2 )
		{
			if ( (*it2).equals( (*it).upURL(), true ) ) skip = true;
			if ( (*it).upURL().isParentOf( *it2 ) ) skip = true;
		}

		if ( !skip )
		{
			m_search_list.append( (*it).upURL() );
		}
		else
		{
			incrementProgress();
		}

		appended_urls.push_back( (*it).upURL() );

		if ( it == m_url_list.begin() ) break;
	}

	m_mutex.lock();

	KIO::Job *job = KIO::listDir( m_search_list.last(), false /*progress*/, true /*hidden m_files*/ );
	connect( job, SIGNAL( entries( KIO::Job *, const KIO::UDSEntryList & ) ), (QObject*)this, SLOT( slotFile( KIO::Job *, const KIO::UDSEntryList & ) ) );
	connect( job, SIGNAL( result( KIO::Job * ) ), (QObject*)this, SLOT( slotFinished( KIO::Job * ) ) );
	m_search_list.pop_back();
	m_is_running = true;

	while ( m_is_running && !m_canceled )
	{
		m_mutex.unlock();
		QThread::usleep( 500 );
		m_mutex.lock();
	}

	m_mutex.unlock();

	emitProgress( m_status, -1 );

	if ( !isCanceled() )
	{
		QCustomEvent *event = new QCustomEvent( RESULT_EMPTY_FINISHED );
		QApplication::postEvent( m_parent, event );
	}

	m_initialized = false;
}

void KomparatorEmptyDirectoryJob::slotFile( KIO::Job *job, const KIO::UDSEntryList &list )
{
	m_mutex.lock();
	KIO::UDSEntryListConstIterator end = list.end();
	for ( KIO::UDSEntryListConstIterator it = list.begin(); it != end; ++it )
	{
		KFileItem tmp_item = KFileItem( KIO::UDSEntry( *it ), ((KIO::ListJob *)job)->url(), true, true );
		if ( tmp_item.url().fileName() != ((KIO::ListJob *)job)->url().fileName() && tmp_item.url().fileName() != ".." ) // There is something in the directory
		{
			disconnect( job, SIGNAL( result( KIO::Job * ) ), this, SLOT( slotFinished( KIO::Job * ) ) );
			job->kill( false );

			if ( !m_search_list.isEmpty() )
			{
				KIO::Job *new_job = KIO::listDir( m_search_list.last(), false /*progress*/, true /*hidden m_files*/ );
				connect( new_job, SIGNAL( entries( KIO::Job *, const KIO::UDSEntryList & ) ), (QObject*)this, SLOT( slotFile( KIO::Job *, const KIO::UDSEntryList & ) ) );
				connect( new_job, SIGNAL( result( KIO::Job * ) ), (QObject*)this, SLOT( slotFinished( KIO::Job * ) ) );
				m_search_list.pop_back();
			}
			else
			{
				m_is_running = false;
			}

			m_mutex.unlock();
			incrementProgress();
			m_mutex.lock();

			break;
		}
	}
	m_mutex.unlock();
}

void KomparatorEmptyDirectoryJob::slotFinished( KIO::Job *job )
{
	incrementProgress();

	m_mutex.lock();
	if ( job->error() == 0 )
	{
		m_empty_directory_list.append( ((KIO::SimpleJob*)job)->url() );
	}

	if ( !m_search_list.isEmpty() )
	{
		KIO::Job *new_job = KIO::listDir( m_search_list.last(), false /*progress*/, true /*hidden m_files*/ );
		connect( new_job, SIGNAL( entries( KIO::Job *, const KIO::UDSEntryList & ) ), (QObject*)this, SLOT( slotFile( KIO::Job *, const KIO::UDSEntryList & ) ) );
		connect( new_job, SIGNAL( result( KIO::Job * ) ), (QObject*)this, SLOT( slotFinished( KIO::Job * ) ) );
		m_search_list.pop_back();
	}
	else
	{
		m_is_running = false;
	}

	m_mutex.unlock();
}

const KURL::List &KomparatorEmptyDirectoryJob::getEmptyDirectoryList()
{
	return m_empty_directory_list;
}
