
#ifndef MODULE_H
#define MODULE_H

#include <kore/kore.h>

namespace kore
{

class Version;
class ModuleManager;

/**
 * class kore::Module - the base structural and functional unit.
 * Basically everything (expcept for some helper classes) is a module.
 * This is an abstract class you may use to create your own modules.
 * Also setting up a full module Info is mandatory.
 */
class KORE_API Module
{
public:

    /**
     * class kore::Module::Info - provides usefull information on a module.
     * helper class.
     */
    class KORE_API Info
    {
    public:
        /**
         * Create a Info object
         * @param module - the Module object this Info is associated to
         * @param name - the name of this module
         * @param type - the type (class/API) of this module
         * @param description - short description of the module
         * @param version - the version of this module
         * @param api - kore API version required by this module
         */
        Info(Module* module, const char* name = 0, const char* type = 0, const char* description = 0, const Version* version = 0, const Version* api = 0);
        virtual ~Info();

        /**
         * Gets the kore::Module this instance is associated to.
         * @return - the module.
         */
        virtual Module* module() const;
        /**
         * Gets the name of the module.
         * @return - module name
         */
        virtual const char* name() const;
        /**
         * Gets the module type.
         * @return - module type.
         */
        virtual const char* type() const;
        /**
         * Gets the module description string.
         * @return - module description.
         */
        virtual const char* description() const;
        /**
         * Gets the version of the module.
         * @return - module version.
         */
        virtual const Version* version() const;
        /**
         * Gets the API version requiered by this module.
         * @return - API version
         */
        virtual const Version* APIVersion() const;
    protected:
        /**
         * Default constructor, for creating an empty Info object
         */
        Info();
        /**
         * Convenience method for setting the Module this Info is associated to.
         * @param module - the module.
         */
        void setModule(Module* module = 0);
        /**
         * Convenience method for setting the Module name.
         * @param name - module name.
         */
        void setName(const char* name = 0);
        /**
         * Convenience method for setting the Module type.
         * @param type - module type
         */
        void setType(const char* type = 0);
        /**
         * Convenience method for setting the Module description.
         * @param descr - module description
         */
        void setDescription(const char* descr = 0);
        /**
         * Conveninence method for setting the Module version.
         * @param version - module version.
         */
        void setVersion(const Version* version = 0);
        /**
         * Convenience method for setting the API version required by this Module
         * @param api - API version.
         */
        void setAPIVersion(const Version* api = 0);
        /**
         * Convenience method for setting all the Module information "in one shot"
         * @param module - the Module object this Info is associated to
         * @param name - the name of this module
         * @param type - the type (class/API) of this module
         * @param description - short description of the module
         * @param version - the version of this module
         * @param api - kore API version required by this module
         */
        void setInfo(Module* module = 0, const char* name = 0, const char* type = 0, const char* description = 0, const Version* version = 0, const Version* api = 0);
    private:
        // the module this Info is associated to
        Module* _module;
        // module name
        const char* _name;
        // module type
        const char* _type;
        // module description
        const char* _description;
        // module version
        const Version* _version;
        // api version required by this module
        const Version* _api;
    };

    /**
     * Callback method triggered right BEFORE REGISTERING this module to the
     * ModuleManager.
     * @param mm - the ModuleManager this Module will get registered to.
     */
    virtual void registeringModule(ModuleManager* mm);
    /**
     * Callback method triggered right AFTER REGISTERING this module to the
     * ModuleManager.
     * @param mm - the ModuleManager this Module got registered to.
     */
    virtual void moduleRegistered(ModuleManager* mm);
    /**
     * Callback method triggered right BEFORE UNREGISTERING this module from the
     * ModuleManager.
     * @param mm - the ModuleManager this Module will get unregistered from.
     */
    virtual void unregisteringModule(ModuleManager* mm);
    /**
     * Callback method triggered right AFTER UNREGISTERING this module from the
     * ModuleManager.
     * @param mm - the ModuleManager this Module will get unregistered from.
     */
    virtual void moduleUnregistered(ModuleManager* mm);
    /**
     * Gets the info for this module. The default implemetation returns the
     * private _info member data, but subclasses may override that.
     * @return the kore::Module::Info instance associated to this Module
     */
    virtual const Info* info() const;
protected:
    /**
     * Default constructor. Creates an empty Module instance.
     */
    Module();
    /**
     * Creates a module and sets its Info.
     * @param info - the module info
     */
    Module(const Info* info);
    /**
     * Destructor.
     */
    virtual ~Module();
    /**
     * Convenience method for setting the module info.
     * @param info - module info.
     */
    void setInfo(const Info* info = 0);
private:
    // module info
    const Info* _info;
};

};

#endif
