
#include <kore/kore.h>
#include <kore/version.h>
#include <kore/pluginloader.h>
#include <kore/plugin.h>
#if defined( KORE_WIN32 )
#elif defined( KORE_BEOS )
#include <image.h>
#elif defined( KORE_ATHEOS )
#include <atheos/image.h>
#else
#include <dlfcn.h>
#endif

#define PLDR_MAJOR 0
#define PLDR_MINOR 0
#define PLDR_REVISION 1
#define PLDR_VERSION "0.0.1"
#define PLDR_API_MAJOR 0
#define PLDR_API_MINOR 0
#define PLDR_API_REVISION 1
#define PLDR_API_VERSION "0.0.1"
#define PLDR_NAME "Plugin Loader"
#define PLDR_TYPE "Plugin Loader"
#define PLDR_DESCRIPTION "Default Kore Plugin Loader"
#define PLDR_SERVICE "Kore/Kernel/Plugin Loader"
#define PLDR_SERVICE_DESCRIPTION "Kore Plugin Loader"

using namespace kore;

PluginLoader::PluginLoader()
{
    _loaderVersion = new Version(PLDR_MAJOR,PLDR_MINOR,PLDR_REVISION,PLDR_VERSION);
    _loaderAPIVersion = new Version(PLDR_API_MAJOR,PLDR_API_MINOR,PLDR_API_REVISION,PLDR_API_VERSION);
    _loaderInfo = new Info(this, PLDR_NAME, PLDR_TYPE, PLDR_DESCRIPTION, _loaderVersion, _loaderAPIVersion);
    setInfo(_loaderInfo);
    _loaderService = new Service(this, PLDR_SERVICE, PLDR_SERVICE_DESCRIPTION);
    addService(_loaderService);
}
PluginLoader::~PluginLoader()
{
    delete _loaderInfo;
    delete _loaderVersion;
    delete _loaderAPIVersion;
    delete _loaderService;
}

// "my_plugin" -> "libmy_plugin.so"/"my_plugin.dll"
char* PluginLoader::libName2fileName(const char* libName)
{
    int n = strlen(libName);
#ifdef KORE_WIN32
	char* fn = new char[n+1+4];
	return strcat(strcpy(fn,libName),".dll");
#else
    char* fn = new char[n+1+6];
    return strcat(strcat(strcpy(fn,"lib"),libName),".so");
#endif
}
// "libmy_plugin.so"/"my_plugin.dll" -> "my_plugin"
char* PluginLoader::fileName2libName(const char* fileName)
{
#ifdef KORE_WIN32
    int n = strlen(libName) - 4;
	char* ln = new char[n+1];
#else
    char* pos = strstr(fileName, ".so");
    int n = pos - fileName - 3;
    char* ln = new char[n+1];
    fileName += 3;
#endif
    return strncpy(ln, fileName, n);
}

Plugin* PluginLoader::runPlugin(const char* libName, const char* libPath, int libFlags)
{
    Plugin* plugin = openPlugin(libName, libPath, libFlags);
    if( plugin )
        plugin->initPlugin();
    return plugin;
}
Plugin* PluginLoader::openPlugin(const char* libName, const char* libPath, int libFlags)
{
    HMODULE handle;
    char* fileName = libName2fileName(libName);
    char* filePath = 0;
    Plugin* res;
    if( libPath && libPath[0] )
    {
        int n = strlen(libPath);
        int slash = libPath[n-1] == '/' ? 0 : 1;
        filePath = new char[n+strlen(fileName)+slash+1];
        strcpy(filePath, libPath);
        if( slash )
        {
            filePath[n] = '/';
            filePath[n+1] = 0;
        }
    }
    else
    {
        filePath = new char[strlen(fileName)+1];
        filePath[0] = 0;
    }
    strcat( filePath, fileName );
    //cout << "Loading: " << filePath << endl;
    delete[] fileName;
#if defined( KORE_WIN32 )
	handle = LoadLibrary(filePath);
#elif defined( KORE_BEOS )
    handle = load_add_on(filePath);
#elif defined( KORE_ATHEOS )
    handle = load_library( filePath, 0 );
#else
    handle = dlopen(filePath, libFlags | RTLD_NOW );
#endif
    delete[] filePath;
    //cout << "Loaded: " << handle << endl;
#if defined( KORE_BEOS )
    if( handle < B_NO_ERROR )
#elif defined( KORE_ATHEOS )
    if( handle < 0 )
#else
    if( !handle )
#endif
        return 0;
	//cout << "Found: " << handle << endl;
	PluginFuncType sym = 0;
#if defined( KORE_WIN32 )
	sym = (PluginFuncType) GetProcAddress( handle, "plugin" );
#elif defined( KORE_BEOS )
    if( B_OK != get_image_symbol( handle, "plugin", B_SYMBOL_TYPE_TEXT, reinterpret_cast<void**>(&sym) ) )
        sym = 0;
#elif defined( KORE_ATHEOS )
    if( 0 != get_symbol_address( handle, "plugin", -1, reinterpret_cast<void**>(&sym) ) )
	sym = 0;
#else
    sym = (PluginFuncType) dlsym( const_cast<void*>(handle), "plugin" );
#endif
    //cout << "Done: " << sym << endl;
    if( !sym )
        res = new Plugin(handle,libName,libPath,libFlags);
    else
        res = sym(handle,libName,libPath,libFlags);
    //cout << "Called: " << sym << endl;
    res->pluginLoaded();
    return res;
}

void PluginLoader::closePlugin(Plugin* plugin)
{
    plugin->unloadingPlugin();
#if defined( KORE_WIN32 )
	FreeLibrary( plugin->libHandle() );
#elif defined( KORE_BEOS )
    unload_add_on( plugin->libHandle() );
#elif defined( KORE_ATHEOS )
    unload_library( plugin->libHandle() );
#else
    dlclose(const_cast<void*> (plugin->libHandle()));
#endif
}

const char* PluginLoader::lastError()
{
    _lastError[0]=0;
#if defined( KORE_WIN32 )
    FormatMessage( FORMAT_MESSAGE_FROM_SYSTEM | FORMAT_MESSAGE_IGNORE_INSERTS, 0,
        GetLastError(), MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), _lastError, 100, NULL );
#elif defined( KORE_BEOS )
    strncpy(_lastError, strerror(errno), 100);
#elif defined( KORE_ATHEOS )
    strncpy(_lastError, strerror(errno), 100);
#else
    strncpy(_lastError, dlerror(), 100);
#endif
    return _lastError;
}
