//
// C++ Implementation: kpgdatatable
//
// Description: 
//
//
// Author: Lumir Vanek <lvanek@users.sourceforge.net>, (C) 2006
//
// Copyright: See COPYING file that comes with this distribution
//
//
#include "kpgdatatable.h"

// include files for Qt
#include <qtextcodec.h>

// include files for KDE
#include <klocale.h>
#include <kmessagebox.h>
#include <kcursor.h>
#include <kfinddialog.h>
#include <kfind.h>
#include <kdebug.h>
#include <kfiledialog.h>

// include files for libpgxx 
#include <pqxx/binarystring>

// application specific includes
#include "../kpogreview.h"
#include "kpgdatatablechildview.h"
#include "../kpgconfiguration.h"
#include "kpgdatatablesettings.h"
#include "../DbObjects/kpgconnection.h"
#include "../DbObjects/kpgdatabase.h"
#include "../DbObjects/kpgtable.h"
#include "../DbObjects/kpgtablecolumnsfolder.h"
#include "../DbObjects/kpgtablecolumn.h"
#include "kpgdatatabledialog.h"
#include "../kpgsqldialog.h"
#include "kpgtexteditdialog.h"
#include "../kpgutil.h"
//#include "../XmlSupport/kxetexteditordialog.h"
#include "kpgkatexmleditordialog.h"


KPGDataTable::KPGDataTable(KPGDataTableChildView *parent, 
    KPoGreView *pPoGreView,
    KXMLGUIFactory *pXmlGuiFactory,
    const PGSTD::string& strConnectionOptions,
    const QString & strNamespaceName,
    const QString & strTableName,
    const ListTableColumns & listTableColumns,
    const MapIndexKey & mapIndexKey,
    const QString &strWherePredicate,
    const QString &strOrderByPredicate,
    KPGKateXmlEditorDialog *pKateXmlEditorDialog
    )
    : 
    KXMLGUIClient(pPoGreView),
    QTable(parent, "DataTable"),
    m_pXmlGuiFactory(pXmlGuiFactory),
    m_pFind(0),
    m_pKateXmlEditorDialog(pKateXmlEditorDialog)
{
    m_bOpeningReload = true;
    m_state = refresh;
    m_editedRow = -1;
    m_bInplaceEditing = false;
    
    m_strConnectionOptions = strConnectionOptions;
    m_strNamespaceName = strNamespaceName;
    m_strTableName = strTableName;
    m_strWherePredicate = strWherePredicate;
    m_strOrderByPredicate = strOrderByPredicate;
            
    // Setup GUI
    //setFactory(pXmlGuiFactory);
    setXMLFile("kpgdatatable.rc", true);
    m_bIsAddedToGuiFactory = false;

    //-----------------------------------------------------------------
    // Edit actions    
    m_pActEditFind = KStdAction::find( this, SLOT(slotEditFind()), actionCollection(), "edit_find" );
    
    m_pActEditFindNext = KStdAction::findNext( this, SLOT(slotEditFindNext()), actionCollection(), "edit_find_next" );

    //-----------------------------------------------------------------
    // Datatable actions       
    
    m_pActFirstRow = new KAction(i18n("First row"), "player_start", 0, this, SLOT(slotFirstRow()), actionCollection(), "datatable_first_row");
    
    m_pActPreviousRow = new KAction(i18n("Previous row"), "1leftarrow", 0, this, SLOT(slotPreviousRow()), actionCollection(), "datatable_previous_row");
    
    m_pActNextRow = new KAction(i18n("Next row"), "1rightarrow", 0, this, SLOT(slotNextRow()), actionCollection(), "datatable_next_row");
    
    m_pActLastRow = new KAction(i18n("Last row"), "player_end", 0, this, SLOT(slotLastRow()), actionCollection(), "datatable_last_row");
    
    m_pActEditCell = new KAction(i18n("Edit cell"), "edittool", Key_F2, this, SLOT(slotEditCell()), actionCollection(), "datatable_edit_cell");
    
    m_pActClearCell = new KAction(i18n("Clear cell"), "editclear", Key_Delete, this, SLOT(slotClearCell()), actionCollection(), "datatable_clear_cell");
    
    m_pActInsertRow = new KAction(i18n("Insert row"), "insrow", CTRL+Key_Insert, this, SLOT(slotInsertRow()), actionCollection(), "datatable_insert_row");
    
    m_pActDeleteRow = new KAction(i18n("Delete row"), "remrow", CTRL+Key_Delete, this, SLOT(slotDeleteRow()), actionCollection(), "datatable_delete_row");
    
    m_pActCommitChanges = new KAction(i18n("Update changes"), "ok", CTRL+Key_Return, this, SLOT(slotCommitChanges()), actionCollection(), "datatable_commit_changes");
    
    m_pActCancelChanges = new KAction(i18n("Cancel changes"), "cancel", Key_Escape, this, SLOT(slotCancelChanges()), actionCollection(), "datatable_cancel_changes");
    
    m_pActReloadData = new KAction(i18n("Reload data ..."), "reload", Key_F8, this, SLOT(slotReloadData()), actionCollection(), "datatable_reload_data"); 
    
    m_pActStopReload = new KAction(i18n("Stop reload data"), "stop", 0, this, SLOT(slotStopReload()), actionCollection(), "datatable_stop_reload");
    
    m_pActLoadLob = new KAction(i18n("Load from file..."), "fileopen", 0, this, SLOT(slotLoadLob()), actionCollection(), "datatable_load_lob");
    
    m_pActSaveLob = new KAction(i18n("Save to file..."), "filesaveas", 0, this, SLOT(slotSaveLob()), actionCollection(), "datatable_save_lob");
    
    m_pActEditLobInEditor = new KAction(i18n("Edit..."), "pencil", 0, this, SLOT(slotEditLobInEditor()), actionCollection(), "datatable_edit_lob_editor");

    //---------------------------------------------------------------------------------------------
                
    //--- Copy info about primary key columns
    for(MapIndexKey::ConstIterator it = mapIndexKey.begin(); it != mapIndexKey.end(); ++it ) 
    {
        m_mapIndexKey[it.key()] = it.data();
    }
        
    m_listTableColumns = listTableColumns;
    
    //--- Look, if PK is auto generated by sequence
    m_bAutoGeneratedPrimaryKey = false;
    
    if(mapIndexKey.size() == 1)
    {
        MapIndexKey::ConstIterator it = mapIndexKey.begin();
        
        QString strPKColumnName(it.data());
        
        for(ListTableColumns::const_iterator cit = m_listTableColumns.begin(); cit != m_listTableColumns.end(); ++cit)
        { 
            if(strPKColumnName == (*cit).name())
            {
                if((*cit).hasDefault() && ((*cit).defaultValue().startsWith("nextval")))
                {
                    m_strNextValueForPrimaryKey = (*cit).defaultValue();
                    m_bAutoGeneratedPrimaryKey = true;
                    m_strPKColumnName = strPKColumnName;
                    kdDebug() << k_funcinfo "Found autogenerated PK: " <<  m_strNextValueForPrimaryKey << endl;
                }
                break;
            }
        }
    }
        
    //-------------------------------------------------------------------------
    setSorting( false );
    setSelectionMode( QTable::SingleRow );
    
    m_connectionInThread.setEventReceiver(this, &m_pqxxResult);
        
    // Open connection
    try
    {
        m_connectionInThread.connectToServer(m_strConnectionOptions);
    }
    catch (const std::exception &e)
    {
        kdError() << "Failed to open connection " << e.what() << endl;
        KMessageBox::sorry(this, e.what());
        return;
    }
    
    disableAllActions();
    
    connect(this, SIGNAL(currentChanged(int, int)), this, SLOT(slotCurrentChanged(int, int)));
    
    connect(this, SIGNAL(valueChanged(int, int)), this, SLOT(slotValueChanged(int, int)));
    
    connect(this, SIGNAL(doubleClicked(int, int, int, const QPoint &)), this, SLOT(slotDoubleClicked(int, int, int, const QPoint &)));
    
    connect(this, SIGNAL(contextMenuRequested(int, int, const QPoint &)), this, SLOT(slotContextMenuRequested(int, int, const QPoint &)));
    
    connect(horizontalHeader(), SIGNAL(clicked(int)), this, SLOT(slotHorzHeaderClicked(int)));
}


KPGDataTable::~KPGDataTable()
{
    // Close connection
    m_connectionInThread.disconnectFromServer(); // close connection if any
    
    if(m_pFind != 0) delete m_pFind;

    removeFromGuiFactory();
}

// Add yourself to GUI factory
void KPGDataTable::addToGuiFactory()
{
    if(!m_bIsAddedToGuiFactory)
    {
        //setFactory(m_pXmlGuiFactory);
        m_pXmlGuiFactory->addClient(this);
        m_bIsAddedToGuiFactory = true;
        //kdDebug() << k_funcinfo << endl;
    }
}

// Remove yourself from GUI factory
void KPGDataTable::removeFromGuiFactory()
{
    if(m_bIsAddedToGuiFactory)
    {
        //setFactory(m_pXmlGuiFactory);
        m_pXmlGuiFactory->removeClient(this);
        m_bIsAddedToGuiFactory = false;
        //kdDebug() << k_funcinfo << endl;
    }
}

void KPGDataTable::slotEditFind()
{
    findFirst(m_listOfResultQuerySearchHistory);
}

void KPGDataTable::slotEditFindNext()
{
    findNext();
}

void KPGDataTable::slotFirstRow()
{
    firstRow();
}

void KPGDataTable::slotPreviousRow()
{
    previousRow();
}

void KPGDataTable::slotNextRow()
{
    nextRow();
}

void KPGDataTable::slotLastRow()
{
    lastRow();
}

void KPGDataTable::slotEditCell()
{
    editCell();
}


void KPGDataTable::slotClearCell()
{
    clearCell();
}


void KPGDataTable::slotInsertRow()
{
    insertRow();
}


void KPGDataTable::slotDeleteRow()
{
    deleteRow();
}

void KPGDataTable::slotCommitChanges()
{
    commitChanges();
}


void KPGDataTable::slotCancelChanges()
{
    cancelChanges();
}

void KPGDataTable::slotReloadData()
{
    reloadData();
}

void KPGDataTable::slotStopReload()
{
    stopReload();
}

void KPGDataTable::slotLoadLob()
{
    KURL url = KFileDialog::getOpenURL(QString::null,   i18n("*.*|All files"), this, i18n("Open File..."));
        
    if(!url.isEmpty())
    {
        loadLob(url);
    }
}

void KPGDataTable::slotSaveLob()
{
    KURL url = KFileDialog::getSaveURL(QDir::currentDirPath(),
            i18n("*.*|All files"), this, i18n("Save as..."));
        
    if(QFile::exists(url.path()))
    {
         // Is it already existing ?
         QString error(url.path());     
         error.append(i18n(" already exists!\n Do you wish to overwrite it ?"));
         if (KMessageBox::questionYesNo(this, error, i18n("Overwrite File ?")) != KMessageBox::Yes) 
         {
            return;
         }
    }
    
    if(!url.isEmpty())
    {
        saveLob(url);
    }
}

void KPGDataTable::slotEditLobInEditor()
{
    editLobInEditor();
}


// Filters events - This is necessary filter moving current row to force call commitChanges()
// before continuing. If commitChanges() fails, return true to stay on current row
bool KPGDataTable::eventFilter(QObject *pObject, QEvent *pEvent)
{
    switch(pEvent->type()) 
    {
        case QEvent::KeyPress: 
        {
            QKeyEvent *pKeyEvent = (QKeyEvent*) pEvent;
                
            if(pKeyEvent->key() == Key_Up || pKeyEvent->key() == Key_Prior ||
                    pKeyEvent->key() == Key_Home || pKeyEvent->key() == Key_Down ||
                    pKeyEvent->key() == Key_Next || pKeyEvent->key() == Key_End)
            {
                if(isEditing())
                {
                    QWidget *editorWidget = cellWidget(currentRow(), currentColumn());
                    if(editorWidget && pObject == editorWidget)
                    {
                        endEdit(currentRow(), currentColumn(), true, true); // editMode() != Editing
                    }
                }
                
                if((m_state == update) || (m_state == insert))
                {
                    //kdDebug() << "Keyboard event filter fired" << endl;
                    if(commitChanges() == false) return true; // not allow change current cell, if commitChanges() fails
                }
            }
            
            break;
        }
        default: break;
    }
    

    bool bResult = QTable::eventFilter(pObject, pEvent);
    setActions();
    return bResult;
}

// This event handler is called whenever the QScrollView receives a mousePressEvent(): the press position in e is translated to be a point on the contents.
void KPGDataTable::contentsMousePressEvent(QMouseEvent* pMouseEvent)
{
    if (! isEditing()) 
    {
        int nRow = rowAt( pMouseEvent->pos().y() );
        fixRow2( nRow, pMouseEvent->pos().y() );
                        
        // If we in Edit or Insert mode, send dirty row to database
        if((m_editedRow >= 0) && ((m_state == update) || (m_state == insert)) && (m_editedRow != nRow))
        {
            //kdDebug() << "Mouse event filter fired" << endl;
                
            if(commitChanges() == false)
            {
                return; // not allow change current cell, if commitChanges() fails
            }
        }
    }
    
    QTable::contentsMousePressEvent(pMouseEvent);
}

/**
  * Reimplemented for update KActions state purpose. 
  * m_bInplaceEditing must be used, because QTable::EditMode is changed after QTable::beginEdit is processed, but this is too late.
*/ 
QWidget *KPGDataTable::beginEdit(int nRow, int nCol, bool bReplace)
{
    QWidget *pWidget = QTable::beginEdit(nRow, nCol, bReplace);
    if(pWidget != 0)
    {
        m_bInplaceEditing = true;
    }
    setActions();
    return pWidget;
}

// Reimplemented for update KActions state purpose
void KPGDataTable::endEdit(int nRow, int nCol, bool bAccept, bool bReplace)
{
    QTable::endEdit(nRow, nCol, bAccept, bReplace);
    m_bInplaceEditing = false;
    
    setActions();
}

// It's here, because QTable::fixRow is private
void KPGDataTable::fixRow2( int &row, int y )
{
    if ( row == -1 ) 
    {
    if ( y < 0 )
        row = 0;
    else
        row = numRows() - 1;
    }
}

// Disable all Datatable related KActions
void KPGDataTable::disableAllActions()
{
    m_pActFirstRow->setEnabled(false);
    m_pActPreviousRow->setEnabled(false);
    m_pActNextRow->setEnabled(false);
    m_pActLastRow->setEnabled(false);
    m_pActEditCell->setEnabled(false);
    m_pActClearCell->setEnabled(false);
    m_pActInsertRow->setEnabled(false);
    m_pActDeleteRow->setEnabled(false);
    m_pActCommitChanges->setEnabled(false);
    m_pActCancelChanges->setEnabled(false);
    m_pActReloadData->setEnabled(false);
    m_pActStopReload->setEnabled(false);
    m_pActEditFind->setEnabled(false); 
    m_pActEditFindNext->setEnabled(false);
}

// Enable/disable Datatable related KActions with regards to datatable state
void KPGDataTable::setActions()
{
    if(m_state == refresh)
    {
        disableAllActions();
        m_pActStopReload->setEnabled(true);
        return;
    }
    else
    {
        m_pActStopReload->setEnabled(false);
    }
    
    if(currentRow() >= 0)
    {
        m_pActFirstRow->setEnabled((currentRow() > 0) && !m_bInplaceEditing);
        m_pActPreviousRow->setEnabled((currentRow() > 0) && !m_bInplaceEditing);
        m_pActNextRow->setEnabled((currentRow() < numRows() - 1) && !m_bInplaceEditing);
        m_pActLastRow->setEnabled((currentRow() < numRows() - 1) && !m_bInplaceEditing);
        
        m_pActEditCell->setEnabled(!m_bInplaceEditing);
        m_pActClearCell->setEnabled(!m_bInplaceEditing);
        m_pActDeleteRow->setEnabled(((m_state == view) && !m_bInplaceEditing));
        
        m_pActCommitChanges->setEnabled((m_state != view) && !m_bInplaceEditing);
        m_pActCancelChanges->setEnabled((m_state != view));
    }
    else
    {
        disableAllActions();
    }
    
    m_pActInsertRow->setEnabled((m_state == view) && !m_bInplaceEditing);
    m_pActReloadData->setEnabled((m_state == view) && !m_bInplaceEditing);
    
    m_pActEditFind->setEnabled(numRows() > 0);
}

void KPGDataTable::reloadData()
{
    if(m_bOpeningReload == false)
    {
        // If it's not opening load, give chance to user to correct WHERE predicate
        KPGDataTableDialog dlg(this);
        dlg.setWherePredicate(m_strWherePredicate);
        dlg.setOrderByPredicate(m_strOrderByPredicate);
        if( dlg.exec() != QDialog::Accepted )
        {
            return;
        }
        m_strWherePredicate = dlg.wherePredicate();
        m_strOrderByPredicate = dlg.orderByPredicate();
    }
    else
    {
        m_bOpeningReload = false; // This is first reload, KPGDataTableDialog was already launched
    }
    
    // clear previous result
    setNumRows(0);
    setNumCols(0);
        
    m_state = refresh;
    m_editedRow = -1;
    disableAllActions();
    m_pActStopReload->setEnabled(true);
    setEnabled(false);
    ((QWidget *) parent())->setCursor(KCursor::waitCursor());
        
    QString strSql(sqlForSelectAllRows());
    m_connectionInThread.setSQL(strSql, KPGConnection::eTransNormal);
    m_connectionInThread.start();
}

void KPGDataTable::customEvent(QCustomEvent *pEvent)
{
    KPGQueryResultEvent *pQueryResultEvent = static_cast <KPGQueryResultEvent *> (pEvent);
    
    // display result
    displayResult();
    
    if(pQueryResultEvent->error().isEmpty() == false)
    {
        KMessageBox::error(this, pQueryResultEvent->error());
    }
            
    ((QWidget *) parent())->setCursor(KCursor::arrowCursor());
    setEnabled(true);
    m_connectionInThread.wait();
    
    m_state = view;
    setActions();
}

// Display popup menu
void KPGDataTable::popupContextMenu(const QString & strMenuName, const QPoint & pos)
{
    QWidget * pContainer = factory()->container(strMenuName, this);

    if ( ! pContainer )
    {
        kdError() << k_funcinfo << " Couldn't get a container widget for the given menu name (" << strMenuName << ")" << endl;
        return;
    }

    if ( ! pContainer->inherits("KPopupMenu") )
    {
        kdError() << k_funcinfo << " Wrong container widget" << endl;
        return;
    }

    KPopupMenu * pMenu = static_cast <KPopupMenu*> (pContainer);
    pMenu->popup( pos );
}

// Display SQL result
void KPGDataTable::displayResult()
{
    // Fill output area with query result
    unsigned int nTotalRows = m_pqxxResult.size();
    unsigned int nTotalCols = m_pqxxResult.columns();
        
    setNumRows(nTotalRows);
    setNumCols(nTotalCols);
    
    // Set field names into table header
    QHeader* hHeader = horizontalHeader();
    
    for(unsigned int nCol = 0; nCol < nTotalCols; nCol++)
    {
        QString strColumnName(m_pqxxResult.column_name(nCol));
        
        QPixmap *pTypePixmap = KPGTableColumn::getPixmapForType(m_listTableColumns[nCol].typName());
        
        bool bColumnIsInPrimaryKey = false;
        
        for(MapIndexKey::ConstIterator it = m_mapIndexKey.begin(); it != m_mapIndexKey.end(); ++it ) 
        {
            if(it.data() == strColumnName)
            {
                bColumnIsInPrimaryKey = true;
                break;
            }
        }
        
        if(!bColumnIsInPrimaryKey) // is column part of primary key ?
        {   
            // No - set icon for type on header
            hHeader->setLabel(nCol, QIconSet(*pTypePixmap), strColumnName);
        }
        else
        {
            // Yes - set icon for type + Primary Key icon on header
            QPixmap joinedPixmap(32, 16, -1, QPixmap::BestOptim);
            
            copyBlt (&joinedPixmap, 0, 0, pTypePixmap);
            copyBlt (&joinedPixmap, 16, 0, KPGTreeItem::m_pIconPrimaryKeyConstr);
            
            QIconSet iconSet(joinedPixmap);
            iconSet.setIconSize (QIconSet::Small, QSize(32, 16));
            
            hHeader->setLabel(nCol, iconSet, strColumnName);
        }
    }
        
    // Set checkboxes to boolean columns
    for(int currentCol = 0; currentCol < numCols(); currentCol++)
    {
       // Disallow direct editing BLOBs
       if(m_listTableColumns[currentCol].typName() == "text")
       {
           setColumnReadOnly(currentCol, true);
       }
       else if(m_listTableColumns[currentCol].typName() == "xml")
       {
           setColumnReadOnly(currentCol, true);
       }
       else if(m_listTableColumns[currentCol].typName() == "bytea")
       {
           setColumnReadOnly(currentCol, true);
       }
    }
    
    // data
    int nTableDataRow = 0;
    for(unsigned int nRow = 0; nRow < nTotalRows; nRow++)
    {
        displayOneRow(nTableDataRow++, nRow, nTotalCols, m_pqxxResult);
    }
    
    for(unsigned int nCol = 0; nCol < nTotalCols; nCol++)
    {
        adjustColumn(nCol);
    }
    
    m_pqxxResult.clear(); // free memory
}

// Display data from one resultset row
void KPGDataTable::displayOneRow(int nTableDataRow, unsigned int nRow, unsigned int nTotalCols, pqxx::result &pqxxResult)
{
    for(unsigned int nCol = 0; nCol < nTotalCols; nCol++)
    {
        QString strValue(m_connectionInThread.connection()->textCodec()->toUnicode(pqxxResult[nRow][nCol].c_str()));
        
        // Truncate long BLOBs
       if((m_listTableColumns[nCol].typName() == "text") ||
            (m_listTableColumns[nCol].typName() == "xml") ||
            (m_listTableColumns[nCol].typName() == "bytea"))
       {
           if(strValue.length() > 30)
           {
                strValue.truncate(30);
                strValue.append(" ...");
           }
       }
        
        setText(nTableDataRow, nCol, strValue);
    }
}

void KPGDataTable::stopReload()
{
    if(m_connectionInThread.running ())
    {
        m_connectionInThread.terminate();
        m_connectionInThread.wait();
        m_connectionInThread.leaveConnection();
                
        m_pqxxResult.clear(); // free memory
                
        m_state = view;
        
        try
        {
            m_connectionInThread.connectToServer(m_strConnectionOptions);
            setActions();
        }
        catch (const std::exception &e)
        {
            kdError() << "Failed to open connection " << e.what() << endl;
            KMessageBox::sorry(this, e.what());
            return;
        }
        
        ((QWidget *) parent())->setCursor(KCursor::arrowCursor());
    } 
}

// Create prepared statement for DML INSERT operation
void KPGDataTable::prepareInsertStatement(const std::string &stdstrPrepStmtName)
{
    QString strDml("INSERT INTO " + KPGUtil::fullyQualifiedName(m_strNamespaceName, m_strTableName) + " VALUES (");
    
    bool bFirst = true;
    int iParameterIndex = 1;
    for(int currentCol = 0; currentCol < numCols(); currentCol++)
    { 
        if(!bFirst) { strDml.append(","); }
        bFirst = false;
        
        if((text(m_editedRow, currentCol) == QString::null) || text(m_editedRow, currentCol).isEmpty())
        {
            if(m_listTableColumns[currentCol].hasDefault())
            {
                if(m_bAutoGeneratedPrimaryKey && (m_strPKColumnName == m_listTableColumns[currentCol].name()))
                {
                    strDml.append(QString(" $%1").arg(iParameterIndex++));
                }
                else
                {
                    strDml.append(" DEFAULT");
                }
            }
            else if(m_listTableColumns[currentCol].isNotNull() == false)
            {
                strDml.append(" NULL");
            }
            else
            {
                kdError() << k_funcinfo << "Field cannot be DEFAULT or NULL. " << m_listTableColumns[currentCol].name() << endl;
            }
        }
        else
        {
            strDml.append(QString(" $%1").arg(iParameterIndex++));
        }
    }
    
    strDml.append(")");
    kdDebug() << strDml << endl;
    
    pqxx::prepare::declaration pqxxPrepDecl = m_connectionInThread.connection()->prepare(stdstrPrepStmtName, strDml);
    
    // Add declaring parameters to prepared statements - INSERTed fields
    for(int currentCol = 0; currentCol < numCols(); currentCol++)
    { 
        if((text(m_editedRow, currentCol) == QString::null) || text(m_editedRow, currentCol).isEmpty())
        {
            if(m_listTableColumns[currentCol].hasDefault())
            {
                if(m_bAutoGeneratedPrimaryKey && (m_strPKColumnName == m_listTableColumns[currentCol].name()))
                {
                    m_connectionInThread.connection()->setPrepareDeclaration(pqxxPrepDecl, m_listTableColumns[currentCol].typName());
                }
            }
         }
         else
         {
                m_connectionInThread.connection()->setPrepareDeclaration(pqxxPrepDecl, m_listTableColumns[currentCol].typName());
         }
    }
}

// Create prepared statement for DML UPDATE operation
void KPGDataTable::prepareUpdateStatement(const std::string &stdstrPrepStmtName)
{
    QString strDml("UPDATE " + KPGUtil::fullyQualifiedName(m_strNamespaceName, m_strTableName) + " SET ");
                
    // Add dirty field names to strDml
    int currentCol = 0;
    bool bFirst = true;
    int iParamsIndex = 0;
    int nDirtyFields = 0; // only for debugging
    
    for(ListRowFields::iterator it = m_listRowFields.begin(); it != m_listRowFields.end(); ++it)
    { 
        if((*it).isDirty())
        {
            if(!bFirst) { strDml.append(","); }
            bFirst = false;
            
            if((text(m_editedRow, currentCol) == QString::null) || text(m_editedRow, currentCol).isEmpty())
            {
                if(m_listTableColumns[currentCol].isNotNull() == false)
                {
                    strDml.append(KPGUtil::quotedName(m_listTableColumns[currentCol].name()) + "=NULL");
                }
                else
                {
                    kdError()  << k_funcinfo << "Field cannot be DEFAULT or NULL. " << m_listTableColumns[currentCol].name() << endl;
                }
            }
            else
            {
                strDml.append(" " + KPGUtil::quotedName(m_listTableColumns[currentCol].name()) + QString("=$%1").arg(++iParamsIndex));
            }
            
            nDirtyFields++;
        }
        currentCol++;
    }
    
    if(nDirtyFields == 0)
        kdError() << "Count of dirty fields is 0 !!!" << endl;
    
    strDml.append(wherePredicateForPrimaryKey(iParamsIndex));
    kdDebug() << strDml << endl;
    
    pqxx::prepare::declaration pqxxPrepDecl = m_connectionInThread.connection()->prepare(stdstrPrepStmtName, strDml);
    
    // Add declaring parameters to prepared statements - UPDATEd fields
    currentCol = 0;
    for(ListRowFields::const_iterator it = m_listRowFields.begin(); it != m_listRowFields.end(); ++it)
    { 
        if((*it).isDirty())
        {
            if((text(m_editedRow, currentCol) != QString::null) && (text(m_editedRow, currentCol).isEmpty() == false))
            {   
               m_connectionInThread.connection()->setPrepareDeclaration(pqxxPrepDecl, m_listTableColumns[currentCol].typName());
            }
        }
        currentCol++;
    }
    
    // Add declaring parameters to prepared statements - primary key fields
    for(MapIndexKey::ConstIterator it = m_mapIndexKey.begin(); it != m_mapIndexKey.end(); it++) 
    {
        pqxxPrepDecl("varchar", pqxx::prepare::treat_string);
    }
}

// Create prepared statement for DML DELETE operation
void KPGDataTable::prepareDeleteStatement(const std::string &stdstrPrepStmtName)
{
    QString strDml("DELETE FROM " + KPGUtil::fullyQualifiedName(m_strNamespaceName, m_strTableName));
    strDml.append(wherePredicateForPrimaryKey(0));
    kdDebug() << strDml << endl;
    
    pqxx::prepare::declaration pqxxPrepDecl = m_connectionInThread.connection()->prepare(stdstrPrepStmtName, strDml);
    
    // Add declaring parameters to prepared statements - primary key fields
    for(MapIndexKey::ConstIterator it = m_mapIndexKey.begin(); it != m_mapIndexKey.end(); it++) 
    {
        pqxxPrepDecl("varchar", pqxx::prepare::treat_string);
    }
}

// Create prepared statement for SQL SELECT operation
void KPGDataTable::prepareSelectStatement(const std::string &stdstrPrepStmtName)
{
    QString strSql(sqlForSelectOneRow());
    kdDebug() << strSql << endl;
    
    pqxx::prepare::declaration pqxxPrepDecl = m_connectionInThread.connection()->prepare(stdstrPrepStmtName, strSql);
    
    // Add declaring parameters to prepared statements - primary key fields
    for(MapIndexKey::ConstIterator it = m_mapIndexKey.begin(); it != m_mapIndexKey.end(); it++) 
    {
        pqxxPrepDecl("varchar", pqxx::prepare::treat_string);
    }
}

// Move current row in data table to first
void KPGDataTable::firstRow()
{
    // If current row is changed, commit changed row to DB
    if((m_state == update) || (m_state == insert))
    {
        if(commitChanges() == false) return;
    }
    
    if(numRows() > 0)
        setCurrentCell(0, currentColumn());
        
    m_pActFirstRow->setEnabled(false);
    m_pActPreviousRow->setEnabled(false);
}

// Move current row in data table to previous
void KPGDataTable::previousRow()
{
    // If current row is changed, commit changed row to DB
    if((m_state == update) || (m_state == insert))
    {
        if(commitChanges() == false) return;
    }
    
    if(currentRow() < 0) return;
    
    if((numRows() > 0) && (currentRow() > 0))
        setCurrentCell(currentRow() - 1, currentColumn());

    m_pActPreviousRow->setEnabled(currentRow() > 0);
}

// Move current row in data table to next
void KPGDataTable::nextRow()
{
    // If current row is changed, commit changed row to DB
    if((m_state == update) || (m_state == insert))
    {
        if(commitChanges() == false) return;
    }
    
    if(currentRow() < 0) return;
    if((numRows() > 0) && (currentRow() < numRows() - 1))
        setCurrentCell(currentRow() + 1, currentColumn());

    m_pActNextRow->setEnabled(currentRow() < numRows() - 1);
}

// Move current row in data table to last
void KPGDataTable::lastRow()
{
    // If current row is changed, commit changed row to DB
    if((m_state == update) || (m_state == insert))
    {
        if(commitChanges() == false) return;
    }
    
    if(numRows() > 0)
        setCurrentCell(numRows(), currentColumn());
        
    m_pActNextRow->setEnabled(false);
    m_pActLastRow->setEnabled(false);
}

// Edit selected cell in data table
void KPGDataTable::editCell()
{
    int nRow = currentRow();
    int nCol = currentColumn();
    if(nRow < 0) return;
    if(nCol < 0) return;
    
    if((m_listTableColumns[nCol].typName() == "text") || 
        (m_listTableColumns[nCol].typName() == "xml") ||
        (m_listTableColumns[nCol].typName() == "bytea"))
    {
        slotDoubleClicked(nRow, nCol, 0, QPoint(0, 0));
    }
    else
    {
        QTable::editCell(nRow, nCol);
    }
}

// Clear selected cell in data table
void KPGDataTable::clearCell()
{
    int nRow = currentRow();
    int nCol = currentColumn();
    if(nRow < 0) return;
    if(nCol < 0) return;
    
    setText(nRow, nCol, "");
    slotValueChanged(nRow, nCol);
}
    
// Insert new row into data table
void KPGDataTable::insertRow()
{
    if(m_state != view)
    {
        kdError()  << k_funcinfo << "Wrong state, expect view" <<  endl;
        return;
    }
    
    int nRows = numRows();
    
    // Insert new row at end of table
    insertRows(nRows);
    
    setCurrentCell(nRows, 0);
    m_state = insert;
    m_editedRow = nRows;
    
    setActions();
}
    
// Delete selected row from data table
void KPGDataTable::deleteRow()
{
    if(m_state != view)
    {
        kdError() << k_funcinfo "Wrong state, expect view" <<  endl;
        return;
    }
    
    bool bConfirmDelRow = KPoGreView::configuration()->datatable()->confirmDelRow();
    if(bConfirmDelRow && (KMessageBox::questionYesNo(this, i18n("Really delete row ?")) != KMessageBox::Yes)) return ;
        
    int nRow = currentRow();
    int nCol = currentColumn();
    if(nRow < 0) return;
    
    bool bSuccess = runDml(dmlDelete);
    if(bSuccess)
    {
        removeRow(currentRow());
        
        if((numRows() > 0) && (nRow < numRows() - 1))
        {
            setCurrentCell(nRow, nCol);
        }
    }
}
    
// UPDATE or INSERT changes from data table current row to database
bool KPGDataTable::commitChanges()
{
    bool bSuccess = true;
    
    // If current row is changed, commit changed row to DB
    if(m_state == view)
    {
        kdError() << k_funcinfo "Wrong state: view" <<  endl;
    }
    else if(m_state == update)
    {
        //--- UPDATE m_editedRow to database table 
        
        // Look at dirty fields, check nullability
        int nCol = 0;
        for(ListRowFields::const_iterator cit = m_listRowFields.begin(); cit != m_listRowFields.end(); ++cit)
        { 
            if((*cit).isDirty()) 
            {
               if((text(m_editedRow, nCol) == QString::null) || text(m_editedRow, nCol).isEmpty())
                {
                    if(m_listTableColumns[nCol].isNotNull())
                    {
                        KMessageBox::sorry(this, i18n("Field %1 can't be NULL !").arg(m_listTableColumns[nCol].name()));
                        setCurrentCell(m_editedRow, nCol);
                        return false;
                    }
                }
            }
            
            nCol++;
        }
        
        bSuccess = runDml(dmlUpdate);
        if(bSuccess)
        {
            // Sucessfully updated - clear dirty flags
            for(ListRowFields::iterator it = m_listRowFields.begin(); it != m_listRowFields.end(); ++it) 
            { 
                (*it).clearDirty();
            }
        }
        else
        {
            return false;
        }
    }
    else
    {
        // INSERT m_editedRow into database table
        
        // Check fields for nullability
        for(int nCol = 0; nCol < numCols(); nCol++)
        { 
            if((text(m_editedRow, nCol) == QString::null) || text(m_editedRow, nCol).isEmpty())
            {
                if((m_listTableColumns[nCol].hasDefault() == false) &&                                            m_listTableColumns[nCol].isNotNull())
                {
                    KMessageBox::sorry(this, i18n("Field %1 can't be NULL !").arg(m_listTableColumns[nCol].name()));
                        setCurrentCell(m_editedRow, nCol);
                    setCurrentCell(m_editedRow, nCol);
                    return false;
                }
            }
        }
        
        bSuccess = runDml(dmlInsert);
        if(!bSuccess)
        {
             return false;
        }
        else
        {
            // Reload current row from database
            refreshRow();
        }
    }
    
    m_state = view;
    m_editedRow = -1;
    
    setActions();
    return bSuccess;
}

// Generate new primary key value from sequence
int KPGDataTable::generateSerialPrimaryKeyValue(pqxx::work &pqxxXaction)
{
    int iSerial;
    
    if(m_bAutoGeneratedPrimaryKey == false)
    {
        kdError() << k_funcinfo "PK is not autogenerated !" <<  endl;
        return 0;
    }
    
    QString strSql("SELECT ");
    strSql.append(m_strNextValueForPrimaryKey);
    kdDebug() << strSql << endl;
    
    try
    {
        pqxx::result pqxxResult = pqxxXaction.exec((std::string) strSql);
    
        if(pqxxResult.size() != 1)
        {
            kdError() << k_funcinfo "Expect one row in result !" <<  endl;
            return false;
        }
        
        pqxxResult[0][0].to(iSerial);
        
        pqxxXaction.commit();
    }
    catch (const std::exception &e)
    {
        kdError() << k_funcinfo << e.what() << endl;
        KPGSqlDialog dlg(this, strSql, m_connectionInThread.connection()->toUnicode(e.what()));
        dlg.exec();
        throw;
    }
    
    return iSerial;
}

// Generate new primary key value from sequence
long KPGDataTable::generateBigSerialPrimaryKeyValue(pqxx::work &pqxxXaction)
{
    long lSerial;

    if(m_bAutoGeneratedPrimaryKey == false)
    {
        kdError() << k_funcinfo "PK is not autogenerated !" <<  endl;
        return 0;
    }
    
    QString strSql("SELECT ");
    strSql.append(m_strNextValueForPrimaryKey);
    kdDebug() << strSql << endl;
    
    try
    {
        pqxx::result pqxxResult = pqxxXaction.exec((std::string) strSql);
    
        if(pqxxResult.size() != 1)
        {
            kdError() << k_funcinfo "Expect one row in result !" <<  endl;
            return false;
        }
        
        pqxxResult[0][0].to(lSerial);
        
        pqxxXaction.commit();
    }
    catch (const std::exception &e)
    {
        kdError() << k_funcinfo << e.what() << endl;
        KPGSqlDialog dlg(this, strSql, m_connectionInThread.connection()->toUnicode(e.what()));
        dlg.exec();
        throw;
    }
    
    return lSerial;
}

// Refresh the current row from database
void KPGDataTable::refreshRow()
{
    int nRow = currentRow();
    if(nRow < 0) return;
    
    std::string stdstrPrepStmtName("S_" + m_strNamespaceName + "_" + m_strTableName);
    
    bool bNeedUnprepare = false;
    try
    {   
        ((QWidget *) parent())->setCursor(KCursor::waitCursor());
        
        prepareSelectStatement(stdstrPrepStmtName);
        bNeedUnprepare = true;
                
        work pqxxXaction(*m_connectionInThread.connection(), stdstrPrepStmtName);
        prepare::invocation pqxxInvocation = pqxxXaction.prepared(stdstrPrepStmtName);
                
        // Append primary key values to vParams
        fillParametersForPrimaryKey(nRow, pqxxInvocation);
        
        // Execute query
        pqxx::result pqxxResult = pqxxInvocation.exec();
        
        // Fill output area with query result - one row
        unsigned int nTotalRows = pqxxResult.size();
        unsigned int nTotalCols = pqxxResult.columns();
        
        if(nTotalRows != 1)
        {
            kdError() << k_funcinfo "Expect one row !" <<  endl;
            throw PGSTD::runtime_error(i18n("Expect one row !"));
        }
        
        if(nTotalCols != (unsigned int) numCols())
        {
            kdError() << k_funcinfo "Columns numbers not match !" <<  endl;
            throw PGSTD::runtime_error("Columns numbers not match ! Table structure was changed ?");
        }
        
        displayOneRow(nRow, 0, nTotalCols, pqxxResult);
        
        bNeedUnprepare = false;
        m_connectionInThread.unprepareStatement();
         
        // Refresh also undo buffer
        m_listRowFields.clear();
        for(int nCol = 0; nCol < numCols(); nCol++)
        {
            m_listRowFields.append(KPGDataTableField(text(nRow, nCol)));
        } 
         
        ((QWidget *) parent())->setCursor(KCursor::arrowCursor());
    }
    catch (const std::exception &e)
    {
        if(bNeedUnprepare)
        {
            m_connectionInThread.unprepareStatement(stdstrPrepStmtName); 
        }
        
        ((QWidget *) parent())->setCursor(KCursor::arrowCursor());
        kdDebug() << k_funcinfo "Exception: " <<  e.what() << endl;
        KMessageBox::error(this, m_connectionInThread.connection()->toUnicode(e.what()));
        return;
    } 
}
 
// Cancel changes in data table
void KPGDataTable::cancelChanges()
{
    if(m_state == insert)
    {
        // Remove last row
        removeRow(numRows() - 1);
    }
    else if(m_state == update)
    {
        // Get back old values 
        ListRowFields::iterator it;
        int currentCol = 0;
        for(it = m_listRowFields.begin(); it != m_listRowFields.end(); ++it)
        { 
            if((*it).isDirty())
            {
                setText(m_editedRow, currentCol, (*it).oldValue());
                (*it).clearDirty();
            }
            currentCol++;
        }
    }
    else
    {
        kdError() << k_funcinfo "Wrong state: view" <<  endl;
    }
    
    // Clear dirty flags
    for(ListRowFields::iterator it = m_listRowFields.begin(); it != m_listRowFields.end(); ++it) 
    { 
        (*it).clearDirty();
    }
            
    m_editedRow = -1;
    m_state = view;
    
    setActions();
}

// Load LOB from file
void KPGDataTable::loadLob(const KURL& url)
{
    int nRow = currentRow();
    int nCol = currentColumn();
    if(nRow < 0) return;
    if(nCol < 0) return;
    
    if((m_listTableColumns[nCol].typName() != "text") &&
    (m_listTableColumns[nCol].typName() != "xml") &&
    (m_listTableColumns[nCol].typName() != "bytea"))
    {
        kdError() << k_funcinfo << "Field is not text, xml or bytea " << endl;
        return;
    }
    
    QString strPrepStmtName("LB_" + m_strNamespaceName + "_" + m_strTableName);
    std::string stdstrPrepStmtName(strPrepStmtName.latin1());
    
    try
    {
        ((QWidget *) parent())->setCursor(KCursor::waitCursor());
        
        work pqxxXaction(*m_connectionInThread.connection(), stdstrPrepStmtName);
        
        pqxx::largeobject pqxxLargeObject(pqxxXaction);
        pqxxLargeObject = largeobject(pqxxXaction, url.path());
        
        // NOTE: Loaded file is expected in Connection encoding !
        if(setLargeObject(pqxxLargeObject, nRow, nCol, stdstrPrepStmtName, pqxxXaction) == false)
        {
          pqxxXaction.abort();
          return;
        }
        
        refreshRow();
    }
    catch (const std::exception &e)
    {
        ((QWidget *) parent())->setCursor(KCursor::arrowCursor());
        kdDebug() << k_funcinfo "Exception: " <<  e.what() << endl;
        KMessageBox::error(this, m_connectionInThread.connection()->toUnicode(e.what()));
        return;
    } 
}

// Save LOB to file
void KPGDataTable::saveLob(const KURL& url)
{
    int nRow = currentRow();
    int nCol = currentColumn();
    if(nRow < 0) return;
    if(nCol < 0) return;
    
    if((m_listTableColumns[nCol].typName() != "text") &&
    (m_listTableColumns[nCol].typName() != "xml") &&
    (m_listTableColumns[nCol].typName() != "bytea"))
    {
        kdError() << k_funcinfo << "Field is not text, xml or bytea " << endl;
        return;
    }
    
    QString strPrepStmtName("SB_" + KPGUtil::unspacedName(m_strNamespaceName) + "_" + KPGUtil::unspacedName(m_strTableName));
    std::string stdstrPrepStmtName(strPrepStmtName.latin1());
    
    try
    {
        ((QWidget *) parent())->setCursor(KCursor::waitCursor());
        
        work pqxxXaction(*m_connectionInThread.connection(), stdstrPrepStmtName);
        pqxx::largeobject pqxxLargeObject = getLargeObject(nRow, nCol, stdstrPrepStmtName, pqxxXaction);
        
        // TODO: Warn when LOB IS NULL
        
        // NOTE: File is saved in Connection encoding !
        pqxxLargeObject.to_file(pqxxXaction, url.path());
        
        ((QWidget *) parent())->setCursor(KCursor::arrowCursor());
    }
    catch (const std::exception &e)
    {
        ((QWidget *) parent())->setCursor(KCursor::arrowCursor());
        kdDebug() << k_funcinfo "Exception: " <<  e.what() << endl;
        KMessageBox::error(this, m_connectionInThread.connection()->toUnicode(e.what()));
        return;
    } 
}

// Edit text LOB in build in editor
void KPGDataTable::editLobInEditor()
{
    int nRow = currentRow();
    int nCol = currentColumn();
    if(nRow < 0) return;
    if(nCol < 0) return;
    
    if((m_listTableColumns[nCol].typName() != "text") &&
    (m_listTableColumns[nCol].typName() != "xml") &&
    (m_listTableColumns[nCol].typName() != "bytea"))
    {
        kdError() << k_funcinfo << "Field is not text, xml or bytea " << endl;
        return;
    }
    
    try
    {
        QString strPrepStmtNameS("EL1_" + KPGUtil::unspacedName(m_strNamespaceName) + "_" + KPGUtil::unspacedName(m_strTableName));
        std::string stdstrPrepStmtNameS(strPrepStmtNameS.latin1());
        
        work pqxxXaction(*m_connectionInThread.connection(), stdstrPrepStmtNameS);
        
        pqxx::largeobject pqxxLargeObjectIn = getLargeObject(nRow, nCol, stdstrPrepStmtNameS, pqxxXaction);
        largeobjectaccess pqxxLargeObjectAccessIn(pqxxXaction, pqxxLargeObjectIn, PGSTD::ios::in);
        
        
        // Get LOB buffer size in MB from settings
        int iLobBufferSize = KPoGreView::configuration()->datatable()->lobBufferSize();
        
        char buffer[iLobBufferSize * 1048576]; 
        const largeobjectaccess::size_type bufSize = sizeof(buffer) - 1;
        largeobjectaccess::size_type nBytes = pqxxLargeObjectAccessIn.read(buffer, bufSize);
        buffer[nBytes] = '\0';
        
        if(nBytes >= bufSize)
        {
            ((QWidget *) parent())->setCursor(KCursor::arrowCursor());  
            KMessageBox::sorry(this, i18n("Insufficient buffer space ! Inscrease buffer size in settings."));
            return;
        }
        
        QString strText(m_connectionInThread.connection()->textCodec()->toUnicode(buffer));
        
        if(m_listTableColumns[nCol].typName() == "text")
        {
            /*KPGTextEditDialog dlg(this, strText);
            if( dlg.exec() != QDialog::Accepted )
            {
                return;
            }
            strText = dlg.text();*/
            
            m_pKateXmlEditorDialog->setEditorText(strText, false);
            if(m_pKateXmlEditorDialog->exec() != QDialog::Accepted)
            {
                return;
            }
            strText = m_pKateXmlEditorDialog->editorText();
        }
        else if (m_listTableColumns[nCol].typName() == "xml")
        {
            //KXETextEditorDialog dlg(this);
            //dlg.setEditorText(strText);
            
            //KPGKateXmlEditorDialog dlg(this, (static_cast <KPoGreView *> (parentWidget()->parentWidget()))->createKatePartFactory());
            
            m_pKateXmlEditorDialog->setEditorText(strText, true);
            if(m_pKateXmlEditorDialog->exec() != QDialog::Accepted)
            {
                return;
            }
            strText = m_pKateXmlEditorDialog->editorText();
        }
       
        pqxx::largeobject pqxxLargeObjectOut(pqxxXaction);
        largeobjectaccess pqxxLargeObjectAccessOut(pqxxXaction, pqxxLargeObjectOut, PGSTD::ios::out);
            
        pqxxLargeObjectAccessOut.write((std::string) m_connectionInThread.connection()->textCodec()->fromUnicode(strText));
        
        QString strPrepStmtNameU("EL2_" + KPGUtil::unspacedName(m_strNamespaceName) + "_" + KPGUtil::unspacedName(m_strTableName));
        std::string stdstrPrepStmtNameU(strPrepStmtNameU.latin1());
        
        if(setLargeObject(pqxxLargeObjectOut, nRow, nCol, stdstrPrepStmtNameU, pqxxXaction) == false)
        {
            pqxxXaction.abort();
            return;
        }
        
        refreshRow();
    }
    catch (const std::exception &e)
    {
        ((QWidget *) parent())->setCursor(KCursor::arrowCursor());
        kdDebug() << k_funcinfo "Exception: " <<  e.what() << endl;
        KMessageBox::error(this, m_connectionInThread.connection()->toUnicode(e.what()));
        return;
    } 
}

// Called, when the current cell in m_pTableData has changed to row, col
void KPGDataTable::slotCurrentChanged (int nRow, int)
{
    if(m_connectionInThread.connection() == 0)
    {
        disableAllActions();
        return;
    }
        
    // Copy row to m_listRowFields to have field values before user edit it
    if((m_state == insert) || (m_state == update))
    {
        if(m_editedRow != nRow)
        {
            // NOTE: this situation occured, when user starts edit cell by typing, and leave it and keyboard or mouse filter not detect it 
            kdError()  << k_funcinfo << "nRow != currentRow()" << endl;
            commitChanges(); 
        }
    }
    
    if(m_state == view)
    {
        m_listRowFields.clear();
        for(int nCol = 0; nCol < numCols(); nCol++)
        {
            m_listRowFields.append(KPGDataTableField(text(nRow, nCol)));
        }
    }
    
    setActions();
}

// Called, when user changed the value in the m_pTableData
void KPGDataTable::slotValueChanged(int row, int col)
{
    if(m_state == refresh)
    {
        return;
    }
    else if(m_state == insert)
    {
        if(m_editedRow != row)
        {
            kdError() << k_funcinfo "Rows not match !" <<  endl;
        }
    }
    else if(m_state == update)
    {
        if(m_editedRow != row)
        {
            kdError() << k_funcinfo "Rows not match !" <<  endl;
        }
        
        m_listRowFields[col].setDirty();
    }
    else // view
    {
        m_state = update;
        m_editedRow = row;
        m_listRowFields[col].setDirty();
    }
    
    setActions();
}

// Called when mouse button button is double-clicked over m_pTableData
void KPGDataTable::slotDoubleClicked(int row, int col, int, const QPoint &)
{
    if(m_state == refresh)
    {
        return;
    }
    
    // Make correction to display context menu on proper place
    QRect r = cellGeometry(row, col);
        
    if(m_listTableColumns[col].typName() == "text")
    {
        if(m_state != view)
        {
            KMessageBox::sorry(this, i18n("Can't edit text field during editing or inserting row !"));
            return;
        }
                            
        popupContextMenu("popupDataTableText", viewport()->mapToGlobal(contentsToViewport(r.center())));
    }
    else if(m_listTableColumns[col].typName() == "xml")
    {
        if(m_state != view)
        {
            KMessageBox::sorry(this, i18n("Can't edit xml field during editing or inserting row !"));
            return;
        }
                            
        popupContextMenu("popupDataTableText", viewport()->mapToGlobal(contentsToViewport(r.center())));
    }      
    else if(m_listTableColumns[col].typName() == "bytea")
    {
       if(m_state != view)
        {
            KMessageBox::sorry(this, i18n("Can't edit binary field during editing or inserting row !"));
            return;
        }
        
        popupContextMenu("popupDataTableBytea", viewport()->mapToGlobal(contentsToViewport(r.center())));
    }
}

// Called, when user make right click over data table. Display popup menu
void KPGDataTable::slotContextMenuRequested(int, int, const QPoint & pos)
{
    popupContextMenu("popupDataTable", pos);
}

// Called when user click on horizontal header 
void KPGDataTable::slotHorzHeaderClicked(int iSection)
{
    kdDebug() << k_funcinfo << iSection << endl;
    
    if(iSection >= 0)
    {
        KPGDataTableColumn &column = m_listTableColumns[iSection];
        
        QString strInfo(column.name() + " " + column.typName());
        if(column.isNotNull())
        {
            strInfo.append(" NOT NULL\n");
        }
        else
        {
            strInfo.append(" NULL\n");
        }
        
        if(column.createdAs().length() > 0)
        {
            strInfo.append(i18n("Created as: ") + column.createdAs() + "\n");
        }
        
        if(column.defaultValue().length() > 0)
        {
            strInfo.append(i18n("Default value: ") + column.defaultValue() + "\n");
        }
        
        if(column.description().length() > 0)
        {
            strInfo.append(i18n("Description: ") + column.description() + "\n");
        }
        
        KMessageBox::information(this, strInfo);
    }
}


// Build WHERE predicate for primary key columns. 
const QString KPGDataTable::wherePredicateForPrimaryKey(int iParamsIndex) const
{
    QString strWhere(" WHERE ");
    
    // field names
    QHeader* pHeader = horizontalHeader();
    
    for(MapIndexKey::ConstIterator it = m_mapIndexKey.begin(); it != m_mapIndexKey.end(); ) 
    {
        // Obtain column index in m_pTableData for this column name         
        int nCol = -1;
        for(int i = 0; i < pHeader->count(); i++)
        {
            if(pHeader->label(i) == it.data()) // If column names match, we found nCol
            {
                nCol = i; 
                break;
            }
        }
        
        if(nCol < 0)
        {
            kdDebug() << k_funcinfo "Can't obtain column index for field:" <<  it.data() << endl;
            return QString::null;
        }
        
        strWhere.append(KPGUtil::quotedName(it.data())); // column name
        strWhere.append(QString("=$%1").arg(++iParamsIndex));
                
                
        if(++it != m_mapIndexKey.end())
        {
            strWhere.append(" AND ");
        }
    }
    
    return strWhere;
}

//  Fill key values into vParams, from iParamsIndex position
void KPGDataTable::fillParametersForPrimaryKey(int nRow, prepare::invocation &pqxxInvocation) const
{
    // field names
    QHeader* pHeader = horizontalHeader();
    
    for(MapIndexKey::ConstIterator it = m_mapIndexKey.begin(); it != m_mapIndexKey.end(); it++) 
    {
        // Obtain column index in m_pTableData for this column name         
        int nCol = -1;
        for(int i = 0; i < pHeader->count(); i++)
        {
            if(pHeader->label(i) == it.data()) // If column names match, we found nCol
            {
                nCol = i; 
                break;
            }
        }
        
        if(nCol < 0)
        {
            kdDebug() << k_funcinfo "Can't obtain column index for field:" <<  it.data() << endl;
            return;
        }
                
        pqxxInvocation((std::string) m_connectionInThread.connection()->textCodec()->fromUnicode(text(nRow, nCol)));
    }
}

// Run DML on DB server. Return true, if command succesfully performed
bool KPGDataTable::runDml(EDmlType dmlType)
{
    if(m_connectionInThread.connection() == 0)
    {
        KMessageBox::error(this, i18n("Not connected"));
        return false;
    }
    
    bool bNeedUnprepare = false;
    std::string stdstrPrepStmtName; 
            
    try
    { 
        ((QWidget *) parent())->setCursor(KCursor::waitCursor());
        
        // Create prepared statement, if not yet
        switch(dmlType)
        {
            case dmlInsert: 
            {
                stdstrPrepStmtName = "I_" +  KPGUtil::unspacedName(m_strNamespaceName) + "_" + KPGUtil::unspacedName(m_strTableName);
                // INSERT statement is cached, it is the same every time
                prepareInsertStatement(stdstrPrepStmtName); 
                bNeedUnprepare = true; 
                break;
            }
            case dmlUpdate: 
            {
                stdstrPrepStmtName = "U_" + KPGUtil::unspacedName(m_strNamespaceName) + "_" + KPGUtil::unspacedName(m_strTableName);
                // UPDATE statement is prepared each time, because updated columns varies 
                prepareUpdateStatement(stdstrPrepStmtName); 
                bNeedUnprepare = true; 
                break; 
            }
            case dmlDelete: 
            {
                stdstrPrepStmtName = "D_" + KPGUtil::unspacedName(m_strNamespaceName) + "_" + KPGUtil::unspacedName(m_strTableName);
                // DELETE statement is cached, it is the same every time
                prepareDeleteStatement(stdstrPrepStmtName); 
                bNeedUnprepare = true; 
                break;
            }
        }
        
        work pqxxXaction(*m_connectionInThread.connection(), stdstrPrepStmtName);
        prepare::invocation pqxxInvocation = pqxxXaction.prepared(stdstrPrepStmtName);
        
        if(dmlType == dmlUpdate)
        {
            // Set dirty field values to vParams
            int currentCol = 0;
            for(ListRowFields::iterator it = m_listRowFields.begin(); it != m_listRowFields.end(); ++it)
            { 
                if((*it).isDirty())
                {
                    if((text(m_editedRow, currentCol) == QString::null) 
                    || text(m_editedRow, currentCol).isEmpty()) continue;
                    
                    bool bSuccess = m_connectionInThread.connection()->setInvocationValue(pqxxInvocation, m_listTableColumns[currentCol].typName(), text(m_editedRow, currentCol));
                    if(!bSuccess)
                    {
                        setCurrentCell(m_editedRow, currentCol);
                        m_connectionInThread.unprepareStatement(stdstrPrepStmtName); 
                        throw PGSTD::runtime_error(i18n("Conversion for column %1 failed !").arg(m_listTableColumns[currentCol].name()));
                    }
                }
                currentCol++;
            }
        }
        else if(dmlType == dmlInsert)
        {
            for(int currentCol = 0; currentCol < numCols(); currentCol++)
            {
                if((text(m_editedRow, currentCol) == QString::null) || text(m_editedRow, currentCol).isEmpty())
                {
                    if(m_bAutoGeneratedPrimaryKey && (m_strPKColumnName == m_listTableColumns[currentCol].name()))
                    {
                        // Get primary key value from sequencer
                        if(m_listTableColumns[currentCol].typName() == "int4")
                        {
                            int iNextValue = generateSerialPrimaryKeyValue(pqxxXaction);
                                                    
                            // Set PK value to datatable
                            QString strNextValue;
                            strNextValue.setNum(iNextValue);
                            setText(m_editedRow, currentCol, strNextValue);
                            
                            pqxxInvocation(iNextValue); 
                        }
                        else if(m_listTableColumns[currentCol].typName() == "int8")
                        {
                            long lNextValue = generateBigSerialPrimaryKeyValue(pqxxXaction);
                            
                            // Set PK value to datatable
                            QString strNextValue;
                            strNextValue.setNum(lNextValue);
                            setText(m_editedRow, currentCol, strNextValue);
                            
                            pqxxInvocation(lNextValue); 
                        }
                        else
                        {
                            kdDebug() << k_funcinfo "Uncovered datatype: " <<  m_listTableColumns[currentCol].typName() << ", " << m_listTableColumns[currentCol].name() << endl;   
                            m_connectionInThread.unprepareStatement(stdstrPrepStmtName); 
                            return false;
                        }
                    }
                    else continue;
                }
                else
                {
                    bool bSuccess = m_connectionInThread.connection()->setInvocationValue(pqxxInvocation, m_listTableColumns[currentCol].typName(), text(m_editedRow, currentCol));
                    if(!bSuccess)
                    {
                        setCurrentCell(m_editedRow, currentCol);
                        m_connectionInThread.unprepareStatement(stdstrPrepStmtName); 
                        throw PGSTD::runtime_error(i18n("Conversion failed !"));
                    }
                }
            }
        }
        
        if(dmlType == dmlUpdate)
        {
            // Append primary key values to pqxxInvocation
            fillParametersForPrimaryKey(m_editedRow, pqxxInvocation);
        }
        else if(dmlType == dmlDelete)
        {
            // Append primary key values to pqxxInvocation
            fillParametersForPrimaryKey(currentRow(), pqxxInvocation);
        }
        
        pqxxInvocation.exec();
        pqxxXaction.commit();
        
        bNeedUnprepare = false;      
        m_connectionInThread.unprepareStatement(stdstrPrepStmtName); 
        
        
        ((QWidget *) parent())->setCursor(KCursor::arrowCursor());
        return true;
    }
    catch (const std::exception &e)
    {   
        ((QWidget *) parent())->setCursor(KCursor::arrowCursor());
        kdDebug() << k_funcinfo "Exception: " <<  e.what() << endl;
        KMessageBox::error(this, m_connectionInThread.connection()->toUnicode(e.what()));
    }
     
    if(bNeedUnprepare)
    {
        m_connectionInThread.unprepareStatement();
    }
        
    return false;
}

// Return SQL query for selecting edited row
const QString KPGDataTable::sqlForSelectOneRow() const
{
    QString strSql("SELECT ");
    
    bool bFirst = true;
    for(ListTableColumns::const_iterator it = m_listTableColumns.begin(); it != m_listTableColumns.end(); ++it)
    { 
        if(!bFirst) { strSql.append(","); }
        bFirst = false;
        
        strSql.append(KPGUtil::quotedName((*it).name()));
    }
    
    strSql.append(" FROM " + KPGUtil::fullyQualifiedName(m_strNamespaceName, m_strTableName));
    strSql.append(wherePredicateForPrimaryKey(0));
        
    return strSql;
}

// Return SQL query for selecting edited row
const QString KPGDataTable::sqlForSelectAllRows() const
{
    QString strSql("SELECT ");
    
    bool bFirst = true;
    for(ListTableColumns::const_iterator it = m_listTableColumns.begin(); it != m_listTableColumns.end(); ++it)
    { 
        if(!bFirst) { strSql.append(", "); }
        bFirst = false;
        
        strSql.append(KPGUtil::quotedName((*it).name()));
    }
    
    strSql.append(" FROM " + KPGUtil::fullyQualifiedName(m_strNamespaceName, m_strTableName));
    
    if(m_strWherePredicate.length() > 0)
    {
        strSql.append(" WHERE " + m_strWherePredicate);
    }
    
    if(m_strOrderByPredicate.length() > 0)
    {
        strSql.append(" ORDER BY " + m_strOrderByPredicate);
    }
        
    return strSql;
}

// Retreive largeobject of LOB for given row and column
pqxx::largeobject KPGDataTable::getLargeObject(int nRow, int nCol, std::string &stdstrPrepStmtName, work &pqxxXaction)
{
    QString strSql("SELECT ");
    strSql.append(KPGUtil::quotedName(m_listTableColumns[nCol].name()));
    strSql.append(" FROM ");
    strSql.append(KPGUtil::fullyQualifiedName(m_strNamespaceName, m_strTableName));
    strSql.append(wherePredicateForPrimaryKey(0));
    kdDebug() << strSql << endl;
        
    bool bNeedUnprepare = false;    
        
    try
    {
        // Select LOB
        pqxx::prepare::declaration pqxxPrepDecl = m_connectionInThread.connection()->prepare(stdstrPrepStmtName, strSql);
        bNeedUnprepare = true;
        
        // Add declaring parameters to prepared statements - primary key fields
        for(MapIndexKey::ConstIterator it = m_mapIndexKey.begin(); it != m_mapIndexKey.end(); it++) 
        {
            pqxxPrepDecl("varchar", pqxx::prepare::treat_string);
        }
                            
        prepare::invocation pqxxInvocation = pqxxXaction.prepared(stdstrPrepStmtName);
                    
        // Append primary key values to vParams
        fillParametersForPrimaryKey(nRow, pqxxInvocation);
            
        // Execute query
        pqxx::result pqxxResult = pqxxInvocation.exec();
                    
        // And unprepare it
        bNeedUnprepare = false;
        m_connectionInThread.unprepareStatement(stdstrPrepStmtName); 
        
        // Check query result - one row, one column
        unsigned int nTotalRows = pqxxResult.size();
        unsigned int nTotalCols = pqxxResult.columns();
            
        if(nTotalRows != 1)
        {
            kdError() << k_funcinfo "Expect one row !" <<  endl;
            throw PGSTD::runtime_error(i18n("Expect one row !"));
        }
            
        if(nTotalCols != 1)
        {
            kdError() << k_funcinfo "Expect one column !" <<  endl;
            throw PGSTD::runtime_error(i18n("Expect one column !"));
        }
            
        pqxx::largeobject pqxxLargeObject(pqxxXaction);
        largeobjectaccess pqxxLargeObjectAccess(pqxxXaction, pqxxLargeObject, PGSTD::ios::out);
        
        if(m_listTableColumns[nCol].typName() == "bytea")
        {
            pqxx::binarystring pqxxBinaryString(pqxxResult[0][0]);
            pqxxLargeObjectAccess.write((const char *) pqxxBinaryString.data(), pqxxBinaryString.size());
            
            kdDebug() << "Readed bytea size: " + to_string(pqxxBinaryString.size()) << endl;
        }
        else if((m_listTableColumns[nCol].typName() == "text") ||
                (m_listTableColumns[nCol].typName() == "xml"))
        {
            std::string charstring = pqxxResult[0][0].c_str();
            long nBytes = pqxxLargeObjectAccess.cwrite(charstring.c_str(), charstring.size());
            
            kdDebug() << "Readed text size: " + to_string(nBytes) << endl;
        }
        
        return pqxxLargeObject;
    }
    catch (const std::exception &e)
    {
        if(bNeedUnprepare)
        {
            m_connectionInThread.unprepareStatement(stdstrPrepStmtName); 
        }
        
        kdError() << k_funcinfo << e.what() << endl;
        throw;
    }
}


// Store largeobject of LOB to database, for given row and column
/*bool KPGDataTable::setLargeObject(pqxx::largeobject &pqxxLargeObject, int nRow, int nCol, std::string &stdstrPrepStmtName, work &pqxxXaction)
{
    QString strSql("UPDATE ");
    strSql.append(KPGUtil::fullyQualifiedName(m_strNamespaceName, m_strTableName));
    strSql.append(" SET ");
    strSql.append("\"" + m_listTableColumns[nCol].name() + "\"");
        
    bool bNeedUnprepare = false;    
        
    try
    {
        largeobjectaccess pqxxLargeObjectAccess(pqxxXaction, pqxxLargeObject, PGSTD::ios::in);
        
        // Get LOB buffer size in MB from settings
        int iLobBufferSize = KPoGreView::configuration()->datatable()->lobBufferSize();
        
        char buffer[iLobBufferSize * 1048576]; 
        const size_t bufSize = sizeof(buffer) - 1;
        largeobjectaccess::size_type nBytes = pqxxLargeObjectAccess.read(buffer, bufSize);
        buffer[nBytes] = '\0';
        
        if(nBytes >= bufSize)
        {
            ((QWidget *) parent())->setCursor(KCursor::arrowCursor());
            KMessageBox::sorry(this, i18n("Insufficient buffer space ! Inscrease buffer size in settings."));
            return false;
        }
        
        if(m_listTableColumns[nCol].typName() == "bytea")
        {
            const std::string stdstrEscaped = pqxxXaction.esc_raw(reinterpret_cast <const unsigned char *> (buffer), nBytes);
            kdDebug() << "Writed bytea size: " + to_string(nBytes) << endl;
            strSql.append("='" + stdstrEscaped + "'");
        }
        else if(m_listTableColumns[nCol].typName() == "text")
        {
            const std::string stdstrText = buffer;
            strSql.append("='" + stdstrText + "'");
        }
        
        strSql.append(wherePredicateForPrimaryKey(0));
        kdDebug() << strSql << endl;
        
        // ---
        pqxx::prepare::declaration pqxxPrepDecl = m_connectionInThread.connection()->prepare(stdstrPrepStmtName, strSql);
        
        bNeedUnprepare = true;
        
        // Add declaring parameters to prepared statements - primary key fields
        for(MapIndexKey::ConstIterator it = m_mapIndexKey.begin(); it != m_mapIndexKey.end(); it++) 
        {
            pqxxPrepDecl("varchar", pqxx::prepare::treat_string);
        }
                            
        prepare::invocation pqxxInvocation = pqxxXaction.prepared(stdstrPrepStmtName);
                
        // Append primary key values to vParams
        fillParametersForPrimaryKey(nRow, pqxxInvocation);
            
        // Execute query
        pqxxInvocation.exec();
        pqxxXaction.commit();
                    
        // And unprepare it
        bNeedUnprepare = false;
        m_connectionInThread.unprepareStatement(stdstrPrepStmtName); 
    
        return true;
    }
    catch (const std::exception &e)
    {
        if(bNeedUnprepare)
        {
            m_connectionInThread.unprepareStatement(stdstrPrepStmtName); 
        }
        
        kdError() << k_funcinfo << e.what() << endl;
        throw;
    }
}

*/

// Store largeobject of LOB to database, for given row and column
bool KPGDataTable::setLargeObject(pqxx::largeobject &pqxxLargeObject, int nRow, int nCol, std::string &stdstrPrepStmtName, work &pqxxXaction)
{
    QString strSql("UPDATE ");
    strSql.append(KPGUtil::fullyQualifiedName(m_strNamespaceName, m_strTableName));
    strSql.append(" SET ");
    strSql.append(KPGUtil::quotedName(m_listTableColumns[nCol].name()));
    strSql.append("=$1");
    strSql.append(wherePredicateForPrimaryKey(1));
    kdDebug() << strSql << endl;
        
    bool bNeedUnprepare = false;    
        
    try
    {
        // Update LOB
        pqxx::prepare::declaration pqxxPrepDecl = m_connectionInThread.connection()->prepare(stdstrPrepStmtName, strSql);
        
        m_connectionInThread.connection()->setPrepareDeclaration(pqxxPrepDecl, m_listTableColumns[nCol].typName());
        bNeedUnprepare = true;
        
        // Add declaring parameters to prepared statements - primary key fields
        for(MapIndexKey::ConstIterator it = m_mapIndexKey.begin(); it != m_mapIndexKey.end(); it++) 
        {
            pqxxPrepDecl("varchar", pqxx::prepare::treat_string);
        }
                            
        prepare::invocation pqxxInvocation = pqxxXaction.prepared(stdstrPrepStmtName);
    
        largeobjectaccess pqxxLargeObjectAccess(pqxxXaction, pqxxLargeObject, PGSTD::ios::in);
        
        // Get LOB buffer size in MB from settings
        int iLobBufferSize = KPoGreView::configuration()->datatable()->lobBufferSize();
        
        char buffer[iLobBufferSize * 1048576]; 
        const largeobjectaccess::size_type bufSize = sizeof(buffer) - 1;
        
        // Read LOB from pqxxLargeObjectAccess into buffer 
        largeobjectaccess::size_type nBytes = pqxxLargeObjectAccess.read(buffer, bufSize);
        buffer[nBytes] = '\0';
        
        if(nBytes >= bufSize)
        {
            ((QWidget *) parent())->setCursor(KCursor::arrowCursor());
            KMessageBox::sorry(this, i18n("Insufficient buffer space ! Inscrease buffer size in settings."));
            return false;
        }
        
        if(m_listTableColumns[nCol].typName() == "bytea")
        {
            const std::string stdstrText(buffer, nBytes);
            pqxxInvocation(stdstrText, nBytes);
            
            kdDebug() << "Writed bytea size: " + to_string(nBytes) << endl;
        }
        else if((m_listTableColumns[nCol].typName() == "text") ||
                (m_listTableColumns[nCol].typName() == "xml"))
        {
            const std::string stdstrText = buffer;
            pqxxInvocation(stdstrText);
        }
                
        // Append primary key values to vParams
        fillParametersForPrimaryKey(nRow, pqxxInvocation);
            
        // Execute query
        pqxxInvocation.exec();
        pqxxXaction.commit();
                    
        // And unprepare it
        bNeedUnprepare = false;
        m_connectionInThread.unprepareStatement(stdstrPrepStmtName); 
    
        return true;
    }
    catch (const std::exception &e)
    {
        /*if(bNeedUnprepare)
        {
            m_connectionInThread.unprepareStatement(stdstrPrepStmtName); 
        } L.V. cause exception: current TX is aborted */
        
        kdError() << k_funcinfo << e.what() << endl;
        throw;
    }
}

// Find first occurence of text
void KPGDataTable::findFirst(QStringList &listOfSearchHistory)
{
    // If current row is changed, commit changed row to DB
    if((m_state == update) || (m_state == insert))
    {
        if(commitChanges() == false) return;
    }
    
    KFindDialog dlg(this, "", 0, listOfSearchHistory, false);
    dlg.setHasCursor(false);
        
    int c = dlg.exec();
    
    if(c != QDialog::Accepted)
        return;

    listOfSearchHistory = dlg.findHistory();
        
    
    if(m_pFind != 0) delete m_pFind;
    m_pFind = new KFind(dlg.pattern(), dlg.options(), this);

    // Connect highlight signal to code which handles highlighting
    // of found text.
    connect( m_pFind, SIGNAL( highlight( const QString &, int, int ) ),
             this, SLOT( slotHighlight( const QString &, int, int ) ) );
             
    // Connect findNext signal - called when pressing the button in the dialog
    connect( m_pFind, SIGNAL( findNext() ), this, SLOT( slotFindNext() ) );

    m_iRowToSearch = (m_pFind->options() & KFindDialog::FindBackwards) ? numRows() - 1 : 0;
    m_iColToSearch = (m_pFind->options() & KFindDialog::FindBackwards) ? numCols() - 1 : 0;
    m_pFind->setData(text(m_iRowToSearch, m_iColToSearch));

    slotFindNext();
}

// Find first occurence of text
void KPGDataTable::findNext()
{
    m_iRowToSearch = currentRow();
    m_iColToSearch = currentColumn();
    if(m_iRowToSearch < 0) return;
    if(m_iColToSearch < 0) return;
    
    slotFindNext();
}

// Find next occurence of text
void KPGDataTable::slotFindNext()
{
    // If current row is changed, commit changed row to DB
    if((m_state == update) || (m_state == insert))
    {
        if(commitChanges() == false) return;
    }
    
    if (!m_pFind) // shouldn't be called before find is activated
        return;


    KFind::Result res = KFind::NoMatch;
    while ( res == KFind::NoMatch &&
            ((m_pFind->options() & KFindDialog::FindBackwards) ? 
                    (m_iRowToSearch >= 0) : (m_iRowToSearch < numRows())
            ) 
          ) 
        {
            //kdDebug() << "searching  1:" << m_iParaToSearch << " 2: " << m_iEndPara << " 3: " << m_iStartPara << endl;
            
            Q_ASSERT(m_iRowToSearch >= 0);
            Q_ASSERT(m_iRowToSearch < numRows());
            Q_ASSERT(m_iColToSearch >= 0);
            Q_ASSERT(m_iColToSearch < numCols());
                
            if(m_pFind->needData()) 
            {
                m_pFind->setData(text(m_iRowToSearch, m_iColToSearch));
            }
        
            // Let KFind inspect the text fragment, and display a dialog if a match is found
            res = m_pFind->find();
        
            if( res == KFind::NoMatch ) 
            {
                if((m_pFind->options() & KFindDialog::FindBackwards))
                {
                    if(m_iColToSearch == 0)
                    {
                        m_iColToSearch = numCols() - 1;
                        m_iRowToSearch--;
                    }
                    else
                    {
                        m_iColToSearch--;
                    }
                }
                else
                {
                    if(m_iColToSearch == numCols() - 1)
                    {
                        m_iColToSearch = 0;
                        m_iRowToSearch++;
                    }
                    else
                    {
                        m_iColToSearch++;
                    }
                }
            }
            //kdDebug() << "2searching  1:" << m_iParaToSearch << " 2: " << m_iEndPara << " 3: " << m_iStartPara << endl;
        }

    if( res == KFind::NoMatch ) 
    { // i.e. at end
        m_pFind->displayFinalDialog();
        m_pFind->resetCounts();
        removeSelection(0);
    }
}

// Highligth found text
void KPGDataTable::slotHighlight( const QString &, int, int)
{
    //kdDebug() << "highlight: " << index << " " << length << endl;
    setCurrentCell(m_iRowToSearch, m_iColToSearch);
}

void KPGDataTable::virtual_hook(int id, void* data)
{ 
    KXMLGUIClient::virtual_hook(id, data); 
}
    
/* TESTING 
bool KPGDataTable::setLargeObject(pqxx::largeobject &pqxxLargeObject, int nRow, int nCol, std::string &stdstrPrepStmtName, work &T)
 {
   //T.exec("DROP TABLE pqxxbin");
 
    const int bufSize = 6;
    char buffer1[bufSize]; // some raw data - begin of PNG file
    buffer1[0] = 0x89; //  \211
    buffer1[1] = 0x50; //  P
    buffer1[2] = 0x4E; //  N
    buffer1[3] = 0x47; //  G
    buffer1[4] = 0x00; //  0x0
    buffer1[5] = '.';  //  .  
  
    T.exec("CREATE TEMP TABLE pqxxbin (binfield bytea)");
  
    // Make string from buffer data
    const std::string myStr(buffer1, bufSize);
    const std::string prepStmtName = "test";
    ASSERT(myStr[myStr.size() - 1] == '.');
  
    pqxx::prepare::declaration D = m_connectionInThread.connection()->prepare(prepStmtName, "INSERT INTO pqxxbin VALUES ($1)");
    D("bytea", pqxx::prepare::treat_binary);
  
    prepare::invocation I = T.prepared(prepStmtName);
    I(myStr); // pass string as invocation parameter for $1. 
  
    I.exec();
  
    result R = T.exec("SELECT binfield from pqxxbin");
  
    binarystring B(R.at(0).at(0));
  
    kdDebug() << "original: " << T.esc_raw(reinterpret_cast <const unsigned char *> (&buffer1), bufSize) << endl << "returned: " << R.at(0).at(0).c_str() << endl;
  
    if (B.size() != bufSize)
      kdError() << "Binary string got changed from " + to_string(bufSize) + " to " + to_string(B.size()) + " bytes"  << endl;
  
    // Output is:
    // original: \\211PNG\\000.
    // returned: \211PNG
    // Binary string got changed from 6 to 4 bytes
      
    T.commit();
    return true;
 }
 */
