//
//   File : kvi_dcc_chat.cpp (/usr/build/NEW_kvirc/kvirc/src/kvirc/kvi_dcc_chat.cpp)
//   Last major modification : Sat Apr 17 1999 18:54:50 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

//#define _KVI_DEBUG_CHECK_RANGE_
//#define _KVI_DEBUG_CLASS_NAME_ "KviDccChat"
#include "kvi_debug.h"

#include "kvi_dcc_chat.h"
#include "kvi_dcc_core.h"
#include "kvi_dcc_manager.h"
#include "kvi_app.h"
#include "kvi_ircview.h"
#include "kvi_input.h"
#include "kvi_frame.h"
#include "kvi_options.h"
#include "kvi_locale.h"

#include "kvi_ircsocket.h"
#include "kvi_netutils.h"
#include "kvi_settings.h"
#include "kvi_popupmenu.h"
#include "kvi_uparser.h"

#include "kvi_event.h"

#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include <fcntl.h>
#include "kvi_error.h"


extern KviPopupMenu * g_pDccChatPopup;

// Time to sleep if no data has to be read or sent
// In DCC Chat it can be a bit longer...
#define KVI_DCC_IDLE_TIME_USECS 400

//
// Friendly suggestion for programmers :
//     If you want your life to be simple and plain : do NOT use threads.
//     If you like risk , and you are patient : DO use threads , and get fun with it :)
//

//Declared in kvi_app.cpp and managed by KviApp class
//extern QPixmap * g_pWindowIcon[KVI_WND_NUM_ICONS];
extern QPixmap * g_pixViewOut[KVI_OUT_NUM_IMAGES];
//Global event dispatcher
extern KviThreadEventDispatcher *g_pThreadEventDispatcher;

extern KviEventManager * g_pEventManager;

//#warning "Override saveProperties()"

//============ KviDccChat ============//

KviDccChat::KviDccChat(KviFrame *lpFrm,const char *name)
:KviWindow(name,KVI_WND_TYPE_CHAT,lpFrm)
{
	g_pThreadEventDispatcher->registerObject(this);
	m_pSplitter  = new QSplitter(QSplitter::Horizontal,this);

	m_pView      = new KviIrcView(m_pSplitter,lpFrm,this);
	m_pInput     = new KviInput(this,lpFrm->m_pUserParser);
	connect(m_pView,SIGNAL(contextPopupRequested(KviIrcView *)),this,SLOT(viewRightClicked(KviIrcView *)));

	m_bThreadRunning = false;

	m_pDccChatStruct = new KviDccChatStruct;
	m_pDccChatStruct->dccChatParent = this;
	m_pDccChatStruct->outQueue = new QQueue<KviStr>;
	m_pDccChatStruct->outQueue->setAutoDelete(true);

	m_szLocalNick = m_pFrm->m_global.szCurrentNick;
	m_szLocalMask = m_pFrm->m_global.szCurrentMaskFromServer;
	if(m_szLocalNick.isEmpty())m_szLocalNick = __tr("me");
	if(m_szLocalMask.isEmpty())m_szLocalMask = __tr("me!me@localhost");

	kvi_threadMutexInit(&(m_pDccChatStruct->outMutex),0);



	setFocusHandler(m_pInput);
}

//============ ~KviDccChat ============//

KviDccChat::~KviDccChat()
{
	g_pThreadEventDispatcher->unregisterObject(this);
	if(m_bThreadRunning)kvi_threadCancel(m_thread);
	if(m_pDccChatStruct){
		delete m_pDccChatStruct->outQueue;
		delete m_pDccChatStruct;
	}
	kvi_threadMutexDestroy(&(m_pDccChatStruct->outMutex));
}

void KviDccChat::closeEvent(QCloseEvent *e)
{
	if(g_pEventManager->eventEnabled(KviEvent_OnDccChatTerminated) && m_szRemoteInfo.hasData()){
		m_pFrm->m_pUserParser->callEvent(KviEvent_OnDccChatTerminated,this,m_szRemoteInfo);
	}
	KviWindow::closeEvent(e);
}

void KviDccChat::viewRightClicked(KviIrcView *)
{
	g_pDccChatPopup->doPopup(this,SLOT(viewPopupClicked(const KviStr &)));
}

void KviDccChat::viewPopupClicked(const KviStr &dataBuffer)
{
	m_pFrm->m_pUserParser->parseCommand(dataBuffer.ptr(),this);
}


//================ myIconPtr =================//

QPixmap * KviDccChat::myIconPtr()
{
	return g_pixViewOut[KVI_OUT_WND_CHAT];
}

//=============== applyOptions ================//

void KviDccChat::applyOptions()
{
	m_pView->setFont(g_pOptions->m_fntView);
	m_pView->setShowImages(g_pOptions->m_bShowImages,false);
	m_pView->setTimestamp(g_pOptions->m_bTimestamp);
	m_pView->setMaxBufferSize(g_pOptions->m_iViewMaxBufferSize);
}

bool KviDccChat::event(QEvent * e)
{
	if(e->type() != QEvent::User)return KviWindow::event(e);
	switch(((KviDccEvent *)e)->m_type){
		case KVI_DCC_EVENT_ERROR:
			outputNoFmt(KVI_OUT_DCCERROR,((KviDccEvent *)e)->m_dataString.ptr());
			if(m_bThreadRunning)m_bThreadRunning = false;
			// The thread is dead now...m_pDccChatStruct is safe.
			// m_szRemoteIpAddress.hasData() is true if we were connected!
			if(g_pEventManager->eventEnabled(KviEvent_OnDccChatTerminated) && m_szRemoteInfo.hasData()){
				KviStr tmp(KviStr::Format,"%s %s",m_szRemoteInfo.ptr(),((KviDccEvent *)e)->m_dataString.ptr());
				if(m_pFrm->m_pUserParser->callEvent(KviEvent_OnDccChatTerminated,this,tmp))return true;
			}
			output(KVI_OUT_DCCINFO,__tr("Connection terminated (%s:%d)"),m_pDccChatStruct->szAddress.ptr(),m_pDccChatStruct->uPort);
		break;
		case KVI_DCC_EVENT_MSG:
			outputNoFmt(KVI_OUT_DCCINFO,((KviDccEvent *)e)->m_dataString.ptr());
		break;
		case KVI_DCC_EVENT_DATA:
			if(g_pEventManager->eventEnabled(KviEvent_OnDccChatMessage)){
				KviStr tmp(KviStr::Format,"%s %s",m_szRemoteInfo.ptr(),((KviDccEvent *)e)->m_dataString.ptr());
				if(m_pFrm->m_pUserParser->callEvent(KviEvent_OnDccChatMessage,this,tmp))return true;
			}
			if(*(((KviDccEvent *)e)->m_dataString.ptr()) == 0x01){
				//CTCP ? Only ACTION supported
				const char *aux = ((KviDccEvent *)e)->m_dataString.ptr();
				aux++;
				if(kvi_strEqualCIN(aux,"ACTION",6)){
					while(*aux)aux++;
					aux--;
					if(*aux == 0x01)((KviDccEvent *)e)->m_dataString.cutRight(1);
					((KviDccEvent *)e)->m_dataString.cutLeft(7);
					((KviDccEvent *)e)->m_dataString.stripWhiteSpace();
					output(KVI_OUT_ACTION,"%s %s",m_szRemoteNick.ptr(),((KviDccEvent *)e)->m_dataString.ptr());
					return true;
				}
			}
			m_pFrm->outputPrivmsg(this,KVI_OUT_NONE,m_szRemoteNick.ptr(),m_szRemoteMask.ptr(),((KviDccEvent *)e)->m_dataString.ptr());
		break;
		case KVI_DCC_EVENT_CONNECTIONESTABILISHED:
			m_szRemoteInfo = ((KviDccEvent *)e)->m_dataString.ptr();
			if(g_pEventManager->eventEnabled(KviEvent_OnDccChatConnected)){
//				KviStr tmp(KviStr::Format,"%s",m_szRemoteNick.ptr(),((KviDccEvent *)e)->m_dataString.ptr());
				m_pFrm->m_pUserParser->callEvent(KviEvent_OnDccChatConnected,this,m_szRemoteInfo);
			}
			// following toggles beeping from default of false, to true.
			if (m_pView->initBeeping(m_szRemoteNick.ptr()))m_pView->toggleBeeping();
		break;
		case KVI_DCC_EVENT_LISTENING:
			m_pFrm->m_pSocket->sendData(((KviDccEvent *)e)->m_dataString.ptr(),((KviDccEvent *)e)->m_dataString.len());
			outputNoFmt(KVI_OUT_DCCINFO,__tr("Sent DCC chat request, waiting for reply..."));
		break;
		default:
			return KviWindow::event(e);
		break;
	}
	return true;
}

bool KviDccChat::sendData(const char * buffer)
{
	if(!m_bThreadRunning)return false; //No connection at all..
	// otherwise the socket is connected
	// or is waiting for connection
	// If it is waiting , just store the text in the queue..
	// It will be all sent when a connection is estabilished
	KviStr * tmp = new KviStr(buffer);
	// Need to lock the mutex while appending...
	kvi_threadMutexLock(&(m_pDccChatStruct->outMutex));
	m_pDccChatStruct->outQueue->enqueue(tmp);
	kvi_threadMutexUnlock(&(m_pDccChatStruct->outMutex));
	return true;
}

//================ resizeEvent ===============//

void KviDccChat::resizeEvent(QResizeEvent *)
{
	int inputSize = m_pInput->heightHint();
	m_pSplitter->setGeometry(0,0,width(),height() - inputSize);
	m_pInput->setGeometry(0,height() - inputSize,width(),inputSize);
}

//=====================================================================================================
//
// Main routines for the slave DCC Chat thread 
//

static void kvi_threadChatCheckForIncomingData(int sock,KviDccChatStruct * dcc)
{
	// Check if there are data available on sock,
	// read it , split into messages and post data events
	// to the parent DCC window class.

	if(!kvi_dccCore_selectForRead(sock)){
		kvi_threadTestCancel();
		usleep(KVI_DCC_IDLE_TIME_USECS);
		return;
	} // else data to read...

	kvi_threadTestCancel();

	//read data
	char buffer[1025];
	int readLength = read(sock,buffer,1024); //read() is not atomic :)

	kvi_threadTestCancel(); //Yes..I like this function call

	if(readLength <= 0){
		// oooops ?
		if(readLength == 0)kvi_dccCore_errorExitThread(__tr("Remote end has closed the connection"),dcc->dccChatParent);
		else { // error ?
			if((errno == EINTR)||(errno == EAGAIN))return;
			// Yes...error :(
			int iError;
			switch(errno){
				case ECONNREFUSED: iError = KVI_ERROR_ConnectionRefused;     break;
				case ENOTSOCK:     iError = KVI_ERROR_KernelNetworkingPanic; break;
				case ETIMEDOUT:    iError = KVI_ERROR_ConnectionTimedOut;    break;
				case ENETUNREACH:  iError = KVI_ERROR_NetworkUnreachable;    break;
				case EPIPE:        iError = KVI_ERROR_BrokenPipe;            break;
				// Unhandled error...pass errno to the strerror function
				default:           iError = -errno;                          break;
			}
			// post the error event to the parent and exit
			dcc->tmpBuffer.sprintf(__tr("READ ERROR: %s"),kvi_getErrorString(iError));
			kvi_dccCore_errorExitThread(dcc->tmpBuffer.ptr(),dcc->dccChatParent);
		}
	}

	// successfull read.

	*(buffer + readLength) = '\0';       // terminate the buffer
	dcc->szPartialBuffer.append(buffer); // and merge with the latest unterminated data

	kvi_threadTestCancel(); //Yeah!

	// Now split the data into messages
	// A message MUST be terminated by a LF.
	int idx = dcc->szPartialBuffer.findFirstIdx('\n');
	while(idx != -1){
		// strip CR after the message...if any
		KviStr tmp = dcc->szPartialBuffer.left(idx);
		if(idx > 0){
			if(dcc->szPartialBuffer.at(idx - 1) == '\r'){
				tmp.cutRight(1);
			}
		}
		// And now post the data event to the parent...
		KviDccEvent *dt = new KviDccEvent(KVI_DCC_EVENT_DATA,tmp.ptr());
		g_pThreadEventDispatcher->postEvent(dt,dcc->dccChatParent);

		kvi_threadTestCancel();
		// Remove the stuff that has been processed...
		dcc->szPartialBuffer.cutLeft(idx+1);
		// And look for the next message
		idx = dcc->szPartialBuffer.findFirstIdx('\n');
	}

	if(dcc->szPartialBuffer.len() > 512){
		// oooops...no LF in the first 512 bytes ?
		// a looooong message...
		// Hehe..if I were a hacker , I could use it as
		// a system attack once a dcc chat has been estabilished.
		// Just write a script that sends an infinite sequence of senseless
		// data without LF's inside...
		// Sooner or later an unhandled ENOMEM will jump out :)
		// Anyway , KVirc is immune now...
		KviDccEvent *dt = new KviDccEvent(KVI_DCC_EVENT_MSG,__tr("Unterminated message longer than 512 bytes: Auto-splitting"));
		g_pThreadEventDispatcher->postEvent(dt,dcc->dccChatParent);

		kvi_threadTestCancel();

		dt = new KviDccEvent(KVI_DCC_EVENT_DATA,dcc->szPartialBuffer.ptr());
		g_pThreadEventDispatcher->postEvent(dt,dcc->dccChatParent);
		dcc->szPartialBuffer = "";
	}
}

static void kvi_threadChatCheckForOutgoingData(int sock,KviDccChatStruct * dcc)
{
	// Need to lock a mutex while managing the outQueue.
	// No cancel available while mutex is locked , so
	// there (should be) no need for another cleanup routine

	// Usually there is only one message per 1-2 seconds (unless
	// the user paste something in the input line) , so this stuff
	// can be done only once per loop...

	KviStr *str = 0;
	kvi_threadMutexLock(&(dcc->outMutex));
	str = dcc->outQueue->head();
	kvi_threadMutexUnlock(&(dcc->outMutex));

	kvi_threadTestCancel();

	if(!str){
		usleep(KVI_DCC_IDLE_TIME_USECS); //take a break
		return;      //No data to be sent
	}

	int wrLen = write(sock,str->ptr(),str->len());

	kvi_threadTestCancel();
	// Managing str is now safe...the ponter and the object are never changed
	// by other threads...
	if(wrLen == str->len()){
		// All data sent...
		// Send a CRLF too
		write(sock,"\r\n",2);
		// now relock the mutex , and remove the string from the queue
		kvi_threadTestCancel();
		kvi_threadMutexLock(&(dcc->outMutex));
		dcc->outQueue->remove();
		kvi_threadMutexUnlock(&(dcc->outMutex));	
		kvi_threadTestCancel();
	} else {
		if(wrLen >= 0)str->cutLeft(wrLen);
		else {
			// oooooops...error while writing ?
			if((errno == EINTR)||(errno == EAGAIN))return;
			// Yes...error :(
			int iError;
			switch(errno){
				case ECONNREFUSED: iError = KVI_ERROR_ConnectionRefused;     break;
				case ENOTSOCK:     iError = KVI_ERROR_KernelNetworkingPanic; break;
				case ETIMEDOUT:    iError = KVI_ERROR_ConnectionTimedOut;    break;
				case ENETUNREACH:  iError = KVI_ERROR_NetworkUnreachable;    break;
				case EPIPE:        iError = KVI_ERROR_BrokenPipe;            break;
				case EFAULT:       iError = KVI_ERROR_OutOfAddressSpace;     break;
				case EBADF:        iError = KVI_ERROR_BadFileDescriptor;     break;
				// Unhandled error...pass errno to the strerror function
				default:           iError = -errno;                          break;
			}
			// post the error event to the parent and exit
			dcc->tmpBuffer.sprintf(__tr("WRITE ERROR: %s"),kvi_getErrorString(iError));
			kvi_dccCore_errorExitThread(dcc->tmpBuffer.ptr(),dcc->dccChatParent);
		}
	}
}

static void * kvi_threadChatLoop(int sock,KviDccChatStruct * dcc)
{
	// Main DCC Chat loop
	// This funciton newer returns.
	// It is terminated when pthread_exit() is called
	// from one of the internal calls
	// Or if the parent object calls pthread_cancel().
	for(;;){ //forever
		kvi_threadTestCancel();
		kvi_threadChatCheckForIncomingData(sock,dcc);
		kvi_threadTestCancel();
		kvi_threadChatCheckForOutgoingData(sock,dcc);
		kvi_threadTestCancel();
	}
	return 0; //Newer reached
}

//=====================================================================================================
//
// Accept a DCC CHAT REQUEST from another user on IRC
//

static void * kvi_threadAcceptDccRequest(void * data)
{
	kvi_threadInitialize();
	// Specific routine for accepting DCC Chat requests
	// from users on IRC.
	// An user sent us a CTCP DCC with a port and an address
	// Now we try to contact that host and initiate a DCC Chat...
	KviDccChatStruct * dcc = (KviDccChatStruct *)data; //Maybe convert it in just casts...
	// Prepare the target data
	struct in_addr     inAddress;
	struct sockaddr_in hostSockAddr;

	inAddress.s_addr        = dcc->uAddress;
	hostSockAddr.sin_family = AF_INET;
	hostSockAddr.sin_port   = htons(dcc->uPort);
	hostSockAddr.sin_addr   = inAddress;

	kvi_threadTestCancel();
	// Let's go...
	int sock = ::socket(PF_INET,SOCK_STREAM,0);
	// Don't forget to close the socket when terminating.
	kvi_threadCleanupPush(kvi_dccCore_threadCleanupCloseSocket,&sock);

	if(sock < 0)kvi_dccCore_errorExitThread(__tr("Unable to create a stream socket. DCC chat failed"),dcc->dccChatParent);

	kvi_threadTestCancel();

	if(fcntl(sock, F_SETFL, O_NONBLOCK) < 0)kvi_dccCore_errorExitThread(__tr("Unable to create a non-blocking stream socket. DCC chat failed"),dcc->dccChatParent);

	kvi_threadTestCancel();

	// Ok...now try to connect
	int iError = kvi_dccCore_connect(sock,(struct sockaddr *)(&hostSockAddr),sizeof(hostSockAddr));

	if(iError != KVI_ERROR_Success){
		dcc->tmpBuffer.sprintf(__tr("CONNECT ERROR: %s"),kvi_getErrorString(iError));
		kvi_dccCore_errorExitThread(dcc->tmpBuffer.ptr(),dcc->dccChatParent);
	}

	kvi_threadTestCancel();

	// Now wait for connection...

	KviStr tmp(KviStr::Format,__tr("Connecting to %s on port %u"),dcc->szAddress.ptr(),dcc->uPort);
	KviDccEvent * msg = new KviDccEvent(KVI_DCC_EVENT_MSG,tmp.ptr());
	g_pThreadEventDispatcher->postEvent(msg,dcc->dccChatParent);

	kvi_threadTestCancel();

	iError = kvi_dccCore_waitForOutgoingConnection(sock);
	if(iError != KVI_ERROR_Success){
		dcc->tmpBuffer.sprintf(__tr("CONNECT ERROR: %s"),kvi_getErrorString(iError));
		kvi_dccCore_errorExitThread(dcc->tmpBuffer.ptr(),dcc->dccChatParent);
	}

	kvi_threadTestCancel();

	msg = new KviDccEvent(KVI_DCC_EVENT_MSG,__tr("Connection established"));
	g_pThreadEventDispatcher->postEvent(msg,dcc->dccChatParent);

	dcc->tmpBuffer.sprintf("%s %s %s %s",dcc->nick.ptr(),dcc->username.ptr(),dcc->host.ptr(),dcc->szAddress.ptr());
	msg = new KviDccEvent(KVI_DCC_EVENT_CONNECTIONESTABILISHED,dcc->tmpBuffer.ptr());
	g_pThreadEventDispatcher->postEvent(msg,dcc->dccChatParent);

	kvi_threadTestCancel();

	kvi_threadChatLoop(sock,dcc);

	// Code that is newer reached...
	kvi_threadCleanupPop(0); //Do not execute

	return 0;
}

void KviDccChat::acceptDccRequest(const char *nick,const char *username,const char *host,unsigned long uAddress,unsigned short uPort)
{
	m_szRemoteNick = nick;
	m_szRemoteMask.sprintf("%s!%s@%s",nick,username,host);
	// Called from KviDccManager
	// An user sent us a CTCP DCC , and we accepted it.
	// Connect to his host
#ifdef _KVI_DEBUG_CHECK_RANGE_
	if(m_bThreadRunning)debug("WARNING: acceptDccRequest was called twice!");
#endif
	m_pDccChatStruct->nick      = nick;
	m_pDccChatStruct->username  = username;
	m_pDccChatStruct->host      = host;
	m_pDccChatStruct->uAddress  = uAddress;
	struct in_addr addr;
	addr.s_addr = uAddress;
	kvi_binaryIpToString(addr,m_pDccChatStruct->szAddress);
	m_pDccChatStruct->uPort     = uPort;
	// Create the slave thread...
	m_bThreadRunning = true;

	if(kvi_threadCreate(&m_thread,0,kvi_threadAcceptDccRequest,m_pDccChatStruct) != 0){
		m_bThreadRunning = false;
		outputNoFmt(KVI_OUT_DCCERROR,__tr("Cannot create slave thread. DCC chat failed"));
	}

//	kvi_threadAttrDestroy(&attr);
}

//=====================================================================================================
//
// Request a DCC to an irc user
//

static void * kvi_threadRequestDccChat(void * data)
{
	__debug("slaveThread");
	kvi_threadInitialize();
	// with pthread_cleanup_push/pop we need to declare the variables at the beginning of the function...
	int newsock = -1;

	KviDccChatStruct * dcc = (KviDccChatStruct *)data; //Maybe convert it in just casts...
	KviDccEvent * e = 0;

	struct sockaddr_in sockAddress;
	sockAddress.sin_family      = AF_INET;
	sockAddress.sin_port        = htons(dcc->uPortToListenOn); // :)
	sockAddress.sin_addr.s_addr = INADDR_ANY;

	kvi_threadTestCancel();
	// Let's go...
	int sock = ::socket(PF_INET,SOCK_STREAM,0);
	// Don't forget to close the socket when terminating.
	kvi_threadCleanupPush(kvi_dccCore_threadCleanupCloseSocket,&sock);

	if(sock < 0)kvi_dccCore_errorExitThread(__tr("Unable to create a listening socket. DCC chat failed"),dcc->dccChatParent);

	kvi_threadTestCancel();

	if((bind(sock,(struct sockaddr *)&sockAddress,sizeof(sockAddress))<0)||(listen(sock,100)<0) )
		kvi_dccCore_errorExitThread(__tr("Unable to set up a listening socket. DCC chat failed"),dcc->dccChatParent);

	kvi_threadTestCancel();

	_this_should_be_socklen_t iSize = sizeof(sockAddress);
	getsockname(sock,(struct sockaddr*)&sockAddress,&iSize);
	dcc->uPort = ntohs(sockAddress.sin_port);

	kvi_threadTestCancel();

	dcc->tmpBuffer.sprintf(__tr("Listening on port %u"),dcc->uPort);
	e = new KviDccEvent(KVI_DCC_EVENT_MSG,dcc->tmpBuffer.ptr());
	g_pThreadEventDispatcher->postEvent(e,dcc->dccChatParent);

	kvi_threadTestCancel();

	dcc->tmpBuffer.sprintf("PRIVMSG %s :%cDCC CHAT chat %u %u%c",dcc->nick.ptr(),0x01,ntohl(dcc->uAddress),dcc->uPort,0x01);
	e = new KviDccEvent(KVI_DCC_EVENT_LISTENING,dcc->tmpBuffer.ptr());
	g_pThreadEventDispatcher->postEvent(e,dcc->dccChatParent);

	kvi_threadTestCancel();

	newsock = kvi_dccCore_waitForIncomingConnection(sock,&(dcc->uAddress),&(dcc->uPort),&(dcc->szAddress));

	kvi_threadCleanupPop(0); //Do not execute
	close(sock); //close the old socket...
	kvi_threadCleanupPush(kvi_dccCore_threadCleanupCloseSocket,&newsock);

	kvi_threadTestCancel();

	dcc->tmpBuffer.sprintf(__tr("Connected to %s on port %u"),dcc->szAddress.ptr(),dcc->uPort);
	e = new KviDccEvent(KVI_DCC_EVENT_MSG,dcc->tmpBuffer.ptr());
	g_pThreadEventDispatcher->postEvent(e,dcc->dccChatParent);

	dcc->tmpBuffer.sprintf("%s %s %s %s",dcc->nick.ptr(),dcc->username.ptr(),dcc->host.ptr(),dcc->szAddress.ptr());
	e = new KviDccEvent(KVI_DCC_EVENT_CONNECTIONESTABILISHED,dcc->tmpBuffer.ptr());
	g_pThreadEventDispatcher->postEvent(e,dcc->dccChatParent);

	kvi_threadTestCancel();

	kvi_threadChatLoop(newsock,dcc);

	// Code that is newer reached...
	kvi_threadCleanupPop(0); //Do not execute

	return 0;
}

void KviDccChat::requestDcc(const char *nick,const char *username,const char *host)
{
	m_szRemoteNick = nick;
	m_szRemoteMask.sprintf("%s!%s@%s",nick,username,host);
	// Called from KviDccManager
	// Request a DCC connection to an user
#ifdef _KVI_DEBUG_CHECK_RANGE_
	if(m_bThreadRunning)debug("WARNING: requestDcc was called twice!");
#endif
	m_pDccChatStruct->nick      = nick;
	m_pDccChatStruct->username  = username;
	m_pDccChatStruct->host      = host;
	m_pDccChatStruct->uAddress  = m_pFrm->m_pSocket->getSockAddress();
	m_pDccChatStruct->uPortToListenOn = m_pFrm->m_pDccManager->getDccSendListenPort();

	if(g_pOptions->m_bUseUserDefinedIpForDccRequests){
		struct in_addr addr;
		if(kvi_stringIpToBinaryIp(g_pOptions->m_szDccLocalIpAddress.ptr(),&addr)){
			output(KVI_OUT_DCCINFO,__tr("Using %s as localhost address"),g_pOptions->m_szDccLocalIpAddress.ptr());
			m_pDccChatStruct->uAddress = addr.s_addr;
		} else {
			output(KVI_OUT_DCCERROR,__tr("IP %s is invalid : Using standard IP address"),g_pOptions->m_szDccLocalIpAddress.ptr());
			g_pOptions->m_bUseUserDefinedIpForDccRequests = false;
		}
	}

	if(m_pDccChatStruct->uAddress == 0){
//#ifdef _KVI_DEBUG_CHECK_RANGE_
//	#warning "Try to use the USER SUPPLIED LOCAL HOST IP ADDRESS"
//#endif
		outputNoFmt(KVI_OUT_DCCERROR,__tr("Cannot resolve localhost. DCC chat failed"));
		return;
	}
	m_pDccChatStruct->szAddress = __tr("unknown");
	m_pDccChatStruct->uPort     = 0;
	// Create the slave thread...
	m_bThreadRunning = true;

	if(kvi_threadCreate(&m_thread,0,kvi_threadRequestDccChat,m_pDccChatStruct) != 0){
		m_bThreadRunning = false;
		outputNoFmt(KVI_OUT_DCCERROR,__tr("Cannot create slave thread. DCC chat failed"));
	}
}

#include "m_kvi_dcc_chat.moc"
