//
//   File : kvi_dirbrowser.cpp (/usr/build/NEW_kvirc/kvirc/src/kvirc/kvi_dirbrowser.cpp)
//   Last major modification : Sun Apr 25 1999 13:58:06 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviDirStuff"

#include "kvi_debug.h"
#include "kvi_dirbrowser.h"
#include "kvi_defines.h"
#include "kvi_options.h"
#include "kvi_locale.h"
#include "kvi_frame.h"
#include "kvi_mime.h"
#include "kvi_app.h"
#include "kvi_statusbar.h"
#include "kvi_error.h"
#include "kvi_fileutils.h"

#include <qfileinfo.h>
#include <qdir.h>
#include <qlist.h>
#include <qdragobject.h>
#include <qstringlist.h>
#include <qevent.h>
#include <qframe.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
#include <unistd.h>
#include <stdio.h>


//Declared in kvi_app.cpp and managed by KviApp class
//extern QPixmap * g_pWindowIcon[KVI_WND_NUM_ICONS];
extern QPixmap * g_pixViewOut[KVI_OUT_NUM_IMAGES];
extern QPixmap * g_pBrowserIcons[KVI_BROWSER_ICONS_NUM_IMAGES];
extern QPixmap * g_pFileIcons[KVI_FILE_ICONS_NUM_IMAGES];

// Declared and managed by KviApp (kvi_app.cpp)
extern KviThreadEventDispatcher * g_pThreadEventDispatcher;


KviDirectoryView::KviDirectoryView(QWidget * parent,KviFrame *lpFrm)
:QIconView(parent)
{
	m_pFrm = lpFrm;
	setGridX(70);
	setSelectionMode(QIconView::Extended);
	setAutoArrange(true);
	setWordWrapIconText(false);
	setResizeMode(QIconView::Adjust);
	setAcceptDrops(true);
	viewport()->setAcceptDrops(true);
	parent->setAcceptDrops(true);
	setItemTextPos(QIconView::Bottom);

//	setShowToolTips(true);

	setSorting(true);
	m_szDirPath = "/";
	connect(this,SIGNAL(dropped(QDropEvent *,const QValueList<QIconDragItem> &)),this,SLOT(itemDropped(QDropEvent *,const QValueList<QIconDragItem> &)));
	connect(this,SIGNAL(moved()),this,SLOT(emitNeedUpdate()));
	connect(this,SIGNAL(rightButtonPressed(QIconViewItem *,const QPoint &)),this,SLOT(showContextMenu(QIconViewItem *,const QPoint &)));
	connect(this,SIGNAL(itemRenamed(QIconViewItem *,const QString &)),this,SLOT(dirItemRenamed(QIconViewItem *,const QString &)));
	m_pContextPopup = new QPopupMenu();
	applyOptionsInternal();
}

void KviDirectoryView::applyOptionsInternal()
{
//	setUseSingleClickMode(g_pOptions->m_bDirBrowserUseSingleClickMode);
//	setSingleClickConfiguration(0,0,0,0,0,500);
}

void KviDirectoryView::applyOptions()
{
	applyOptionsInternal();	
	emit needUpdate();
}

KviDirectoryView::~KviDirectoryView()
{
	delete m_pContextPopup;
//	delete m_pItalicFont;
}

void KviDirectoryView::dirItemRenamed(QIconViewItem * i,const QString &text)
{
	KviStr tmp = text;
	KviDirectoryViewItem * it = (KviDirectoryViewItem *)i;
	if(tmp.findFirstIdx('/') != -1){
		g_pApp->warningBox(__tr("Unable to rename file %s.\nPlease use drag-and-drop to move\nthe file across directories."),it->m_data->szFileName.ptr());
		it->setText(it->m_data->szFileName.ptr());
		return;
	}
	QFileInfo fi(text);
	if(fi.exists()){
		g_pApp->warningBox(__tr("Unable to rename file %s.\nThe file %s already exists."),it->m_data->szFileName.ptr(),tmp.ptr());
		it->setText(it->m_data->szFileName.ptr());
	} else {
		KviStr newPath = it->m_data->szDirPath;
		newPath.ensureLastCharIs('/');
//		if(!newPath.lastCharIs('/'))newPath.append('/');
		newPath.append(text);
		if(!kvi_renameFile(it->m_data->szAbsFilePath.ptr(),newPath.ptr()))
		{
			g_pApp->warningBox(__tr("Unable to rename file %s.\nThe destination may already exist."),it->m_data->szFileName.ptr());
			// failed
			it->setText(it->m_data->szFileName.ptr());
		} else {
			it->m_data->szAbsFilePath = newPath;
			it->m_data->szFileName = text;
		}
	}
}

QDragObject * KviDirectoryView::dragObject()
{
    if(!currentItem())return 0;

	QStringList fnamelist;

	for(KviDirectoryViewItem * item = (KviDirectoryViewItem*)firstItem();item;item = (KviDirectoryViewItem*)item->nextItem())
	{
		if(item->isSelected())
		{
			fnamelist.append(_CHAR_2_QSTRING(item->m_data->szAbsFilePath.ptr()));
		}
	}	

	// QPoint orig = viewportToContents( viewport()->mapFromGlobal( QCursor::pos() ) );
	QUriDrag * drag = new QUriDrag( viewport() );
	drag->setFilenames(fnamelist);
	
	if(fnamelist.count() == 1){
		drag->setPixmap( QPixmap(*(currentItem()->pixmap())),
 		     QPoint( currentItem()->pixmapRect().width() / 2, currentItem()->pixmapRect().height() / 2 ));
	} else {
		drag->setPixmap( *g_pFileIcons[KVI_FILE_ICON_MULTIPLE],
 		     QPoint( currentItem()->pixmapRect().width() / 2, currentItem()->pixmapRect().height() / 2 ));	
	}

    return drag;
}

void KviDirectoryView::itemDropped(QDropEvent *e,const QValueList<QIconDragItem> &lst)
{
	QStringList list;
	bool bNeedReload = false;
	if(QUriDrag::decodeLocalFiles(e,list)){
		if(!list.isEmpty()){
			QStringList::ConstIterator it = list.begin(); //kewl ! :)
    		for( ; it != list.end(); ++it ){
				KviStr filename = *it;
				KviStr newPath = filename;
				newPath.cutToLast('/',true);
				KviStr dirPath = m_szDirPath;
				dirPath.ensureLastCharIs('/');
				newPath.prepend(dirPath);
				if(!kvi_renameFile(filename.ptr(),newPath.ptr()))
				{
					g_pApp->warningBox(__tr("Unable to move file %s.\nThe destination may already exist."),filename.ptr());
					// failed
				} else {
					bNeedReload = true;
				}
			}
		}
	}
	if(bNeedReload)emit needUpdate();
}

void KviDirectoryView::emitNeedUpdate()
{
	emit needUpdate();
}

void KviDirectoryView::showContextMenu(QIconViewItem * i,const QPoint & pos)
{
	m_pContextPopup->clear();
	if(i){
		// some items are selected
		if(g_pOptions->m_szDirBrowserOpenWithEditorCmd.hasData() && (((KviDirectoryViewItem *)i)->m_data->bIsFile))
		{
			m_pContextPopup->insertItem(__tr("&Open with Editor"),this,SLOT(openWithEditorSelectedFiles()));
			m_pContextPopup->insertSeparator();
		}
		m_pContextPopup->insertItem(__tr("Move to &Trash"),this,SLOT(trashSelectedFiles()));
	}
	m_pContextPopup->popup(pos);
}

void KviDirectoryView::trashSelectedFiles()
{
	bool bNeedUpdate = false;
	for(KviDirectoryViewItem * item = (KviDirectoryViewItem*)firstItem();item;item = (KviDirectoryViewItem*)item->nextItem())
	{
		if(item->isSelected())
		{
			if(!(item->m_data->bIsDirectory && 
				(kvi_strEqualCI(item->m_data->szFileName.ptr(),"..") || kvi_strEqualCI(item->m_data->szFileName.ptr(),"."))))
			{
				if(!g_pApp->trashFile(item->m_data->szAbsFilePath.ptr()))
				{
					g_pApp->warningBox(__tr("Unable to move file %s to trash.\n"),item->m_data->szAbsFilePath.ptr());
				} else bNeedUpdate = true;
			}
		}
	}
	if(bNeedUpdate)emit needUpdate();
}

void KviDirectoryView::openWithEditorSelectedFiles()
{
	g_pOptions->m_szDirBrowserOpenWithEditorCmd.stripWhiteSpace();

	for(KviDirectoryViewItem * item = (KviDirectoryViewItem*)firstItem();item;item = (KviDirectoryViewItem*)item->nextItem())
	{
		if(item->isSelected() && item->m_data->bIsFile)
		{
			g_pApp->executeFileWithCommand(g_pOptions->m_szDirBrowserOpenWithEditorCmd.ptr(),
					item->m_data->szAbsFilePath.ptr(),m_pFrm);
		}
	}
}


KviDirectoryViewItem::KviDirectoryViewItem(KviDirectoryView * view,KviDirBrowserEntry * e)
:QIconViewItem(view)
{
	m_view = view;
	m_data = e;
	setText(m_data->szFileName.ptr());	
	setDragEnabled(true);
	setDropEnabled(false);
//	QObject::connect(this,SIGNAL(renamed(const QString &)),this,SLOT(itemRenamed(const QString &)));

	if(!(m_data->bIsReadable))
	{
		if(m_data->bIsDirectory)
		{
			setPixmap(*g_pFileIcons[KVI_FILE_ICON_UNREADABLEDIR]);
		} else {
			setPixmap(*g_pFileIcons[KVI_FILE_ICON_UNREADABLEFILE]);
		}
		return;
	}

	if(m_data->szIconPath.hasData())
	{
		QPixmap * pix = g_pApp->getFileIcon(m_data->szIconPath.ptr());
		if(pix){
			setPixmap(*pix);
			return;
		}
	}

	if(m_data->bIsDirectory)
	{
		setPixmap(*g_pFileIcons[KVI_FILE_ICON_DIRECTORY]);
		setDropEnabled(true);
//		if(m_data->bIsSymLink)setFont(*(view->m_pItalicFont));
	} else if(m_data->bIsSymLink){
		setPixmap(*g_pFileIcons[KVI_FILE_ICON_SYMLINK]);
//		setFont(*(view->m_pItalicFont));
	} else if(m_data->bIsSocket){
		setPixmap(*g_pFileIcons[KVI_FILE_ICON_SOCKET]);
	} else if(m_data->bIsDevice){
		setPixmap(*g_pFileIcons[KVI_FILE_ICON_DEVICE]);
	} else setPixmap(*g_pFileIcons[KVI_FILE_ICON_UNKNOWN]);

}

KviDirectoryViewItem::~KviDirectoryViewItem()
{
	delete m_data;
}

//void KviDirectoryViewItem::itemRenamed(const QString &text)
//{
//	KviStr tmp = text;
//	if(tmp.findFirstIdx('/') != -1){
//		g_pApp->warningBox(__tr("Unable to rename file %s.\nPlease use Drag & Drop to move\nthe file across directories."),m_data->szFileName.ptr());
//		setText(m_data->szFileName.ptr());
//		return;
//	}
//	QFileInfo fi(text);
//	if(fi.exists()){
//		g_pApp->warningBox(__tr("Unable to rename file %s.\nThe file %s already exists."),m_data->szFileName.ptr(),tmp.ptr());
//		setText(m_data->szFileName.ptr());
//	} else {
//		KviStr newPath = m_data->szDirPath;
//		newPath.ensureLastCharIs('/');
////		if(!newPath.lastCharIs('/'))newPath.append('/');
//		newPath.append(text);
//		if(!kvi_renameFile(m_data->szAbsFilePath.ptr(),newPath.ptr()))
//		{
//			g_pApp->warningBox(__tr("Unable to rename file %s.\nProbably the destination already exists."),m_data->szFileName.ptr());
//			// failed
//			setText(m_data->szFileName.ptr());
//		} else {
//			m_data->szAbsFilePath = newPath;
//			m_data->szFileName = text;
//		}
//	}
//}


KviDirectoryBrowserWidget::KviDirectoryBrowserWidget(QWidget * parent,KviFrame * lpFrm,const char *dirname)
:QWidget(parent)
{
	m_pFrm = lpFrm;
	setBackgroundMode(NoBackground);
	m_pToolBar      = new QToolBar("toolbar",lpFrm,this);
//	m_pToolBar->setFocusPolicy(QWidget::NoFocus);
	m_pToolBar->setBackgroundMode(QWidget::PaletteBackground);

	m_pCdUp         = new QToolButton(*g_pBrowserIcons[KVI_BROWSER_ICON_CDUP],__tr("Go Up"),
		__tr("Go one directory up"),this,SLOT(cdUp()),m_pToolBar);
	m_pCdUp->setUsesBigPixmap(true);
	m_pCdUp->setFocusPolicy(QWidget::NoFocus);
	m_pCdUp->setUsesBigPixmap(g_pOptions->m_bUseBigToolbarPixmaps);
	m_pCdUp->setUsesTextLabel(g_pOptions->m_bUseTextToolbarComment);

	QToolButton * b = new QToolButton(*g_pBrowserIcons[KVI_BROWSER_ICON_RELOAD],__tr("Reload"),
		__tr("Reload current directory"),this,SLOT(cdReload()),m_pToolBar);
	b->setUsesBigPixmap(true);
	b->setFocusPolicy(QWidget::NoFocus);
	b->setUsesBigPixmap(g_pOptions->m_bUseBigToolbarPixmaps);
	b->setUsesTextLabel(g_pOptions->m_bUseTextToolbarComment);

	b               = new QToolButton(*g_pBrowserIcons[KVI_BROWSER_ICON_HOME],__tr("Home"),
		__tr("Enter your home directory"),this,SLOT(cdHome()),m_pToolBar);
	b->setUsesBigPixmap(true);
	b->setFocusPolicy(QWidget::NoFocus);
	b->setUsesBigPixmap(g_pOptions->m_bUseBigToolbarPixmaps);
	b->setUsesTextLabel(g_pOptions->m_bUseTextToolbarComment);

	b               = new QToolButton(*g_pBrowserIcons[KVI_BROWSER_ICON_KVIRCHOME],__tr("KVIrc Home"),
		__tr("Enter local KVIrc directory"),this,SLOT(cdKviHome()),m_pToolBar);
	b->setUsesBigPixmap(true);
	b->setFocusPolicy(QWidget::NoFocus);
	b->setUsesBigPixmap(g_pOptions->m_bUseBigToolbarPixmaps);
	b->setUsesTextLabel(g_pOptions->m_bUseTextToolbarComment);

	m_pShowBookmarks = new QToolButton(*g_pBrowserIcons[KVI_BROWSER_ICON_BOOKMARK],__tr("Bookmarks"),
		__tr("Show bookmarks list"),this,SLOT(showBookmarksPopup()),m_pToolBar);
	m_pShowBookmarks->setUsesBigPixmap(true);
	m_pShowBookmarks->setFocusPolicy(QWidget::NoFocus);
	m_pShowBookmarks->setUsesBigPixmap(g_pOptions->m_bUseBigToolbarPixmaps);
	m_pShowBookmarks->setUsesTextLabel(g_pOptions->m_bUseTextToolbarComment);


	QFrame *f = new QFrame(m_pToolBar);
	m_pToolBar->setStretchableWidget(f);

	m_pBookmarksPopup = new QPopupMenu();

	m_pDirView      = new KviDirectoryView(this,m_pFrm);
	m_pDirView->setFocusPolicy(QWidget::ClickFocus);
	m_pDirView->viewport()->setFocusPolicy(QWidget::ClickFocus);
	connect(m_pDirView,SIGNAL(doubleClicked(QIconViewItem *)),this,SLOT(itemDoubleClicked(QIconViewItem *)));
//	connect(m_pDirView,SIGNAL(clicked(QIconViewItem *)),this,SLOT(itemClicked(QIconViewItem *it)));
	connect(m_pDirView,SIGNAL(selectionChanged()),this,SLOT(selectionChanged()));
	connect(m_pDirView,SIGNAL(needUpdate()),this,SLOT(cdReload()));
	m_pDirPathEdit  = new QLineEdit(this);
	m_pDirPathEdit->setText("/");
	m_pDirPathEdit->setFocusPolicy(QWidget::ClickFocus);
	connect(m_pDirPathEdit,SIGNAL(returnPressed()),this,SLOT(returnPressed()));
	m_szDirPath     = "/";
//	m_pJob          = 0;

	m_pToolBar->show();

	g_pApp->refreshMimeTypeIcons();
	g_pThreadEventDispatcher->registerObject(this);

	if(!loadDirectory(dirname))
	{
		// failed to read the first directory
		if(dirname)loadDirectory(0); // try the last browsd , or home
		else debug("ooops....can't find a suitable start directory");
	}

}

//============ ~KviDirectoryBrowser ============//

KviDirectoryBrowserWidget::~KviDirectoryBrowserWidget()
{
	stop();
	g_pThreadEventDispatcher->unregisterObject(this);
	g_pOptions->m_szLastBrowsedDirectory = m_szDirPath;
	delete m_pBookmarksPopup;
}

bool KviDirectoryBrowserWidget::loadDirectory(const char *dirname)
{
	stop();
	if(setDirPathAndEnsureReadable(dirname))
	{
		m_pDirPathEdit->setText(m_szDirPath.ptr());
		if(beginReadCurrentDirectory())
		{
			g_pOptions->m_szLastBrowsedDirectory = m_szDirPath;
			m_pDirView->m_szDirPath = m_szDirPath;
			return true;
		}
	}
	return false;
}

bool KviDirectoryBrowserWidget::setDirPathAndEnsureReadable(const char *dirname)
{
	if(dirname)
	{
		if(isDirReadable(dirname))
		{
			m_szDirPath = dirname;
			return true;
		} else return false; // tried ,but had no success
	}
	// dirname is 0...try the defaults
	if(isDirReadable(g_pOptions->m_szLastBrowsedDirectory.ptr()))
	{
		m_szDirPath = g_pOptions->m_szLastBrowsedDirectory;
		return true;
	}
	// last browsed dir disappeared ?
	KviStr tmp = QDir::homeDirPath();
	if(isDirReadable(tmp.ptr()))
	{
		m_szDirPath = tmp;
		return true;
	}
	if(isDirReadable("/"))
	{
		m_szDirPath = "/";
		return true;
	}
	return false;
}

bool KviDirectoryBrowserWidget::isDirReadable(const char *dirname)
{
	QFileInfo fi(dirname);
	return (fi.exists() && fi.isDir() && fi.isReadable());
}


static void kvi_killDirObject(void *data)
{
	closedir((DIR *)data);
}

static void kvi_killDirJob(void * data)
{
	if(((KviDirBrowserJob *)data)->entries)delete ((KviDirBrowserJob *)data)->entries;
	delete ((KviDirBrowserJob *)data);
}

static void * kvi_syncReadDirectory(void * data)
{
	kvi_threadInitialize();
	// copy the data to avoid sharing problems
	KviDirBrowserJob * job      = (KviDirBrowserJob *)data;
	QWidget * pParent           = job->parent;
//	pthread_mutex_t * pJobMutex = job->pJobMutex;

	DIR	     *dir;
    dirent   *file;

//	debug("Entering directory %s",job->szDirPath.ptr());

	dir = opendir(job->szDirPath.ptr());
	job->entries = new QList<KviDirBrowserEntry>;
	job->entries->setAutoDelete(true);

	if(dir){
//		debug("God directory descriptor");
		kvi_threadCleanupPush(kvi_killDirJob,job);
		kvi_threadCleanupPush(kvi_killDirObject,dir);

		while((file = readdir(dir)))
		{
//			debug("Readed a file");
			kvi_threadTestCancel();
			KviDirBrowserEntry * e = new KviDirBrowserEntry;
			e->szAbsFilePath = job->szDirPath.ptr();
			e->szAbsFilePath.ensureLastCharIs('/');
			e->szAbsFilePath.append(file->d_name);
			struct stat st;
			if(lstat(e->szAbsFilePath.ptr(),&st) != 0)
			{
				// Can not stat the file
//				debug("Can't stat it");
				delete e;
				continue;
			}
			e->szFileName    = file->d_name;
			e->szDirPath     = job->szDirPath.ptr();
			if(kvi_strEqualCS(e->szFileName.ptr(),".") && S_ISDIR(st.st_mode))
			{
//				debug("Current dir");
				// current dir inode
				delete e;
				continue;
			}
			if(!(job->bShowHidden))
			{
				if(e->szFileName.firstCharIs('.'))
				{
					// Unwanted hidden file
					delete e;
					continue;
				}
			}
			e->uSize         = st.st_size;
			e->bIsDirectory  = S_ISDIR(st.st_mode);
			e->bIsReadable   = (getuid() == 0) || (S_IROTH & st.st_mode) || ((S_IRGRP & st.st_mode) && (getgid() == st.st_gid)) || ((S_IRUSR & st.st_mode) && (getuid() == st.st_uid));
			e->bIsSymLink    = S_ISLNK(st.st_mode);
			e->bIsFile       = S_ISREG(st.st_mode);
			e->bIsSocket     = S_ISFIFO(st.st_mode) || S_ISSOCK(st.st_mode);
			e->bIsDevice     = S_ISBLK(st.st_mode) || S_ISCHR(st.st_mode);
			if(e->bIsFile){
//				debug("A reg file...looking for mime(%s)",e->szAbsFilePath.ptr());
				KviMimeType * m  = g_pOptions->m_pMimeManager->findMatch(e->szAbsFilePath.ptr(),true);
//				debug("Mime found");
				e->szMimeName    = m->mimeName;
				e->szCommandline = m->commandline;
				e->szIconPath    = m->iconPath;
			} else if(e->bIsSymLink){
				// read the link
				char buffer[256];
				int len = readlink(e->szAbsFilePath.ptr(),buffer,256);
				if(len > -1)
				{
					e->szMimeName.setStr(buffer,len);
					if(stat(e->szAbsFilePath.ptr(),&st) == 0)
					{
						e->bIsDirectory = S_ISDIR(st.st_mode);
					}
				}
			}
			// sort
			if(job->bGroupByExtension)
			{
				if(job->bSortBySize)
				{
					// Group by extension then sort by size in each group
					// Directories first , then files without extension , then extensions in alphabetic order
					// Inside each group of extensions sort by size
					int idx = e->szFileName.findLastIdx('.');
					KviStr tmp(KviStr::Format,"%u",e->uSize);
					if(idx >= 1){ // no hidden files
						// has extension
						if(e->bIsDirectory)e->szKey.sprintf("1 %s %s",e->szFileName.ptr() + idx,file->d_name);
						else e->szKey.sprintf("3 %s %d %u",e->szFileName.ptr() + idx,tmp.len(),e->uSize);
//						e->szKey.sprintf("%s %s %d %u",e->bIsDirectory ? "1" : "3",,tmp.len(),e->uSize);
					} else {
						// no extension
						if(e->bIsDirectory)e->szKey.sprintf("1 _ %s",file->d_name);
						else e->szKey.sprintf("2 _ %d %u",tmp.len(),e->uSize);
//						e->szKey.sprintf("%s _ %d %u",e->bIsDirectory ? "1" : "2",tmp.len(),e->uSize);
					}
				} else {
					int idx = e->szFileName.findLastIdx('.');
					if(idx >= 1){ // no hidden files
						e->szKey.sprintf("%s %s %s",e->bIsDirectory ? "1" : "3",e->szFileName.ptr() + idx,file->d_name);
					} else {
						e->szKey.sprintf("%s _ %s",e->bIsDirectory ? "1" : "2",file->d_name);
					}
				}
			} else {
				if(job->bSortBySize)
				{
					// Sort by size : directories first , then by size string length then by size string
					KviStr tmp(KviStr::Format,"%u",e->uSize);
					if(e->bIsDirectory)e->szKey.sprintf("1 %s",file->d_name);
					else e->szKey.sprintf("2 %d %u",tmp.len(),e->uSize);
				} else {
					// Sort by name : directories first , then by name
					e->szKey.sprintf("%s %s",e->bIsDirectory ? "1" : "2",file->d_name);
				}
			}
			if(job->bCaseInsensitive)e->szKey.toLower();
//			debug("Appending entry");
			job->entries->append(e);
		}


		kvi_threadCleanupPop(1); // will close the dir
		kvi_threadCleanupPop(0); // will do nothing

		kvi_threadTestCancel(); //ensure that we're alive
	}

	QCustomEvent *ev = new QCustomEvent(QEvent::User,job);
	g_pThreadEventDispatcher->postEvent(ev,pParent);
	return 0;
}

bool KviDirectoryBrowserWidget::beginReadCurrentDirectory()
{
	__range_invalid(m_pJob);
	m_pJob                    = new KviDirBrowserJob;
	m_pJob->szDirPath         = m_szDirPath;
	m_pJob->parent            = this;
	m_pJob->entries           = 0;
	m_pJob->bShowHidden       = g_pOptions->m_bDirBrowserShowHiddenFiles;
	m_pJob->bGroupByExtension = g_pOptions->m_bDirBrowserGroupFilesByExtension;
	m_pJob->bSortBySize       = g_pOptions->m_bDirBrowserSortBySize;
	m_pJob->bCaseInsensitive  = g_pOptions->m_bDirBrowserSortCaseInsensitive;
	// m_pJob is readonly from now on
	// no mutex should be needed
	if(kvi_threadCreate(&m_childThread,0,kvi_syncReadDirectory,m_pJob) != 0){
		delete m_pJob;
		m_pJob = 0;
		debug("Failed to start the I/O child thread");
		return false;
	}
	// m_pJob is owned by the child thread now , until the use event is received
	return true;
}

void KviDirectoryBrowserWidget::stop()
{
	if(!m_pJob)return; // no need to stop (readonly access if the child thread is running: no mutex needed)
	// kill the running thread
	kvi_threadCancel(m_childThread);
	m_pJob = 0; // this is dead for us
}


bool KviDirectoryBrowserWidget::event(QEvent *e)
{
	if(e->type() == QEvent::User){
		// Ensure that we're in sync
		if(m_pJob != ((KviDirBrowserJob *)((QCustomEvent *)e)->data())){
			// Cancelled!...
			// Do not leak the job
			KviDirBrowserJob * j = (KviDirBrowserJob *)((QCustomEvent *)e)->data();
			if(j->entries)delete j->entries;
			delete j;
			return true;
		}
		if(m_pJob->entries){
			m_pDirView->clear();
			KviDirectoryViewItem * it = 0;
			m_pJob->entries->setAutoDelete(false);
			for(KviDirBrowserEntry * e = m_pJob->entries->first();e;e=m_pJob->entries->next())
			{
				it = new KviDirectoryViewItem(m_pDirView,e);
				it->setKey(e->szKey.ptr());
			}
			delete m_pJob->entries;
		}
		delete m_pJob;
		m_pJob = 0;
		return true;
	} else return QWidget::event(e);
}

//================ resizeEvent ===============//

void KviDirectoryBrowserWidget::resizeEvent(QResizeEvent *)
{
	int hght  = m_pToolBar->sizeHint().height();
	int hght2 = m_pDirPathEdit->sizeHint().height();
	m_pToolBar->setGeometry(0,0,width(),hght);
	m_pDirPathEdit->setGeometry(0,hght,width(),hght2);
	m_pDirView->setGeometry(0,hght + hght2,width(),height() - (hght + hght2));
}

void KviDirectoryBrowserWidget::cdUp()
{
	QDir d(m_szDirPath.ptr());
	d.cdUp();
	m_szDirPath = d.absPath();
	loadDirectory(m_szDirPath.ptr());
}

void KviDirectoryBrowserWidget::itemDoubleClicked(QIconViewItem * it)
{
	KviDirectoryViewItem * i = (KviDirectoryViewItem *)it;
	if(!(i->m_data->bIsReadable))return;
	if(i->m_data->bIsDirectory){
		QDir d(m_szDirPath.ptr());
		d.cd(i->m_data->szFileName.ptr());
		m_szDirPath = d.absPath();
		loadDirectory(m_szDirPath.ptr());
	} else if(i->m_data->bIsFile){
		g_pApp->executeFileWithCommand(i->m_data->szCommandline.ptr(),i->m_data->szAbsFilePath.ptr(),m_pFrm);
	}
}

void KviDirectoryBrowserWidget::selectionChanged()
{
	QList<KviDirectoryViewItem> l;
	l.setAutoDelete(false);
	for(KviDirectoryViewItem * it = (KviDirectoryViewItem *)m_pDirView->firstItem();it;it=(KviDirectoryViewItem *)it->nextItem())
	{
		if(it->isSelected())l.append(it);
	}

	if(l.count() <= 0)return;
	if(l.count() == 1){
		KviStr tmp;
		KviDirectoryViewItem *it = l.first();
		if(it->m_data->bIsReadable)
		{
			if(it->m_data->bIsSymLink){
				if(it->m_data->bIsDirectory)
				{
					tmp.sprintf(__tr("%s [Symbolic link to %s (Directory)][Double-click to open]"),
						it->m_data->szFileName.ptr(),it->m_data->szMimeName.ptr());
				} else {
					tmp.sprintf(__tr("%s [Symbolic link to %s]"),
						it->m_data->szFileName.ptr(),it->m_data->szMimeName.ptr());
				}
			} else if(it->m_data->bIsDirectory){
				tmp.sprintf(__tr("%s [Directory][Double-click to open]"),it->m_data->szFileName.ptr());
			} else if(it->m_data->bIsFile){
				// Regular file
				tmp.sprintf(__tr("%s [%s][%u bytes][Double-click: %s]"),
					it->m_data->szFileName.ptr(),it->m_data->szMimeName.ptr(),it->m_data->uSize,
					it->m_data->szCommandline.hasData() ? it->m_data->szCommandline.ptr() : __tr("No command specified"));
			} else if(it->m_data->bIsSocket){
				tmp.sprintf(__tr("%s [Socket or FIFO]"),
					it->m_data->szFileName.ptr());
			} else if(it->m_data->bIsDevice){
				tmp.sprintf(__tr("%s [Block or character device]"),
					it->m_data->szFileName.ptr());
			} else {
				tmp.sprintf(__tr("%s [Unknown][%u bytes]"),
					it->m_data->szFileName.ptr(),it->m_data->uSize);
			}
			m_pFrm->m_pStatusBar->tempText(tmp.ptr());
		} else {
			// Not readable
			if(it->m_data->bIsDirectory)
			{
				tmp.sprintf(__tr("%s [Unreadable directory][%u bytes]"),
					it->m_data->szFileName.ptr(),it->m_data->uSize);
			} else {
				tmp.sprintf(__tr("%s [Unreadable file][%u bytes]"),
					it->m_data->szFileName.ptr(),it->m_data->uSize);
			}
		}
	} else {
		unsigned int mb = 0;
		unsigned int kb = 0;
		unsigned int bytes = 0;
		unsigned int by;
		for(KviDirectoryViewItem * it = l.first();it;it=l.next())
		{
			if(!it->m_data->bIsDirectory){
				by = it->m_data->uSize;
				mb += by / (1024 * 1024);
				by = by % (1024 * 1024);
				kb += by / 1024;
				bytes += by % 1024;
				if(bytes > 1024)
				{
					kb += bytes / 1024;
					bytes = bytes % 1024;
				}
				if(kb > 1024)
				{
					mb += kb / 1024;
					kb = kb % 1024;
				}
			}
		}
		KviStr tmp(KviStr::Format,__tr("Multiple selection [%u files][%u mb %u kb %u bytes]"),l.count(),mb,kb,bytes);
		m_pFrm->m_pStatusBar->tempText(tmp.ptr());
	}
}

void KviDirectoryBrowserWidget::cdReload()
{
	loadDirectory(m_szDirPath.ptr());
}

void KviDirectoryBrowserWidget::cdHome()
{
	KviStr tmp = QDir::homeDirPath();
	loadDirectory(tmp.ptr());
}

void KviDirectoryBrowserWidget::cdKviHome()
{
	KviStr tmp;
	g_pApp->getLocalKVircDirectory(tmp);
	loadDirectory(tmp.ptr());
}

void KviDirectoryBrowserWidget::returnPressed()
{
	KviStr tmp = m_pDirPathEdit->text();
	if(kvi_strEqualCIN("file:/",tmp.ptr(),6))tmp.cutLeft(6);
	loadDirectory(tmp.ptr());
}

void KviDirectoryBrowserWidget::showBookmarksPopup()
{
	m_pBookmarksPopup->clear();
	m_pBookmarksPopup->insertItem(__tr("&Add Bookmark"));
	m_pBookmarksPopup->insertSeparator();
	for(KviStr * s = g_pOptions->m_pDirBookmarks->first();s;s = g_pOptions->m_pDirBookmarks->next()){
		m_pBookmarksPopup->insertItem(_CHAR_2_QSTRING(s->ptr()));
	}
	m_pBookmarksPopup->popup(m_pShowBookmarks->mapToGlobal(QPoint(0,m_pShowBookmarks->height())));
	connect(m_pBookmarksPopup,SIGNAL(activated(int)),this,SLOT(bookmarkSelected(int)));
}

void KviDirectoryBrowserWidget::applyOptions()
{
	m_pDirView->applyOptions();
}

void KviDirectoryBrowserWidget::bookmarkSelected(int id)
{
	KviStr tmp=m_pBookmarksPopup->text(id);
	if(kvi_strEqualCSN("/",tmp.ptr(),1))
	{
		// A path item
		loadDirectory(tmp.ptr());
	} else {
		// Non path item , "Add to bookmarks"
		// Add to bookmarks the current path
		// Avoid double entries
		KviStr tmp = m_szDirPath;
		if((tmp.len() > 1) && (tmp.lastCharIs('/')))tmp.cutRight(1);
		tmp.stripWhiteSpace();
		if(tmp.isEmpty())return;
		for(KviStr * s = g_pOptions->m_pDirBookmarks->first();s;s = g_pOptions->m_pDirBookmarks->next())
		{
			if(kvi_strEqualCS(s->ptr(),m_szDirPath.ptr()))return;
		}
		if(g_pOptions->m_pDirBookmarks->count()  == KVI_DIRBROWSER_MAX_BOOKMARKS)
		{
			g_pOptions->m_pDirBookmarks->removeFirst();
		}
		g_pOptions->m_pDirBookmarks->append(new KviStr(m_szDirPath));
	}
}


///////////////////////


KviDirectoryBrowser::KviDirectoryBrowser(KviFrame *lpFrm,const char *dirname)
:KviWindow(KVI_DIRECTORYBROWSER_WINDOW_NAME,KVI_WND_TYPE_DIRBROWSER,lpFrm)
{
	m_pBrowser = new KviDirectoryBrowserWidget(this,lpFrm,dirname);

	setFocusHandler(m_pBrowser->m_pDirPathEdit);
}

//============ ~KviDirectoryBrowser ============//

KviDirectoryBrowser::~KviDirectoryBrowser()
{
}

void KviDirectoryBrowser::saveProperties()
{
	KviWindowProperty p;
	p.rect = externalGeometry();
	p.isDocked = isAttached();
	p.splitWidth1 = 0;
	p.splitWidth2 = 0;
	p.timestamp = false;
	p.imagesVisible = false;
	p.isMaximized = isAttached() && isMaximized();
	p.topSplitWidth1 = 0;
	p.topSplitWidth2 = 0;
	p.topSplitWidth3 = 0;
	g_pOptions->m_pWinPropertiesList->setProperty(caption(),&p);
}

void KviDirectoryBrowser::setProperties(KviWindowProperty *)
{
}


QPixmap * KviDirectoryBrowser::myIconPtr()
{
	return g_pixViewOut[KVI_OUT_WND_DIRBROWSER];
}

//=============== applyOptions ================//

void KviDirectoryBrowser::applyOptions()
{
	m_pBrowser->applyOptions();
}

//================ resizeEvent ===============//

void KviDirectoryBrowser::resizeEvent(QResizeEvent *)
{
	m_pBrowser->setGeometry(0,0,width(),height());
}


#include "m_kvi_dirbrowser.moc"
