//
//   File : kvi_input.cpp (/usr/build/NEW_kvirc/kvirc/kvirc/kvi_input.cpp)
//   Last major modification : Sun Jan 3 1999 23:11:50 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#define _KVI_DEBUG_CHECK_RANGE_
#include "kvi_debug.h"

#define _KVI_INPUT_CPP_

#include "kvi_input.h"

#include "kvi_options.h"
#include "kvi_app.h"
#include "kvi_ircview.h"
#include "kvi_colorwin.h"
#include "kvi_window.h"
#include "kvi_uparser.h"
#include "kvi_locale.h"
#include "kvi_listbox.h"
#include "kvi_ircuser.h"
#include "kvi_frame.h"
#include "kvi_event.h"
#include "kvi_statusbar.h"
#include "kvi_strsub.h"

#include <X11/Xlib.h>
#ifdef COMPILE_USE_AA_FONTS
	#include <X11/Xft/Xft.h>
#endif

#include <ctype.h>
#include <stdlib.h>

#include <qnamespace.h>
#include <qpopupmenu.h>
#include <qclipboard.h>
#include <qdragobject.h>
#include <qstringlist.h>


/*
	@quickhelp: KviInput
	@widget: Commandline input

	This is the commandline input widget.<br>
	<docsubtitle>Default key bindings:</docsubtitle><br>
	Ctrl+B : Inserts the 'bold' mIRC control character<br>
	Ctrl+K : Inserts the 'color' mIRC control character<br>
	Ctrl+R : Inserts the 'reverse' mIRC control character<br>
	Ctrl+U : Inserts the 'underline' mIRC control character<br>
	Ctrl+O : Inserts the 'reset' mIRC control character<br>
	Ctrl+C : Copy selected text to clipboard<br>
	Ctrl+X : Cuts the selected text<br>
	Ctrl+V : Paste the clipboard contents (same as middle mouse click)<br>
	CursorUp : Moves backward in the commandline history<br>
	CursorDown : Moves forward in the commandline history<br>
	CursorRight : Moves cursor right<br>
	CursorLeft : Moves cursor left :)<br>
	Shift+CursorLeft : Moves selection left<br>
	Shift+RightCursor : Moves selection right<br>
	Tab : Nick completion or function/command completion (see below)<br>
	Shift+Tab : Mask completion or function/command completion (see below)<br>
	Alt+&lt;numeric_sequence&gt; : Inserts the character by ASCII code<br>
	<example>
	Alt+32 : Inserts the ASCII character 32 = space
	Alt+00032 : Same as above :)
	Alt+13 : Inserts the Carriage Return (CR) control character
	Alt+77 : Inserts the ASCII character 77 : M
	</example>
	Take a look also at the <a href="shortcuts.kvihelp">global shortcuts</a> description.<br>
	If you drop a file on this widget , a <a href="parse.kvihelp">/PARSE &lt;filename&gt;</a> will be executed.<br>
	You can enable word substitution in the misc options dialog.<br>
	For example, if you choose to substitute "afaik" with "As far as I know",<br>
	when you will type "afaik" somewhere in the commandline , and then
	press space or return , that word will be replaced with "As far as I know".<br>
	Experiment with it :)<br>
	The Tab key activates the completion of the current word.<br>
	If the word start with a '/' sign , it is treated as a command to be completed,
	if it starts with a '$' sign it is treated as a function or identifier to be completed,
	otherwise it is treated as a nickname to be completed.<br>
	<example>
		/ec&lt;tab&gt; will produce /echo&lt;space&gt
		/echo $loca&lt;tab&gt; will produce /echo $localhost
	</example>
	Multiple matches are listed in the view window and the word is completed
	to the common part of all the matches.<br>
	<example>
		$sel&lt;tab;&gt; will find multiple matches and produce $selected
	</example>
	Experiment with that too :)
*/

//This comes from kvi_app.cpp
extern KviColorWindow  *g_pColorWindow;
extern QPopupMenu      *g_pInputPopup;
extern KviEventManager *g_pEventManager;

// This comes from the KviIrcView class
extern const char * getColorBytes(const char *data_ptr,char *byte_1,char *byte_2);

#ifdef COMPILE_USE_AA_FONTS
//	extern XftDraw * g_pInputFtDraw;
	extern XftFont        * g_pXftFont;
	extern XftDraw        * g_pXftDraw;
	extern int qt_use_xft (void); // qpainter_x11.cpp
	extern void *qt_ft_font (const QFont *f); // qfont_x11.cpp
	extern XftDraw * qt_lookup_ft_draw (Drawable draw, bool paintEventClipOn, QRegion *crgn);
#endif

//=============== Construktor ==============//

KviInput::KviInput(KviWindow *parent,KviUserParser *pParser,KviListBox *listBox)
:QWidget(parent,"input")
{
	m_szTextBuffer         = "";                            //Main data buffer
	m_szSaveTextBuffer     = "";                            //Save the data while browsing history
	m_pMemBuffer           = new QPixmap(width(),height()); //Pixmap buffer for painting
	m_iCursorPosition      = 0;                             //Index of the char AFTER the cursor
	m_iFirstVisibleChar    = 0;                             //Index of the first visible character
	m_iSelectionBegin      = -1;                            //Index of the first char in the selection
	m_iSelectionEnd        = -1;                            //Index of the last char in the selection
	m_bCursorOn            = false;                         //Cursor state
	m_iCursorTimer         = 0;                             //Timer that iverts the cursor state
	m_iDragTimer           = 0;                             //Timer for drag selection updates
	m_iLastCursorXPosition = KVI_INPUT_BORDER;              //Calculated in paintEvent
	m_iSelectionAnchorChar = -1;                            //Character clicked at the beginning of the selection process
	m_pHistory             = new QList<KviStr>;             //History buffer
	m_pHistory->setAutoDelete(true);
	m_iCurHistoryIdx       = -1;                            //No data in the history
	m_bUpdatesEnabled      = true;
	m_pUserParser          = pParser;
	m_pParent              = parent;
	m_szAltKeyCode         = "";
	m_pListBoxToControl    = listBox;
	m_iLastCompletionIndex = -1;
	m_szStringBeforeSubstitution = "";
	m_bLastEventWasASubstitution = false;
	setBackgroundMode(NoBackground);
	setFocusPolicy(StrongFocus);
	setAcceptDrops(true);
}

//================= Destruktor ================//

KviInput::~KviInput()
{
	delete m_pHistory;
	if(m_iCursorTimer)killTimer(m_iCursorTimer);
	killDragTimer();
	delete m_pMemBuffer;
}

void KviInput::dragEnterEvent(QDragEnterEvent *e)
{
	if(QUriDrag::canDecode(e)){
		e->accept(true);
		m_pParent->m_pFrm->m_pStatusBar->tempText(__tr("Drop the file to /PARSE it"),5000);
	} else e->accept(false);
}

void KviInput::dropEvent(QDropEvent *e)
{
	QStringList list;
	if(QUriDrag::decodeLocalFiles(e,list)){
		if(!list.isEmpty()){
			QStringList::ConstIterator it = list.begin(); //kewl ! :)
    		for( ; it != list.end(); ++it ){
				KviStr tmp = *it; //wow :)
				if(*(tmp.ptr()) != '/')tmp.prepend("/"); //HACK HACK HACK for Qt bug (?!?)
				tmp.prepend("/PARSE ");
				m_pUserParser->parseUserCommand(tmp,m_pParent);
			}
		}
	}
}

int KviInput::heightHint() const
{
	return g_pOptions->m_iInputFontLineSpacing+(KVI_INPUT_BORDER*2);
}

#define KVI_INPUT_DEF_BACK 100
#define KVI_INPUT_DEF_FORE 101

Display * dpy;

void KviInput::paintEvent(QPaintEvent *)
{
	if(!isVisibleToTLW())return;
	dpy                   = m_pMemBuffer->x11Display();
	HANDLE hMemBuffer     = m_pMemBuffer->handle();
	GC gc_aux             = XCreateGC(dpy,hMemBuffer,0,0);
	int widgetWidth       = width();
	int widgetHeight      = height();
	//Draw the background
	XSetLineAttributes(dpy,gc_aux,0,LineSolid,CapButt,JoinMiter);
	if(g_pOptions->m_pixInputBack->isNull()){
		XSetForeground(dpy,gc_aux,g_pOptions->m_clrInputBack.pixel());
		XSetBackground(dpy,gc_aux,g_pOptions->m_clrInputBack.pixel());
		XSetFillStyle(dpy,gc_aux,FillSolid );
	} else {
		XSetTile(dpy,gc_aux,g_pOptions->m_pixInputBack->handle());
		XSetFillStyle(dpy,gc_aux,FillTiled);
	}
   	XFillRectangle(dpy,hMemBuffer,gc_aux,0,0,widgetWidth,widgetHeight);
	XSetFillStyle(dpy,gc_aux,FillSolid);

#ifdef COMPILE_USE_AA_FONTS
	if(qt_use_xft())
	{
		g_pXftFont = (XftFont *)qt_ft_font(&(g_pOptions->m_fntInput));
		g_pXftDraw = qt_lookup_ft_draw (hMemBuffer,false,0);
		if(!g_pXftDraw)
		{
			XSetFont(dpy,gc_aux,g_pOptions->m_fntInput.handle());
			g_pXftFont = 0;
		}
	} else {
#endif
	XSetFont(dpy,gc_aux,g_pOptions->m_fntInput.handle());
#ifdef COMPILE_USE_AA_FONTS
		g_pXftFont = 0;
		g_pXftDraw = 0;
	}
#endif

	int curXPos      = KVI_INPUT_BORDER;
	int maxXPos      = widgetWidth - KVI_INPUT_BORDER;
	m_iCurBack       = KVI_INPUT_DEF_BACK; //transparent
	m_iCurFore       = KVI_INPUT_DEF_FORE; //normal fore color
	m_bCurBold       = false;
	m_bCurUnderline  = false;
	int bottom       = widgetHeight-(KVI_INPUT_BORDER+g_pOptions->m_iInputFontBaseLineOffset);

	register char *p = runUpToTheFirstVisibleChar();
	int charIdx = m_iFirstVisibleChar;

	XRectangle r[1];
	r[0].x=KVI_INPUT_BORDER;
	r[0].y=KVI_INPUT_BORDER;
	r[0].width=widgetWidth-(KVI_INPUT_BORDER * 2);
	r[0].height=widgetHeight-(KVI_INPUT_BORDER * 2)+1;
	XSetClipRectangles(dpy,gc_aux,0,0,r,1,Unsorted);

	//Control the selection state
	if((m_iSelectionEnd < m_iSelectionBegin)||(m_iSelectionEnd == -1)||(m_iSelectionBegin == -1)){
		m_iSelectionEnd = -1;
		m_iSelectionBegin = -1;
	}

	while(extractNextBlock(p,curXPos,maxXPos) && (curXPos < maxXPos)){
		if(m_bControlBlock){
			//Only a control char
			int topLine = widgetHeight-(KVI_INPUT_BORDER+g_pOptions->m_iInputFontHeight);

			if((charIdx >= m_iSelectionBegin) && (charIdx <= m_iSelectionEnd)){
				//Paint the selected control char
				XSetForeground(dpy,gc_aux,g_pOptions->m_clrInputSeleBack.pixel());
				XSetBackground(dpy,gc_aux,g_pOptions->m_clrInputSeleBack.pixel());
				XFillRectangle(dpy,hMemBuffer,gc_aux,curXPos,topLine,m_iBlockWidth,g_pOptions->m_iInputFontHeight);
			}
			XSetForeground(dpy,gc_aux,g_pOptions->m_clrInputCursor.pixel());
			char s=getSubstituteChar(*p);
#ifdef COMPILE_USE_AA_FONTS
			if(g_pXftFont)
			{
				XftColor color;
				QColor * clr = &(g_pOptions->m_clrInputCursor);
				color.color.red = clr->red() | clr->red() << 8;
				color.color.green = clr->green() | clr->green() << 8;
				color.color.blue = clr->blue() | clr->blue() << 8;
				color.color.alpha = 0xffff;
				color.pixel = clr->pixel();
				XftDrawString8(g_pXftDraw,&color,g_pXftFont,curXPos + 2,bottom,(unsigned char *)&s,1);
			} else
#endif
			XDrawString(dpy,hMemBuffer,gc_aux,curXPos+2,bottom,&s,1);
			XSetLineAttributes(dpy,gc_aux,1,LineSolid,CapButt,JoinMiter);
			XDrawLine(dpy,hMemBuffer,gc_aux,curXPos,topLine,curXPos+m_iBlockWidth-1,topLine);
			XDrawLine(dpy,hMemBuffer,gc_aux,curXPos,topLine,curXPos,topLine+g_pOptions->m_iInputFontHeight);
			XDrawLine(dpy,hMemBuffer,gc_aux,curXPos,topLine+g_pOptions->m_iInputFontHeight,curXPos+m_iBlockWidth,topLine+g_pOptions->m_iInputFontHeight);
			XDrawLine(dpy,hMemBuffer,gc_aux,curXPos+m_iBlockWidth-1,topLine,curXPos+m_iBlockWidth-1,topLine+g_pOptions->m_iInputFontHeight);
			p       += m_iBlockLen;
			curXPos += m_iBlockWidth;
			charIdx += m_iBlockLen;
		} else {
			//Paint a normal string
			//First check if it is in the selection
			if((m_iSelectionBegin != -1)&&(m_iSelectionEnd != -1)&&
				(charIdx <= m_iSelectionEnd)&&(charIdx+m_iBlockLen > m_iSelectionBegin)){
				//Selection in this block!
				int auxWidth = 0;
				int auxLen   = 0;
				char *aux_ptr=p;
				if(charIdx < m_iSelectionBegin){
					//But have non selected stuff before
					while((charIdx < m_iSelectionBegin)&&(auxLen < m_iBlockLen)){
						auxWidth +=g_pOptions->m_iInputFontCharacterWidth[(unsigned char)*aux_ptr];
						aux_ptr++;
						charIdx++;
						auxLen++;
					}
					drawTextBlock(gc_aux,hMemBuffer,curXPos,bottom,p,auxLen,auxWidth);
					curXPos+=auxWidth;
					p=aux_ptr;
					m_iBlockLen -= auxLen;
					auxWidth = 0;
					auxLen   = 0;
				}
				//Selection
				while((charIdx <= m_iSelectionEnd)&&(auxLen < m_iBlockLen)){
					auxWidth +=g_pOptions->m_iInputFontCharacterWidth[(unsigned char)*aux_ptr];
					aux_ptr++;
					charIdx++;
					auxLen++;
				}
#ifdef COMPILE_USE_AA_FONTS
				if(g_pXftFont)
				{
					XSetForeground(dpy,gc_aux,g_pOptions->m_clrInputSeleBack.pixel());
					XFillRectangle(dpy,hMemBuffer,gc_aux,curXPos,KVI_INPUT_BORDER,auxWidth,widgetHeight - (KVI_INPUT_BORDER * 2));
					XftColor color;
					QColor * clr = &(g_pOptions->m_clrInputSeleFore);
					color.color.red = clr->red() | clr->red() << 8;
					color.color.green = clr->green() | clr->green() << 8;
					color.color.blue = clr->blue() | clr->blue() << 8;
					color.color.alpha = 0xffff;
					color.pixel = clr->pixel();
					XftDrawString8(g_pXftDraw, &color,g_pXftFont,curXPos,bottom,(unsigned char *)p,auxLen);
				} else {
#endif
					XSetForeground(dpy,gc_aux,g_pOptions->m_clrInputSeleFore.pixel());
					XSetBackground(dpy,gc_aux,g_pOptions->m_clrInputSeleBack.pixel());
					XDrawImageString(dpy,hMemBuffer,gc_aux,curXPos,bottom,p,auxLen);
#ifdef COMPILE_USE_AA_FONTS
				}
#endif

				curXPos+=auxWidth;
				p=aux_ptr;
				m_iBlockLen -= auxLen;
				//Last part
				if(m_iBlockLen > 0){
					auxWidth = 0;
					auxLen   = 0;
					while(auxLen < m_iBlockLen){
						auxWidth +=g_pOptions->m_iInputFontCharacterWidth[(unsigned char)*aux_ptr];
						aux_ptr++;
						charIdx++;
						auxLen++;
					}
					drawTextBlock(gc_aux,hMemBuffer,curXPos,bottom,p,auxLen,auxWidth);
					curXPos+=auxWidth;
					p=aux_ptr;
				}
			} else {
				//No selection
				drawTextBlock(gc_aux,hMemBuffer,curXPos,bottom,p,m_iBlockLen,m_iBlockWidth);
				p       += m_iBlockLen;
				curXPos += m_iBlockWidth;
				charIdx += m_iBlockLen;
			}
		}
	}
	XSetLineAttributes(dpy,gc_aux,1,LineSolid,CapButt,JoinMiter);
	XSetClipMask(dpy,gc_aux,None);
	//Now the cursor
	m_iLastCursorXPosition = KVI_INPUT_BORDER;
	m_iBlockLen = m_iFirstVisibleChar;
	p = m_szTextBuffer.ptr()+m_iFirstVisibleChar;
	while(m_iBlockLen < m_iCursorPosition && *p){
		m_iLastCursorXPosition+=g_pOptions->m_iInputFontCharacterWidth[(unsigned char) *p];
		m_iBlockLen++;
		p++;
	}
	if(m_bCursorOn){
		XSetForeground(dpy,gc_aux,g_pOptions->m_clrInputCursor.pixel());
		XDrawLine(dpy,hMemBuffer,gc_aux,m_iLastCursorXPosition,KVI_INPUT_BORDER-1,m_iLastCursorXPosition,(widgetHeight-KVI_INPUT_BORDER)+2);
	} else XSetForeground(dpy,gc_aux,g_pOptions->m_clrInputFore.pixel());
	XDrawLine(dpy,hMemBuffer,gc_aux,m_iLastCursorXPosition-2,KVI_INPUT_BORDER-2,m_iLastCursorXPosition+3,KVI_INPUT_BORDER-2);
	XDrawLine(dpy,hMemBuffer,gc_aux,m_iLastCursorXPosition-3,KVI_INPUT_BORDER-3,m_iLastCursorXPosition+4,KVI_INPUT_BORDER-3);
	XDrawLine(dpy,hMemBuffer,gc_aux,m_iLastCursorXPosition-1,KVI_INPUT_BORDER-1,m_iLastCursorXPosition+2,KVI_INPUT_BORDER-1);
	XDrawLine(dpy,hMemBuffer,gc_aux,m_iLastCursorXPosition,KVI_INPUT_BORDER,m_iLastCursorXPosition+1,KVI_INPUT_BORDER);

	//Need to draw the sunken rect around the view now...
	XSetForeground(dpy,gc_aux,colorGroup().dark().pixel());
	XDrawLine(dpy,hMemBuffer,gc_aux,0,0,widgetWidth,0);
	XDrawLine(dpy,hMemBuffer,gc_aux,0,0,0,widgetHeight);
	XSetForeground(dpy,gc_aux,colorGroup().light().pixel());
	XDrawLine(dpy,hMemBuffer,gc_aux,1,widgetHeight-1,widgetWidth-1,widgetHeight-1);
	XDrawLine(dpy,hMemBuffer,gc_aux,widgetWidth-1,1,widgetWidth-1,widgetHeight);
	//COPY TO THE DISPLAY
	XCopyArea(dpy,hMemBuffer,handle(),gc_aux,0,0,widgetWidth,widgetHeight,0,0);
	XFreeGC(dpy,gc_aux);
}

void KviInput::drawTextBlock(GC gc_aux,HANDLE hMemBuffer,int curXPos,int bottom,char *p,int len,int wdth)
{
#ifdef COMPILE_USE_AA_FONTS
	if(g_pXftFont)
	{
		if(m_iCurBack != KVI_INPUT_DEF_BACK)
		{
			if(((unsigned char)m_iCurBack) > 16)
			{
				XSetForeground(dpy,gc_aux,
					g_pOptions->m_clrInputFore.pixel());
			} else {
				XSetForeground(dpy,gc_aux,
					g_pOptions->m_pMircColor[(unsigned char)m_iCurBack]->pixel());
			}
			XFillRectangle(dpy,hMemBuffer,gc_aux,curXPos,KVI_INPUT_BORDER,wdth,bottom+g_pOptions->m_iInputFontBaseLineOffset - KVI_INPUT_BORDER);
		}

		XftColor color;
		QColor * clr;
		if(m_iCurFore == KVI_INPUT_DEF_FORE)clr = &(g_pOptions->m_clrInputFore);
		else {
			if(((unsigned char)m_iCurFore) > 16)clr = &(g_pOptions->m_clrInputBack);
			else clr = g_pOptions->m_pMircColor[(unsigned char)m_iCurFore];
		}
		color.color.red = clr->red() | clr->red() << 8;
		color.color.green = clr->green() | clr->green() << 8;
		color.color.blue = clr->blue() | clr->blue() << 8;
		color.color.alpha = 0xffff;
		color.pixel = clr->pixel();
		XftDrawString8(g_pXftDraw, &color,g_pXftFont,curXPos,bottom,(unsigned char *)p,len);
		if(m_bCurBold)XftDrawString8(g_pXftDraw, &color,g_pXftFont,curXPos + 1,bottom,(unsigned char *)p,len);

		if(m_bCurUnderline)
		{
			XSetForeground(dpy,gc_aux,clr->pixel());
			XSetLineAttributes(dpy,gc_aux,g_pOptions->m_iInputFontLineWidth,LineSolid,CapButt,JoinMiter);
			XDrawLine(dpy,hMemBuffer,gc_aux,curXPos,bottom+g_pOptions->m_iInputFontBaseLineOffset,curXPos+wdth,bottom+g_pOptions->m_iInputFontBaseLineOffset);
		}
	} else {
#endif
	if(m_iCurFore == KVI_INPUT_DEF_FORE)XSetForeground(dpy,gc_aux,g_pOptions->m_clrInputFore.pixel());
	else if(m_iCurFore > 16)XSetForeground(dpy,gc_aux,g_pOptions->m_clrInputBack.pixel());
	else XSetForeground(dpy,gc_aux,g_pOptions->m_pMircColor[(unsigned char)m_iCurFore]->pixel());
	if(m_iCurBack == KVI_INPUT_DEF_BACK)XDrawString(dpy,hMemBuffer,gc_aux,curXPos,bottom,p,len);
	else {
		if(m_iCurBack > 16)XSetBackground(dpy,gc_aux,g_pOptions->m_clrInputFore.pixel());
		else XSetBackground(dpy,gc_aux,g_pOptions->m_pMircColor[(unsigned char)m_iCurBack]->pixel());
		XDrawImageString(dpy,hMemBuffer,gc_aux,curXPos,bottom,p,len);
	}
	if(m_bCurBold)XDrawString(dpy,hMemBuffer,gc_aux,curXPos+1,bottom,p,len);
	if(m_bCurUnderline){
		XSetLineAttributes(dpy,gc_aux,g_pOptions->m_iInputFontLineWidth,LineSolid,CapButt,JoinMiter);
		XDrawLine(dpy,hMemBuffer,gc_aux,curXPos,bottom+g_pOptions->m_iInputFontBaseLineOffset,curXPos+wdth,bottom+g_pOptions->m_iInputFontBaseLineOffset);
	}
#ifdef COMPILE_USE_AA_FONTS
	}
#endif
}

char KviInput::getSubstituteChar(char control_code)
{
	switch(control_code){
		case KVI_TEXT_COLOR:
			return 'K';
			break;
		case KVI_TEXT_BOLD:
			return 'B';
			break;
		case KVI_TEXT_RESET:
			return 'O';
			break;
		case KVI_TEXT_REVERSE:
			return 'R';
			break;
		case KVI_TEXT_UNDERLINE:
			return 'U';
			break;
		default:
			return control_code;
			break;
	}
}

bool KviInput::extractNextBlock(char *p,int curXPos,int maxXPos)
{
	m_iBlockLen = 0;
	m_iBlockWidth = 0;
	if(*p){
		m_bControlBlock = true;
		//Control code
		switch(*p){
			case KVI_TEXT_BOLD:
				m_iBlockLen = 1;
				m_iBlockWidth = g_pOptions->m_iInputFontCharacterWidth[(unsigned char)*p];
				m_bCurBold = ! m_bCurBold;
				break;
			case KVI_TEXT_UNDERLINE:
				m_iBlockLen = 1;
				m_iBlockWidth = g_pOptions->m_iInputFontCharacterWidth[(unsigned char)*p];
				m_bCurUnderline = ! m_bCurUnderline;
				break;
			case KVI_TEXT_RESET:
				m_iBlockLen = 1;
				m_iBlockWidth = g_pOptions->m_iInputFontCharacterWidth[(unsigned char)*p];
				m_iCurFore = KVI_INPUT_DEF_FORE;
				m_iCurBack = KVI_INPUT_DEF_BACK;
				m_bCurBold = false;
				m_bCurUnderline = false;
				break;
			case KVI_TEXT_REVERSE:{
				m_iBlockLen = 1;
				m_iBlockWidth = g_pOptions->m_iInputFontCharacterWidth[(unsigned char)*p];
				char auxClr = m_iCurFore;
				m_iCurFore  = m_iCurBack;
				m_iCurBack  = auxClr;}
				break;
			case KVI_TEXT_COLOR:{
				m_iBlockLen = 1;
				m_iBlockWidth = g_pOptions->m_iInputFontCharacterWidth[(unsigned char)*p];
				p++;
				char fore;
				char back;
				getColorBytes(p,&fore,&back);
				if(fore != KVI_NOCHANGE)m_iCurFore = fore;
				else m_iCurFore = KVI_INPUT_DEF_FORE;
				if(back != KVI_NOCHANGE)m_iCurBack = back;
				else m_iCurBack = KVI_INPUT_DEF_BACK;}
				break;
			default:
				m_bControlBlock = false;
				//Not a control code...run..
				while((*p) && (curXPos < maxXPos) && (*p != KVI_TEXT_COLOR) &&
					(*p != KVI_TEXT_BOLD) && (*p != KVI_TEXT_UNDERLINE) &&
					(*p != KVI_TEXT_RESET) && (*p != KVI_TEXT_REVERSE)){
					m_iBlockLen++;
					m_iBlockWidth+=g_pOptions->m_iInputFontCharacterWidth[(unsigned char)*p];
					curXPos      +=g_pOptions->m_iInputFontCharacterWidth[(unsigned char)*p];
					p++;
				}
				break;
		}
	} else return false;
	return true;
}

char * KviInput::runUpToTheFirstVisibleChar()
{
	register int idx = 0;
	register char *p = m_szTextBuffer.ptr();
	while(idx < m_iFirstVisibleChar){
		if(*p < 32){
			switch(*p){
				case KVI_TEXT_BOLD:
					m_bCurBold = ! m_bCurBold;
					p++;
					break;
				case KVI_TEXT_UNDERLINE:
					m_bCurUnderline = ! m_bCurUnderline;
					p++;
					break;
				case KVI_TEXT_RESET:
					m_iCurFore = KVI_INPUT_DEF_FORE;
					m_iCurBack = KVI_INPUT_DEF_BACK;
					m_bCurBold = false;
					m_bCurUnderline = false;
					p++;
					break;
				case KVI_TEXT_REVERSE:{
					char auxClr = m_iCurFore;
					m_iCurFore  = m_iCurBack;
					m_iCurBack  = auxClr;
					p++;}
					break;
				case KVI_TEXT_COLOR:{
					p++;
					char fore;
					char back;
					getColorBytes(p,&fore,&back);
					if(fore != KVI_NOCHANGE)m_iCurFore = fore;
					if(back != KVI_NOCHANGE)m_iCurBack = back;}
					break;
				//case '0':
				//	debug("KviInput::Encountered invisible end of the string!");
				//	exit(0);
				//	break;
				default:
					p++;
					break;
			}
		} else p++;
		idx++;
	}
	return p;
}

void KviInput::resizeEvent(QResizeEvent *)
{
	m_pMemBuffer->resize(width(),height());
}

void KviInput::mousePressEvent(QMouseEvent *e)
{
	if(e->button() & LeftButton){
		m_iCursorPosition = charIndexFromXPosition(e->pos().x());
		//move the cursor to
		int anchorX =  xPositionFromCharIndex(m_iCursorPosition);
		if(anchorX > (width()-KVI_INPUT_BORDER))m_iFirstVisibleChar++;
		m_iSelectionAnchorChar = m_iCursorPosition;
		selectOneChar(-1);
		grabMouse(QCursor(crossCursor));
		repaintWithCursorOn();
		killDragTimer();
		m_iDragTimer = startTimer(KVI_INPUT_DRAG_TIMEOUT);
	} else if(e->button() & RightButton){
		//Popup menu
		g_pInputPopup->clear();
		g_pInputPopup->insertItem(_i18n_("Copy"),this,SLOT(copy()),0,1);
		g_pInputPopup->setItemEnabled(1,hasSelection());
		g_pInputPopup->insertItem(_i18n_("Paste"),this,SLOT(paste()),0,2);
		g_pInputPopup->setItemEnabled(2,(g_pApp->clipboard()->text() != 0));
		g_pInputPopup->insertItem(_i18n_("Cut"),this,SLOT(cut()),0,3);
		g_pInputPopup->setItemEnabled(3,hasSelection());
		g_pInputPopup->insertSeparator();
		g_pInputPopup->insertItem(_i18n_("Select All"),this,SLOT(selectAll()),0,4);
		g_pInputPopup->setItemEnabled(4,(m_szTextBuffer.hasData()));
		g_pInputPopup->insertItem(_i18n_("Clear"),this,SLOT(clear()),0,5);
		g_pInputPopup->setItemEnabled(5,(m_szTextBuffer.hasData()));
		g_pInputPopup->popup(mapToGlobal(e->pos()));
	} else paste();
}

bool KviInput::hasSelection()
{
	return ((m_iSelectionBegin != -1)&&(m_iSelectionEnd != -1));
}

void KviInput::copy()
{
	if(hasSelection()){
		g_pApp->clipboard()->setText(
			m_szTextBuffer.middle(m_iSelectionBegin,(m_iSelectionEnd-m_iSelectionBegin)+1).ptr()
		);
	}
	repaintWithCursorOn();
}

void KviInput::moveCursorTo(int idx,bool bRepaint)
{
	if(idx < 0)idx = 0;
	if(idx > m_szTextBuffer.len())idx = m_szTextBuffer.len();
	if(idx > m_iCursorPosition){
		while(m_iCursorPosition < idx){
			moveRightFirstVisibleCharToShowCursor();
			m_iCursorPosition++;
		}
	} else {
		m_iCursorPosition = idx;
		if(m_iFirstVisibleChar > m_iCursorPosition)m_iFirstVisibleChar = m_iCursorPosition;
	}
	if(bRepaint)repaintWithCursorOn();
}

void KviInput::cut()
{
	if(hasSelection()){
		m_szTextBuffer.cut(m_iSelectionBegin,(m_iSelectionEnd-m_iSelectionBegin)+1);
		if((m_iFirstVisibleChar > 0)&&(xPositionFromCharIndex(m_szTextBuffer.len()) <= width()-KVI_INPUT_BORDER))
			m_iFirstVisibleChar-=(m_iSelectionEnd-m_iSelectionBegin)+1;
		moveCursorTo(m_iSelectionBegin,false);
		selectOneChar(-1);
		repaintWithCursorOn();
	}
}

void KviInput::paste()
{
	KviStr szText=g_pApp->clipboard()->text();
	if(szText.hasData()){
		szText.replaceAll('\t'," "); //Do not paste tabs
		m_bUpdatesEnabled = false;
		cut();
		m_bUpdatesEnabled = true;
		if(szText.findFirstIdx('\n')==-1){
			m_szTextBuffer.insert(m_iCursorPosition,szText.ptr());
			if(m_szTextBuffer.len() > KVI_INPUT_MAX_BUFFER_SIZE){
				m_szTextBuffer.setLen(KVI_INPUT_MAX_BUFFER_SIZE);
			}
			moveCursorTo(m_iCursorPosition + szText.len());
		} else {
			//Multiline paste...do not execute commands here
			KviStr szBlock;
			while(szText.hasData()){
				int idx = szText.findFirstIdx('\n');
				if(idx != -1){
					szBlock = szText.left(idx);
					szText.cutLeft(idx+1);
				} else {
					szBlock = szText.ptr();
					szText  = "";
				}
				m_szTextBuffer.insert(m_iCursorPosition,szBlock.ptr());
				if(m_szTextBuffer.len() > KVI_INPUT_MAX_BUFFER_SIZE){
					m_szTextBuffer.setLen(KVI_INPUT_MAX_BUFFER_SIZE);
				}
//				moveCursorTo(m_iCursorPosition + szBlock.len());
				const char * aux = m_szTextBuffer.ptr();
				if(*aux < 33)while(*aux && ((unsigned char)*aux) < 33)++aux;
				if(*aux == '/')m_szTextBuffer.insert(aux - m_szTextBuffer.ptr(),"\\");
				returnPressed(idx != -1);
			}
		}
	}
}

void KviInput::selectAll()
{
	if(m_szTextBuffer.len() > 0){
		m_iSelectionBegin = 0;
		m_iSelectionEnd = m_szTextBuffer.len()-1;
	}
	end();
}

void KviInput::clear()
{
	m_szTextBuffer = "";
	selectOneChar(-1);
	home();
}

void KviInput::setText(const char *text)
{
	m_szTextBuffer = text;
	if(m_szTextBuffer.len() > KVI_INPUT_MAX_BUFFER_SIZE){
		m_szTextBuffer.setLen(KVI_INPUT_MAX_BUFFER_SIZE);
	}
	selectOneChar(-1);
	end();
}

void KviInput::mouseReleaseEvent(QMouseEvent *)
{
	copy();
	if(m_iDragTimer){
		m_iSelectionAnchorChar =-1;
		releaseMouse();
		killDragTimer();
	}
}

void KviInput::killDragTimer()
{
	if(m_iDragTimer){
		killTimer(m_iDragTimer);
		m_iDragTimer = 0;
	}
}

void KviInput::timerEvent(QTimerEvent *e)
{
	if(e->timerId() == m_iCursorTimer){
		if(!hasFocus() || !isVisibleToTLW()){
			killTimer(m_iCursorTimer);
			m_iCursorTimer = 0;
			m_bCursorOn = false;
		} else m_bCursorOn = ! m_bCursorOn;
		paintEvent(0);
	} else {
		//Drag timer
		handleDragSelection();
	}
}

void KviInput::handleDragSelection()
{
	if(m_iSelectionAnchorChar == -1)return;
	Window root,child;
	int rootX,rootY,winX,winY;
	unsigned int mask;
	if(XQueryPointer(x11Display(),handle(),&root,&child,&rootX,&rootY,&winX,&winY,&mask)){
		if(mask & Button1Mask){ //Button 1 is pressed
			if(winX <= 0){
				//Left side dragging
				if(m_iFirstVisibleChar > 0)m_iFirstVisibleChar--;
				m_iCursorPosition = m_iFirstVisibleChar;
			} else if(winX >= width()){
				//Right side dragging...add a single character to the selection on the right
				if(m_iCursorPosition < m_szTextBuffer.len()){
					moveRightFirstVisibleCharToShowCursor();
					m_iCursorPosition++;
				} //else at the end of the selection...don't move anything
			} else {
				//Inside the window...
				m_iCursorPosition = charIndexFromXPosition(winX);
			}
			if(m_iCursorPosition == m_iSelectionAnchorChar)selectOneChar(-1);
			else {
				if(m_iCursorPosition > m_iSelectionAnchorChar){
					m_iSelectionBegin = m_iSelectionAnchorChar;
						m_iSelectionEnd   = m_iCursorPosition-1;
				} else {
					m_iSelectionBegin = m_iCursorPosition;
					m_iSelectionEnd   = m_iSelectionAnchorChar-1;
				}
			}
			repaintWithCursorOn();
			return;
		}
	}
	mouseReleaseEvent(0);
}

void KviInput::returnPressed(bool bRepaint)
{
	if(g_pEventManager->eventEnabled(KviEvent_OnIdleStart) || g_pEventManager->eventEnabled(KviEvent_OnIdleStop))
			m_pParent->m_pFrm->textInput(&m_szTextBuffer);
	KviStr *pHist = new KviStr(m_szTextBuffer.ptr());
	if(m_pHistory->count() > 0){
		if(!kvi_strEqualCI(m_szTextBuffer.ptr(),m_pHistory->at(0)->ptr()))
			m_pHistory->insert(0,pHist);
	} else m_pHistory->insert(0,pHist);
	__range_valid(KVI_INPUT_HISTORY_ENTRIES > 1); //ABSOLUTELY NEEDED, if not, pHist will be destroyed...
	if(m_pHistory->count() > KVI_INPUT_HISTORY_ENTRIES)m_pHistory->removeLast();
	m_iCurHistoryIdx = -1;
	m_szTextBuffer="";
	selectOneChar(-1);
	m_iCursorPosition = 0;
	m_iFirstVisibleChar = 0;
	if(bRepaint)repaintWithCursorOn();
	m_pUserParser->parseUserCommand(*pHist,m_pParent); //the parsed string must be NOT THE SAME AS m_szTextBuffer
}

void KviInput::focusInEvent(QFocusEvent *)
{
	if(m_iCursorTimer==0){
		m_iCursorTimer = startTimer(KVI_INPUT_BLINK_TIME);
		m_bCursorOn = true;
		paintEvent(0);
	}
}

void KviInput::focusOutEvent(QFocusEvent *)
{
	if(m_iCursorTimer)killTimer(m_iCursorTimer);
	m_iCursorTimer = 0;
	m_bCursorOn = false;
	paintEvent(0);
}

void KviInput::keyPressEvent(QKeyEvent *e)
{
//	if(g_pEventManager->eventEnabled(KviEvent_OnIdleStart) || g_pEventManager->eventEnabled(KviEvent_OnIdleStop))
//			m_pParent->m_pFrm->inputKeyPressed();


	if((e->key() == Qt::Key_Tab) || (e->key() == Qt::Key_BackTab)){
		completion(e->state() & ShiftButton);
		return;
	} else if(e->key() != Qt::Key_Shift)m_iLastCompletionIndex = -1;

	if (e->key()!=Qt::Key_Backspace)m_bLastEventWasASubstitution=false;

	if(e->state() & ControlButton){
		switch(e->key()){
			case Qt::Key_K:{
				insertChar(KVI_TEXT_COLOR);
				int xPos = xPositionFromCharIndex(m_iCursorPosition);
				if(xPos > 24)xPos-=24;
				if(xPos+g_pColorWindow->width() > width())xPos = width()-(g_pColorWindow->width()+2);
				g_pColorWindow->move(mapToGlobal(QPoint(xPos,-35)));
				g_pColorWindow->popup(this);}
				break;
			case Qt::Key_B:
				insertChar(KVI_TEXT_BOLD);
				break;
			case Qt::Key_O:
				insertChar(KVI_TEXT_RESET);
				break;
			case Qt::Key_U:
				insertChar(KVI_TEXT_UNDERLINE);
				break;
			case Qt::Key_R:
				insertChar(KVI_TEXT_REVERSE);
				break;
			case Qt::Key_C:
				copy();
				break;
			case Qt::Key_X:
				cut();
				break;
			case Qt::Key_V:
				paste();
				break;
			default:
				e->ignore();
				break;
		}
		return;
	}

	if(e->state() & AltButton){
		// Qt::Key_Meta seems to substitute Key_Alt on some keyboards
		if((e->key() == Qt::Key_Alt) || (e->key() == Qt::Key_Meta)){
			m_szAltKeyCode = "";
			return;
		} else if((e->ascii() >= '0') && (e->ascii() <= '9')){
			m_szAltKeyCode += e->ascii();
			return;
		}
	}

	switch(e->key()){
		case Qt::Key_Right:
			if(m_iCursorPosition < m_szTextBuffer.len()){
				moveRightFirstVisibleCharToShowCursor();
				if(e->state() & ShiftButton){
					//Grow the selection if needed
					if((m_iSelectionBegin > -1)&&(m_iSelectionEnd > -1)){
						if(m_iSelectionEnd == m_iCursorPosition-1)m_iSelectionEnd++;
						else if(m_iSelectionBegin == m_iCursorPosition)m_iSelectionBegin++;
						else selectOneChar(m_iCursorPosition);
					} else selectOneChar(m_iCursorPosition);
				}
				m_iCursorPosition++;
				repaintWithCursorOn();
			}
			break;
		case Qt::Key_Left:
			if(m_iCursorPosition > 0){
				if(e->state() & ShiftButton){
					if((m_iSelectionBegin > -1)&&(m_iSelectionEnd > -1)){
						if(m_iSelectionBegin == m_iCursorPosition)m_iSelectionBegin--;
						else if(m_iSelectionEnd == m_iCursorPosition-1)m_iSelectionEnd--;
						else selectOneChar(m_iCursorPosition - 1);
					} else selectOneChar(m_iCursorPosition - 1);
				}
				m_iCursorPosition--;
				if(m_iFirstVisibleChar > m_iCursorPosition)m_iFirstVisibleChar--;
				repaintWithCursorOn();
			}
			break;
		case Qt::Key_Backspace:
			if(m_iCursorPosition <= m_szTextBuffer.len()){
				if (m_bLastEventWasASubstitution) {
					////
					// restore the old string
					m_szTextBuffer=m_szStringBeforeSubstitution;
					m_iCursorPosition=m_iCursorPositionBeforeSubstitution;
					m_iLastCursorXPosition=m_iCursorXPositionBeforeSubstitution;
					m_iFirstVisibleChar=m_iFirstVisibleCharBeforeSubstitution;
					///
					insertChar(m_cDetonationChar);
					repaintWithCursorOn();
					m_bLastEventWasASubstitution=false;
					break;
				}
				if(hasSelection()&&(m_iSelectionEnd == m_iCursorPosition-1)||(m_iSelectionBegin == m_iCursorPosition)){
					//remove the selection
					m_szTextBuffer.cut(m_iSelectionBegin,(m_iSelectionEnd-m_iSelectionBegin)+1);
					m_iCursorPosition = m_iSelectionBegin;
					if((m_iFirstVisibleChar > 0)&&(xPositionFromCharIndex(m_szTextBuffer.len()) <= width()-KVI_INPUT_BORDER))
						m_iFirstVisibleChar-=(m_iSelectionEnd-m_iSelectionBegin)+1;
					else if(m_iFirstVisibleChar > m_iCursorPosition)m_iFirstVisibleChar = m_iCursorPosition;
				} else if(m_iCursorPosition > 0){
					m_iCursorPosition--;
					m_szTextBuffer.cut(m_iCursorPosition,1);
					if((m_iFirstVisibleChar > 0)&&(xPositionFromCharIndex(m_szTextBuffer.len()) <= width()-KVI_INPUT_BORDER))
						m_iFirstVisibleChar--;
					else if(m_iFirstVisibleChar > m_iCursorPosition)m_iFirstVisibleChar--;
				}
				selectOneChar(-1);
				repaintWithCursorOn();
			}
			break;
		case Qt::Key_Delete:
			if(m_iCursorPosition < m_szTextBuffer.len()){
				if(hasSelection()&&(m_iSelectionEnd == m_iCursorPosition-1)||(m_iSelectionBegin == m_iCursorPosition)){
					m_szTextBuffer.cut(m_iSelectionBegin,(m_iSelectionEnd-m_iSelectionBegin)+1);
					m_iCursorPosition = m_iSelectionBegin;
					if((m_iFirstVisibleChar > 0)&&(xPositionFromCharIndex(m_szTextBuffer.len()) <= width()-KVI_INPUT_BORDER))
						m_iFirstVisibleChar-=(m_iSelectionEnd-m_iSelectionBegin)+1;
					else if(m_iFirstVisibleChar > m_iCursorPosition)m_iFirstVisibleChar = m_iCursorPosition;
				} else {
					m_szTextBuffer.cut(m_iCursorPosition,1);
					if((m_iFirstVisibleChar > 0)&&(xPositionFromCharIndex(m_szTextBuffer.len()) < width()-KVI_INPUT_BORDER))
						m_iFirstVisibleChar--;
				}
				moveRightFirstVisibleCharToShowCursor();
				selectOneChar(-1);
				repaintWithCursorOn();
			}
			break;
		case Qt::Key_Home:
			if(m_iCursorPosition > 0){
				if(e->state() & ShiftButton){
					if((m_iSelectionBegin == -1)&&(m_iSelectionEnd == -1))m_iSelectionEnd = m_iCursorPosition - 1;
					m_iSelectionBegin = 0;
				}
				home();
			}
			break;
		case Qt::Key_End:
			if(m_iCursorPosition < m_szTextBuffer.len()){
				if(e->state() & ShiftButton){
					if((m_iSelectionBegin == -1)&&(m_iSelectionEnd == -1))m_iSelectionBegin = m_iCursorPosition;
					m_iSelectionEnd = m_szTextBuffer.len();
				}
				end();
			}
			break;
		case Qt::Key_Up:
			if(m_pHistory->count() > 0){
				if(m_iCurHistoryIdx < 0){
					m_szSaveTextBuffer = m_szTextBuffer;
					m_szTextBuffer = m_pHistory->at(0)->ptr();
					m_iCurHistoryIdx = 0;
				} else if(m_iCurHistoryIdx >= (int)(m_pHistory->count()-1)){
					m_szTextBuffer=m_szSaveTextBuffer;
					m_iCurHistoryIdx = -1;
				} else {
					m_iCurHistoryIdx++;
					m_szTextBuffer = m_pHistory->at(m_iCurHistoryIdx)->ptr();
				}
				selectOneChar(-1);
				if(g_pOptions->m_bInputHistoryCursorAtEnd)end();
				else home();
			}
			break;
		case Qt::Key_Down:
			if(m_pHistory->count() > 0){
				if(m_iCurHistoryIdx < 0){
					m_szSaveTextBuffer = m_szTextBuffer;
					m_szTextBuffer = m_pHistory->at(m_pHistory->count()-1)->ptr();
					m_iCurHistoryIdx = m_pHistory->count()-1;
				} else if(m_iCurHistoryIdx == 0){
					m_szTextBuffer=m_szSaveTextBuffer;
					m_iCurHistoryIdx = -1;
				} else {
					m_iCurHistoryIdx--;
					m_szTextBuffer = m_pHistory->at(m_iCurHistoryIdx)->ptr();
				}
				selectOneChar(-1);
				if(g_pOptions->m_bInputHistoryCursorAtEnd)end();
				else home();
			}
			break;
		case Qt::Key_Enter:
		case Qt::Key_Return:
			checkForSubstitution();
			m_bLastEventWasASubstitution=false;
			returnPressed();
			break;
		case Qt::Key_Alt:
		case Qt::Key_Meta:
			m_szAltKeyCode = "";
			break;
		case Qt::Key_Space:
			m_cDetonationChar=' ';
			checkForSubstitution();
			insertChar(e->ascii());
			break;
		case Qt::Key_Period:
			m_cDetonationChar='.';
			checkForSubstitution();
			insertChar(e->ascii());
			break;
		case Qt::Key_Comma:
			m_cDetonationChar=',';
			checkForSubstitution();
			insertChar(e->ascii());
			break;
		default:
			if(e->ascii() > 0)insertChar(e->ascii());
			else e->ignore();
			break;
	}
}

void KviInput::keyReleaseEvent(QKeyEvent *e)
{
	if((e->key() == Qt::Key_Alt) || (e->key() == Qt::Key_Meta)){
		if(m_szAltKeyCode.hasData()){
			bool bOk;
			char ch = m_szAltKeyCode.toChar(&bOk);
			if(bOk && ch != 0){
				insertChar(ch);
				e->accept();
			}
		}
		m_szAltKeyCode = "";
	}
	e->ignore();
}

void KviInput::completion(bool bShift)
{
	if(m_szTextBuffer.isEmpty() || (m_iCursorPosition == 0))return;
	// get the word before the cursor
	KviStr word = m_szTextBuffer.left(m_iCursorPosition);

	int idx = word.findLastIdx(' ');
	int idx2 = word.findLastIdx(',');
	if(idx2 > idx)idx = idx2;
	idx2 = word.findLastIdx('(');
	if(idx2 > idx)idx = idx2;
	bool bFirstWordInLine = false;
	if(idx > -1)word.cutLeft(idx+1);
	else bFirstWordInLine = true;
	word.stripWhiteSpace();

	KviStr match;
	KviStr multiple;
	int iMatches = 0;

	if((*(word.ptr()) == '/') || (*(word.ptr()) == g_pOptions->m_cPersonalCommandPrefix))
	{
		// command completion
		word.cutLeft(1);
		if(word.isEmpty())return;
		iMatches = m_pUserParser->completeCommand(word,match,multiple);
	} else if(*(word.ptr()) == '$')
	{
		// function/identifer completion
		word.cutLeft(1);
		if(word.isEmpty())return;
		iMatches = m_pUserParser->completeFunctionOrIdentifier(word,match,multiple);
	} else {
		// empty word will end up here
		nickCompletion(bShift,word,bFirstWordInLine);
		repaintWithCursorOn();
		return;
	}

	if(iMatches != 1)
	{
		if(iMatches == 0)m_pParent->output(KVI_OUT_INTERNAL,__tr("[Command completion]: No matches"));
		else {
			multiple.toLower();
			m_pParent->output(KVI_OUT_INTERNAL,__tr("[Command completion]: Multiple matches: %s"),multiple.ptr());
		}
	}

	if(match.hasData())
	{
		match.toLower();
		m_iCursorPosition -= word.len();
		m_szTextBuffer.cut(m_iCursorPosition,word.len());
		m_szTextBuffer.insert(m_iCursorPosition,match.ptr());
		if(m_szTextBuffer.len() > KVI_INPUT_MAX_BUFFER_SIZE){
			m_szTextBuffer.setLen(KVI_INPUT_MAX_BUFFER_SIZE);
		}
		moveCursorTo(m_iCursorPosition + match.len());
	}

	repaintWithCursorOn();
}

void KviInput::nickCompletion(bool bAddMask,KviStr &word,bool bFirstWordInLine)
{
	if(!m_pListBoxToControl)return;
	if(m_iLastCompletionIndex < 0){
		// new completion session
		// save the buffer for the next time
		m_szLastCompletionBuffer = m_szTextBuffer;
		m_iLastCompletionCursorPosition = m_iCursorPosition;
		m_iLastCompletionCursorXPosition = m_iLastCursorXPosition;
		m_iLastCompletionFirstVisibleChar = m_iFirstVisibleChar;
		if(word.isEmpty())return; // new completion with an empty word
	} else {
		// old completion...swap the buffers
		m_szTextBuffer = m_szLastCompletionBuffer;
		m_iCursorPosition = m_iLastCompletionCursorPosition;
		m_iLastCursorXPosition = m_iLastCompletionCursorXPosition;
		m_iFirstVisibleChar = m_iLastCompletionFirstVisibleChar;
		// re-extract 
		word = m_szTextBuffer.left(m_iCursorPosition);
		// trailing spaces allowed this time!
		int iOriginalCursorPos = m_iCursorPosition;

		while(word.lastCharIs(' ')){
			m_iCursorPosition--;
			word.cutRight(1);
		}

		int idx = word.findLastIdx(' ');
		int idx2 = word.findLastIdx(',');
		if(idx2 > idx)idx = idx2;
		idx2 = word.findLastIdx('(');
		if(idx2 > idx)idx = idx2;
		bFirstWordInLine = false;
		if(idx > -1)word.cutLeft(idx+1);
		else bFirstWordInLine = true;
		word.stripWhiteSpace();

		if(word.isEmpty()){
			// no word to complete
			m_iCursorPosition = iOriginalCursorPos;
			return;
		}
	}


	// k....now look for a nick that matches...

	selectOneChar(-1);

	int index = 0;	
	for(KviIrcUser * u = m_pListBoxToControl->firstUser();u;u=m_pListBoxToControl->nextUser()){
		if(index > m_iLastCompletionIndex){
			if(kvi_strEqualCIN(u->nick(),word.ptr(),word.len())){
				// the first part matches word
				m_iLastCompletionIndex = index;
				KviStr missingPart = u->nick();
				if(bAddMask){
					missingPart.append('!');
					missingPart.append(u->username());
					missingPart.append('@');
					missingPart.append(u->host());
				}
				if(g_pOptions->m_bCompletionReplaceWholeWord){
					m_iCursorPosition -= word.len();
					m_szTextBuffer.cut(m_iCursorPosition,word.len());
				} else missingPart.cutLeft(word.len());
				if(bFirstWordInLine || (!g_pOptions->m_bApplyCompletionPostfixForFirstWordOnly))
					missingPart+=g_pOptions->m_szStringToAddAfterCompletedNick;
				m_szTextBuffer.insert(m_iCursorPosition,missingPart.ptr());
				if(m_szTextBuffer.len() > KVI_INPUT_MAX_BUFFER_SIZE){
					m_szTextBuffer.setLen(KVI_INPUT_MAX_BUFFER_SIZE);
				}
				moveCursorTo(m_iCursorPosition + missingPart.len());
				// REPAINT CALLED FROM OUTSIDE!
				return;
			}
		}
		index++;
	}
	// nope...no matches...
	// just reset to the old buffer and next time start from beginning
	m_iLastCompletionIndex = -1;
	// REPAINT CALLED FROM OUTSIDE!
}

//Funky helpers

void KviInput::end()
{
	m_iLastCursorXPosition = KVI_INPUT_BORDER;
	m_iCursorPosition = 0;
	m_iFirstVisibleChar = 0;
	while(m_iCursorPosition < m_szTextBuffer.len()){
		moveRightFirstVisibleCharToShowCursor();
		m_iCursorPosition++;
	}
	repaintWithCursorOn();
}

void KviInput::home()
{
	m_iFirstVisibleChar = 0;
	m_iCursorPosition   = 0;
	repaintWithCursorOn();
}

void KviInput::insertChar(char c)
{
	// Kill the selection
	if((m_iSelectionBegin > -1)||(m_iSelectionEnd > -1))
	{
		if((m_iSelectionEnd == m_iCursorPosition-1)||(m_iSelectionBegin == m_iCursorPosition)){
			m_szTextBuffer.cut(m_iSelectionBegin,(m_iSelectionEnd-m_iSelectionBegin)+1);
			m_iCursorPosition = m_iSelectionBegin;
			if((m_iFirstVisibleChar > 0)&&(xPositionFromCharIndex(m_szTextBuffer.len()) <= width()-KVI_INPUT_BORDER))
				m_iFirstVisibleChar-=(m_iSelectionEnd-m_iSelectionBegin)+1;
		}
	}
	selectOneChar(-1);
	if(m_szTextBuffer.len()<KVI_INPUT_MAX_BUFFER_SIZE){
		m_szTextBuffer.insert(m_iCursorPosition,c);
		moveRightFirstVisibleCharToShowCursor();
		m_iCursorPosition++;
		repaintWithCursorOn();
	}
}

void KviInput::moveRightFirstVisibleCharToShowCursor()
{
	// :)
	m_iLastCursorXPosition += g_pOptions->m_iInputFontCharacterWidth[(unsigned char)m_szTextBuffer.at(m_iCursorPosition)];
	while(m_iLastCursorXPosition >= width()-KVI_INPUT_BORDER){
		m_iLastCursorXPosition -= g_pOptions->m_iInputFontCharacterWidth[(unsigned char)m_szTextBuffer.at(m_iFirstVisibleChar)];
		m_iFirstVisibleChar++;
	}
}

void KviInput::repaintWithCursorOn()
{
	// :)
	if(m_bUpdatesEnabled){
		m_bCursorOn = true;
		paintEvent(0);
	}
}

void KviInput::selectOneChar(int pos)
{	
	m_iSelectionBegin = pos;
	m_iSelectionEnd   = pos;
}

int KviInput::charIndexFromXPosition(int xPos)
{
	int curXPos = KVI_INPUT_BORDER;
	int curChar = m_iFirstVisibleChar;
	int bufLen  = m_szTextBuffer.len();
	while(curChar < bufLen){
		int widthCh = g_pOptions->m_iInputFontCharacterWidth[(unsigned char)m_szTextBuffer.at(curChar)];
		if(xPos < (curXPos+(widthCh/2)))return curChar;
		else if(xPos < (curXPos+widthCh))return (curChar+1);
		{
			curXPos+=widthCh;
			curChar++;
		}
	}
	return curChar;
}

int KviInput::xPositionFromCharIndex(int chIdx)
{
	int curXPos = KVI_INPUT_BORDER;
	int curChar = m_iFirstVisibleChar;
	while(curChar < chIdx){
		curXPos += g_pOptions->m_iInputFontCharacterWidth[(unsigned char)m_szTextBuffer.at(curChar)];
		curChar++;
	}
	return curXPos;
}

void KviInput::checkForSubstitution() {

	if(!g_pOptions->m_bUseStringSubstitution || m_szTextBuffer.isEmpty() || !m_iCursorPosition)return;
	KviStr szAuxTextBuffer = m_szTextBuffer;
	szAuxTextBuffer.prepend(' ');
	for(KviStrSubItem * m=g_pOptions->m_pStrSub->m_pList->first();m;m=g_pOptions->m_pStrSub->m_pList->next()) {
		KviStr szAuxOriginal=m->szOriginal;
		szAuxOriginal.prepend(' ');
		if (!kvi_strMatchRevCS(szAuxTextBuffer.ptr(),szAuxOriginal.ptr(),m_iCursorPosition)) {
			//////
			// save buffer in case of backspace
			m_szStringBeforeSubstitution = m_szTextBuffer;
			m_iCursorPositionBeforeSubstitution=m_iCursorPosition;
			m_iCursorXPositionBeforeSubstitution=m_iLastCursorXPosition;
			m_iFirstVisibleCharBeforeSubstitution=m_iFirstVisibleChar;
			///  
			m_iCursorPosition -= m->szOriginal.len();
			m_szTextBuffer.cut(m_iCursorPosition,m->szOriginal.len());
			if((m_iFirstVisibleChar > 0)&&(xPositionFromCharIndex(m_szTextBuffer.len()) <= width()-KVI_INPUT_BORDER))
				m_iFirstVisibleChar -= m->szOriginal.len();
			m_szTextBuffer.insert(m_iCursorPosition,m->szSubstitute.ptr());
			if(m_szTextBuffer.len() > KVI_INPUT_MAX_BUFFER_SIZE){
				m_szTextBuffer.setLen(KVI_INPUT_MAX_BUFFER_SIZE);
			}
			moveCursorTo(m_iCursorPosition + m->szSubstitute.len());
			m_bLastEventWasASubstitution=true;
			repaintWithCursorOn();
			return;
		}
	}
}

#include "m_kvi_input.moc"
