//
//   File : kvi_links.cpp (/usr/build/NEW_kvirc/kvirc/src/kvirc/kvi_links.cpp)
//   Last major modification : Tue Jul 6 1999 14:36:15 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#define _KVI_DEBUG_CHECK_RANGE_
#include "kvi_debug.h"

#include "kvi_links.h"
#include "kvi_app.h"
#include "kvi_ircview.h"
#include "kvi_ircsocket.h"
#include "kvi_frame.h"
#include "kvi_locale.h"
#include "kvi_options.h"
#include "kvi_popupmenu.h"

#include <qvaluelist.h>
#include <qpopupmenu.h>

#include <ctype.h>

//Declared in kvi_app.cpp and managed by KviApp class
//extern QPixmap * g_pWindowIcon[KVI_WND_NUM_ICONS];
extern QPixmap * g_pixViewOut[KVI_OUT_NUM_IMAGES];

//============ KviLinksWindow ============//

KviLinksWindow::KviLinksWindow(KviFrame *lpFrm)
:KviWindow(KVI_LINKS_WINDOW_NAME,KVI_WND_TYPE_LINKS,lpFrm)
{
	m_pSplitter  = new QSplitter(QSplitter::Vertical,this);
	m_pListView  = new QListView(m_pSplitter);
	m_pListView->addColumn(__tr("Link"));
	m_pListView->addColumn(__tr("Hops"));
	m_pListView->addColumn(__tr("Description"));
	m_pListView->setRootIsDecorated(true);
	m_pListView->setAllColumnsShowFocus(true);
	connect(m_pListView,SIGNAL(rightButtonPressed(QListViewItem *,const QPoint &,int)),
			this,SLOT(showHostPopup(QListViewItem *,const QPoint &,int)));

	m_pView      = new KviIrcView(m_pSplitter,lpFrm,this);

	m_pLinkList = new QList<KviLink>;
	m_pLinkList->setAutoDelete(true);

	m_pHostPopup = new QPopupMenu();
	connect(m_pHostPopup,SIGNAL(activated(int)),this,SLOT(hostPopupClicked(int)));

	setFocusHandler(m_pListView);
}

//============ ~KviLinksWindow ============//

KviLinksWindow::~KviLinksWindow()
{
	delete m_pLinkList;
	delete m_pHostPopup;
}

void KviLinksWindow::endOfLinks()
{
	m_pListView->clear();
	m_pListView->setSorting(-1);

	outputNoFmt(KVI_OUT_INTERNAL,"======================");
	outputNoFmt(KVI_OUT_INTERNAL,__tr("Received end of links."));
	outputNoFmt(KVI_OUT_INTERNAL,"======================");

	QListViewItem * it   = 0;
	QListViewItem * root = 0;

	int totalHosts  = 0;
	int totalHops   = 0;
	int maxHops     = 0;
	int avgHops     = 0;
	int directLinks = 0;
	int nearLinks   = 0;
	int brokenLinks = 0;
	int maxLinks    = 0;
	int farLinks    = 0;

	KviStr szMaxHop,szMaxLinks;

	m_pListView->setUpdatesEnabled(false);
	for(KviLink *l=m_pLinkList->first();l;l=m_pLinkList->next()){
		totalHosts++;
		if(l->hops == 0)root = new QListViewItem(m_pListView,_CHAR_2_QSTRING(l->host.ptr()),__tr("0"),_CHAR_2_QSTRING(l->description.ptr()));
		else {
			totalHops += l->hops;
			if(l->hops < 4){
				nearLinks++;
				if(l->hops == 1)directLinks++;
			} else {
				if(l->hops >= 7)farLinks++;
			}
			if(l->hops == maxHops){
				szMaxHop.append(',');
				szMaxHop.append(l->host);
			} else if(l->hops > maxHops){
				maxHops = l->hops;
				szMaxHop = l->host;
			}
			it = insertLink(l);
			if(!it){
				output(KVI_OUT_ERROR,__tr("Broken link: Missing parent (%s) for %s (%d hops): %s (used /LINKS <mask> ?)"),
					l->parent.ptr(),l->host.ptr(),l->hops,l->description.ptr());
				brokenLinks++;
				KviStr tmp(KviStr::Format,__tr("%s : Parent link %s"),l->description.ptr(),l->parent.ptr());
				KviStr tmp2(KviStr::Format,"%d",l->hops);
				if(root)it = new QListViewItem(m_pListView,root,_CHAR_2_QSTRING(l->host.ptr()),_CHAR_2_QSTRING(tmp2.ptr()),_CHAR_2_QSTRING(tmp.ptr()));
				else {
					outputNoFmt(KVI_OUT_ERROR,__tr("Warning: No root link was sent by the server : Stats may be invalid"));
					it = new QListViewItem(m_pListView,_CHAR_2_QSTRING(l->host.ptr()),_CHAR_2_QSTRING(tmp2.ptr()),_CHAR_2_QSTRING(tmp.ptr()));
				}
			} else {
				it = it->parent();
				if(it){
					int links = it->childCount() + 1;
					if(links == maxLinks){
						szMaxLinks.append(',');
						szMaxLinks.append(it->text(0));
					} else if(links > maxLinks){
						maxLinks = links;
						szMaxLinks = it->text(0);
					}
				}
			}
		}
	}
	
	avgHops = ((totalHosts > 0) ? ((totalHops * 100) / totalHosts) : 0 );
	int nearProcent = ((totalHosts > 0) ? ((nearLinks * 10000) / totalHosts) : 0);
	int directProcent = ((totalHosts > 0) ? ((directLinks * 10000) / totalHosts) : 0);
	int midLinks = totalHosts - (farLinks + nearLinks + 1 + brokenLinks);
	if(midLinks < 0)midLinks = 0;
	int midProcent = ((totalHosts > 0) ? ((midLinks * 10000) / totalHosts) : 0);
	int farProcent = ((totalHosts > 0) ? ((farLinks * 10000) / totalHosts) : 0);

	outputNoFmt(KVI_OUT_INTERNAL,"======================");

//	if(!root)root = m_pListView->firstChild();
	if(root){
		KviStr tmp = root->text(0);
		output(KVI_OUT_INTERNAL,__tr("%c%cLinks for for %s"),KVI_TEXT_BOLD,KVI_TEXT_UNDERLINE,tmp.ptr());
		outputNoFmt(KVI_OUT_INTERNAL,"======================");
		output(KVI_OUT_INTERNAL,__tr("Total hosts: %d"),totalHosts);
		output(KVI_OUT_INTERNAL,__tr("Direct links: %d (~%d.%d %)"),directLinks,directProcent / 100, directProcent % 100);
		output(KVI_OUT_INTERNAL,__tr("Near links (1-3 hops): %d (~%d.%d %)"),nearLinks,nearProcent / 100,nearProcent % 100);
		output(KVI_OUT_INTERNAL,__tr("Mid-range links (4-6 hops): %d (~%d.%d %)"),midLinks,midProcent / 100,midProcent % 100);
		output(KVI_OUT_INTERNAL,__tr("Far links (7+ hops): %d (~%d.%d %)"),farLinks,farProcent / 100,farProcent % 100);
		output(KVI_OUT_INTERNAL,__tr("Broken (unknown) links: %d"),brokenLinks);
		output(KVI_OUT_INTERNAL,__tr("Max links per host: %d [%s]"),maxLinks,szMaxLinks.ptr());
		output(KVI_OUT_INTERNAL,__tr("Total links: %d"),totalHops);
		output(KVI_OUT_INTERNAL,__tr("Maximum hops: %d [%s]"),maxHops,szMaxHop.ptr());
		output(KVI_OUT_INTERNAL,__tr("Average hops: ~%d.%d"),avgHops / 100,avgHops % 100);
	} else outputNoFmt(KVI_OUT_INTERNAL,__tr("Partial LINKS result : No stats available"));
	outputNoFmt(KVI_OUT_INTERNAL,"======================");


	while(!m_pLinkList->isEmpty())m_pLinkList->removeFirst();

	m_pListView->setUpdatesEnabled(true);
	m_pListView->repaint();
}

QListViewItem * KviLinksWindow::insertLink(KviLink *l)
{
	__range_valid(l->hops > 0);
	QListViewItem * i = getItemByHost(l->parent.ptr(),0);
	QListViewItem * it = 0;
	if(!i)return 0;
	else {
		KviStr hops(KviStr::Format,"%d",l->hops);
		it = new QListViewItem(i,_CHAR_2_QSTRING(l->host.ptr()),_CHAR_2_QSTRING(hops.ptr()),_CHAR_2_QSTRING(l->description.ptr()));
		i->setOpen(true);
	}
	return it;
}

QListViewItem * KviLinksWindow::getItemByHost(const char *host,QListViewItem * par)
{
	QListViewItem * i = (par ? par->firstChild() : m_pListView->firstChild());
	if(!i)return 0;
	while(i){
		KviStr tmp = i->text(0);
		if(kvi_strEqualCI(tmp.ptr(),host))return i;
		QListViewItem * ch = getItemByHost(host,i);
		if(ch)return ch;
		i = i->nextSibling();
	}
	return 0;
}

void KviLinksWindow::showHostPopup(QListViewItem *i,const QPoint &p,int)
{
	if(!i)return;
	KviStr host=i->text(0);
	if(host.isEmpty())return;
	m_pHostPopup->clear();
	KviStr tmp(KviStr::Format,"LINKS %s *",host.ptr());
	m_pHostPopup->insertItem(tmp.ptr());
	m_pHostPopup->insertSeparator();
	tmp.sprintf("TIME %s",host.ptr());
	m_pHostPopup->insertItem(tmp.ptr());
	tmp.sprintf("ADMIN %s",host.ptr());
	m_pHostPopup->insertItem(tmp.ptr());
	tmp.sprintf("INFO %s",host.ptr());
	m_pHostPopup->insertItem(tmp.ptr());
	tmp.sprintf("MOTD %s",host.ptr());
	m_pHostPopup->insertItem(tmp.ptr());
	tmp.sprintf("VERSION %s",host.ptr());
	m_pHostPopup->insertItem(tmp.ptr());
	tmp.sprintf("TRACE %s",host.ptr());
	m_pHostPopup->insertItem(tmp.ptr());
	tmp.sprintf("USERS %s",host.ptr());
	m_pHostPopup->insertItem(tmp.ptr());
	m_pHostPopup->insertSeparator();
	tmp.sprintf("STATS c %s",host.ptr());
	m_pHostPopup->insertItem(tmp.ptr());
	tmp.sprintf("STATS h %s",host.ptr());
	m_pHostPopup->insertItem(tmp.ptr());
	tmp.sprintf("STATS i %s",host.ptr());
	m_pHostPopup->insertItem(tmp.ptr());
	tmp.sprintf("STATS k %s",host.ptr());
	m_pHostPopup->insertItem(tmp.ptr());
	tmp.sprintf("STATS l %s",host.ptr());
	m_pHostPopup->insertItem(tmp.ptr());
	tmp.sprintf("STATS m %s",host.ptr());
	m_pHostPopup->insertItem(tmp.ptr());
	tmp.sprintf("STATS o %s",host.ptr());
	m_pHostPopup->insertItem(tmp.ptr());
	tmp.sprintf("STATS y %s",host.ptr());
	m_pHostPopup->insertItem(tmp.ptr());
	tmp.sprintf("STATS u %s",host.ptr());
	m_pHostPopup->insertItem(tmp.ptr());
	m_pHostPopup->popup(p);
}

void KviLinksWindow::hostPopupClicked(int id)
{
	KviStr tmp = m_pHostPopup->text(id);
	if(tmp.hasData()){
		if(!m_pFrm->m_pSocket->sendData(tmp.ptr()))output(KVI_OUT_ERROR,__tr("Not connected to server"));
	}
}

void KviLinksWindow::reset()
{
	outputNoFmt(KVI_OUT_INTERNAL,__tr("Reset"));
	while(!m_pLinkList->isEmpty())m_pLinkList->removeFirst();
}

void KviLinksWindow::processLink(const char *msg)
{
	output(KVI_OUT_INTERNAL,__tr("Processing link: %s"),msg);
	KviLink * l = new KviLink;
	msg = kvi_extractToken(l->host,msg,' ');
	msg = kvi_extractToken(l->parent,msg,' ');
	while(*msg && (*msg == ' '))msg++;
	while(*msg && (*msg == ':'))msg++;
	if(isdigit(*msg)){
		const char *aux = msg;
		while(*msg && (isdigit(*msg)))msg++;
		KviStr tmp(aux,msg - aux);
		l->hops = tmp.toInt();
	} else {
		outputNoFmt(KVI_OUT_ERROR,__tr("Broken message syntax : Can't extract hops number, assuming 0"));
		l->hops = 0;
	}
	while(*msg && (*msg == ' '))msg++;
	l->description = msg;
	uint idx=0;
	for(KviLink *m=m_pLinkList->first();m;m=m_pLinkList->next()){
		if(m->hops >= l->hops){
			m_pLinkList->insert(idx,l);
			return;
		}
		idx++;
	}
	m_pLinkList->append(l);
}

void KviLinksWindow::saveProperties()
{
	KviWindowProperty p;
	p.rect = externalGeometry();
	p.isDocked = isAttached();
	QValueList<int> l(m_pSplitter->sizes());
	if(l.count() >= 1)p.splitWidth1 = *(l.at(0));
	if(l.count() >= 2)p.splitWidth2 = *(l.at(1));
	p.timestamp = m_pView->timestamp();
	p.imagesVisible = m_pView->imagesVisible();
	p.isMaximized = isAttached() && isMaximized();
	p.topSplitWidth1 = 0;
	p.topSplitWidth2 = 0;
	p.topSplitWidth3 = 0;
	g_pOptions->m_pWinPropertiesList->setProperty(caption(),&p);
}

void KviLinksWindow::setProperties(KviWindowProperty *p)
{
	QValueList<int> l;
	l.append(p->splitWidth1);
	l.append(p->splitWidth2);
	m_pSplitter->setSizes(l);
	m_pView->setTimestamp(p->timestamp);
	m_pView->setShowImages(p->imagesVisible);
}

//================ myIconPtr =================//

QPixmap * KviLinksWindow::myIconPtr()
{
	return g_pixViewOut[KVI_OUT_WND_LINKS];
}

//=============== applyOptions ================//

void KviLinksWindow::applyOptions()
{
	m_pView->setFont(g_pOptions->m_fntView);
	m_pView->setShowImages(g_pOptions->m_bShowImages,false);
	m_pView->setTimestamp(g_pOptions->m_bTimestamp);
	m_pView->setMaxBufferSize(g_pOptions->m_iViewMaxBufferSize);
}

//================ resizeEvent ===============//

void KviLinksWindow::resizeEvent(QResizeEvent *)
{
	m_pSplitter->setGeometry(0,0,width(),height());
}

#include "m_kvi_links.moc"
