//
//   File : kvi_plugin.cpp (/usr/build/KVIrc/kvirc/src/kvilib/kvi_plugin.cpp)
//   Last major modification : Wed Jul 21 1999 16:41:14 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
#define _KVI_DEBUG_CHECK_RANGE_
#include "kvi_debug.h"

#include "kvi_plugin.h"

#ifdef COMPILE_PLUGIN_SUPPORT

#include "kvi_locale.h"
#include "kvi_config.h"
#include "kvi_event.h"
#include "kvi_app.h"
#include "kvi_frame.h"
#include "kvi_window.h"
#include "kvi_style.h"
#include "kvi_statusbar.h"
#include "kvi_systray.h"

extern KviEventManager * g_pEventManager;

#include <dlfcn.h>

#include <qobjcoll.h>
#include <qobjectdict.h>
#include <qmetaobject.h>


#include "kvi_statusbar.h"

KviPluginManager::KviPluginManager()
{
	m_pPluginList = new QList<KviPluginData>;
	m_pPluginList->setAutoDelete(true);
	m_pCommandList = new QList<KviPluginCommandHandler>;
	m_pCommandList->setAutoDelete(true);
	m_pFunctionList = new QList<KviPluginFunctionHandler>;
	m_pFunctionList->setAutoDelete(true);
}

KviPluginManager::~KviPluginManager()
{
	while(m_pPluginList->first())unloadPlugin(m_pPluginList->first()->filename.ptr());
	delete m_pPluginList;
	__range_valid(m_pCommandList->count() == 0);
	delete m_pCommandList;
	__range_valid(m_pFunctionList->count() == 0);
	delete m_pFunctionList;
}

//#include <qasciidict.h>

void KviPluginManager::unregisterMetaObject(const char * metaObjName)
{
	if(!objectDict)return;
	QMetaObject * ob = objectDict->find(metaObjName);
	if(ob)
	{
//		debug("QMetaObject(\"%s\") found...destroying",metaObjName);
		// WARNING: HACK for a Qt bug
		// ob->d->slotAccess is never zeroed , and we get sigsegvs here..

// mmmmmh...some versions of qt seem to not have it ? (set_slot_access)

		ob->set_slot_access(0);
		objectDict->remove(metaObjName);
//		debug("QMetaObject(\"%s\") destroyed",metaObjName);

	}
}


void KviPluginManager::getLastError(KviStr &buffer)
{
	buffer = dlerror();
}

KviPluginData * KviPluginManager::loadPlugin(const char * fName,bool bAutoload,KviStr * pErrorRet)
{
//	unloadPlugin(fName); //unload the file with the same name...
	KviPluginData * d = findPlugin(fName);
	if(d)return d; //already there

	void * handle = dlopen(fName,RTLD_NOW);

	if(!handle){
		if(pErrorRet)pErrorRet->sprintf(__tr("dlopen error: %s"),dlerror());
		return 0;
	}

	KviPlugin * description = (KviPlugin *)dlsym(handle,"kvirc_plugin");
	if(!description){
		// not a kvirc plugin
		if(pErrorRet)*pErrorRet = __tr("Plugin loaded but exports no 'kvirc_plugin' symbol");
		dlclose(handle);
		return 0;
	}

	d = new KviPluginData;
	d->filename = fName;
	d->handle = handle;
	d->description = description;
	d->bAutoload = bAutoload;
	d->bInitExecuted = false;

	m_pPluginList->append(d);

	return d;
}

KviPluginData * KviPluginManager::findPlugin(const char * fName)
{
	for(KviPluginData *d=m_pPluginList->first();d;d=m_pPluginList->next()){
		if(kvi_strEqualCI(fName,d->filename.ptr()))return d;
	}
	return 0;
}

KviPluginData * KviPluginManager::findPluginByModuleName(const char * name)
{
	for(KviPluginData *d=m_pPluginList->first();d;d=m_pPluginList->next()){
		if(kvi_strEqualCI(name,d->description->module_name))return d;
	}
	return 0;
}

KviPluginData * KviPluginManager::findPlugin(void * plugin_handle)
{
	for(KviPluginData *d=m_pPluginList->first();d;d=m_pPluginList->next()){
		if(d->handle == plugin_handle)return d;
	}
	return 0;
}

bool KviPluginManager::executeInitRoutine(KviPluginData * data,KviPluginCommandStruct * cmd)
{
	__range_valid(data);
	__range_valid(findPlugin(data->filename.ptr()));
	if(data->bInitExecuted)return true; //already done with success!
	data->bInitExecuted = (data->description->init_routine)(cmd);
	return data->bInitExecuted;
}

bool KviPluginManager::unloadPlugin(const char * fName)
{
	KviPluginData * d = findPlugin(fName);
	if(!d)return false;
	unloadPluginInternal(d);
	return true;
}

bool KviPluginManager::unloadPluginByModuleName(const char * mName)
{
	KviPluginData * d = findPluginByModuleName(mName);
	if(!d)return false;
	unloadPluginInternal(d);
	return true;	
}
/*
bool KviPluginManager::unloadPlugin(void * plugin_handle)
{
	KviPluginData * d = findPlugin(plugin_handle);
	if(!d)return false;
	unloadPluginInternal(d);
	return true;
}
*/
void KviPluginManager::unloadPluginInternal(KviPluginData *d)
{
	__range_valid(d);
//	debug("Unloading plugin %s",d->filename.ptr());
	if(d->bInitExecuted)
	{
		if(d->description->cleanup_routine){
			// call the cleanup routine
			(d->description->cleanup_routine)();
		}
		g_pApp->unregisterStylesFor(d->handle);
		unregisterCommandsFor(d->handle);
		unregisterFunctionsFor(d->handle);
		unregisterHooksFor(d->handle);
		removeAllPluginWindows(d->handle);
		removeAllPluginDockedWidgets(d->handle);
		removeAllPluginSysTrayWidgets(d->handle);
	}
	dlclose(d->handle);
	m_pPluginList->removeRef(d);
}

void KviPluginManager::unregisterCommandsFor(void * plugin_handle)
{
	QList<KviPluginCommandHandler> l;
	l.setAutoDelete(false);
	// find all the handlers owned by this plugin
	for(KviPluginCommandHandler * c=m_pCommandList->first();c;c=m_pCommandList->next()){
		if(c->plugin_handle == plugin_handle)l.append(c);
	}
	// now kill it
	for(KviPluginCommandHandler * h=l.first();h;h=l.next())m_pCommandList->removeRef(h);
}

void KviPluginManager::registerCommand(void * plugin_handle,const char * cmdname,
			bool (*handler_routine)(KviPluginCommandStruct *))
{
	__range_valid(findPlugin(plugin_handle));
	// kill any previous handler with this name
	KviPluginCommandHandler * d = findCommandHandler(cmdname);
	if(d)m_pCommandList->removeRef(d);
	// now add it
	d = new KviPluginCommandHandler;
	d->plugin_handle   = plugin_handle;
	d->handler_routine = handler_routine;
	d->cmd_name        = cmdname;
	m_pCommandList->append(d);
}

KviPluginCommandHandler * KviPluginManager::findCommandHandler(const char *cmdname)
{
	for(KviPluginCommandHandler * c=m_pCommandList->first();c;c=m_pCommandList->next()){
		if(kvi_strEqualCI(c->cmd_name.ptr(),cmdname))return c;
	}
	return 0;
}

void KviPluginManager::unregisterCommand(void * plugin_handle,const char *cmdname)
{
	KviPluginCommandHandler * c = findCommandHandler(cmdname);
	if(c){
		__range_valid(c->plugin_handle = plugin_handle);
		m_pCommandList->removeRef(c);
	}
}

void KviPluginManager::registerFunction(void * plugin_handle,const char * fncname,
			bool (*handler_routine)(KviPluginCommandStruct *,KviStr *))
{
	__range_valid(findPlugin(plugin_handle));
	// kill any previous handler with this name
	KviPluginFunctionHandler * d = findFunctionHandler(fncname);
	if(d)m_pFunctionList->removeRef(d);
	// now add it
	d = new KviPluginFunctionHandler;
	d->plugin_handle   = plugin_handle;
	d->handler_routine = handler_routine;
	d->fnc_name        = fncname;
	m_pFunctionList->append(d);
}

void KviPluginManager::unregisterFunction(void * plugin_handle,const char *fncname)
{
	KviPluginFunctionHandler * c = findFunctionHandler(fncname);
	if(c){
		__range_valid(c->plugin_handle = plugin_handle);
		m_pFunctionList->removeRef(c);
	}
}

void KviPluginManager::unregisterFunctionsFor(void * plugin_handle)
{
	QList<KviPluginFunctionHandler> l;
	l.setAutoDelete(false);
	// find all the handlers owned by this plugin
	for(KviPluginFunctionHandler * c=m_pFunctionList->first();c;c=m_pFunctionList->next()){
		if(c->plugin_handle == plugin_handle)l.append(c);
	}
	// now kill it
	for(KviPluginFunctionHandler * h=l.first();h;h=l.next())m_pFunctionList->removeRef(h);
}

KviPluginFunctionHandler * KviPluginManager::findFunctionHandler(const char *fncname)
{
	for(KviPluginFunctionHandler * c=m_pFunctionList->first();c;c=m_pFunctionList->next()){
		if(kvi_strEqualCI(c->fnc_name.ptr(),fncname))return c;
	}
	return 0;
}

bool KviPluginManager::load(const char *configName,KviStr &error)
{
	error = "";

	KviConfig cfg(configName);
	int nPlugins = cfg.readIntEntry("nPlugins",0);
	for(int i=0;i<nPlugins;i++){
		KviStr tmp(KviStr::Format,"plugin%d",i);
		KviStr fName = cfg.readEntry(tmp.ptr(),"");
		if(fName.hasData()){
			KviPluginData * d = loadPlugin(fName.ptr());
			if(!d){
				error.append("Could not load plugin ");
				error.append(fName);
				error.append('\n');
			} else {
				KviPluginCommandStruct plgcmd;

				plgcmd.app     = 0;
				plgcmd.handle  = d->handle;
				plgcmd.params  = 0;
				plgcmd.window  = 0;
				plgcmd.frame   = 0;
				plgcmd.error   = 0;
				plgcmd.sock    = 0;
				plgcmd.console = 0;

				if(!executeInitRoutine(d,&plgcmd)){
					error.append("Could not initialize plugin ");
					error.append(fName);
					error.append('\n');
					unloadPlugin(d->filename.ptr());
				}
			}
		}
	}
	return error.isEmpty();
}

bool KviPluginManager::save(const char *configName)
{
	KviConfig cfg(configName);
	cfg.clear();
	int nPlugins = 0;
	for(KviPluginData *d=m_pPluginList->first();d;d=m_pPluginList->next()){
		if(d->bAutoload){
			KviStr tmp(KviStr::Format,"plugin%d",nPlugins);
			cfg.writeEntry(tmp.ptr(),d->filename.ptr());
			nPlugins++;
		}
	}
	cfg.writeEntry("nPlugins",nPlugins);
	return cfg.sync();
}

void KviPluginManager::registerHook(void * plugin_handle,int eventindex,
			bool (*handler_routine)(KviPluginCommandStruct *))
{
	__range_valid(findPlugin(plugin_handle));
	if(isHookRegistered(plugin_handle,eventindex)){
		unregisterHook(plugin_handle,eventindex); // kill the previous hook that belongs to this plugin
	}

	KviPluginEventHandlerStruct * s= new KviPluginEventHandlerStruct;
	s->handler_routine = handler_routine;
	s->plugin_handle = plugin_handle;

	if(!g_pEventManager->m_pEventHandlerList[eventindex]){
		g_pEventManager->m_pEventHandlerList[eventindex] = new QList<KviPluginEventHandlerStruct>;
		g_pEventManager->m_pEventHandlerList[eventindex]->setAutoDelete(true);
	}
	g_pEventManager->m_pEventHandlerList[eventindex]->append(s);
}

bool KviPluginManager::isHookRegistered(void * plugin_handle,int eventindex)
{
	if(!g_pEventManager->m_pEventHandlerList[eventindex])return false; //no hooks registered	
	for(KviPluginEventHandlerStruct *s=g_pEventManager->m_pEventHandlerList[eventindex]->first();s;s=g_pEventManager->m_pEventHandlerList[eventindex]->next()){
		if(s->plugin_handle == plugin_handle)return true;
	}
	return false;
}

void KviPluginManager::unregisterHook(void * plugin_handle,int eventindex)
{
	__range_valid(findPlugin(plugin_handle));

	if(!g_pEventManager->m_pEventHandlerList[eventindex])return; //no hooks registered

	KviPluginEventHandlerStruct *s=g_pEventManager->m_pEventHandlerList[eventindex]->first();
	while(s){
		if(s->plugin_handle == plugin_handle){
			g_pEventManager->m_pEventHandlerList[eventindex]->removeRef(s);
			s = 0;
		} else s = g_pEventManager->m_pEventHandlerList[eventindex]->next();
	}

	if(g_pEventManager->m_pEventHandlerList[eventindex]->count() == 0){
		// and cleanup if the list is empty
		delete g_pEventManager->m_pEventHandlerList[eventindex];
		g_pEventManager->m_pEventHandlerList[eventindex] = 0;
	}
}

void KviPluginManager::unregisterHooksFor(void * plugin_handle)
{
	for(int i=0;i<KVI_NUM_EVENTS;i++)unregisterHook(plugin_handle,i);
}

void KviPluginManager::addPluginWindow(void * plugin_handle,KviFrame * frame,KviWindow *wnd,bool bShow)
{
	if(wnd->type() != KVI_WND_TYPE_PLUGIN){
		wnd->m_type = KVI_WND_TYPE_PLUGIN; //fix silently
#ifdef _KVI_DEBUG_CHECK_RANGE_
		debug("The window exported by the plugin is not of type KVI_WND_TYPE_PLUGIN : fixing");
#endif
	}
	KviStr tmp = wnd->caption();
	if(!kvi_strEqualCIN("!",tmp.ptr(),1)){
		tmp.prepend("!");
		wnd->setWindowCaption(tmp.ptr());
		wnd->setName(tmp.ptr());
#ifdef _KVI_DEBUG_CHECK_RANGE_
		debug("The window exported by the plugn has a wrong name : fixing");
#endif
	}
	wnd->m_pluginHandle = plugin_handle;
	frame->addWindow(wnd,bShow);
}

void KviPluginManager::removeAllPluginWindows(void * plugin_handle)
{
	if(!g_pApp->m_pFrameList)return; //shutdown...no more windows here.
	for(KviFrame * f=g_pApp->m_pFrameList->first();f;f=g_pApp->m_pFrameList->next()){
		QList<KviWindow> l;
		l.setAutoDelete(false);
		for(KviWindow *wnd=f->m_pWinList->first();wnd;wnd=f->m_pWinList->next()){
			if(wnd->type() == KVI_WND_TYPE_PLUGIN){
				if(wnd->m_pluginHandle == plugin_handle)l.append(wnd);
			}
		}
		for(KviWindow *w=l.first();w;w=l.next())f->closeWindow(w);	
	}
}

void KviPluginManager::addPluginDockedWidget(void * plugin_handle,KviFrame * frame,KviDockableWidget * w)
{
	w->m_plugin_handle = plugin_handle;
	frame->m_pStatusBar->addWidget(w,0,true);
/*
	const QObjectList * l = frame->m_pStatusBar->children();
	QObjectListIt it(*l);
	QObject *o;
	while((o=it.current())){
		KviStr tmp =o->className();
		if(kvi_strEqualCI(tmp.ptr(),"QSizeGrip")){
			((QWidget *)o)->setMaximumWidth(((QWidget *)o)->sizeHint().width());
		}
		++it;
	}
	delete l;
*/
}

void KviPluginManager::removeAllPluginDockedWidgets(void * plugin_handle)
{
	if(!g_pApp->m_pFrameList)return; //shutdown...no more windows here.
	QList<KviDockableWidget> * list = new QList<KviDockableWidget>;
	list->setAutoDelete(true);
	for(KviFrame * f=g_pApp->m_pFrameList->first();f;f=g_pApp->m_pFrameList->next()){
	//	f->m_pStatusBar->killDockedWidgetsByPluginHandle(plugin_handle);
		QObjectList * l = f->m_pStatusBar->queryList("KviDockableWidget");
		l->setAutoDelete(false); //!!!
		QObjectListIt it(*l);
		QObject *o;
		while((o=it.current())){
			if(((KviDockableWidget *)o)->m_plugin_handle == plugin_handle){
				f->m_pStatusBar->removeWidget((QWidget *)o);
				list->append((KviDockableWidget *)o);
			}
			++it;
		}
		delete l;

	}
	delete list;
}

void KviPluginManager::addPluginSysTrayWidget(void * plugin_handle,KviFrame * frame,KviSysTrayWidget * w, bool _show)
{
	w->m_plugin_handle = plugin_handle;
	frame->m_pSysTrayBar->m_pSysTray->addPluginWidget(w, _show);
}

void KviPluginManager::removeAllPluginSysTrayWidgets(void * plugin_handle)
{
	if(!g_pApp->m_pFrameList)return; //shutdown...no more windows here.
	QList<KviSysTrayWidget> * list = new QList<KviSysTrayWidget>;
	list->setAutoDelete(true);
	for(KviFrame * f=g_pApp->m_pFrameList->first();f;f=g_pApp->m_pFrameList->next()){
	//	f->m_pStatusBar->killDockedWidgetsByPluginHandle(plugin_handle);
		f->m_pSysTrayBar->m_pSysTray->killPluginWidgets(plugin_handle);
	}
	delete list;
}

#else //!COMPILE_PLUGIN_SUPPORT

	#warning "Plugin support is disabled"

#endif
