#ifndef _KVI_SCRIPTOBJECT_H_INCLUDED_
#define _KVI_SCRIPTOBJECT_H_INCLUDED_

//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#include <qlist.h>
#include <qasciidict.h>

#include <qobject.h>

#include "kvi_string.h"

// Fwd...
class KviUserParser;
class KviScriptObjectController;
class KviVariableCache;
class KviWindow;
class KviScriptObject;
class KviFrame;

// Event handler structure...each object has a list of these structs
typedef struct KviScriptEventStructTag
{
	KviStr szName;
	KviStr szBuffer;
} KviScriptEventStruct;

// An object function callback
typedef int (KviScriptObject::*scriptObjectFunction)(QList<KviStr> *,KviStr &);

// Flags for the functions
#define KVI_SCRIPTOBJECTFUNCTION_FLAG_INHERITED 1
#define KVI_SCRIPTOBJECTFUNCTION_FLAG_BUILTIN 2
#define KVI_SCRIPTOBJECTFUNCTION_FLAG_OVERRIDE 4

// Function data struct : the class definition has a list of these structs
typedef struct KviScriptObjectFunctionStructTag
{
	scriptObjectFunction fncHandler;
	int                  flags;
	KviStr               szBuffer;
} KviScriptObjectFunctionStruct;

// A definition of a class (class descriptor)
class KviScriptObjectClassDefinition
{
	public:
		KviScriptObjectClassDefinition(const char *className,KviScriptObjectClassDefinition * inheritedClass,bool bBuiltin);
		~KviScriptObjectClassDefinition();
	protected:
		QAsciiDict<KviScriptObjectFunctionStruct> * m_pFunctionDict;   // class functions
		KviStr                                      m_szClassName;     // class name
		KviScriptObjectClassDefinition            * m_pInheritedClass; // the nearest inherited class
		QList<KviScriptObjectClassDefinition>     * m_pChildClasses;   // child classes list
		QList<KviScriptEventStruct>               * m_pEventList;      // list of default event handlers
		bool                                        m_bBuiltin;        // is this a builtin class ?
	public:
		// Global class data
		QAsciiDict<KviScriptObjectFunctionStruct> * functionDict()  { return m_pFunctionDict; };
		QList<KviScriptEventStruct>               * eventList()     { return m_pEventList; };
		KviScriptObjectClassDefinition            * inheritedClass(){ return m_pInheritedClass; };
		QList<KviScriptObjectClassDefinition>     * childClasses()  { return m_pChildClasses; };
		const char                                * getClass()      { return m_szClassName.ptr(); };
		bool                                        isBuiltin()     { return m_bBuiltin; };
		// Lookup utils
		KviScriptObjectClassDefinition            * lookupChildClassDefinition(const char *szClass);
		KviScriptObjectClassDefinition            * lookupParentClassDefinition(const char *szClass);
		KviScriptObjectFunctionStruct             * lookupFunction(const char *szFncName){ return m_pFunctionDict->find(szFncName); };
		// Management
		void addChildClassDefinition(KviScriptObjectClassDefinition * d){ m_pChildClasses->append(d); };
		void addDefaultEvent(KviScriptEventStruct *s);
		void addFunction(const char *fncName,const char * buffer,bool bBuiltin = false);
		void addBuiltinFunction(const char *fncName,scriptObjectFunction fncHandler);
		// debug FIXME : Remove it ?
		void dump(KviWindow * wnd,const char * margin = 0);
};

typedef struct KviScriptObjectSlotStruct
{
	KviScriptObject * pObject;    // destination object
	KviStr            szFunction; // slot function name
} KviScriptObjectSlot;

typedef QList<KviScriptObjectSlot> KviScriptObjectSlotList;

typedef struct KviScriptObjectSignalStruct
{
	KviStr            szSignalName;
	KviStr            szSlotName;
	KviScriptObject * pObject;
} KviScriptObjectSignal;

// A script object : base class
class KviScriptObject : public QObject
{
	Q_OBJECT
	public:
		KviScriptObject(KviScriptObjectController * cntrl,KviScriptObject * p,const char *name,KviScriptObjectClassDefinition * pDef);
		~KviScriptObject();
	private:
		KviStr                           m_szId;               // Unique ID string (@id)
		QList<KviScriptObject>         * m_pChildList;         // list of children objects
		KviScriptObjectController      * m_pController;        // object controller (unique for each frame)
		KviVariableCache               * m_pVarCache;          // object member variables
		QAsciiDict<KviStr>             * m_pEventDict;         // object event handlers
		QList<KviStr>                  * m_pInheritanceList;   // list of inherited classes
		KviScriptObjectClassDefinition * m_pClassDefinition;   // pointer to the global class definition
//		bool                             m_bInEventOrFunction; // actually executing an event or function
		bool                             m_bInDelayedDestroy;  // In DelayedDestroy state!
		bool                             m_bDying;             // In the destructor ?
		bool                             m_bNeedRecreate;      // Need to rise from the ashes after being deleted ?
		QAsciiDict<KviScriptObjectSlotList> * m_pSlotDict;   // dictionary of slots connected to the signals of this object (key = signal name)
		QList<KviScriptObjectSignal>        * m_pSignalList;
	private:
		void                             registerSlot(const char *signalName,const char *slotName,KviScriptObject *object);
		void                             unregisterSlot(const char *signalName,const char *slotName,KviScriptObject *object);
		void                             registerSignal(const char *signalName,const char *slotName,KviScriptObject *object);
		void                             unregisterSignal(const char *signalName,const char *slotName,KviScriptObject *object);
	public:
		static void                      connectSignalSlot(KviScriptObject *src,const char *signalName,KviScriptObject *dst,const char *slotName);
		static void                      disconnectSignalSlot(KviScriptObject *src,const char *signalName,KviScriptObject *dst,const char *slotName);
	public:
		KviScriptObjectController      * controller(){ return m_pController; };

		void                             die(bool bDelayed);
		void                             dieAndRecreate(bool bDelayed);
		void                             dieOutOfThisEventStep(); // sets m_bInDelayedDestroy to true
		bool                             inDelayedDestroy(){ return m_bInDelayedDestroy; };
//		bool                             inEventOrFunction(){ return m_bInEventOrFunction; };

		KviVariableCache       * varCache(){ return m_pVarCache; };
		KviScriptObject        * parentObject(){ return (KviScriptObject *)parent(); };
		QList<KviScriptObject> * childrenList(){ return m_pChildList; };
		KviScriptObject        * childByName(const char * szName);
		const char             * getName(){ return name(); };
		const char             * getClass(){ return m_pClassDefinition->getClass(); };
		KviScriptObjectClassDefinition * classDefinition(){ return m_pClassDefinition; };
		const char             * id(){ return m_szId.ptr(); };
		KviScriptObject        * findObjectByName(const char *szName);
		KviScriptObject        * findObjectByClass(const char *szClass);
		KviScriptObject        * findObjectById(const char *szId);
		void                     triggerEvent(const char *evName,const KviStr &parms);
		bool                     emitSignal(const char *sigName,QList<KviStr> * params,KviStr &buffer);
		void                     removeEventHandler(const char *evName){ m_pEventDict->remove(evName); };
		void                     setEventHandler(const char *evName,const char *evBuffer);
		bool                     hasEventHandler(const char *evName){ return (m_pEventDict->find(evName) != 0); };
		int                      callFunction(const char *fncName,QList<KviStr> * params,KviStr &buffer,const char *classOverride = 0);
		void                     dump(KviWindow *wnd,const char *margin = 0);

		static void              initializeClassDefinition(KviScriptObjectClassDefinition *d);

		int                      builtinFunction_NAME(QList<KviStr> * params,KviStr &buffer);
		int                      builtinFunction_CLASS(QList<KviStr> * params,KviStr &buffer);
		int                      builtinFunction_PARENT(QList<KviStr> * params,KviStr &buffer);
		int                      builtinFunction_INHERITS(QList<KviStr> * params,KviStr &buffer);
		int                      builtinFunction_INHERITEDCLASSES(QList<KviStr> * params,KviStr &buffer);
		int                      builtinFunction_CHILDREN(QList<KviStr> * params,KviStr &buffer);
		int                      builtinFunction_FINDDIRECTCHILD(QList<KviStr> * params,KviStr &buffer);
		int                      builtinFunction_FINDCHILD(QList<KviStr> * params,KviStr &buffer);
		int                      builtinFunction_EMIT(QList<KviStr> * params,KviStr &buffer);

	private:
		void addChild(KviScriptObject * o);
		void removeChild(KviScriptObject * o,bool bDelete = true);
	private slots:
		virtual void delayedDie();
};


class KviScriptObjectController
{
	friend class KviScriptObject;
	public:
		KviScriptObjectController(KviUserParser * pParser);
		~KviScriptObjectController();
	protected:
		KviUserParser          * m_pUserParser;
		KviScriptObject        * m_pTopLevelObject;
	public:
		KviFrame * mainFrame();
		KviScriptObject * topLevelObject(){ return m_pTopLevelObject; };
		void getUniqueId(KviStr &idBuf);
//		KviScriptObject * findObjectByName(const char *szName);
		KviScriptObject * findObjectById(const char *szId); // Lookup a GLOBAL object
		KviScriptObject * findLocalObjectByClass(const char *szClass);
//		KviScriptObject * findGlobalObjectByName(const char *szName);
//		KviScriptObject * findGlobalObjectById(const char *szId);
//		KviScriptObject * findGlobalObjectByClass(const char *szClass);
		bool isKnownObjectClass(const char *szClass);
		KviScriptObject * allocateObject(const char *szClass,const char *szName,KviScriptObject * par,QList<KviStr> * params,bool * pbDeletedParams);
		KviScriptObjectClassDefinition * lookupClassDefinition(const char *szClass);

		// Kill all the object of all the controllers
		// Remove all the classes and recreate the built in ones
		// Recreate the toplevel objects then
		void globalReset();

		// Kill all objects with a specific class
		bool killClass(const char *szClassName);
		void dumpClasses(KviWindow *wnd);
		void dumpObjects(KviWindow *wnd);
	private:
		void killClass(KviScriptObjectClassDefinition * d);
		void killAllLocalObjectsWithClass(const char * szClassName);
		void allocateBuiltInClassDefinitions();
		void localCleanup();
		void createTopLevelObject();
};

inline bool KviScriptObjectController::isKnownObjectClass(const char *szClass)
{
	return (lookupClassDefinition(szClass) != 0);
}

#endif //_KVI_SCRIPTOBJECT_H_DEFINED_
