//
//   File : kvi_systray.cpp (/usr/build/NEW_kvirc/kvirc/src/kvirc/kvi_systray.cpp)
//   Last major modification : Tue Apr 13 1999 17:55:15 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

//#define _KVI_DEBUG_CLASS_NAME_ "KviSysTray"
#define _KVI_DEBUG_CHECK_RANGE_
#include "kvi_debug.h"
#include "kvi_systray.h"
#include "kvi_options.h"
#include "kvi_locale.h"
#include "kvi_frame.h"

#include <qnamespace.h>

#include <qbitmap.h>
#include <qtooltip.h>


/*
	@quickhelp: KviSysTray
	@widget: System display
		This is the system display.<br>
		It can act as a clock , a GMT clock , an On-Line timer or an I/O Led.<br>
		By clicking in the left corner of this window you will cyclically
		switch between off and these four modes.<br>
		<br>
		Position 1 : Off<br>
		Position 2 : Local time clock<br>
		&nbsp;&nbsp;&nbsp;&nbsp;Should be in sync with your system clock.<br>
		&nbsp;&nbsp;&nbsp;&nbsp;Shows the current date and time.<br>
		&nbsp;&nbsp;&nbsp;&nbsp;If you see three letters (DST) on the right side<br>
		&nbsp;&nbsp;&nbsp;&nbsp;that means that the daylight saving time is in effect.<br>
		Position 3 : GMT clock<br>
		&nbsp;&nbsp;&nbsp;&nbsp;If your timezone is set correctly<br>
		&nbsp;&nbsp;&nbsp;&nbsp;this clock shows the current Universal Coordinated Time and date.<br>
		Position 4 : On-Line timer<br>
		&nbsp;&nbsp;&nbsp;&nbsp;This one works only when you are connected to a server.<br>
		&nbsp;&nbsp;&nbsp;&nbsp;It shows the time that the connection started and the<br>
		&nbsp;&nbsp;&nbsp;&nbsp;duration.<br>
		Position 5 : I/O Led<br>
		&nbsp;&nbsp;&nbsp;&nbsp;Input - Output "main socket traffic".<br>
		&nbsp;&nbsp;&nbsp;&nbsp;The upper indicator show the received packets of data,<br>
		&nbsp;&nbsp;&nbsp;&nbsp;and the lower the sent ones.<br>
		<br>
		Other modes can be added by plugins.<br>
*/

extern Display *g_display;
// declared and managed in kvi_app.cpp
extern QPixmap *g_pSysTrayBigIcons[4];

#ifdef COMPILE_USE_AA_FONTS
	extern XftFont        * g_pXftFont;
	extern XftDraw        * g_pXftDraw;
	extern int qt_use_xft (void); // qpainter_x11.cpp
	extern void *qt_ft_font (const QFont *f); // qfont_x11.cpp
	extern XftDraw * qt_lookup_ft_draw (Drawable draw, bool paintEventClipOn, QRegion *crgn);
#endif

static char * days_of_week[7]={
	_i19n_("Sun"),_i19n_("Mon"),_i19n_("Tue"),_i19n_("Wed"),
	_i19n_("Thu"),_i19n_("Fri"),_i19n_("Sat"),
};

static char * months_of_year[12]={
	_i19n_("Jan"),_i19n_("Feb"),_i19n_("Mar"),
	_i19n_("Apr"),_i19n_("May"),_i19n_("Jun"),
	_i19n_("Jul"),_i19n_("Aug"),_i19n_("Sep"),
	_i19n_("Oct"),_i19n_("Nov"),_i19n_("Dec")
};

void kvi_timeFormatDuration(time_t diff,KviStr &buffer)
{
	int days = diff / 86400; //86400 secs in a day---days
	diff = diff % 86400;
	int hours = diff / 3600; //3600 secs in a hour
	diff = diff % 3600;
	int mins = diff / 60; //60 secs in a minute
	diff = diff % 60; //tim = secs!
	buffer.sprintf(__tr("%d d %d h %d m %d s"),days,hours,mins,diff);
}



///////////////////////////////////////////////////////////////////////////////////////////////

KviSysTrayIconDisplay::KviSysTrayIconDisplay(KviSysTray * parent)
:QWidget(parent,"systray_icon_display")
{
	m_pIcon = 0;
	m_pSysTray = parent;
	setBackgroundMode(NoBackground);
	QToolTip::add(this,__tr("Change System Tray display mode"));
}

KviSysTrayIconDisplay::~KviSysTrayIconDisplay()
{
	if(m_pIcon)delete m_pIcon;
}

void KviSysTrayIconDisplay::setIcon(QPixmap * pix)
{
	if(m_pIcon)delete m_pIcon;
	m_pIcon = 0;
	if(pix)m_pIcon = new QIconSet(*pix);
	update();
}

void KviSysTrayIconDisplay::paintEvent(QPaintEvent *)
{
	if(!isVisibleToTLW())return;

	HANDLE hMemBuf   = m_pSysTray->getMemoryBufferHandle();
	GC the_gc        = XCreateGC(g_display,hMemBuf,0,0);

	int widgetWidth  = width();
	int widgetHeight = height();

	XSetBackground(g_display,the_gc,m_pSysTray->getBackgroundPixel());
	XSetForeground(g_display,the_gc,m_pSysTray->getBackgroundPixel());
	XSetFillStyle(g_display,the_gc,FillSolid);
	XFillRectangle(g_display,hMemBuf,the_gc,0,0,widgetWidth,widgetHeight);

	if(m_pIcon){
		QPixmap pix = m_pIcon->pixmap((height() > 32) ? QIconSet::Large : QIconSet::Small,true);
		int pos = (widgetHeight - pix.height())/2;
		if(pix.mask()){
			XSetClipMask(g_display,the_gc,pix.mask()->handle());
			XSetClipOrigin(g_display,the_gc,pos,pos);
		}
		//Draw the pixmap
		XCopyArea(g_display,pix.handle(),hMemBuf,the_gc,0,0,pix.width(),pix.height(),pos,pos);
		XSetClipMask(g_display,the_gc,None);
	}

	XSetLineAttributes(g_display,the_gc,1,LineSolid,CapButt,JoinMiter);
	XSetForeground(g_display,the_gc,colorGroup().dark().pixel());
	XDrawLine(g_display,hMemBuf,the_gc,0,0,widgetWidth-1,0);
	XDrawLine(g_display,hMemBuf,the_gc,0,0,0,widgetHeight);
	XSetForeground(g_display,the_gc,colorGroup().light().pixel());
	XDrawLine(g_display,hMemBuf,the_gc,1,widgetHeight-1,widgetWidth,widgetHeight-1);
	XDrawLine(g_display,hMemBuf,the_gc,widgetWidth-1,0,widgetWidth-1,widgetHeight);

	XCopyArea(g_display,hMemBuf,this->handle(),the_gc,0,0,widgetWidth,widgetHeight,0,0);
	XFreeGC(g_display,the_gc);
}

void KviSysTrayIconDisplay::mousePressEvent(QMouseEvent *)
{
	m_pSysTray->rotateMode();
}

////////////////////////////////////////////////////////////////////////////////////////

KviSysTrayWidget::KviSysTrayWidget(KviSysTray * parent,const char *tooltip,const char * name)
:QWidget(parent,name ? name : "unnamed_systray_widget")
{
	m_pSysTray = parent;
	hide();
	if(tooltip)QToolTip::add(this,tooltip);
	m_plugin_handle = 0;
}

KviSysTrayWidget::~KviSysTrayWidget()
{
}

QPixmap * KviSysTrayWidget::icon()
{
	return 0;
}

void KviSysTrayWidget::paintDefaultBackground(HANDLE hMemBuf,GC the_gc)
{
	XSetBackground(g_display,the_gc,m_pSysTray->getBackgroundPixel());
	XSetForeground(g_display,the_gc,m_pSysTray->getBackgroundPixel());
	XSetFillStyle(g_display,the_gc,FillSolid);
	XFillRectangle(g_display,hMemBuf,the_gc,0,0,width(),height());
}

void KviSysTrayWidget::paintDefaultFrame(HANDLE hMemBuf,GC the_gc)
{
	XSetLineAttributes(g_display,the_gc,1,LineSolid,CapButt,JoinMiter);
	XSetForeground(g_display,the_gc,colorGroup().dark().pixel());
	XDrawLine(g_display,hMemBuf,the_gc,0,0,width()-1,0);
	XDrawLine(g_display,hMemBuf,the_gc,0,0,0,height());
	XSetForeground(g_display,the_gc,colorGroup().light().pixel());
	XDrawLine(g_display,hMemBuf,the_gc,1,height()-1,width(),height()-1);
	XDrawLine(g_display,hMemBuf,the_gc,width()-1,0,width()-1,height());
}

////////////////////////////////////////////////////////////////////////////////////////

KviSysTrayLogo::KviSysTrayLogo(KviSysTray * parent)
:KviSysTrayWidget(parent,__tr("Logo"),"systray_logo")
{
}

KviSysTrayLogo::~KviSysTrayLogo()
{
}

void KviSysTrayLogo::paintEvent(QPaintEvent *)
{
	if(!isVisibleToTLW())return;

	HANDLE hMemBuf   = m_pSysTray->getMemoryBufferHandle();
	GC the_gc        = XCreateGC(g_display,hMemBuf,0,0);

	paintDefaultBackground(hMemBuf,the_gc);
	paintDefaultFrame(hMemBuf,the_gc);

	XCopyArea(g_display,hMemBuf,this->handle(),the_gc,0,0,width(),height(),0,0);
	XFreeGC(g_display,the_gc);
}

////////////////////////////////////////////////////////////////////////////////////////

KviSysTrayClock::KviSysTrayClock(KviSysTray * parent)
:KviSysTrayWidget(parent,__tr("Local Clock"),"systray_local_clock")
{
	m_timer = 0;
	setBackgroundMode(NoBackground);
}

KviSysTrayClock::KviSysTrayClock(KviSysTray * parent,const char *tooltip,const char * name)
:KviSysTrayWidget(parent,tooltip,name)
{
	m_timer = 0;
	setBackgroundMode(NoBackground);
}

KviSysTrayClock::~KviSysTrayClock()
{
	if(m_timer)killTimer(m_timer);
}

void KviSysTrayClock::show()
{
	m_timer = startTimer(1000);
	KviSysTrayWidget::show();
}

void KviSysTrayClock::hide()
{
	if(m_timer){
		killTimer(m_timer);
		m_timer = 0;
	}
	KviSysTrayWidget::hide();
}

void KviSysTrayClock::timerEvent(QTimerEvent *e)
{
	if(e->timerId() == m_timer)paintEvent(0);
}

struct tm * KviSysTrayClock::getTime(const time_t * timep)
{
	return localtime(timep);
}

QPixmap * KviSysTrayClock::icon()
{
	return g_pSysTrayBigIcons[0];
}


void KviSysTrayClock::paintEvent(QPaintEvent *)
{
	if(!isVisibleToTLW())return;

	HANDLE hMemBuf   = m_pSysTray->getMemoryBufferHandle();
	GC the_gc        = XCreateGC(g_display,hMemBuf,0,0);

	int widgetWidth  = width();
	int widgetHeight = height();

	paintDefaultBackground(hMemBuf,the_gc);

	XSetForeground(g_display,the_gc,m_pSysTray->getForegroundPixel());
	time_t tim = ::time(0);
	struct tm *t = getTime(&tim);
	KviStr timeStr(KviStr::Format,"%d%d:%d%d:%d%d",t->tm_hour / 10,t->tm_hour % 10,
				t->tm_min  / 10,t->tm_min  % 10,t->tm_sec  / 10,t->tm_sec  % 10);
	if(t->tm_isdst > 0)timeStr += __tr(" (DST)");
	KviStr dateStr(KviStr::Format,__tr("%s %d %s %d (Day %d)"),_i18n_translateNoText(days_of_week[t->tm_wday]),t->tm_mday,
			_i18n_translateNoText(months_of_year[t->tm_mon]),1900 + t->tm_year,t->tm_yday);


	if(height() > 32)
	{
		// Big mode
		g_pOptions->m_fntSysTray.setPointSize(16);
#ifdef COMPILE_USE_AA_FONTS
		if(qt_use_xft())
		{
			g_pXftFont = (XftFont *)qt_ft_font(&(g_pOptions->m_fntSysTray));
			g_pXftDraw = qt_lookup_ft_draw (hMemBuf,false,0);
			if(!g_pXftDraw)
			{
				XSetFont(g_display,the_gc,g_pOptions->m_fntSysTray.handle());
				g_pXftFont = 0;
			}
			
		} else {
#endif
			XSetFont(g_display,the_gc,g_pOptions->m_fntSysTray.handle());
#ifdef COMPILE_USE_AA_FONTS
			g_pXftFont = 0;
			g_pXftDraw = 0;
		}
		if(g_pXftFont){
			XftColor color;
			QColor * clr = &(g_pOptions->m_clrSysTrayFore);
			color.color.red = clr->red() | clr->red() << 8;
			color.color.green = clr->green() | clr->green() << 8;
			color.color.blue = clr->blue() | clr->blue() << 8;
			color.color.alpha = 0xffff;
			color.pixel = clr->pixel();
			XftDrawString8(g_pXftDraw,&color,g_pXftFont,4,height() - 6,
				(unsigned char *)timeStr.ptr(),timeStr.len());
		} else
#endif
			XDrawString(g_display,hMemBuf,the_gc,4,height() - 6,timeStr.ptr(),timeStr.len());
			// kvi_getDate(t,tmp);
			g_pOptions->m_fntSysTray.setPointSize(10);
#ifdef COMPILE_USE_AA_FONTS
		if(qt_use_xft())
		{
			g_pXftFont = (XftFont *)qt_ft_font(&(g_pOptions->m_fntSysTray));
			g_pXftDraw = qt_lookup_ft_draw (hMemBuf,false,0);
			if(!g_pXftDraw)
			{
				XSetFont(g_display,the_gc,g_pOptions->m_fntSysTray.handle());
				g_pXftFont = 0;
			}
			
		} else {
#endif
			XSetFont(g_display,the_gc,g_pOptions->m_fntSysTray.handle());
#ifdef COMPILE_USE_AA_FONTS
			g_pXftFont = 0;
			g_pXftDraw = 0;
		}
		if(g_pXftFont){
			XftColor color;
			QColor * clr = &(g_pOptions->m_clrSysTrayFore);
			color.color.red = clr->red() | clr->red() << 8;
			color.color.green = clr->green() | clr->green() << 8;
			color.color.blue = clr->blue() | clr->blue() << 8;
			color.color.alpha = 0xffff;
			color.pixel = clr->pixel();
			XftDrawString8(g_pXftDraw,&color,g_pXftFont,4,14,
				(unsigned char *)dateStr.ptr(),dateStr.len());
		} else
#endif
			XDrawString(g_display,hMemBuf,the_gc,4,14,dateStr.ptr(),dateStr.len());
	} else {
		// Small mode
		timeStr.append(" - ");
		timeStr.append(dateStr);
		g_pOptions->m_fntSysTray.setPointSize(14);
#ifdef COMPILE_USE_AA_FONTS
		if(qt_use_xft())
		{
			g_pXftFont = (XftFont *)qt_ft_font(&(g_pOptions->m_fntSysTray));
			g_pXftDraw = qt_lookup_ft_draw (hMemBuf,false,0);
			if(!g_pXftDraw)
			{
				XSetFont(g_display,the_gc,g_pOptions->m_fntSysTray.handle());
				g_pXftFont = 0;
			}
			
		} else {
#endif
			XSetFont(g_display,the_gc,g_pOptions->m_fntSysTray.handle());
#ifdef COMPILE_USE_AA_FONTS
			g_pXftFont = 0;
			g_pXftDraw = 0;
		}
		if(g_pXftFont){
			XftColor color;
			QColor * clr = &(g_pOptions->m_clrSysTrayFore);
			color.color.red = clr->red() | clr->red() << 8;
			color.color.green = clr->green() | clr->green() << 8;
			color.color.blue = clr->blue() | clr->blue() << 8;
			color.color.alpha = 0xffff;
			color.pixel = clr->pixel();
			XftDrawString8(g_pXftDraw,&color,g_pXftFont,4,(height() + 12) / 2,
				(unsigned char *)timeStr.ptr(),timeStr.len());
		} else
#endif
			XDrawString(g_display,hMemBuf,the_gc,4,(height() + 12) / 2,timeStr.ptr(),timeStr.len());
	}

	paintDefaultFrame(hMemBuf,the_gc);

	XCopyArea(g_display,hMemBuf,this->handle(),the_gc,0,0,widgetWidth,widgetHeight,0,0);
	XFreeGC(g_display,the_gc);
}

////////////////////////////////////////////////////////////////////////////////////////

KviSysTrayUtcClock::KviSysTrayUtcClock(KviSysTray * parent)
:KviSysTrayClock(parent,__tr("UTC Clock"),"systray_utc_clock")
{
}

KviSysTrayUtcClock::~KviSysTrayUtcClock()
{
}

struct tm * KviSysTrayUtcClock::getTime(const time_t * timep)
{
	return gmtime(timep);
}

QPixmap * KviSysTrayUtcClock::icon()
{
	return g_pSysTrayBigIcons[1];
}

////////////////////////////////////////////////////////////////////////////////////////

KviSysTrayOnLineTimer::KviSysTrayOnLineTimer(KviSysTray * parent)
:KviSysTrayWidget(parent,__tr("Online Timer"),"systray_online_timer")
{
	m_timer = 0;
	m_iConnectionStartTime = 0;
	setBackgroundMode(NoBackground);
}

KviSysTrayOnLineTimer::~KviSysTrayOnLineTimer()
{
	if(m_timer)killTimer(m_timer);
}

QPixmap * KviSysTrayOnLineTimer::icon()
{
	return g_pSysTrayBigIcons[2];
}


void KviSysTrayOnLineTimer::show()
{
	m_timer = startTimer(1000);
	KviSysTrayWidget::show();
}

void KviSysTrayOnLineTimer::hide()
{
	if(m_timer){
		killTimer(m_timer);
		m_timer = 0;
	}
	KviSysTrayWidget::hide();
}

void KviSysTrayOnLineTimer::timerEvent(QTimerEvent *e)
{
	if(e->timerId() == m_timer)paintEvent(0);
}

void KviSysTrayOnLineTimer::run()
{
	m_iConnectionStartTime = time(0);
	if(isVisibleToTLW())paintEvent(0);
}

void KviSysTrayOnLineTimer::stop()
{
	m_iConnectionStartTime = 0;
	if(isVisibleToTLW())paintEvent(0);
}

void KviSysTrayOnLineTimer::getConnectionTime(KviStr &buffer)
{
	if(m_iConnectionStartTime==0)kvi_timeFormatDuration((time_t)0,buffer);
	else kvi_timeFormatDuration((time_t)(::time(0) - m_iConnectionStartTime),buffer);
}

int KviSysTrayOnLineTimer::getIntConnectionTime()
{
	if(m_iConnectionStartTime==0) return 0;
	else return (int)(::time(0) - m_iConnectionStartTime);
}

void KviSysTrayOnLineTimer::paintEvent(QPaintEvent *)
{
	if(!isVisibleToTLW())return;

	HANDLE hMemBuf   = m_pSysTray->getMemoryBufferHandle();
	GC the_gc        = XCreateGC(g_display,hMemBuf,0,0);

	int widgetWidth  = width();
	int widgetHeight = height();

	paintDefaultBackground(hMemBuf,the_gc);

	KviStr timeStr;
	KviStr dateStr;

	XSetForeground(g_display,the_gc,m_pSysTray->getForegroundPixel());
	getConnectionTime(timeStr);
	if(m_iConnectionStartTime == 0)dateStr=__tr("Not connected");
	else {
		struct tm *t = localtime(&(m_iConnectionStartTime));
		dateStr.sprintf(__tr("Started at %d%d:%d%d:%d%d"),t->tm_hour / 10,t->tm_hour % 10,
			t->tm_min  / 10,t->tm_min  % 10,t->tm_sec  / 10,t->tm_sec  % 10);
		if(t->tm_isdst > 0)dateStr += __tr(" (DST)");
	}

	if(height() > 32)
	{
		// Big mode
		g_pOptions->m_fntSysTray.setPointSize(16);
#ifdef COMPILE_USE_AA_FONTS
		if(qt_use_xft())
		{
			g_pXftFont = (XftFont *)qt_ft_font(&(g_pOptions->m_fntSysTray));
			g_pXftDraw = qt_lookup_ft_draw (hMemBuf,false,0);
			if(!g_pXftDraw)
			{
				XSetFont(g_display,the_gc,g_pOptions->m_fntSysTray.handle());
				g_pXftFont = 0;
			}
			
		} else {
#endif
			XSetFont(g_display,the_gc,g_pOptions->m_fntSysTray.handle());
#ifdef COMPILE_USE_AA_FONTS
			g_pXftFont = 0;
			g_pXftDraw = 0;
		}
		if(g_pXftFont){
			XftColor color;
			QColor * clr = &(g_pOptions->m_clrSysTrayFore);
			color.color.red = clr->red() | clr->red() << 8;
			color.color.green = clr->green() | clr->green() << 8;
			color.color.blue = clr->blue() | clr->blue() << 8;
			color.color.alpha = 0xffff;
			color.pixel = clr->pixel();
			XftDrawString8(g_pXftDraw,&color,g_pXftFont,4,height() - 6,
				(unsigned char *)timeStr.ptr(),timeStr.len());
		} else
#endif
			XDrawString(g_display,hMemBuf,the_gc,4,height() - 6,timeStr.ptr(),timeStr.len());
		// kvi_getDate(t,tmp);
		g_pOptions->m_fntSysTray.setPointSize(10);
#ifdef COMPILE_USE_AA_FONTS
		if(qt_use_xft())
		{
			g_pXftFont = (XftFont *)qt_ft_font(&(g_pOptions->m_fntSysTray));
			g_pXftDraw = qt_lookup_ft_draw (hMemBuf,false,0);
			if(!g_pXftDraw)
			{
				XSetFont(g_display,the_gc,g_pOptions->m_fntSysTray.handle());
				g_pXftFont = 0;
			}
			
		} else {
#endif
			XSetFont(g_display,the_gc,g_pOptions->m_fntSysTray.handle());
#ifdef COMPILE_USE_AA_FONTS
			g_pXftFont = 0;
			g_pXftDraw = 0;
		}
		if(g_pXftFont){
			XftColor color;
			QColor * clr = &(g_pOptions->m_clrSysTrayFore);
			color.color.red = clr->red() | clr->red() << 8;
			color.color.green = clr->green() | clr->green() << 8;
			color.color.blue = clr->blue() | clr->blue() << 8;
			color.color.alpha = 0xffff;
			color.pixel = clr->pixel();
			XftDrawString8(g_pXftDraw,&color,g_pXftFont,4,14,
				(unsigned char *)dateStr.ptr(),dateStr.len());
		} else
#endif
			XDrawString(g_display,hMemBuf,the_gc,4,14,dateStr.ptr(),dateStr.len());
	} else {
		// Small mode
		timeStr.append(" - ");
		timeStr.append(dateStr);
		g_pOptions->m_fntSysTray.setPointSize(14);
#ifdef COMPILE_USE_AA_FONTS
		if(qt_use_xft())
		{
			g_pXftFont = (XftFont *)qt_ft_font(&(g_pOptions->m_fntSysTray));
			g_pXftDraw = qt_lookup_ft_draw (hMemBuf,false,0);
			if(!g_pXftDraw)
			{
				XSetFont(g_display,the_gc,g_pOptions->m_fntSysTray.handle());
				g_pXftFont = 0;
			}
			
		} else {
#endif
			XSetFont(g_display,the_gc,g_pOptions->m_fntSysTray.handle());
#ifdef COMPILE_USE_AA_FONTS
			g_pXftFont = 0;
			g_pXftDraw = 0;
		}
		if(g_pXftFont){
			XftColor color;
			QColor * clr = &(g_pOptions->m_clrSysTrayFore);
			color.color.red = clr->red() | clr->red() << 8;
			color.color.green = clr->green() | clr->green() << 8;
			color.color.blue = clr->blue() | clr->blue() << 8;
			color.color.alpha = 0xffff;
			color.pixel = clr->pixel();
			XftDrawString8(g_pXftDraw,&color,g_pXftFont,4,(height() + 12) / 2,
				(unsigned char *)timeStr.ptr(),timeStr.len());
		} else
#endif
			XDrawString(g_display,hMemBuf,the_gc,4,(height() + 12) / 2,timeStr.ptr(),timeStr.len());
	}


	paintDefaultFrame(hMemBuf,the_gc);

	XCopyArea(g_display,hMemBuf,this->handle(),the_gc,0,0,widgetWidth,widgetHeight,0,0);
	XFreeGC(g_display,the_gc);
}

////////////////////////////////////////////////////////////////////////////////////////

KviSysTrayIoLed::KviSysTrayIoLed(KviSysTray * parent)
:KviSysTrayWidget(parent,__tr("I/O Meter\nUpper bar: Input packets\nLower bar: Output packets"),"systray_io_led")
{
	m_timer   = 0;
	m_iOutLed = 0;
	m_iInLed  = 0;
	setBackgroundMode(NoBackground);
}

KviSysTrayIoLed::~KviSysTrayIoLed()
{
	if(m_timer)killTimer(m_timer);
}

QPixmap * KviSysTrayIoLed::icon()
{
	return g_pSysTrayBigIcons[3];
}


void KviSysTrayIoLed::show()
{
	m_timer = startTimer(1000);
	KviSysTrayWidget::show();
}

void KviSysTrayIoLed::hide()
{
	if(m_timer){
		killTimer(m_timer);
		m_timer   = 0;
		m_iOutLed = 0;
		m_iInLed  = 0;
	}
	KviSysTrayWidget::hide();
}

void KviSysTrayIoLed::clear()
{
	m_iOutLed = 0;
	m_iInLed = 0;
	update();
}

void KviSysTrayIoLed::timerEvent(QTimerEvent *e)
{
	if(e->timerId() == m_timer){
		if(m_iInLed | m_iOutLed){
			if(m_iInLed > 0)m_iInLed--;
			if(m_iOutLed > 0)m_iOutLed--;
			paintEvent(0);
		}
	}
}

void KviSysTrayIoLed::dataMessageReceived()
{
	if(!isVisibleToTLW())return;
	m_iInLed++;
	paintEvent(0);
}

void KviSysTrayIoLed::dataMessageSent()
{
	if(!isVisibleToTLW())return;
	m_iOutLed+=2;
	paintEvent(0);
}

void KviSysTrayIoLed::paintEvent(QPaintEvent *)
{
	if(!isVisibleToTLW())return;

	HANDLE hMemBuf   = m_pSysTray->getMemoryBufferHandle();
	GC the_gc        = XCreateGC(g_display,hMemBuf,0,0);

	int widgetWidth  = width();
	int widgetHeight = height();

	paintDefaultBackground(hMemBuf,the_gc);

	int hght = (widgetHeight-12)/2;
	XSetForeground(g_display,the_gc,Qt::green.pixel());
	if(m_iInLed > 0){
		XSetLineAttributes(g_display,the_gc,2,LineSolid,CapButt,JoinMiter);
		for(int i=0;i<m_iInLed;i++)
		{
			if(i == 10)XSetForeground(g_display,the_gc,Qt::yellow.pixel());
			else if(i == 20)XSetForeground(g_display,the_gc,Qt::red.pixel());
			int x1 = 4 + (i * 3);
			if(x1 > widgetWidth){
				m_iInLed = i; //stop increasing...
				break;
			}
			XDrawLine(g_display,hMemBuf,the_gc,x1,4,x1,4+hght);
		}
	}

	int y1 = hght+8;
	XSetForeground(g_display,the_gc,Qt::green.pixel());
	if(m_iOutLed > 0){
		XSetLineAttributes(g_display,the_gc,2,LineSolid,CapButt,JoinMiter);
		for(int i=0;i<m_iOutLed;i++)
		{
			if(i == 10)XSetForeground(g_display,the_gc,Qt::yellow.pixel());
			else if(i == 20)XSetForeground(g_display,the_gc,Qt::red.pixel());
			int x1 = 4 + (i * 3);
			if(x1 > widgetWidth)break;
			XDrawLine(g_display,hMemBuf,the_gc,x1,y1,x1,y1+hght);
		}
	}

	paintDefaultFrame(hMemBuf,the_gc);

	XCopyArea(g_display,hMemBuf,this->handle(),the_gc,0,0,widgetWidth,widgetHeight,0,0);
	XFreeGC(g_display,the_gc);
}


////////////////////////////////////////////////////////////////////////////////////////

KviSysTray::KviSysTray(QWidget * parent)
:QWidget(parent,"systray")
{
	setBackgroundMode(QWidget::NoBackground);
	m_pIconDisplay = new KviSysTrayIconDisplay(this);

	m_pWidgetList = new QList<KviSysTrayWidget>;
	m_pWidgetList->setAutoDelete(true);

	m_pWidgetList->append(new KviSysTrayLogo(this));
	m_pWidgetList->append(new KviSysTrayClock(this));
	m_pWidgetList->append(new KviSysTrayUtcClock(this));
	m_pWidgetList->append(new KviSysTrayOnLineTimer(this));
	m_pWidgetList->append(new KviSysTrayIoLed(this));

	m_pCurrentWidget = findSysTrayWidget(g_pOptions->m_szSysTrayMode.ptr());
	if(!m_pCurrentWidget){
		m_pCurrentWidget = findSysTrayWidget("KviSysTrayLogo");
		__range_valid(m_pCurrentWidget);
		g_pOptions->m_szSysTrayMode = "KviSysTrayLogo";
	}

	m_pCurrentWidget->show();
	m_pIconDisplay->setIcon(m_pCurrentWidget->icon());
	setMinimumSize(QSize(180,20));
	m_pMemBuffer = new QPixmap(width(),height());
}

KviSysTray::~KviSysTray()
{
	delete m_pWidgetList;
	delete m_pMemBuffer;
}

QSize KviSysTray::sizeHint()
{
	return QSize(minimumSize().height() + 180,minimumSize().height());
}

void KviSysTray::resizeEvent(QResizeEvent *e)
{
	m_pIconDisplay->setGeometry(0,0,height(),height());
	m_pCurrentWidget->setGeometry(height(),0,width() - height(),height());
	m_pMemBuffer->resize(width(),height());
	QWidget::resizeEvent(e);
}

void KviSysTray::addPluginWidget(KviSysTrayWidget *w, bool bShow)
{
	m_pWidgetList->append(w);
	
	if(bShow)setCurrentWidget(w);
}

bool KviSysTray::setCurrentWidget(KviSysTrayWidget *w)
{
	if(m_pCurrentWidget == w)return true; //already current

	for(KviSysTrayWidget *wdg = m_pWidgetList->first();wdg;wdg = m_pWidgetList->next())
	{
		if(wdg == w)
		{
			if(m_pCurrentWidget != 0)m_pCurrentWidget->hide();
			m_pCurrentWidget = w;
			w->setGeometry(height(),0,width() - height(),height());
			w->raise();
			w->show();
			m_pIconDisplay->setIcon(w->icon());
			g_pOptions->m_szSysTrayMode = w->className();
			return true;
		}
	}
	return false; // not found!
}

void KviSysTray::removeWidget(KviSysTrayWidget * w,bool bDelete)
{
	m_pWidgetList->setAutoDelete(bDelete);
	if(w == m_pCurrentWidget)rotateMode();
	if(w == m_pCurrentWidget)m_pCurrentWidget = 0;
	m_pWidgetList->removeRef(w);
	m_pWidgetList->setAutoDelete(true);
}

void KviSysTray::killPluginWidgets(void * plugin_handle)
{
	QList<KviSysTrayWidget> list;
	list.setAutoDelete(false);
	for(KviSysTrayWidget * w = m_pWidgetList->first();w;w=m_pWidgetList->next())
	{
		if(w->m_plugin_handle == plugin_handle)list.append(w);
	}
	for(KviSysTrayWidget * w = list.first();w;w = list.next())
	{
		//autoDelete is true!
		if(w == m_pCurrentWidget)rotateMode();
		if(w == m_pCurrentWidget)m_pCurrentWidget = 0;
		m_pWidgetList->removeRef(w);
	}
}

void KviSysTray::rotateMode()
{
	if(m_pWidgetList->count() <= 1)return;
	for(KviSysTrayWidget * w=m_pWidgetList->first();w;w=m_pWidgetList->next()){
		if(w == m_pCurrentWidget){
			w = m_pWidgetList->next();
			if(!w)w = m_pWidgetList->first();
			__range_valid(w);
			m_pCurrentWidget->hide();
			m_pCurrentWidget = w;
			m_pIconDisplay->setIcon(w->icon());
			w->setGeometry(height(),0,width() - height(),height());
			w->raise();
			w->show();
			g_pOptions->m_szSysTrayMode = w->className();
			return;
		}
	}
}

unsigned long KviSysTray::getBackgroundPixel()
{
	return g_pOptions->m_clrSysTrayBack.pixel();
}

unsigned long KviSysTray::getForegroundPixel()
{
	return g_pOptions->m_clrSysTrayFore.pixel();
}

KviSysTrayWidget * KviSysTray::findSysTrayWidget(const char * classname)
{
	for(KviSysTrayWidget * w = m_pWidgetList->first();w;w=m_pWidgetList->next()){
		KviStr tmp = w->className();
		if(kvi_strEqualCI(tmp.ptr(),classname))return w;
	}
	return 0;
}


KviSysTrayBar::KviSysTrayBar(KviFrame *parent,QMainWindow::ToolBarDock dock)
:QToolBar("SysTrayBar",parent,dock,false,"systray_toolbar")
{
	m_pSysTray = new KviSysTray(this);
	setHorizontalStretchable(true);
	setVerticalStretchable(false);
	setStretchableWidget(m_pSysTray);
	setBackgroundMode(QWidget::PaletteBackground);
}

KviSysTrayBar::~KviSysTrayBar()
{
}


#include "m_kvi_systray.moc"
