// =============================================================================
//
//      --- kvi_ipeditor.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviIpEditor"

#include "kvi_ipeditor.h"
#include "kvi_label.h"
#include "kvi_lineedit.h"
#include "kvi_string.h"

KviIpEditor::KviIpEditor(QWidget *parent, const QString &ipAddr, const char *name)
	: QHBox(parent, name)
{
	QFontMetrics fm(font());
	int minWidth = fm.width("000") + 4;
	m_iSizeHint  = 0;

	for( int i = 0; i < 4; i++ ) {
		m_pEdit[i] = new KviLineEdit(this);
		m_pEdit[i]->installEventFilter(this);
		m_pEdit[i]->setFrame(false);
		m_pEdit[i]->setAlignment(Qt::AlignCenter);
		m_pEdit[i]->setMinimumWidth(minWidth);
		m_pEdit[i]->setMaxLength(3);
		m_iSizeHint += minWidth;

		if( i < 3) {
			m_pLabel[i] = new KviLabel(".", this);
			m_pLabel[i]->setBackgroundMode(PaletteBase);
			m_iSizeHint += fm.width(".") + 4;
		}
	}

	setFrameStyle(QFrame::Sunken | QFrame::WinPanel);
	if( ipAddr.isNull() )
		setAddress("0.0.0.0");
	else
		setAddress(ipAddr);
}

KviIpEditor::~KviIpEditor()
{
	// Nothing here
}

void KviIpEditor::setEnabled(bool bEnabled)
{
	QHBox::setEnabled(bEnabled);
	for( int i = 0; i < 4; i++ ) {
		m_pEdit[i]->setEnabled(bEnabled);
		if( i < 3 ) {
			m_pLabel[i]->setEnabled(bEnabled);
			m_pLabel[i]->setBackgroundMode(isEnabled() ? PaletteBase : PaletteBackground);
		}
	}
}

/**
 * Sets the address displayed by the widget.
 */
bool KviIpEditor::setAddress(const QString &ipAddr)
{
	if( ipAddr.isEmpty() ) return false;

	QCString ip = ipAddr.ascii(); // IP addresses are digits & latin letters abcdef (IPv6)
	ip = ip.stripWhiteSpace();
	const char *c = ip.data();

	if( !c ) return false; // Huh? Should never happen at this point

	for( int i = 0; i < 4; i++ ) {
		const char *anchor = c;
		while( isdigit(*c) ) c++;
		if(c == anchor ) return false; // Invalid empty field
		QCString str(anchor, (c - anchor) + 1);
		bool bOk;
		int num = str.toInt(&bOk);
		if( !bOk ) return false; // Should never happen, but just to be sure
		if( (num < 0) || (num > 255) ) return false; // Invalid field
		m_pEdit[i]->setText(str.data());
		if( i < 3 ) {
			if( *c == '.' ) c++;
			else return false; // Missing separator
		}
	}

	return (*c == '\0'); // Trailing garbage check (we could avoid this)
}

/**
 * Returns the edited IP address in the string form.<br>
 * The string returned is always a valid IP address.<br>
 * In the IPv4 mode, blank fields are treated as zeroes
 * and leading zeroes are removed.
 */
QString KviIpEditor::address()
{
	QString ret;

	for( int i = 0; i < 4; i++ ) {
		QString tmp = m_pEdit[i]->text();
		bool bOk;
		int num = tmp.toInt(&bOk);
		if( !bOk ) num = 0;
		tmp.setNum(num);
		ret.append(tmp);
		if( i < 3 ) ret.append(".");
	}
	return ret;
}

bool KviIpEditor::eventFilter(QObject *o, QEvent *e)
{
	if( o->inherits("KviLineEdit") ) {
		if( e->type() == QEvent::KeyPress ) {
			QString s;
			// Find the editor
			int edIdx = -1;
			for( int i = 0; i < 4; i++ ) {
				if( m_pEdit[i] == o ) {
					edIdx = i;
					break;
				}
			}
			if( edIdx == -1 ) return QHBox::eventFilter(o, e); // User added KviLineEdit child?
			int edMax = 3;
			int cursorPos = ((KviLineEdit *) o)->cursorPosition();
			switch( ((QKeyEvent *) e)->key() ) {
				case Qt::Key_Right:
					s = ((KviLineEdit *) o)->text();
					if( ((unsigned int) cursorPos) == s.length() ) {
						if( edIdx < edMax ) {
							m_pEdit[++edIdx]->setCursorPosition(0);
							m_pEdit[edIdx]->setFocus();
							return true;
						}
					}
					break;
				case Qt::Key_Left:
				case Qt::Key_Backspace:
					if( cursorPos == 0 ) {
						if( edIdx > 0 ) {
							s = m_pEdit[--edIdx]->text();
							m_pEdit[edIdx]->setCursorPosition(s.length());
							m_pEdit[edIdx]->setFocus();
							return true;
						}
					}
					return QHBox::eventFilter(o, e);
					break;
				case Qt::Key_End:
				case Qt::Key_Home:
				case Qt::Key_Delete:
				case Qt::Key_Tab:
					return QHBox::eventFilter(o, e);
					break;
				default:
					// A normal key (this part substitutes a QValidator)
					char c = tolower(((QKeyEvent *) e)->ascii());
					if( (c >= '0') && (c <= '9') ) {
						if( m_pEdit[edIdx]->hasSelectedText() ) m_pEdit[edIdx]->cut();
						cursorPos = m_pEdit[edIdx]->cursorPosition();
						s = m_pEdit[edIdx]->text();
						s.insert(cursorPos, c);
						bool bOk = false;
						int num = s.toInt(&bOk);
						if( !bOk ) return true; // Should never happen, but just to be sure
						if( (num < 0) || (num > 255) ) return true; // Invalid field
						m_pEdit[edIdx]->setText(s);
						if( num > 25 ) {
							// The focus goes to the next editor
							if( edIdx < edMax ) {
								m_pEdit[++edIdx]->selectAll();
								m_pEdit[edIdx]->setCursorPosition(0);
								m_pEdit[edIdx]->setFocus();
								return true;
							}
						}
						m_pEdit[edIdx]->cursorForward(false);
					}
					return true;
				break;
			}
		}
	}
	return QHBox::eventFilter(o, e);
}

QSize KviIpEditor::sizeHint() const
{
	QSize hint = QHBox::sizeHint();
	hint.setWidth(m_iSizeHint);
	return hint;
}

#include "m_kvi_ipeditor.moc"
