;;;;;
;;;
;;; Helper function
;;;
;;;;;

(define (write-string socket string)
  (for-each (lambda (c) (net:socket:putc socket c)) (string->list string)))

(define (read-line socket)
  (let* ([cr   (string-ref "\r" 0)]
         [list (let loop ()
                 (if (not (net:socket:ready? socket 30000))
                     '()
                     (let ([c (net:socket:getc socket)])
                       (cond [(or (eq? c #\newline) (eof-object? c))
                              '()]
                             [(eq? c cr)
                              (net:socket:getc socket)
                              '()]
                             [else
                              (cons c (loop))]))))])
    (list->string list)))

(define (read-sequence socket)
  (let loop ()
    (if (not (net:socket:ready? socket 30000))
        '()
        (let ([c (net:socket:getc socket)])
          (if (eof-object? c)
              '()
              (cons c (loop)))))))

;;;;;
;;;
;;; Second part adapted from mod-lisp.lisp
;;;
;;;;;

(define +apache-port+ 3000)

(define (modlisp-server . port)
  (let ([passive-socket (net:socket:make-server (if (pair? port) (car port) +apache-port+))])
    (apache-listen (net:socket:accept passive-socket))))

(define (apache-listen *apache-socket*)
  (let ([*close-apache-socket*  #t])
    (let loop ()
      (let ([command (get-apache-command *apache-socket*)])
        (and (not (eq? command ()))
             (process-apache-command *apache-socket* command)
             (loop))))
    (display "net:socket:shutdown *apache-socket*")
    (net:socket:shutdown *apache-socket*)))

(define (get-apache-command *apache-socket*)
  (let ([header (let collect ()
                  (let ([key (read-line *apache-socket*)])
                    (if (string=? key "end")
                        '()
                        (cons (list key (read-line *apache-socket*)) (collect)))))])
    (if (assoc "content-length" header)
      (cons (list "posted-content" (read-sequence *apache-socket*)) header)
      header)))

(define (process-apache-command *apache-socket* command)
  (let* ([list   (str:separate (cadr (assoc "url" command)) #\?)]
         [url    (list-ref* list 0)]
         [param  (list-ref* list 1)]
         [list2  (and (string? param) (not (string=? param "")) (str:separate param #\&))]
         [params (and list2 (map (lambda (key-value) (str:separate key-value #\=)) list2))]
         [html   (if (string=? url "/asp/debug")
                     (debug-table command url params)
                     (fixed-html  command url params))])
    (write-header-line *apache-socket* "Status" "200 OK")
    (write-header-line *apache-socket* "Content-Type" "text/html")
    (write-header-line *apache-socket* "Content-Length" (format #f "~d" (string-length html)))
    (write-header-line *apache-socket* "Keep-Socket" "1")
    (write-string *apache-socket* "end")
    (net:socket:putc *apache-socket* #\NewLine)
    (write-string *apache-socket* html)
    #t))

(define (write-header-line *apache-socket* key value)
  (write-string *apache-socket* key)
  (net:socket:putc *apache-socket* #\NewLine)
  (write-string *apache-socket* value)
  (net:socket:putc *apache-socket* #\NewLine))

(define (debug-table command url params)
  (let ([port (open-output-string)])
    (display "<!DOCTYPE HTML PUBLIC \"-//IETF//DTD HTML//EN\"><HTML><HEAD></HEAD><BODY><TABLE bgcolor=\"#c0c0c0\"><TR bgcolor=\"yellow\"><TH COLSPAN=2>mod_lisp 2.0b2 + OpenScheme + Apache + Windows</TH></TR><TR bgcolor=\"yellow\"><TH>Key</TH><TH>Value</TH></TR>" port)
    (for-each
      (lambda (key-value) (format port "<TR bgcolor=\"#F0F0c0\"><TD>~a</TD><TD>~a</TD></TR>" (car key-value) (cadr key-value)))
      command)
    (and params
         (display "<TR bgcolor=\"#c0c0c0\"><TD colspan=2>Parameters</TD></TR>" port)
         (for-each
           (lambda (key-value) (format port "<TR bgcolor=\"#F0F0c0\"><TD>~a</TD><TD>~a</TD></TR>" (car key-value) (cadr key-value)))
           params))
    (display "</TABLE></BODY></HTML>" port)
    (let ([string (get-output-string port)])
      (close-output-port port)
      string)))

(define (fixed-html command url params)
  "<!DOCTYPE HTML PUBLIC \"-//IETF//DTD HTML//EN\"><HTML><HEAD></HEAD><BODY><H1>mod_lisp 2.0</H1><P>This is a constant html string sent by mod_lisp 2.0b2 + OpenScheme + Apache + Windows</P></BODY></HTML>")
