/**
 * @file libgalago-gtk/galago-gtk-contact-chooser-dialog.c
 *       Contact chooser dialog
 *
 * @Copyright (C) 2005-2006 Christian Hammond.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#include <libgalago-gtk/galago-gtk-contact-chooser-dialog.h>
#include <libgalago-gtk/galago-gtk-contact-chooser-widget.h>
#include <gtk/gtk.h>

struct _GalagoGtkContactChooserDialogPriv
{
	GtkWidget *chooser_widget;
};

static void galago_gtk_contact_chooser_dialog_class_init(GalagoGtkContactChooserDialogClass *klass);
static void galago_gtk_contact_chooser_dialog_iface_init(GalagoGtkContactChooserIface *iface);
static void galago_gtk_contact_chooser_dialog_init(GalagoGtkContactChooserDialog *chooser_dialog);
static void galago_gtk_contact_chooser_dialog_finalize(GObject *obj);
static void galago_gtk_contact_chooser_dialog_destroy(GtkObject *obj);
static void galago_gtk_contact_chooser_dialog_style_set(GtkWidget *widget,
														GtkStyle *prev_style);
static void update_response_buttons(GalagoGtkContactChooserDialog *dialog);

static GtkDialogClass *parent_class = NULL;

GType
galago_gtk_contact_chooser_dialog_get_type(void)
{
	static GType type = 0;

	if (!type)
	{
		static const GTypeInfo info =
		{
			sizeof(GalagoGtkContactChooserDialogClass),
			NULL,
			NULL,
			(GClassInitFunc)galago_gtk_contact_chooser_dialog_class_init,
			NULL,
			NULL,
			sizeof(GalagoGtkContactChooserDialog),
			0,
			(GInstanceInitFunc)galago_gtk_contact_chooser_dialog_init
		};

		static const GInterfaceInfo contact_chooser_info =
		{
			(GInterfaceInitFunc)galago_gtk_contact_chooser_dialog_iface_init,
			NULL,
			NULL
		};

		type = g_type_register_static(GTK_TYPE_DIALOG,
									  "GalagoGtkContactChooserDialog", &info, 0);

		g_type_add_interface_static(type, GALAGO_GTK_TYPE_CONTACT_CHOOSER,
									&contact_chooser_info);
	}

	return type;
}

static void
galago_gtk_contact_chooser_dialog_class_init(
	GalagoGtkContactChooserDialogClass *klass)
{
	GObjectClass   *gobject_class = G_OBJECT_CLASS(klass);
	GtkObjectClass *object_class  = GTK_OBJECT_CLASS(klass);
	GtkWidgetClass *widget_class  = GTK_WIDGET_CLASS(klass);

	parent_class = g_type_class_peek_parent(klass);

	gobject_class->finalize = galago_gtk_contact_chooser_dialog_finalize;

	object_class->destroy = galago_gtk_contact_chooser_dialog_destroy;

	widget_class->style_set = galago_gtk_contact_chooser_dialog_style_set;
}

#define WRAP_CONTACT_CHOOSER_SET_FUNC(func, paramtype) \
static void \
_galago_gtk_contact_chooser_dialog_##func(GalagoGtkContactChooser *chooser, \
                                         paramtype value) \
{ \
	GalagoGtkContactChooserDialog *dialog = \
		GALAGO_GTK_CONTACT_CHOOSER_DIALOG(chooser); \
\
	galago_gtk_contact_chooser_##func( \
		GALAGO_GTK_CONTACT_CHOOSER(dialog->priv->chooser_widget), (value)); \
}

#define WRAP_CONTACT_CHOOSER_GET_FUNC(func, rettype) \
static rettype \
_galago_gtk_contact_chooser_dialog_##func(const GalagoGtkContactChooser *chooser) \
{ \
	GalagoGtkContactChooserDialog *dialog = \
		GALAGO_GTK_CONTACT_CHOOSER_DIALOG(chooser); \
\
	return galago_gtk_contact_chooser_##func( \
		GALAGO_GTK_CONTACT_CHOOSER(dialog->priv->chooser_widget)); \
}

WRAP_CONTACT_CHOOSER_SET_FUNC(set_service_type, GalagoGtkServiceListItemType)
WRAP_CONTACT_CHOOSER_SET_FUNC(set_service, GalagoService *)
WRAP_CONTACT_CHOOSER_SET_FUNC(set_contact, GalagoAccount *)
WRAP_CONTACT_CHOOSER_SET_FUNC(set_person, GalagoPerson *)
WRAP_CONTACT_CHOOSER_GET_FUNC(get_contact, GalagoAccount *)
WRAP_CONTACT_CHOOSER_GET_FUNC(get_person, GalagoPerson *)

#undef WRAP_CONTACT_CHOOSER_SET_FUNC
#undef WRAP_CONTACT_CHOOSER_GET_FUNC

static void
galago_gtk_contact_chooser_dialog_iface_init(
	GalagoGtkContactChooserIface *iface)
{
	iface->set_service_type = _galago_gtk_contact_chooser_dialog_set_service_type;
	iface->set_service      = _galago_gtk_contact_chooser_dialog_set_service;
	iface->set_contact      = _galago_gtk_contact_chooser_dialog_set_contact;
	iface->set_person       = _galago_gtk_contact_chooser_dialog_set_person;
	iface->get_contact      = _galago_gtk_contact_chooser_dialog_get_contact;
	iface->get_person       = _galago_gtk_contact_chooser_dialog_get_person;
}

static void
activated_cb(GalagoGtkContactChooserWidget *widget,
			 GalagoGtkContactChooserDialog *chooser_dialog)
{
#if GTK_CHECK_VERSION(2,8,0)
	GtkDialog *dialog;
	GList *children, *l;
#endif

	g_signal_emit_by_name(chooser_dialog, "activated", 0);

	if (gtk_window_activate_default(GTK_WINDOW(chooser_dialog)))
		return;

#if GTK_CHECK_VERSION(2,8,0)
	/*
	 * Follow GtkFileChooserDialog's logic to try to find an acceptable
	 * default response widget.
	 */
	dialog = GTK_DIALOG(chooser_dialog);
	children = gtk_container_get_children(GTK_CONTAINER(dialog->action_area));

	for (l = children; l != NULL; l = l->next)
	{
		GtkWidget *widget;
		int response_id;

		widget = GTK_WIDGET(l->data);

		response_id = gtk_dialog_get_response_for_widget(dialog, widget);

		if (response_id == GTK_RESPONSE_ACCEPT ||
			response_id == GTK_RESPONSE_OK     ||
			response_id == GTK_RESPONSE_YES    ||
			response_id == GTK_RESPONSE_APPLY)
		{
			gtk_widget_activate(widget);
			break;
		}
	}

	g_list_free(children);
#endif /* GTK_CHECK_VERSION(2,8,0) */
}

static void
selection_changed_cb(GalagoGtkContactChooserWidget *widget,
					 GalagoGtkContactChooserDialog *chooser_dialog)
{
	update_response_buttons(chooser_dialog);

	g_signal_emit_by_name(chooser_dialog, "selection-changed", 0);
}

static void
galago_gtk_contact_chooser_dialog_init(
	GalagoGtkContactChooserDialog *chooser_dialog)
{
	GtkWidget *chooser_widget;

	chooser_dialog->priv = g_new0(GalagoGtkContactChooserDialogPriv, 1);

	gtk_dialog_set_has_separator(GTK_DIALOG(chooser_dialog), FALSE);

	chooser_widget = chooser_dialog->priv->chooser_widget =
		galago_gtk_contact_chooser_widget_new();
	gtk_widget_show(chooser_widget);
	gtk_container_add(GTK_CONTAINER(GTK_DIALOG(chooser_dialog)->vbox),
					  chooser_widget);

	g_signal_connect(G_OBJECT(chooser_widget), "activated",
					 G_CALLBACK(activated_cb), chooser_dialog);
	g_signal_connect(G_OBJECT(chooser_widget), "selection-changed",
					 G_CALLBACK(selection_changed_cb), chooser_dialog);

	/* XXX This is kinda bleh, but oh well. */
	gtk_window_resize(GTK_WINDOW(chooser_dialog), 600, 340);
}

static void
galago_gtk_contact_chooser_dialog_style_set(GtkWidget *widget,
											GtkStyle *prev_style)
{
	GtkDialog *dialog;

	if (GTK_WIDGET_CLASS(parent_class)->style_set)
		GTK_WIDGET_CLASS(parent_class)->style_set(widget, prev_style);

	dialog = GTK_DIALOG(widget);

	/* Override to be HIG-compliant */
	gtk_container_set_border_width(GTK_CONTAINER(dialog->vbox), 12);
	gtk_box_set_spacing(GTK_BOX(dialog->vbox), 24);

	gtk_container_set_border_width(GTK_CONTAINER(dialog->action_area), 0);
	gtk_box_set_spacing(GTK_BOX(dialog->action_area), 6);
}

static void
galago_gtk_contact_chooser_dialog_finalize(GObject *obj)
{
	GalagoGtkContactChooserDialog *chooser_dialog;

	g_return_if_fail(obj != NULL);
	g_return_if_fail(GALAGO_GTK_IS_CONTACT_CHOOSER_DIALOG(obj));

	chooser_dialog = GALAGO_GTK_CONTACT_CHOOSER_DIALOG(obj);

	g_free(chooser_dialog->priv);

	if (G_OBJECT_CLASS(parent_class)->finalize)
		G_OBJECT_CLASS(parent_class)->finalize(obj);
}

static void
galago_gtk_contact_chooser_dialog_destroy(GtkObject *obj)
{
	GalagoGtkContactChooserDialog *chooser_dialog;

	g_return_if_fail(obj != NULL);
	g_return_if_fail(GALAGO_GTK_IS_CONTACT_CHOOSER_DIALOG(obj));

	chooser_dialog = GALAGO_GTK_CONTACT_CHOOSER_DIALOG(obj);

	if (GTK_OBJECT_CLASS(parent_class)->destroy)
		GTK_OBJECT_CLASS(parent_class)->destroy(obj);
}

static void
update_response_buttons(GalagoGtkContactChooserDialog *dialog)
{
	gboolean sensitive = (galago_gtk_contact_chooser_get_contact(
		GALAGO_GTK_CONTACT_CHOOSER(dialog)) != NULL);

	/*
	 * We don't know what response types the user set, but set all
	 * the stock ACCEPT-style ones.
	 */
	gtk_dialog_set_response_sensitive(GTK_DIALOG(dialog),
									  GTK_RESPONSE_ACCEPT, sensitive);
	gtk_dialog_set_response_sensitive(GTK_DIALOG(dialog),
									  GTK_RESPONSE_OK, sensitive);
	gtk_dialog_set_response_sensitive(GTK_DIALOG(dialog),
									  GTK_RESPONSE_YES, sensitive);
	gtk_dialog_set_response_sensitive(GTK_DIALOG(dialog),
									  GTK_RESPONSE_APPLY, sensitive);
}

static GtkWidget *
galago_gtk_contact_chooser_dialog_new_valist(
	const gchar *title, GtkWindow *parent,
	GalagoGtkContactChooserFlags flags,
	const gchar *first_button_text, va_list args)
{
	GalagoGtkContactChooserDialog *dialog;
	const char *button_text = first_button_text;
	gint response_id;

	dialog = g_object_new(GALAGO_GTK_TYPE_CONTACT_CHOOSER_DIALOG,
						  "title", title,
						  NULL);
						  /* "flags", flags, */

	if (parent != NULL)
		gtk_window_set_transient_for(GTK_WINDOW(dialog), parent);

	for (button_text = first_button_text;
		 button_text != NULL;
		 button_text = va_arg(args, const gchar *))
	{
		response_id = va_arg(args, gint);
		gtk_dialog_add_button(GTK_DIALOG(dialog), button_text, response_id);
	}

	update_response_buttons(dialog);

	return GTK_WIDGET(dialog);
}

GtkWidget *
galago_gtk_contact_chooser_dialog_new(const gchar *title, GtkWindow *parent,
									  GalagoGtkContactChooserFlags flags,
									  const gchar *first_button_text, ...)
{
	GtkWidget *dialog;
	va_list args;

	va_start(args, first_button_text);
	dialog = galago_gtk_contact_chooser_dialog_new_valist(title, parent,
														  flags,
														  first_button_text,
														  args);
	va_end(args);

	return dialog;
}
