/*******************************************************************************************************************************************
 carchive.h
 Logical sequential read and write on chunks with data dump on disk.
*******************************************************************************************************************************************/

#ifndef __CARCHIVE_H__
#define __CARCHIVE_H__

#include "cchunk.h"
#include "cstring.h"
#include "cfile.h"

// opaque class
class CSerialized;

//------------------------------------------------------------------------------------------------------------------------------------------
// archive process definition
//------------------------------------------------------------------------------------------------------------------------------------------
typedef enum
{
	ARCHIVEPROCESS_NONE=0,		// no current process
	ARCHIVEPROCESS_STORING,		// the CSerialized object is stored
	ARCHIVEPROCESS_LOADING		// the CSerialized object is loaded
} TArchiveProcess;

//------------------------------------------------------------------------------------------------------------------------------------------
// archive disk access definition
//------------------------------------------------------------------------------------------------------------------------------------------
#define ARCHIVE_READ		1	// the archive is loaded from disk
#define	ARCHIVE_WRITE		2	// the archive is written to disk
#define	ARCHIVE_NOFLUSH		4	// the archive is not flushed when its instance is deleted

//------------------------------------------------------------------------------------------------------------------------------------------
// CArchive class
//------------------------------------------------------------------------------------------------------------------------------------------
class CArchive : public CChunk
{
	// instanciation section
	public :

		CArchive			(const CString &inArchiveName, const int inArchiveAccess=0);
		CArchive			(const CArchive &);
		virtual ~CArchive 		();

	// general functions
	public :

		int				GetAccess		() const;
		TArchiveProcess			GetProcess		() const;

		void				Flush			();

	// operators
	public :

		// reverse, forward on data
		CArchive &			operator ++		();
		CArchive &			operator ++		(int);
		CArchive &			operator --		();
		CArchive &			operator --		(int);

		// sequential write
		CArchive &			operator <<		(const SInt8);
		CArchive &			operator <<		(const UInt8);
		CArchive &			operator <<		(const SInt16);
		CArchive &			operator <<		(const UInt16);
		CArchive &			operator <<		(const SInt32);
		CArchive &			operator <<		(const UInt32);
		CArchive &			operator <<		(const SInt64);
		CArchive &			operator <<		(const UInt64);
		CArchive &			operator <<		(const Float32);
		CArchive &			operator <<		(const Float64);
		CArchive &			operator <<		(const SInt8 *);
		CArchive &			operator <<		(CSerialized &);

		// sequential read
		CArchive &			operator >>		(SInt8 &);
		CArchive &			operator >>		(UInt8 &);
		CArchive &			operator >>		(SInt16 &);
		CArchive &			operator >>		(UInt16 &);
		CArchive &			operator >>		(SInt32 &);
		CArchive &			operator >>		(UInt32 &);
		CArchive &			operator >>		(SInt64 &);
		CArchive &			operator >>		(UInt64 &);
		CArchive &			operator >>		(Float32 &);
		CArchive &			operator >>		(Float64 &);
		CArchive &			operator >>		(SInt8 *&);
		CArchive &			operator >>		(CSerialized &);

		// friends...
		friend CArchive &		operator <<		(CSerialized &, CArchive &);
		friend CArchive &		operator >>		(CSerialized &, CArchive &);

	// protected section
	protected :

		int				m_ArchiveAccess;
		TArchiveProcess			m_ArchiveProcess;
		FILE *				m_File;
		SInt16				m_Count;

		// metaclass description
		SECTION_CAPSULE_METACLASS;
};

// metaclass and class tag declaration
DECLARE_CAPSULE_METACLASS ('arch', CArchive, CChunk);

#endif
