/* -*- Mode: C; indent-tabs-mode: nil; c-basic-offset: 4 c-style: "K&R" -*- */

/*
   libgpiv - library for Particle Image Velocimetry

   Copyright (C) 2002, 2003, 2004 Gerber van der Graaf

   This file is part of libgpiv.
   Libgpiv is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  



-----------------------------------------------------------------------
FILENAME:               post.c
LIBRARY:                libgpiv


EXTERNAL FUNCTIONS:
			gpiv_post_manipiv 
                        gpiv_post_savg


LAST MODIFICATION DATE: $Id: post.c,v 1.16 2006/01/31 13:30:13 gerber Exp $
 ------------------------------------------------------------------- */

#include <stdio.h> 
#include <string.h>
#include <math.h>
#include <assert.h>

#include <gpiv.h>


/* 
 * Local functions
 */

static char * 
flip(GpivPivData in_data,
     GpivPivData *out_data,
     GpivPostPar piv_post_par
     )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *    Flips piv data in x-and/or y direction
 *
 * PROTOTYPE LOCATATION:
 *     post.h
 *
 * INPUTS:
 *
 * OUTPUTS:
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/
{
    char *err_msg = NULL;
    int i, j;
    
    switch (piv_post_par.operator_manipiv) {
    case GPIV_FLIP_X:
        for (i=0; i < in_data.ny; i++) {
            for (j=0; j < in_data.nx; j++) {
                out_data->point_x[i][j] = in_data.point_x[i][j];
                out_data->point_y[i][j] = in_data.point_y[i][j];
                out_data->dx[out_data->ny-i-1][j] = in_data.dx[i][j];
/*
 * Change sign
 */
                out_data->dy[out_data->ny-i-1][j] = -in_data.dy[i][j];
                out_data->snr[out_data->ny-i-1][j] = in_data.snr[i][j];
                out_data->peak_no[out_data->ny-i-1][j] = in_data.peak_no[i][j];
            }
        }
        break;
        
    case GPIV_FLIP_Y:
        for (i = 0; i < in_data.ny; i++) {
            for (j = 0; j < in_data.nx; j++) {
                out_data->point_x[i][j] = in_data.point_x[i][j];
                out_data->point_y[i][j] = in_data.point_y[i][j];
/*
 * Change sign
 */
                out_data->dx[i][out_data->nx-j-1] = -in_data.dx[i][j];
                out_data->dy[i][out_data->nx-j-1] = in_data.dy[i][j]; 
                out_data->snr[i][out_data->nx-j-1] = in_data.snr[i][j];
                out_data->peak_no[i][out_data->nx-j-1] = in_data.peak_no[i][j];
            }
        }
        break;

/*
 * Identic to rotating over 180 degr
 */
    case GPIV_ROT180:
        out_data->nx = in_data.ny;
        out_data->ny = in_data.nx;
        for (i = 0; i < in_data.ny; i++) {
            for (j = 0; j < in_data.nx; j++) {
                out_data->point_x[i][j] = in_data.point_x[i][j];
                out_data->point_y[i][j] = in_data.point_y[i][j];
/*
 * Change sign
 */
                out_data->dx[out_data->ny-i-1][in_data.nx-j-1] = 
                    -in_data.dx[i][j];  
/*
 * Change sign
 */
                out_data->dy[out_data->ny-i-1][in_data.nx-j-1] = 
                    -in_data.dy[i][j];  
                out_data->snr[out_data->ny-i-1][in_data.nx-j-1] = 
                    in_data.snr[i][j];
                out_data->peak_no[out_data->ny-i-1][in_data.nx-j-1] = 
                    in_data.peak_no[i][j];
            }
        }
        break;

    case GPIV_ROT90:
        out_data->nx = in_data.nx;
        out_data->ny = in_data.ny;
        for (i = 0; i < in_data.ny; i++) {
            for (j = 0; j < in_data.nx; j++) {
                out_data->point_x[i][j] = in_data.point_y[i][j];
                out_data->point_y[i][j] = in_data.point_x[i][j];
/*
 * Change sign
 */
                out_data->dx[i][out_data->nx-j-1] = -in_data.dy[i][j];
                out_data->dy[i][out_data->nx-j-1] = in_data.dx[i][j];
                out_data->snr[i][out_data->nx-j-1] = in_data.snr[i][j];
                out_data->peak_no[i][out_data->nx-j-1] = in_data.peak_no[i][j];
                
            }
        }

    default:
        err_msg = "FLIP: no valid operation_manipiv";
        gpiv_warning("%s", err_msg);
        return err_msg;
        break;
    }


    return err_msg;
}



static char *
revert_data_col(GpivPivData in_data,
                GpivPivData *out_data
                )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Interchanges column indexes to reverse the order of data
 *
 * PROTOTYPE LOCATATION:
 *     post.h
 *
 * INPUTS:
 *
 * OUTPUTS:
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/
{
    char *err_msg = NULL;
    int i, j;
    
    out_data->nx = in_data.ny;
    out_data->ny = in_data.nx;
    for (i = 0; i < in_data.ny; i++) {
        for (j = 0; j < in_data.nx; j++) {
            out_data->point_x[i][in_data.nx-j-1] = 
                in_data.point_x[i][j];
            out_data->point_y[i][in_data.nx-j-1] = 
                in_data.point_y[i][j];
            out_data->dx[i][in_data.nx-j-1] = in_data.dx[i][j];
            out_data->dy[i][in_data.nx-j-1] = in_data.dy[i][j];
            out_data->snr[i][in_data.nx-j-1] = in_data.snr[i][j];
            out_data->peak_no[i][in_data.nx-j-1] = 
                in_data.peak_no[i][j];
        }
    }
    
    return err_msg;
}



static char *
revert_data_row(GpivPivData in_data,
                GpivPivData *out_data
                )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Interchanges row indexes to reverse the order of data
 *
 * PROTOTYPE LOCATATION:
 *     post.h
 *
 * INPUTS:
 *
 * OUTPUTS:
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/
{
    char *err_msg = NULL;
    int i, j;
    
    out_data->nx = in_data.ny;
    out_data->ny = in_data.nx;
    for (i = 0; i < in_data.ny; i++) {
        for (j = 0; j < in_data.nx; j++) {
            out_data->point_x[in_data.ny-i-1][j] = 
                in_data.point_x[i][j];
            out_data->point_y[in_data.ny-i-1][j] = 
                in_data.point_y[i][j];
            out_data->dx[in_data.ny-i-1][j] = in_data.dx[i][j];
            out_data->dy[in_data.ny-i-1][j] = in_data.dy[i][j];
            out_data->snr[in_data.ny-i-1][j] = in_data.snr[i][j];
            out_data->peak_no[in_data.ny-i-1][j] = 
                in_data.peak_no[i][j];
        }
    }
    
    return err_msg;
}



static char *
revert_data(GpivPivData in_data,
            GpivPivData *out_data
            )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Interchanges array indexes to reverse the order of data
 *
 * PROTOTYPE LOCATATION:
 *     post.h
 *
 * INPUTS:
 *
 * OUTPUTS:
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/
{
    char *err_msg = NULL;
    GpivPivData ldata;
    
    ldata.nx = in_data.ny;
    ldata.ny = in_data.nx;
    gpiv_null_pivdata(&ldata);
    gpiv_alloc_pivdata(&ldata);
    revert_data_col(in_data, &ldata);
    revert_data_row(ldata, out_data);

    gpiv_free_pivdata(&ldata);
    return err_msg;
}



static char *
fasty(GpivPivData in_data,
      GpivPivData *out_data
      )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Copies input to output data; writes with normal gpiv write function
 *
 * PROTOTYPE LOCATATION:
 *     post.h
 *
 * INPUTS:
 *
 * OUTPUTS:
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/
{
    char *err_msg = NULL;
    int i, j;
    
    out_data->nx = in_data.ny;
    out_data->ny = in_data.nx;
    for (i = 0; i < in_data.ny; i++) {
        for (j = 0; j < in_data.nx; j++) {
            out_data->point_x[i][j]=in_data.point_x[i][j];
            out_data->point_y[i][j]=in_data.point_y[i][j];
            out_data->dx[i][j]= in_data.dx[i][j];
            out_data->dy[i][j]=in_data.dy[i][j];
            out_data->snr[i][j]=in_data.snr[i][j];
            out_data->peak_no[i][j]=in_data.peak_no[i][j];
        }
    }

    return err_msg;
}



static int 
del_data_block(GpivPivData in_data,
               GpivPivData *out_data,
               GpivPostPar piv_post_par,
               GpivRoi block
               )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Deletes a block of data from the data stream
 *
 * PROTOTYPE LOCATATION:
 *     post.h
 *
 * INPUTS:
 *
 * OUTPUTS:
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/
{
    int i, j;

    switch (piv_post_par.operator_manipiv) {

/*
 * Filtering out data defined by block
 */
    case GPIV_FILTER_BLOCK:
        for (i = 0; i < in_data.ny; i++) {
            for (j = 0; j < in_data.nx; j++) {
                out_data->point_x[i][j] = in_data.point_x[i][j];
                out_data->point_y[i][j] = in_data.point_y[i][j];

                if (in_data.peak_no[i][j] != -1 &&
                    in_data.point_x[i][j] >= block.x_1 && 
                    in_data.point_x[i][j] <= block.x_2 &&
                    in_data.point_y[i][j] >= block.y_1 && 
                    in_data.point_y[i][j] <= block.y_2) {
                    out_data->dx[i][j] = 0.00;
                    out_data->dy[i][j] = 0.00;
                    out_data->snr[i][j] = 10;
                    out_data->peak_no[i][j] = -1;

                } else {
                    out_data->dx[i][j] = in_data.dx[i][j];
                    out_data->dy[i][j] = in_data.dy[i][j];
                    out_data->snr[i][j] = in_data.snr[i][j];
                    out_data->peak_no[i][j] = in_data.peak_no[i][j];
                }

            }
        }
        break;

/*
 * Passing through data defined by block
 */
    case GPIV_PASS_BLOCK:
        out_data->nx = in_data.ny;
        out_data->ny = in_data.nx;
        for (i = 0; i < in_data.ny; i++) {
            for (j = 0; j < in_data.nx; j++) {
                out_data->point_x[i][j] = in_data.point_x[i][j];
                out_data->point_y[i][j] = in_data.point_y[i][j];

                if (in_data.peak_no[i][j] != -1 &&
                    in_data.point_x[i][j] >= block.x_1 && 
                    in_data.point_x[i][j] <= block.x_2 &&
                    in_data.point_y[i][j] >= block.y_1 && 
                    in_data.point_y[i][j] <= block.y_2) {
                    out_data->dx[i][j] = in_data.dx[i][j];
                    out_data->dy[i][j] = in_data.dy[i][j];
                    out_data->snr[i][j] = in_data.snr[i][j];
                    out_data->peak_no[i][j] = in_data.peak_no[i][j];

                } else {
                    out_data->dx[i][j] = 0.00;
                    out_data->dy[i][j] = 0.00;
                    out_data->snr[i][j] = 10;
                    out_data->peak_no[i][j] = -1;
                }

            }
        }

    default:
        fprintf (stderr,"\n%s error: no valid operation_manipiv\n", LIBNAME);
        return 1;
        break;
    }
  
    return 0;
}


static void
set_value_block(GpivPivData *out_data, 
                GpivPostPar piv_post_par
                )
/* ----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Set displacement values for a block of disabled data
 *
 * PROTOTYPE LOCATATION:
 *     post.h
 *
 * INPUTS:
 *     out_data:       GpivPivData containing all variables in a datafile. 
 *     piv_post_par:   parameters for post processing
 *
 * RETURNS:
 *     out_data:       GpivPivData with resulting set values. 
 *-------------------------------------------------------------------------- */
{
    int i, j;

    for (i = 0; i < out_data->ny; i++) {
        for (j = 0; j < out_data->nx; j++) {
            if (out_data->peak_no[i][j] == -1) {
                out_data->dx[i][j] = piv_post_par.set_dx;
                out_data->dy[i][j] = piv_post_par.set_dx;
            }
        }
    }

}



static char *
read_pivdata_fastx (GpivPivData *piv, 
                    char data_line[GPIV_MAX_LINES][GPIV_MAX_CHARS], 
                    int ndata_lines,
                    char comment_line[GPIV_MAX_LINES_C][GPIV_MAX_CHARS], 
                    int *ncomment_lines
                    )
/* ----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Reads data from stdin
 *
 * PROTOTYPE LOCATATION:
 *     post.h
 *
 * INPUTS:
 *     piv:            GpivPivData containing all variables in a datafile. 
 *                     piv.nx and piv.ny have to be known.
 *
 * RETURNS:
 *     comment_line:   comment lines (lines starting with '#'), except header
 *     ncomment_lines: number of comment lines
 *-------------------------------------------------------------------------- */
{
    char *err_msg = NULL;
    int i = 0, j = 0, nc = 0, line_nr = 0;

    assert(piv->point_x != NULL);
    assert(piv->point_y != NULL);
    assert(piv->dx != NULL);
    assert(piv->dy != NULL);
    assert(piv->snr != NULL);
    assert(piv->peak_no != NULL);
    
  while ((i < piv->nx) && (j < piv->ny)) {

   if (data_line[line_nr][0] == '#' && 
       (strstr(data_line[line_nr],"x(m)") == '\0') && 
       (strstr(data_line[line_nr],"x(px)") == '\0')) {
       strcpy(comment_line[nc],data_line[line_nr]);
       nc++;

   } else if (data_line[line_nr][0] != '\n' && 
	      data_line[line_nr][0] != '\t' && 
	      (strstr(data_line[line_nr],"x(m)") == '\0') && 
	      (strstr(data_line[line_nr],"x(px)") == '\0')) {
       sscanf(data_line[line_nr],"%f %f %f %f %f %d", &piv->point_x[i][j], 
              &piv->point_y[i][j],&piv->dx[i][j], &piv->dy[i][j], 
              &piv->snr[i][j], &piv->peak_no[i][j]);
       if (i < piv->nx - 1) {
           i++; 
       } else if (j < piv->ny - 1) { 
           i=0; 
           j++; 
       } else {
           break; 
       }
   }

   line_nr++;
  }

  *ncomment_lines = nc;

  return err_msg;
}



static char *
fread_pivdata_fastx(char *fname, 
                    GpivPivData *piv, 
                    char comment_line[GPIV_MAX_LINES_C][GPIV_MAX_CHARS], 
                    int *ncomment_lines
                    )
/*  ---------------------------------------------------------------------------
 * DESCRIPTION:
 *     Reads data from ascii data file with fast running x-position variables 
 *     (1st column in data stream)
 *
 * PROTOTYPE LOCATATION:
 *     post.h
 *
 * INPUTS:
 *     fname:          complete filename
 *     piv:            GpivPivData containing all variables in a datafile. 
 *                     piv.nx and piv.ny have to be known.
 * OUTPUTS:
 *
 * RETURNS:
 *-------------------------------------------------------------------------- */
{
    char *err_msg = NULL;
    int i = 0, j = 0, nc = 0, line_nr = 0;
    FILE *fp;
    char line_dum[GPIV_MAX_CHARS];
    char *function_name = "fread_pivdata_fastx";

/*
 * Opens input file and reads all lines
 */
    if((fp=fopen(fname,"r"))==NULL) { 
        err_msg = "FREAD_PIVDATA_FASTX: Failure opening inputfile";
        gpiv_warning("%s", err_msg);
        return err_msg;
    }


    while (fgets(line_dum, GPIV_MAX_CHARS, fp) != NULL) {

        if (line_dum[0] == '#' && (strstr(line_dum,"x(m)") == '\0') && 
            (strstr(line_dum,"x(px)") == '\0')) {
            strcpy(comment_line[nc],line_dum);
            nc++;

        } else if (line_dum[0] != '\n' && line_dum[0] != '\t' && 
                   (strstr(line_dum,"x(m)") == '\0') && 
                   (strstr(line_dum,"x(px)") == '\0')) {
            sscanf(line_dum,"%f %f %f %f %f %d", &piv->point_x[i][j], 
                   &piv->point_y[i][j],&piv->dx[i][j], &piv->dy[i][j], 
                   &piv->snr[i][j], &piv->peak_no[i][j]);
            if (i < piv->nx - 1) {
                i++; 
            } else if (j < piv->ny - 1) { 
                i=0; 
                j++; 
            } else { 
                break; 
            }
        }

        line_nr++;
    }

    fclose(fp);
    *ncomment_lines = nc;
    return err_msg;
}




static void
s_stats(GpivPivData in_data,
        GpivPivData *out_data,
	int *ndata
	)
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Calculates spatial global mean and sdev of piv data
 *
 * PROTOTYPE LOCATATION:
 *     post.h
 *
 * INPUTS:
 *
 * OUTPUTS:
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/
{
    int i, j, count = 0;
    float dx_sum = 0, dy_sum = 0, dx_sum_q = 0, dy_sum_q = 0;

    for (i=0; i<in_data.ny; i++) {
        for (j=0; j<in_data.nx; j++) {
            if (in_data.peak_no[i][j] != -1) {
                if (in_data.dx[i][j] > out_data->max_dx) out_data->max_dx = in_data.dx[i][j];
                if (in_data.dx[i][j] < out_data->min_dx) out_data->min_dx = in_data.dx[i][j];
                if (in_data.dy[i][j] > out_data->max_dy) out_data->max_dy = in_data.dy[i][j];
                if (in_data.dy[i][j] < out_data->min_dx) out_data->min_dy = in_data.dy[i][j];
                dx_sum += in_data.dx[i][j];
                dy_sum += in_data.dy[i][j];
                dx_sum_q += in_data.dx[i][j] * in_data.dx[i][j];
                dy_sum_q += in_data.dy[i][j] * in_data.dy[i][j];
                count++;
            }
        }
    }

    out_data->mean_dx = dx_sum / count;
    out_data->sdev_dx = sqrt(dx_sum_q/(count-1) - dx_sum*dx_sum/(count*(count-1)));
    out_data->mean_dy = dy_sum / count;
    out_data->sdev_dy = sqrt(dy_sum_q/(count-1) - dy_sum*dy_sum/(count*(count-1)));
    *ndata = count;
}



/*
 * Public functions
 */

void 
gpiv_post_subtract_dxdy(GpivPivData in_data, 
                        GpivPivData *out_data, 
                        float z_off_dx, 
                        float z_off_dy
                        )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Subtracts displacements or velocity from piv data
 *
 * PROTOTYPE LOCATATION:
 *     post.h
 *
 * INPUTS:
 *
 * OUTPUTS:
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/
{
    int i, j;

    out_data->ny = in_data.ny;
    out_data->nx = in_data.nx;
    for (i=0; i < in_data.ny; i++) {
        for (j=0; j < in_data.nx; j++) {
            out_data->point_x[i][j] = in_data.point_x[i][j];
            out_data->point_y[i][j] = in_data.point_y[i][j];
            out_data->snr[i][j] = in_data.snr[i][j];
            out_data->peak_no[i][j] =in_data.peak_no[i][j];
            if (in_data.peak_no[i][j] != -1) {
                out_data->dx[i][j] = in_data.dx[i][j] - z_off_dx;
                out_data->dy[i][j] = in_data.dy[i][j] - z_off_dy;
                out_data->peak_no[i][j] =in_data.peak_no[i][j];
            } else {
                out_data->dx[i][j] = in_data.dx[i][j];
                out_data->dy[i][j] = in_data.dy[i][j];
            }
        }
    }

}



char *
gpiv_post_manipiv(char *fname_in_piv,
                  gboolean fname_logic,
                  GpivPivData in_data, 
                  GpivPivData *out_data,
                  GpivPostPar piv_post_par,
                  char d_line[GPIV_MAX_LINES][GPIV_MAX_CHARS], 
                  int nd_lines, 
                  char c_line[GPIV_MAX_LINES_C][GPIV_MAX_CHARS],
                  int *nc_lines
                  )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Piv post processing function to manipulate data; flipping rotating, etc
 *
 * INPUTS:
 *      fname_in_piv:   file name of input piv data
 *      fname_logic:    flag whether files are to be used
 *      in_data:        input piv data
 *      piv_post_par:   post-processing parameters
 *      d_line: 	data lines
 *      nd_lines:       number of data lines
 *      c_line:         commented lines
 *      nc_lines:       number of commented lines
 *      
 *
 * OUTPUTS:
 *      out_data:       output piv data
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/
{
    char *err_msg = NULL;

    assert(in_data.point_x != NULL);
    assert(in_data.point_y != NULL);
    assert(in_data.dx != NULL);
    assert(in_data.dy != NULL);
    assert(in_data.snr != NULL);
    assert(in_data.peak_no != NULL);
    
    assert(out_data->point_x != NULL);
    assert(out_data->point_y != NULL);
    assert(out_data->dx != NULL);
    assert(out_data->dy != NULL);
    assert(out_data->snr != NULL);
    assert(out_data->peak_no != NULL);

/*     fprintf(stderr, "\npost_manipiv:: 1\n"); */

/*
 * Reading input file of piv data 
*/
    if (piv_post_par.operator_manipiv == GPIV_FAST_Y) {
        if(fname_logic == TRUE) {
            
            if ((err_msg = fread_pivdata_fastx(fname_in_piv, &in_data,
                                                  c_line, nc_lines)) != NULL) {
                err_msg = "GPIV_POST_MANIPIV: Failure calling fread_pivdata_fast_x";
                gpiv_warning("%s", err_msg);
                return err_msg;
            }
        } else {
            if ((err_msg = read_pivdata_fastx(&in_data, d_line, nd_lines, c_line, 
                                                 nc_lines)) != NULL) {
                err_msg = "GPIV_POST_MANIPIV: Failure calling read_pivdata_fastx";
                return err_msg;
            }
        }
        
    } else {
        if(fname_logic == TRUE) {
            if ((err_msg = gpiv_fread_pivdata(fname_in_piv, &in_data, 
                                              c_line, nc_lines)) != NULL) {
                err_msg = "GPIV_POST_MANIPIV: Failure calling gpiv_fread_pivdata";
                gpiv_warning("%s", err_msg);
                return err_msg;
            }
        
        } else {
            gpiv_read_pivdata(&in_data, d_line, nd_lines, c_line, nc_lines);
        }
    }
    
/*
 * Manipulating of piv data
*/
    if (piv_post_par.operator_manipiv == GPIV_FLIP_X || 
        piv_post_par.operator_manipiv == GPIV_FLIP_Y || 
        piv_post_par.operator_manipiv == GPIV_ROT90) { 
        if ((err_msg = flip(in_data, out_data, piv_post_par)) != NULL) {
            err_msg = "Failure calling flip\n";
            gpiv_warning("%s", err_msg);
            return err_msg;
        }
        
    } else if (piv_post_par.operator_manipiv == GPIV_REVERT) {
        revert_data(in_data, out_data);
        
    } else if (piv_post_par.operator_manipiv == GPIV_FAST_Y) {
        fasty(in_data, out_data);
        
    } else if (piv_post_par.operator_manipiv == GPIV_FILTER_BLOCK ||
               piv_post_par.operator_manipiv == GPIV_PASS_BLOCK) {
        del_data_block(in_data, out_data, piv_post_par, block);
        gpiv_warning("GPIV_POST_MANIPIV: 1");
        if (piv_post_par.set == TRUE) {
            gpiv_warning("GPIV_POST_MANIPIV: 2");
            set_value_block(out_data, piv_post_par);
            gpiv_warning("GPIV_POST_MANIPIV: 2b");
        }
        gpiv_warning("GPIV_POST_MANIPIV: 3");

    }

/*   gpiv_write_pivdata(&out_data, c_line, nc_lines, scale, RCSID); */
/*     fprintf(stderr, "\npost_manipiv:: 4 LAST\n"); */

    return err_msg;
}



void
gpiv_post_savg(GpivPivData in_data, 
               GpivPivData *out_data,
               GpivPostPar piv_post_par,
               int *ndata
               )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Piv post processing function to calculate spatial mean, variances
 *
 * INPUTS:
 *      in_data:        input piv data
 *      piv_post_par:   post-processing parameters
 *
 * OUTPUTS:
 *      out_data:       output piv data
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/
{
    assert(in_data.point_x != NULL);
    assert(in_data.point_y != NULL);
    assert(in_data.dx != NULL);
    assert(in_data.dy != NULL);
    assert(in_data.snr != NULL);
    assert(in_data.peak_no != NULL);
    
    if (piv_post_par.subtract == 1) {
        assert(out_data->point_x != NULL);
        assert(out_data->point_y != NULL);
        assert(out_data->dx != NULL);
        assert(out_data->dy != NULL);
        assert(out_data->snr != NULL);
        assert(out_data->peak_no != NULL);
    }

    s_stats(in_data, out_data, ndata);

    if (piv_post_par.subtract == 1) {
        gpiv_post_subtract_dxdy(in_data, out_data, out_data->mean_dx, 
                                out_data->mean_dy);
    } else if (piv_post_par.subtract == 2) {
        gpiv_post_subtract_dxdy(in_data, out_data, piv_post_par.z_off_dx, 
                       piv_post_par.z_off_dy);
    }

}



void 
gpiv_post_uvhisto(GpivPivData in_data, 
                 GpivBinData * klass,
                 enum GpivVelComponent velcomp
                 )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Calculating histogram of U (horizontal) or V (vertical) particle 
 *     displacements
 *
 * PROTOTYPE LOCATATION:
 *     valid.h
 *
 * INPUTS:
 *      in_data:        input piv data
 *      velcomp:        velocity component from which histogram is calculated
 *
 * OUTPUTS:
 *      klass:          histogram
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/
{
    gint i, j, k;
    gfloat delta, fract;
    gfloat *bound = klass->bound, *centre = klass->centre;
    gint *count = klass->count, nbins = klass->nbins;
    gfloat lower_bound = 10.0e9, upper_bound = -10.0e9;

    assert(in_data.point_x != NULL);
    assert(in_data.point_y != NULL);
    assert(in_data.dx != NULL);
    assert(in_data.dy != NULL);
    assert(in_data.snr != NULL);
    assert(in_data.peak_no != NULL);

    assert(klass->count != NULL);
    assert(klass->bound != NULL);
    assert(klass->centre != NULL);

/*
 * Determine minimum and maximum displacements
 */
    for (i=0; i<in_data.ny; i++) {
        for (j=0; j<in_data.nx; j++) {
            if (in_data.peak_no[i][j] != -1) {
                if (velcomp == GPIV_U) {
                    if (in_data.dx[i][j] > upper_bound) 
                        upper_bound = in_data.dx[i][j];
                    if (in_data.dx[i][j] < lower_bound) 
                        lower_bound = in_data.dx[i][j];
                } else {
                    if (in_data.dy[i][j] > upper_bound) 
                        upper_bound = in_data.dy[i][j];
                    if (in_data.dy[i][j] < lower_bound) 
                        lower_bound = in_data.dy[i][j];
                }
            }
        }
    }

    delta = (gfloat) (upper_bound - lower_bound) / (gfloat) nbins;
    for (i = 0; i < nbins; i++) {
	centre[i] = lower_bound + delta / 2.0 + (float) i * delta;
	bound[i] = lower_bound + (float) i * delta;
	count[i] = 0;
    }

/*
 * Subdividing particle displacements in bins
 */
    for (i = 0; i < in_data.ny; i++) {
	for (j = 0; j < in_data.nx; j++) {
            if (in_data.peak_no[i][j] != -1) {
                for (k = 0; k < nbins; k++) {
                    if (velcomp == GPIV_U) {
                        if (( in_data.dx[i][j] >= bound[k]) 
                            && (in_data.dx[i][j] < bound[k + 1])) {
                            count[k] = count[k] + 1;
                        }
                    } else {
                        if (( in_data.dy[i][j] >= bound[k]) 
                            && (in_data.dy[i][j] < bound[k + 1])) {
                            count[k] = count[k] + 1;
                        }
                    }
                }
            }
        }
    }

}



int
gpiv_post_scale(GpivPivData in_data, 
		GpivPivData *out_data,
                GpivImagePar image_par
		)
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Scales piv data in x-and/or y direction
 *
 * PROTOTYPE LOCATATION:
 *     post.h
 *
 * INPUTS:
 *
 * OUTPUTS:
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/
{
    int i, j;

    assert(in_data.point_x != NULL);
    assert(in_data.point_y != NULL);
    assert(in_data.dx != NULL);
    assert(in_data.dy != NULL);
    assert(in_data.snr != NULL);
    assert(in_data.peak_no != NULL);

    assert(out_data->point_x != NULL);
    assert(out_data->point_y != NULL);
    assert(out_data->dx != NULL);
    assert(out_data->dy != NULL);
    assert(out_data->snr != NULL);
    assert(out_data->peak_no != NULL);
    
    
    for (i=0; i < in_data.ny; i++) {
        for (j=0; j < in_data.nx; j++) {
            out_data->point_x[i][j] = in_data.point_x[i][j] * 
                image_par.s_scale * 1e-3;
            out_data->point_y[i][j] = in_data.point_y[i][j] * 
                image_par.s_scale * 1e-3;
            out_data->dx[i][j] = in_data.dx[i][j] * image_par.s_scale / 
                image_par.t_scale;
            out_data->dy[i][j] = in_data.dy[i][j] * image_par.s_scale / 
                image_par.t_scale;
            out_data->snr[i][j] = in_data.snr[i][j];
            out_data->peak_no[i][j] = in_data.peak_no[i][j];
        }
    }

/*
 * Zero offset of positions
 */
    if (image_par.z_off_x_logic && image_par.z_off_y_logic) {
        for (i = 0; i < in_data.ny; i++) {
            for (j = 0; j < in_data.nx; j++) {
                out_data->point_x[i][j] = out_data->point_x[i][j] + 
                    image_par.z_off_x;
                out_data->point_y[i][j] = out_data->point_y[i][j] + 
                    image_par.z_off_y;
            }
        }
    }
    
  
    return 1;
}


int
gpiv_post_inverse_scale(GpivPivData in_data, 
                        GpivPivData *out_data,
                        GpivImagePar image_par
                        )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Scales piv data in x-and/or y direction 
 *
 * PROTOTYPE LOCATATION:
 *     post.h
 *
 * INPUTS:
 *
 * OUTPUTS:
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/
{
    int i, j;

    assert(in_data.point_x != NULL);
    assert(in_data.point_y != NULL);
    assert(in_data.dx != NULL);
    assert(in_data.dy != NULL);
    assert(in_data.snr != NULL);
    assert(in_data.peak_no != NULL);
    
    assert(out_data->point_x != NULL);
    assert(out_data->point_y != NULL);
    assert(out_data->dx != NULL);
    assert(out_data->dy != NULL);
    assert(out_data->snr != NULL);
    assert(out_data->peak_no != NULL);


/*
 * Reverse Zero offset of positions
 */
    if (image_par.z_off_x_logic && image_par.z_off_y_logic) {
        for (i=0; i < in_data.ny; i++) {
            for (j=0; j < in_data.nx; j++) {
                in_data.point_x[i][j] = in_data.point_x[i][j] - 
                    image_par.z_off_x;
                in_data.point_y[i][j] = in_data.point_y[i][j] - 
                    image_par.z_off_y;
            }
        }
    }
    
    for (i=0; i < in_data.ny; i++) {
        for (j=0; j < in_data.nx; j++) {
            out_data->point_x[i][j] = in_data.point_x[i][j] / 
                image_par.s_scale * 1e+3;
            out_data->point_y[i][j] = in_data.point_y[i][j] / 
                image_par.s_scale * 1e+3;
            out_data->dx[i][j] = in_data.dx[i][j] / image_par.s_scale * 
                image_par.t_scale;
            out_data->dy[i][j] = in_data.dy[i][j] / image_par.s_scale * 
                image_par.t_scale;
            out_data->snr[i][j] = in_data.snr[i][j];
            out_data->peak_no[i][j] = in_data.peak_no[i][j];
        }
    }
    
    return 1;
}



/*
 * Post processing function to calculate vorticity and strain
 */


char *
gpiv_post_vorstra(GpivPivData in_data, 
		  GpivScalarData * out_data, 
		  GpivPostPar piv_post_par
		  )
/* ----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Calculates vorticity or shear/normal strain quantities from piv data
 * 
 * PROTOTYPE LOCATATION:
 *     post.h
 *
 * INPUTS:
 *     in_data:        piv data
 *     piv_post_par:   piv post parameters
 *
 * OUPUTS:
 *     out_data:       scalar data containing vorticity or strain
 *
 * RETURNS:
 *      NULL on success or *err_msg on failure
 *-------------------------------------------------------------------------- */
{
    char *err_msg = NULL;
    int i, j, k, l, valid=0;
    int diff_order = 0;
    float delta;
    
    assert(in_data.point_x != NULL);
    assert(in_data.point_y != NULL);
    assert(in_data.dx != NULL);
    assert(in_data.dy != NULL);
    assert(in_data.snr != NULL);
    assert(in_data.peak_no != NULL);
    
    assert(out_data->point_x != NULL);
    assert(out_data->point_y != NULL);
    assert(out_data->scalar != NULL);
    
    delta = in_data.point_x[0][1] - in_data.point_x[0][0];


    for (i = 0; i < out_data->ny; i++) {
        for (j = 0; j < out_data->nx; j++) {
	    out_data->point_x[i][j] = in_data.point_x[i][j];
	    out_data->point_y[i][j] = in_data.point_y[i][j];
	    out_data->flag[i][j] = in_data.peak_no[i][j];
	    out_data->scalar[i][j] = 0.0;
	}
    }

/*
 * determine order of differentiation
 */
    if (piv_post_par.diff_type == GPIV_CENTRAL || 
        piv_post_par.diff_type == GPIV_CIRCULATION) {
        diff_order = 1;
    } else if (piv_post_par.diff_type == GPIV_LEAST_SQUARES || 
               piv_post_par.diff_type == GPIV_RICHARDSON) {
        diff_order = 2;
    }


/*
 * exclude all excluded data or invalid points
 */
/*      for (i=0; i < in_data.y; i++) { */
/* 	  for (j=0; j < in_data.nx; j++) { */
/* 	       if (in_data.peak_no[i][j] == -1) { */
/* 		    for (k=i-diff_order; k <= i+diff_order; k++) { */
/* 			 if ((k >= 0) && (k < in_data.ny)) { */
/* 			      for (l=j-diff_order; l <= j+diff_order; l++) { */
/* 				   if ((l >= 0) && (j < in_data.nx)) { */
/* 					out_data->scalar[k][l] = 0; */
/* 					out_data.flag[k][l] = -1; */
/* 				   } */
/* 			      } */
/* 			 } */
/* 		    } */
/* 	       } */
/* 	  } */
/*      } */


/*
 * check all data if valid; peak !=-1
 */
    for (i=diff_order; i < in_data.ny - diff_order; i++) {
        for (j=diff_order; j < in_data.nx - diff_order; j++) {
            valid = 0;
            for (k=i-diff_order; k <= i + diff_order; k++) {
                if ((k >= 0) && (k < in_data.ny)) {
                    for (l=j-diff_order; l <= j + diff_order; l++) {
                        if ((l >= 0) && (j < in_data.nx)) {
/*
 * Only disables actual datapoint
 */

/* 				      if (in_data.peak_no[k][l] != -1)  */
/* 					   valid = in_data.peak_no[k][l];  */

/*
 * Unvalid if one or more of the points used in the derivative is unvalid
 */
                            if (in_data.peak_no[k][l] == -1) 
                                valid = -1; 
                        }
                    }
                }
            }


/*  	for (i=0; i < in_data.ny; i++) { */
/* 	     for (j=0; j < in_data.nx; j++) { */

/*
 * Check all data if valid peak !=-1, i.e. are accepted
 */
            if (valid != -1) {
                
                switch (piv_post_par.diff_type) {

                case GPIV_CENTRAL:
                    switch (piv_post_par.operator_vorstra) {
                    case GPIV_VORTICITY:
                        out_data->scalar[i][j] = 
                            ((in_data.dy[i][j+1] - 
                              in_data.dy[i][j-1]) -
                             (in_data.dx[i+1][j] - 
                              in_data.dx[i-1][j])) / (2 * delta);
                        break;

                    case GPIV_S_STRAIN:
                        out_data->scalar[i][j] = 
                            ((in_data.dx[i+1][j] - 
                              in_data.dx[i-1][j]) + 
                             (in_data.dy[i][j+1] - 
                              in_data.dy[i][j-1])) / (2 * delta);
                        break;

                    case 	GPIV_N_STRAIN:
                        out_data->scalar[i][j] = 
                            ((in_data.dx[i][j+1] - 
                              in_data.dx[i][j]-1) +
                             (in_data.dy[i+1][j] - 
                              in_data.dy[i-1][j])) / (2 * delta);
                        break;

                    default:
                        err_msg = "GPIV_POST_VORSTRA: no valid operation";
                        gpiv_warning("%s", err_msg);
                        return err_msg;
                        break;
                    }
                    break;
                    


                case GPIV_CIRCULATION:
                    switch (piv_post_par.operator_vorstra) {

                    case GPIV_VORTICITY:
                        out_data->scalar[i][j] = 
                            ((in_data.dx[i-1][j-1] + 
                              2 * in_data.dx[i-1][j] + 
                              in_data.dx[i-1][j+1]) +
                             (in_data.dy[i-1][j+1] + 
                              2 * in_data.dy[i][j+1] + 
                              in_data.dy[i+1][j+1]) -
                             (in_data.dx[i+1][j+1] + 
                              2 * in_data.dx[i+1][j] + 
                              in_data.dx[i+1][j-1]) -
                             (in_data.dy[i+1][j-1] + 
                              2 * in_data.dy[i][j-1] + 
                              in_data.dy[i-1][j-1])) / (8 * delta);
                        break;

                    default:
                        err_msg = "GPIV_POST_VORSTRA: no valid operation";
                        gpiv_warning("%s", err_msg);
                        return err_msg;
                        break;
                    }	  
                    break;



                case GPIV_LEAST_SQUARES:
                    switch (piv_post_par.operator_vorstra) {
                        
                    case GPIV_VORTICITY:
                        out_data->scalar[i][j] = 
                            (2 * in_data.dy[i][j+2] + 
                             in_data.dy[i][j+1] - 
                             in_data.dy[i][j-1] - 
                             2 * in_data.dy[i][j-2]) / 
                            (10*delta) - (2 * in_data.dx[i+2][j] + 
                                          in_data.dx[i+1][j] - 
                                          in_data.dx[i-1][j] - 
                                          2 * in_data.dx[i-2][j]) / (10*delta);
                        break;

                    case GPIV_S_STRAIN:
                        out_data->scalar[i][j] =
                            (2 * in_data.dx[i+2][j] + 
                             in_data.dx[i+1][j] - 
                             in_data.dx[i-1][j] - 
                             2 * in_data.dx[i-2][j]) / (10*delta) +
                            (2 * in_data.dy[i][j+2] + 
                             in_data.dy[i][j+1] - 
                             in_data.dy[i][j-1] - 
                             2 * in_data.dy[i][j-2]) / (10*delta);
                        break;

                    case GPIV_N_STRAIN:
                        out_data->scalar[i][j] =
                            (2 * in_data.dx[i][j+2] + 
                             in_data.dx[i][j+1] - 
                             in_data.dx[i][j-1] - 
                             2 * in_data.dx[i][j-2]) / (10*delta) +
                            (2 * in_data.dy[i+2][j] + 
                             in_data.dy[i+1][j] - 
                             in_data.dy[i-1][j] - 
                             2 * in_data.dy[i-2][j]) / (10*delta);
                        break;

                    default:
                        err_msg = "GPIV_POST_VORSTRA: no valid operation\n";
                        gpiv_warning("%s", err_msg);
                        return err_msg;
                        break;
                    }	  
                    break;
                    


                case GPIV_RICHARDSON:
                    switch (piv_post_par.operator_vorstra) {
 
                   case GPIV_VORTICITY:
                        out_data->scalar[i][j] = 
                            (in_data.dy[i][j-2] - 
                             8 * in_data.dy[i][j-1] + 
                             8 * in_data.dy[i][j+1] - 
                             in_data.dy[i][j+2]) / (12*delta) -
                            (in_data.dx[i-2][j] - 
                             8 * in_data.dx[i-1][j] + 
                             8 * in_data.dx[i+1][j] - 
                             in_data.dx[i+2][j]) / (12*delta);
                        break;

                    case GPIV_S_STRAIN:
                        out_data->scalar[i][j] =
                            (in_data.dx[i-2][j] - 
                             8 * in_data.dx[i-1][j] + 
                             8 * in_data.dx[i+1][j] - 
                             in_data.dx[i+2][j]) / (12*delta) + 
                            (in_data.dy[i][j-2] - 
                             8 * in_data.dy[i][j-1] + 
                             8 * in_data.dy[i][j+1] - 
                             in_data.dy[i][j+2]) / (12*delta);
                        break;

                    case GPIV_N_STRAIN:
                        out_data->scalar[i][j] = 
                            (in_data.dx[i][j-2] - 
                             8 * in_data.dx[i][j-1] + 
                             8 * in_data.dx[i][j+1] - 
                             in_data.dx[i][j+2]) / (12*delta) +
                            (in_data.dy[i-2][j] - 
                             8 * in_data.dy[i-1][j] + 
                             8 * in_data.dy[i+1][j] - 
                             in_data.dy[i+2][j]) / (12*delta);
                        break;

                    default:
                        err_msg = "GPIV_POST_VORSTRA: no valid operation";
                        gpiv_warning("%s", err_msg);
                        return err_msg;
                        break;
                    }	  
                    break;
                    
                    

                default:
                    err_msg = "GPIV_POST_VORSTRA: no valid differential operator type";
                    gpiv_warning("%s", err_msg);
                    return err_msg;
                    break;
                }
            } else {
                out_data->flag[i][j] = -1;
            }
            
            
            
        }
    }
    
    

/*
 * exclude all data near the boundaries of the dataset
 */
    for (i=0; i < diff_order; i++) {
        for (j=0; j < in_data.nx; j++) {
            out_data->scalar[i][j] = 0;
            out_data->flag[i][j] = -1;
        }
    }
    
    for (i=0; i < in_data.ny; i++) {
        for (j=0; j < diff_order; j++) {
	       out_data->scalar[i][j] = 0;
	       out_data->flag[i][j] = -1;
        }
    }
    
    for (i=in_data.ny - diff_order; i < in_data.ny; i++) {
        for (j=0; j < in_data.nx; j++) {
            out_data->scalar[i][j] = 0;
            out_data->flag[i][j] = -1;
        }
    }
    
    for (i=0; i < in_data.ny; i++) {
        for (j=in_data.nx - diff_order; j < in_data.nx; j++) {
            out_data->scalar[i][j] = 0;
            out_data->flag[i][j] = -1;
        }
    }
    
    
    return err_msg;
}
