/*******************************************************************************************************************************************
 cfiledialog.c
*******************************************************************************************************************************************/

#include "cfiledialog.h"

//-----------------------------------------------------------------------------------------------------------------------------------------
// metaclass code resolution
//-----------------------------------------------------------------------------------------------------------------------------------------
RESOLVE_DYNAMIC_METACLASS (CFileDialog);

//-----------------------------------------------------------------------------------------------------------------------------------------
// constructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CFileDialog::CFileDialog (CForm *inOwner, const CString &inCaption, const bool inModal, const CDialogListener *inListener)
	    :CDialog	 (NULL, inCaption, _DBNone_, inModal, inListener)
{ 
	// gtk widget instanciation coherence request
	if (inOwner != NULL) SetOwner (inOwner);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// destructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CFileDialog::~CFileDialog ()
{
	// gtk widget deletion coherence request
	CWidget::DestroyWidget (this);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// gtk widget instanciation
//-----------------------------------------------------------------------------------------------------------------------------------------
GtkWidget * CFileDialog::PerformWidgetInstanciate ()
{
	// ok
	return ::gtk_file_selection_new (m_Caption.Get());
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// gtk widget initialization
//-----------------------------------------------------------------------------------------------------------------------------------------
void CFileDialog::PerformWidgetInitialize ()
{
	// generic initialization only
	CDialog::PerformWidgetInitialize ();
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// multiple selection affectation
//-----------------------------------------------------------------------------------------------------------------------------------------
void CFileDialog::SetSelectionMultiple (const bool inMultiple)
{
	// our widget
	GtkWidget *inGtkWidget = GetGtkWidget ();

	// check the pointer
	if (inGtkWidget == NULL) return;

	// gtk property affectation
	::gtk_file_selection_set_select_multiple (GTK_FILE_SELECTION(inGtkWidget), inMultiple);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// multiple selection property reader
//-----------------------------------------------------------------------------------------------------------------------------------------
bool CFileDialog::GetSelectionMultiple () const
{
	// our widget
	GtkWidget *inGtkWidget = GetGtkWidget ();

	// read from the gtk dialog property
	return inGtkWidget != NULL ? ::gtk_file_selection_get_select_multiple (GTK_FILE_SELECTION(inGtkWidget)) : false;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// default selection affectation
//-----------------------------------------------------------------------------------------------------------------------------------------
void CFileDialog::SetFileSelectionOn (const CString &inFileSelection)
{
	// our widget
	GtkWidget *inGtkWidget = GetGtkWidget ();

	// check the pointer and set the property
	if (inGtkWidget != NULL) ::gtk_file_selection_set_filename (GTK_FILE_SELECTION(inGtkWidget), inFileSelection.Get());
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// default selection access
//-----------------------------------------------------------------------------------------------------------------------------------------
CString CFileDialog::GetFileSelectionOn () const
{
	// our gtk widget
	GtkWidget *inGtkWidget = GetGtkWidget();

	// ok ?
	return inGtkWidget != NULL ? CString(::gtk_file_selection_get_filename (GTK_FILE_SELECTION(inGtkWidget))) : CString();
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// pattern affectation
//-----------------------------------------------------------------------------------------------------------------------------------------
void CFileDialog::SetFilePattern (const CString &inPattern)
{
	// our gtk widget
	GtkWidget *inGtkWidget = GetGtkWidget ();

	// check the pointer and set the property
	if (inGtkWidget != NULL) ::gtk_file_selection_complete (GTK_FILE_SELECTION(inGtkWidget), inPattern.Get());
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// dialog selection output
//-----------------------------------------------------------------------------------------------------------------------------------------
CStrings CFileDialog::GetSelection () const
{
	// out file name selection
	CStrings outFileNames;

	// our gtk widget
	GtkWidget *inGtkWidget = GetGtkWidget ();

	// check the pointer and set the property
	if (inGtkWidget == NULL) return outFileNames;

	// multiple selection mode
	if (GetSelectionMultiple ())
	{
		gchar **inFileNames = ::gtk_file_selection_get_selections (GTK_FILE_SELECTION(inGtkWidget));
		gchar **p = inFileNames; while (*p != NULL) outFileNames += CString(*(p++));
		::g_strfreev (inFileNames);
	}
	// single selection mode
	else
	{
		outFileNames += CString(::gtk_file_selection_get_filename (GTK_FILE_SELECTION(inGtkWidget)));
	}

	// ok
	return outFileNames;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// xml serialization
//-----------------------------------------------------------------------------------------------------------------------------------------
void CFileDialog::Serialize (CXMLElementNode *&ioXMLElementNode, const int inMode) THROWABLE
{
	// generic call first
	CDialog::Serialize (ioXMLElementNode, inMode);

	// xml serialization request analye
	switch (inMode)
	{
		// xml dump
		case XML_WRITE :
		{
			// allocate a new xml element
			CXMLElement *newXMLElement = new CXMLElement (ioXMLElementNode, XML_FILEDIALOG_ELEMENT);

			// modify the io xml node pointer so that overloaded definition will continue serialization under this node
			ioXMLElementNode = newXMLElement -> GetXMLElementNode ();

			// set the attributes
			newXMLElement -> AddAttribute (XML_FILEDIALOG_ATTR_SELECTION, CString(GetSelectionMultiple()?"multiple":"single"));
			if (GetFileSelectionOn() != CString())
				newXMLElement -> AddAttribute (XML_FILEDIALOG_ATTR_ROOT, GetFileSelectionOn());
		}
		break;

		// xml load
		case XML_READ :
		{
			// get the xml node the file dialog is expected to load from
			CXMLElementNode *inXMLNode = ::xml_node_get_child (ioXMLElementNode, XML_FILEDIALOG_ELEMENT);

			// check we got it !
			if (inXMLNode == NULL)
				throw new CException (CString("CFileDialog::Serialize, specified xml node is not a \"") + 
							      XML_FILEDIALOG_ELEMENT + CString("\" element one."), __exception(XMLPARSE));

			// modify the io xml node pointer
			ioXMLElementNode = inXMLNode;

			// set the gtk properties
			SetSelectionMultiple (::xml_node_get_attribute 
					      (inXMLNode, XML_FILEDIALOG_ATTR_SELECTION).GetValue() == CString("multiple") ? true : false);
			CString inRoot (::xml_node_get_attribute (inXMLNode, XML_FILEDIALOG_ATTR_ROOT).GetValue()); 
			if (inRoot != CString()) SetFileSelectionOn (inRoot);
		}
		break;
	}
}



