/* usi.cc
 */
#include "osl/record/usi.h"
#include "osl/record/psn.h"
#include "osl/state/simpleState.h"
#include "osl/pieceStand.h"
#include <boost/algorithm/string/replace.hpp>
#include <boost/foreach.hpp>
#include <iostream>
#include <fstream>
#include <sstream>
#include <cctype>

const std::string osl::record::usi::
show(Move m)
{
  if (m.isPass())
    return "pass";
  if (m == Move::DeclareWin())
    return "win";
  if (! m.isNormal())
    return "resign";
  return psn::show(m);
}

const std::string osl::record::usi::
show(PtypeO ptypeo)
{
  if (! isPiece(ptypeo))
    return "";

  char c = psn::show(unpromote(getPtype(ptypeo)));
  if (getOwner(ptypeo) == WHITE)
    c = tolower(c);
  std::string ret(1,c);
  if (isPromoted(ptypeo))
    ret = "+" + ret;
  return ret;
}

const std::string osl::record::usi::
show(Piece p)
{
  return show(p.ptypeO());
}

const std::string osl::record::usi::
show(const SimpleState& state)
{
  std::ostringstream ret;
  if (state == SimpleState(HIRATE)) {
    ret << "startpos";
    return ret.str();
  }
  ret << "sfen ";
  for (int y=1; y<=9; ++y) {
    int empty_count = 0;
    for (int x=9; x>=1; --x) {
      const Piece p = state.pieceOnBoard(Square(x,y));
      if (p.isEmpty()) {
	++empty_count;
	continue;
      }
      if (empty_count) {
	ret << empty_count;
	empty_count = 0;
      }
      ret << show(p);
    }
    if (empty_count)
      ret << empty_count;
    if (y < 9) ret << "/";
  }
  ret << " " << "bw"[state.turn() == WHITE] << " ";
  bool has_any = false;
  for (int z=0; z<2; ++z) {
    const Player player = indexToPlayer(z);
    BOOST_FOREACH(Ptype ptype, PieceStand::order) {
      const int count = state.countPiecesOnStand(player, ptype);
      if (count == 0)
	continue;
      if (count > 1)
	ret << count;
      ret << show(newPtypeO(player, ptype));
      has_any = true;
    }
  }
  if (! has_any)
    ret << "-";
  ret << " 1";
  return ret.str();
}

const osl::Move osl::record::usi::
strToMove(const std::string& str, const SimpleState& s)
{
  if (str == "win")
    return Move::DeclareWin();
  if (str == "pass")
    return Move::PASS(s.turn());
  if (str == "resign")
    return Move::INVALID();
  try {
    return psn::strToMove(str, s);
  }
  catch (...) {
    throw ParseError("usi::strToMove failed for " + str);
  }
}

osl::PtypeO osl::record::usi::
charToPtypeO(char c)
{
  const Ptype ptype = psn::charToPtype(toupper(c));
  if (ptype == PTYPE_EMPTY)
    throw ParseError("Invalid piece character: " + c);
  const Player pl = isupper(c) ? BLACK : WHITE;
  return newPtypeO(pl, ptype);
}

void osl::record::usi::parseBoard(const std::string& word, SimpleState& state)
{
  if (word.empty())
    throw ParseError(word);

  state.init();
  int x=9, y=1;
  for (size_t i=0; i<word.size(); ++i) {
    const char c = word[i];
    if (isalpha(c)) {
      const PtypeO ptypeo = charToPtypeO(c);
      state.setPiece(getOwner(ptypeo), Square(x,y), getPtype(ptypeo));
      --x;
    } else if (c == '+') {
      if ( (i+1) >= word.size() )
        throw ParseError(word);
      const char next = word[i+1];
      if (!isalpha(next))
        throw ParseError(word);
      const PtypeO ptypeo = charToPtypeO(next);
      if (!canPromote(ptypeo))
        throw ParseError(word);
      const PtypeO promoted = promote(ptypeo);
      state.setPiece(getOwner(promoted), Square(x,y), getPtype(promoted));
      --x;
      ++i;
    } else if (c == '/') {
      if (x != 0)
        throw ParseError(word);
      x = 9;
      ++y;
    } else if (isdigit(c)) {
      const int n = c - '0';
      if (n == 0)
        throw ParseError(word);
      x -= n;
    } else {
      throw ParseError("usi: unknown input " + c);
    }
    if (x < 0 || x > 9 || y < 0 || y > 9)
      throw ParseError(word);
  }
}

void osl::record::usi::parse(const std::string& line, NumEffectState& state)
{
  SimpleState board;
  vector<Move> moves;
  parse(line, board, moves);
  state.copyFrom(NumEffectState(board));
  BOOST_FOREACH(Move move, moves) {
    state.makeMove(move);
  }
}

void osl::record::usi::parse(const std::string& line, SimpleState& state, vector<Move>& moves)
{
  moves.clear();
  std::istringstream is(line);
  std::string word;
  is >> word;
  if (word == "position")
    is >> word;
  if (word == "startpos") 
    state.init(HIRATE);
  else {
    if (word != "sfen")
        throw ParseError("sfen not found "+word);
    is >> word;
    parseBoard(word, state);
    is >> word;
    if (word != "b" && word != "w")
        throw ParseError(" turn error "+word);
    state.setTurn((word == "b") ? BLACK : WHITE);
    is >> word;
    if (word != "-") {
      int prefix = 0;
      BOOST_FOREACH(char c, word) {
	if (isalpha(c)) {
	  PtypeO ptypeo = charToPtypeO(c);
	  for (int j=0; j<std::max(1, prefix); ++j)
	    state.setPiece(getOwner(ptypeo), Square::STAND(), getPtype(ptypeo));
	  prefix = 0;
	}
	else {
	  if (!isdigit(c))
            throw ParseError(word);
	  prefix = (c - '0') + prefix*10;
          if (prefix == 0)
            throw ParseError(word);
	}
      }
    }
    state.initPawnMask();
    int move_number; // will not be used
    if (! (is >> move_number))
      return;
    assert(is);
  }
  if (! (is >> word))
    return;
  if (word != "moves")
    throw ParseError("moves not found "+word);
  NumEffectState state_copy(state);
  while (is >> word) {
    Move m = strToMove(word, state_copy);
    moves.push_back(m);
    if (! m.isNormal() || ! state_copy.isValidMove(m))
      throw ParseError("invalid move "+word);
    state_copy.makeMove(m);
  }
} 

void osl::record::usi::
escape(std::string& str)
{
  boost::algorithm::replace_all(str, "/", "_");
  boost::algorithm::replace_all(str, "+", "@");
  boost::algorithm::replace_all(str, " ", ".");
}

void osl::record::usi::
unescape(std::string& str)
{
  boost::algorithm::replace_all(str, "_", "/");
  boost::algorithm::replace_all(str, "@", "+");
  boost::algorithm::replace_all(str, ".", " ");
}


osl::record::usi::
UsiFile::UsiFile(const std::string& filename)
{
  std::ifstream is(filename.c_str());
  std::string line;
  if (! std::getline(is, line))
  {
    const std::string msg = "UsiFile::UsiFile file cannot read ";
    std::cerr << msg << filename << "\n";
    throw usi::ParseError(msg + filename);
  }
  SimpleState initial;
  vector<Move> moves;
  parse(line, initial, moves);
  assert(initial.isConsistent());
  record.setInitialState(initial);
  record::RecordVisitor visitor;
  visitor.setRecord(&record);
  visitor.setState(&initial);
  BOOST_FOREACH(Move move, moves)
    visitor.addMoveAndAdvance(move);
}

osl::record::usi::
UsiFile::~UsiFile()
{
}

const osl::record::Record& osl::record::usi::
UsiFile::getRecord() const
{
  return record;
}

const osl::NumEffectState osl::record::usi::
UsiFile::getInitialState() const
{
  return NumEffectState(record.getInitialState());
}

/* ------------------------------------------------------------------------- */
// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
