/*
 * Written by Bastien Chevreux (BaCh)
 *
 * Copyright (C) 2003 and later by Bastien Chevreux
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the 
 * Free Software Foundation, Inc., 
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA
 * 
 */

// 	$Id$	


#include "io/gbf.H"

#ifndef lint
static char vcid[] = "$Id$";
#endif /* lint */


vector<string>   GBF::GBF_gbffeaturenames;
vector<string>   GBF::GBF_gap4featurenames;


bool GBF::GBF_staticfeaturesinit=false;
// do not delete last (empty) entry in field below, it's needed
//  in GBF::fillFeatureTranslationVectors()
const char * GBF::GBF_featuretranslations[]= {
  "MFSM","MFSM",                // this one does not exist in GenBank !!! just for testing ... should probably use F??? and note=MFSM

  "CDS","FCDS",
  "gene","Fgen",
  "mRNA","FmRN",
  "misc_RNA","Fm-R",
  "precursor_RNA","FpRN",
  "rRNA","FrRN",
  "scRNA","FscR",
  "snRNA","FsnR",
  "tRNA","FtRN",

  "-","F---",
  "-10_signal","F-10",
  "-35_signal","F-35",
  "3'UTR","F3_U",
  "3'clip","F3_c",
  "5'UTR","F5_U",
  "5'clip","F5_c",
  "CAAT_signal","FCAA",
  "C_region","FCre",
  "D-loop","FD-l",
  "D_segment","FD-s",
  "GC_signal","FGCs",
  "J_segment","FJ-s",
  "LTR","FLTR",
  "N_region","FN-r",
  "RBS","FRBS",
  "STS","FSTS",
  "S_region","FS-r",
  "TATA_signal","FTAT",
  "V_region","FV-r",
  "V_segment","FV-s",
  "attenuator","Fatt",
  "conflict","Fcon",
  "enhancer","Fenh",
  "exon","Fexn",
  "iDNA","FiDN",
  "intron","Fint",
  "mat_peptide","Fmat",
  "misc_binding","Fm-b",
  "misc_difference","Fm-d",
  "misc_feature","Fm-f",
  "misc_recomb","Fm-r",
  "misc_signal","Fm-S",
  "misc_structure","Fm-s",
  "modified_base","Fmod",
  "mutation","Fmut",
  "old_sequence","Fold",
  "polyA_signal","FpAS",
  "polyA_site","FpAs",
  "prim_transcript","Fp_t",
  "primer_bind","Fp_b",
  "promoter","Fpro",
  "protein_bind","Fprt",
  "rep_origin","Frpo",
  "repeat_region","Frpr",
  "repeat_unit","Frpu",
  "satellite","Fsat",
  "sig_peptide","Fsig",
  "source","Fsrc",
  "stem_loop","Fstm",
  "terminator","Fter",
  "transit_peptide","Ft-p",
  "unsure","F???",
  "variation","Fvar",
  ""
};



void GBF::foolCompiler()
{
#include "stdinc/foolcompiler.C"
}

// Plain vanilla constructor
GBF::GBF()
{
  FUNCSTART("GBF::GBF()");

  zeroVars();
  init();

  FUNCEND();
}

void GBF::zeroVars()
{
  FUNCSTART("void GBF::zeroVars()");

  GBF_sequencenames.clear();
  GBF_sequences.clear();
  GBF_tags.clear();

  FUNCEND();
}

void GBF::init()
{
  FUNCSTART("void GBF::init()");
  FUNCEND();
}



GBF::~GBF()
{
  FUNCSTART("GBF::~GBF()");

  discard();

  FUNCEND();
}


void GBF::discard()
{
  FUNCSTART("GBF::discard()");

  zeroVars();

  FUNCEND();
}



/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

const string & GBF::getSequenceName(uint32 i) const
{
  FUNCSTART("const string & GBF::getSequenceName(uint32 i) const");

  if(i>=GBF_sequencenames.size()){
    throw Notify(Notify::WARNING, THISFUNC, ": Tried to get out of range sequence name.");
  }

  FUNCEND();
  return GBF_sequencenames[i];
}

/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

const string & GBF::getSequence(uint32 i) const
{
  FUNCSTART("const string & GBF::getSequence(uint32 i) const");

  if(i>=GBF_sequencenames.size()){
    throw Notify(Notify::WARNING, THISFUNC, ": Tried to get out of range sequence.");
  }

  FUNCEND();
  return GBF_sequences[i];
}

/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

const vector<tag_t> & GBF::getTags(uint32 i) const
{
  FUNCSTART("const vector<tag_t> & GBF::getTags(uint32 i) const");

  if(i>=GBF_sequencenames.size()){
    throw Notify(Notify::WARNING, THISFUNC, ": Tried to get out of range tags.");
  }

  FUNCEND();
  return GBF_tags[i];
}

/*************************************************************************
 *
 * Copies keys in "thingstotransfer" from /gene features
 *  to all other features
 *
 * Expects that /gene features precedes the /otherfeature in the GBF
 *
 *************************************************************************/

void GBF::transferGeneInfoToCDSInfo()
{
  FUNCSTART("void GBF::transferGeneInfoToCDSInfo()");

  BUGIFTHROW(GBF_sequences.size()!=GBF_sequencenames.size()
	     || GBF_sequencenames.size() != GBF_tags.size(),
	     "GBF data read is fishy, unequal number of sequences, names and tag vectors?");
  
  vector<string> thingstotransfer;
  thingstotransfer.push_back("/gene");
  thingstotransfer.push_back("/locus_tag");

  for(uint32 i=0; i<GBF_tags.size(); i++){
    vector<tag_t>::iterator J=GBF_tags[i].begin();
    for(uint32 j=1; j< GBF_tags[i].size(); j++){
      if(GBF_tags[i][j].identifier!="Fgen") {
	if(GBF_tags[i][j].from==GBF_tags[i][j-1].from
	   && GBF_tags[i][j].to==GBF_tags[i][j-1].to) {
	  for(uint32 t=0; t<thingstotransfer.size(); t++){
	    string::size_type fpos=GBF_tags[i][j].comment.find(thingstotransfer[t],0);
	    if(fpos==string::npos){
	      fpos=GBF_tags[i][j-1].comment.find(thingstotransfer[t],0);
	      if(fpos!=string::npos){
		string::size_type lpos=GBF_tags[i][j-1].comment.find('\n',0);
		if(lpos==string::npos) lpos=GBF_tags[i][j-1].comment.size();
		string newcomment=GBF_tags[i][j-1].comment.substr(fpos, lpos-fpos);
		if(GBF_tags[i][j].comment.size()){
		  if(newcomment[newcomment.size()-1]!='\n') newcomment+='\n';
		  newcomment+=GBF_tags[i][j].comment;
		}
		GBF_tags[i][j].comment=newcomment;
	      }
	    }
	  }
	}
      }
    }
  }

  FUNCEND();
  return;
}


/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/


void GBF::load(const string & gbfin)
{
  FUNCSTART("GBF::load(const string & gbfin)");

  discard();

  ifstream fin;
  fin.open(gbfin.c_str(), ios::in|ios::ate);
  if(!fin){
    MIRANOTIFY(Notify::WARNING, "File not found: " << gbfin);
  }

  streampos lenfile=fin.tellg();
  // static cast needed for gcc 3.3.3
  if(lenfile==static_cast<streampos>(0)){
    MIRANOTIFY(Notify::FATAL, "Zero length file: " << gbfin);
  }
  fin.seekg(0, ios::beg);

  loadTheFile(fin, lenfile);

  correctForTagPositionErrors();

  //cout << "Name: " << GBF_sequencenames.back() << endl;
  //cout << "Sequence: " << GBF_sequences.back() << endl;

  fin.close();

  FUNCEND();
}


/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void GBF::correctForTagPositionErrors()
{
  FUNCSTART("void GBF::correctForTagPositionErrors()");

  BUGIFTHROW(GBF_sequences.size()!=GBF_sequencenames.size()
	     || GBF_sequencenames.size() != GBF_tags.size(),
	     "GBF data read is fishy, unequal number of sequences, names and tag vectors?");
  
  for(uint32 i=0; i<GBF_tags.size(); i++){
    size_t seqlen=GBF_sequences[i].size();
    vector<tag_t>::iterator J=GBF_tags[i].begin();
    for(;J != GBF_tags[i].end(); J++){
      if(J->to > seqlen) {
	cout << "Fishy tag (to is greater than sequence length of " << seqlen << "):\n" << *J;
	J->to=static_cast<uint32>(seqlen);
      }
    }
  }

  FUNCEND();
  return;
}


/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

// silly gcc 4.3: if lentoreserve is "streampos" type, there's a
// conversion error

//#define CEBUG(bla)   {cout << bla; cout.flush(); }
void GBF::loadTheFile(ifstream & fin, uint64 lentoreserve)
{
  FUNCSTART("void GBF::loadTheFile(ifstream & fin, streampos lentoreserve)");

  ProgressIndicator<uint64> P(0, lentoreserve,1000);
 
  string::size_type linepos;

  uint32 numloci=0;

  tag_t emptytag;
  emptytag.from=0;
  emptytag.to=0;
  emptytag.strand='=';
  emptytag.identifier="";
  emptytag.comment="";

  tag_t acttag=emptytag;
  string actkey;
  string actval;
  vector<int32> fromtopos;

  string line;
  string token="";
  bool haveLocus=false;
  bool haveFeature=false;
  bool haveOrigin=false;
  while (getline(fin,line,'\n')){
    if(P.delaytrigger()) P.progress(fin.tellg());

    // get rid of '\r' from DOS
    while(!line.empty() && line[line.size()-1]=='\r') line.resize(line.size()-1);

    linepos=0;
    getNextToken(line, linepos, token);
    //cout << "line: " << line ;
    //cout << "token\t" << token << endl;
    //cout << "linepos: " << linepos << endl;
    if(token.size()==0) continue;
    if(token=="//") {
      haveLocus=false;
      haveFeature=false;
      haveOrigin=false;
      acttag=emptytag;
      actkey="";
      actval="";
      fromtopos.clear();
    } else if(haveLocus && haveOrigin) {
      // in ORIGIN
      // act token contains the linenumbers ... not interesting to us

      CEBUG("In ORIGIN: line: " << line<<endl);
      string & actseq=GBF_sequences.back();
      for(;linepos<line.size(); linepos++){
	if(!isblank(line[linepos])) {
	  actseq+=line[linepos];
	  CEBUG("Tadaaa: " << line[linepos] << endl);
	}
      }
    } else if(haveLocus && haveFeature) {
      // in feature
      if(token=="ORIGIN" && linepos==6) {
	haveOrigin=true;
	addKeyVal2TagComment(actkey,actval,acttag);
	storeAccumulatedTags(acttag,fromtopos);
      } else if(linepos<=20){
	// new feature

	addKeyVal2TagComment(actkey,actval,acttag);
	storeAccumulatedTags(acttag,fromtopos);

	acttag=emptytag;
	actkey.clear();
	actval.clear();

	string location;
	getNextToken(line, linepos, location);
	fromtopos.clear();
	parseGBFLocation(location,fromtopos,1);

	const char * id= translateGBFfeat2GAP4GBFfeat(token);
	if(strlen(id)>0){
	  acttag.identifier=id;
	  //cout << token << " (" << id << ")" << location << endl;
	}
      } else {
	// feature continuation line
	if(token[0]=='/') {
	  // new key
	  //cout << "Key : " << token << endl;

	  addKeyVal2TagComment(actkey,actval,acttag);

	  actkey.clear();
	  actval.clear();

	  string::size_type equpos=token.find('=');
	  if(equpos!=string::npos && equpos != token.size()-1){
	    actkey=token.substr(0,equpos);
	    actval=token.substr(equpos+1,10000000)+line.substr(linepos,1000000);;
	    if(actkey=="/gene"
	       || actkey=="/locus_tag"
	       || actkey=="/product"
	       || actkey=="/note"
	       || actkey=="/function"
	       || actkey=="/EC_number"
	       || actkey=="/operon"
	       || actkey=="/protein_id"
	       || actkey=="/codon_start"
	       || actkey=="/transl_table"
	       || actkey=="/protein_id"
//	       || actkey=="/translation"
	       || actkey=="/standard_name") {
	      //cout << "Key : " << actkey << "\t" << actval << endl;	    
	      //actkey=substr(1,10000);
	    }else{
	      actkey.clear();
	      actval.clear();
	    }

	  }
	}else{
	  // key continuation line
	  // additionally insert a space to anything but /translation
	  //cout << "cont " << token << endl;
	  if(!actkey.empty()){
	    if(actkey!="/translation") actval+=' ';
	    actval+=token;
	    if(linepos!=string::npos) actval+=line.substr(linepos,100000);
	  }
	}
      }
    } else if(haveLocus) {
      // after locus
      if(token=="FEATURES") {
	haveFeature=true;
      }else if(token=="ORIGIN") {
	haveOrigin=true;
      }
    } else {
      // nothing here yet
      if(token=="LOCUS") {
	haveLocus=true;
	numloci++;
	string seqname;
	getNextToken(line, linepos,seqname);
	GBF_sequencenames.push_back(seqname);
	GBF_sequences.push_back("");
	GBF_sequences.back().reserve(lentoreserve);
	GBF_tags.resize(numloci);
	GBF_tags.back().reserve(10000);
      }else{
	throw Notify(Notify::FATAL, THISFUNC, "Missing LOCUS token as first entry in file ... are you sure that this is a Genbank file?");
      }
    }
  }

  P.finishAtOnce();
  cout << endl;

  FUNCEND();
}

void GBF::addKeyVal2TagComment(const string & actkey, const string & actval, tag_t & thetag) const 
{
  if(!actkey.empty()) {
    if(!thetag.comment.empty()) thetag.comment+='\n';
    thetag.comment+=actkey+'='+actval;
  }
}



/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/


void GBF::storeAccumulatedTags(tag_t & acttag, vector<int32> & fromto) 
{
  FUNCSTART("void GBF::storeAccumulatedTags(tag_t & acttag, vector<int32> & fromto)");

  if(!acttag.identifier.empty()){
    //cout << "Write tag(s):"<<endl;
    //cout << "identifier: " << acttag.identifier << endl;
    //cout << "comment: " << acttag.comment << endl;
    //cout << "for ranges: ";
    if(fromto.size()%3 != 0) {
      throw Notify(Notify::INTERNAL, THISFUNC, "Could not parse feature location in GBF file. This also might be an FATAL error due to buggy GBF!");
    }
    vector<int32>::const_iterator I=fromto.begin();
    for(;I!=fromto.end(); I+=3) {
      if(*I>0) {
	acttag.strand='+';
      }else{
	acttag.strand='-';
      }
      // tags have positions beginning at 0 in MIRA
      acttag.from=*(I+1)-1;
      acttag.to=*(I+2)-1;
      //cout << "Storing: ";
      //acttag.dump();
      GBF_tags.back().push_back(acttag);
      //cout << "Stored: ";
      //GBF_tags.back().back().dump();
      //cout << "strand: " << acttag.strand << endl;
      //cout << "from: " << acttag.from << "\tto: " << acttag.to << endl;
    }
  }

  FUNCEND();
}




/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void GBF::getNextToken(const string & line, string::size_type & linepos, string & token) const 
{
  token="";
  if(linepos>=line.size()) {
    linepos=string::npos;
    return;
  }

  string blanks=" \t";

  string::size_type tokenstart=string::npos;
  string::size_type tokenend=string::npos;

  tokenstart=line.find_first_not_of(blanks,linepos);
  if(tokenstart==string::npos){
    linepos=string::npos;
    return;
  }
  tokenend=line.find_first_of(blanks,tokenstart);
  if(tokenend==string::npos) tokenend=line.size();

  token=line.substr(tokenstart, tokenend-tokenstart);
  linepos=tokenend;
  return;
}



/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/


/*************************************************************************
 *
 *
 *
 *
 *************************************************************************/

void GBF::parseGBFLocation(const string & location, vector<int32> & fromto, int32 direction) const
{
  if(location.empty()) return;

  if(location[0]=='c'
     || location[0]=='j'
     || location[0]=='o') { 
    if(location[0]=='c') direction*=-1;
    string::size_type leftbracket=location.find('(');
    if(leftbracket==string::npos) return;
    string::size_type rightbracket=getCorrespondingRightBracket(location, leftbracket);

    parseGBFLocation(location.substr(leftbracket+1,rightbracket-leftbracket-1),
		     fromto,
		     direction);
    rightbracket++;
    if(rightbracket<location.size())
      parseGBFLocation(location.substr(rightbracket+1,1000000),
		       fromto,
		       1);
  } else {
    string::size_type locpos=0;
    long firstnum=0;
    long secondnum=0;
    if(location[0]=='<') locpos++;
    if(location[0]=='>') locpos++; //??
    if(location[0]==',') locpos++;
    if(location[0]=='('){
      //cout << "Analyse bracket val 1st: " << location << endl;
      string::size_type rightbracket=getCorrespondingRightBracket(location, 0);
      parseGBFLocation(location.substr(1,rightbracket-1),
		       fromto,
		       direction);
//      werte poppen und kleinsten nehmen
      int32 choice2=fromto.back();
      fromto.pop_back();
      int32 choice1=fromto.back();
      fromto.pop_back();
      fromto.pop_back();
      if(choice1<choice2) {
	firstnum=choice1;
	secondnum=choice2;
      } else {
	firstnum=choice2;
	secondnum=choice1;
      }
 
      locpos+=rightbracket+1;
      //cout << "Rest after bracket: " << (char *) &location[locpos] << endl;
    }else{
      //cout << "Analyse firstnum: " << location << endl;
      
      firstnum=strtol(&location[locpos],NULL,10);
      //cout << "Firstnum is " << firstnum << endl;
      while(locpos<location.size() && isdigit(location[locpos])) locpos++;
      //cout << "Eaten until " << location[locpos] << endl;
      //string numbers="0123456789";
      //string::size_type numend=location.find_first_not_of(numbers,locpos);
      //string firstnum=location.substr(locpos,numend);
      secondnum=firstnum;
    }
    if(locpos<location.size()){
      //cout << "Analyse secondpart: " << (char *) &location[locpos] << endl;
      while(locpos<location.size() 
	    && (location[locpos] == '.' 
		|| location[locpos]=='^'
		|| location[locpos]=='<'
		|| location[locpos]=='>')) {
	//cout << "Eaten " << location[locpos] << endl;
	locpos++;
      }
      if(location[locpos]=='('){
	//cout << "Analyse bracket val 2nd: " << location[locpos] << endl;
	string::size_type rightbracket=getCorrespondingRightBracket(location, locpos);
	//cout << location.substr(locpos+1,rightbracket-locpos-1) << " rightbrack: " << location[rightbracket] << endl;
	parseGBFLocation(location.substr(locpos+1,rightbracket-locpos-1),
			 fromto,
			 direction);
//      werte poppen und grten nehmen
	int32 choice2=fromto.back();
	fromto.pop_back();
	int32 choice1=fromto.back();
	fromto.pop_back();
	// pop direction
	fromto.pop_back();
	//cout << "popped: " << choice1 << " " << choice2 << "\n";
	if(choice1>choice2) {
	  secondnum=choice1;
	} else {
	  secondnum=choice2;
	}
      }else{
	secondnum=strtol(&location[locpos],NULL,10);
	while(locpos<location.size() && isdigit(location[locpos])) locpos++;
	//cout << "Eaten until " << location[locpos] << endl;
      }
    }
    //cout << "Secondnum is " << secondnum << endl;
    fromto.push_back(direction);
    fromto.push_back(static_cast<uint32>(firstnum));
    fromto.push_back(static_cast<uint32>(secondnum));
    //cout << "Push direction: " << direction << endl;
    //cout << "Push firstnum: " << firstnum << endl;
    //cout << "Push secondnum: " << secondnum << endl;
    if(locpos<location.size() && location[locpos]==','){
      parseGBFLocation(location.substr(locpos+1,100000000),
		       fromto,
		       direction);
    }
  }
}



/*************************************************************************
 *
 * start must be on first ( bracket
 * gives back position of last character _before_ right bracket
 *
 *************************************************************************/

string::size_type GBF::getCorrespondingRightBracket(const string & chars, string::size_type start) const
{
  FUNCSTART("string::size_type GBF::getCorrespondingRightBracket(const string & chars, const size_type start) const");

  if(start<chars.size()) {
    if(chars[start]!='('){
      cerr << chars << "At start " << start << " in " << chars << " is " << chars[start] << " and not a bracket?\n";
      throw Notify(Notify::FATAL, THISFUNC, "expected a open bracket (. This also might be an INTERNAL error in the parsing routine!");
    }
  } else {
    throw Notify(Notify::INTERNAL, THISFUNC, ": tried to parse after string???");
  }
  start++;
  int32 leftopen=1;
  while(start<chars.size()){
    //cout << (char) chars[start]; 
    if(leftopen==1 && chars[start]==')') break;
    if(chars[start]=='(') leftopen++;
    if(chars[start]==')') leftopen--;
    start++;
  }

  //cout << endl;

  //cout << "start " << start << "\tchars.size() " << chars.size() << endl;

  // this line is for "saving" forgotten close brackets at end of lines
  if(start==chars.size() && chars[start-1]==')')  start--;

  FUNCEND();
  return start;
}

void GBF::fillFeatureTranslationVectors()
{
  FUNCSTART("void GBF::fillFeatureTranslationVectors()");

  if(GBF_gbffeaturenames.size()==0){
    uint32 i=0;
    while( strlen(GBF_featuretranslations[i]) != 0) {
      GBF_gbffeaturenames.push_back(GBF_featuretranslations[i]);
      i++;
      GBF_gap4featurenames.push_back(GBF_featuretranslations[i]);
      i++;
    }
  }

  GBF_staticfeaturesinit=true;

  FUNCEND();
}

const char * GBF::translateGBFfeat2GAP4GBFfeat(const string & feature)
{
  if(!GBF_staticfeaturesinit) fillFeatureTranslationVectors();

  vector<string>::const_iterator gbfI=GBF_gbffeaturenames.begin();
  vector<string>::const_iterator gapI=GBF_gap4featurenames.begin();

  for(; gbfI!=GBF_gbffeaturenames.end(); gbfI++, gapI++){
    if(feature==*gbfI) return gapI->c_str();
  }

  return "";
}

const char * GBF::translateGAP4GBFfeat2GBFfeat(const string & feature)
{
  if(!GBF_staticfeaturesinit) fillFeatureTranslationVectors();

  vector<string>::const_iterator gbfI=GBF_gbffeaturenames.begin();
  vector<string>::const_iterator gapI=GBF_gap4featurenames.begin();

  for(; gapI!=GBF_gap4featurenames.end(); gbfI++, gapI++){
    if(feature==*gapI) return gbfI->c_str();
  }

  return "";
}

bool GBF::checkIfGBFfeature(const string & feature)
{
  if(!GBF_staticfeaturesinit) fillFeatureTranslationVectors();

  vector<string>::const_iterator gbfI=GBF_gbffeaturenames.begin();

  for(; gbfI!=GBF_gbffeaturenames.end(); gbfI++){
    if(feature==*gbfI) return true;
  }

  return false;
}

bool GBF::checkIfGAP4GBFfeaure(const string & feature)
{
  if(!GBF_staticfeaturesinit) fillFeatureTranslationVectors();

  vector<string>::const_iterator gapI=GBF_gap4featurenames.begin();

  for(; gapI!=GBF_gap4featurenames.end(); gapI++){
    if(feature==*gapI) return true;
  }

  return false;
}



//// Copy constructor
////  no discard needed as this object will be freshly created when
////  called through this constructor
//GBF::GBF(GBF const &other)
//{
//  FUNCSTART("GBF::GBF(GBF const &other)");
//
//  GBF_valid=0;
//
//  *this=other;                               // call the copy operator
//
//  FUNCEND();
//}
//
//// Copy operator, needed by copy-constructor
//GBF const & GBF::operator=(GBF const & other)
//{
//  FUNCSTART("GBF const & GBF::operator=(GBF const & other)");
//  ERROR("Not implemented yet.");
//  FUNCEND();
//  return *this;
//}

//ostream & operator<<(ostream &ostr, GBF const &gbf)
//{
//  FUNCSTART("friend ostream & GBF::operator<<(ostream &ostr, const  &gbf)");
//  ERROR("Not implemented yet.");
//
//  FUNCEND();
//  return ostr;
//}
