//=========================================================
//  MusE
//  Linux Music Editor
//    $Id: mess.h,v 1.10 2002/02/27 08:48:09 muse Exp $
//  (C) Copyright 2001 Werner Schweer (ws@seh.de)
//=========================================================

#ifndef __MESS_H__
#define __MESS_H__

#include <config.h>

#include <list>

#ifdef ALSACVS
#include <alsa/asoundlib.h>
#else
#include <sys/asoundlib.h>
#endif
#include "globaldefs.h"

//---------------------------------------------------------
//   MidiPatch
//---------------------------------------------------------

struct MidiPatch {
      char typ;                     // 1 - GM  2 - GS  4 - XG
      char hbank, lbank, prog;
      const char* name;
      };

//---------------------------------------------------------
//   RawMidiEvent
//    midi event
//---------------------------------------------------------

class RawMidiEvent {
      int _type;           // alsa event type
      char _chan;
      int _dataA;
      int _dataB;
      unsigned char* _data;
      int _dataLen;

   public:
      RawMidiEvent() {
            _data = 0;
            }
      RawMidiEvent(int t, char c, char a, char b, unsigned char* d = 0,
         int l = 0) {
            _type    = t;
            _chan    = c;
            _dataA   = a;
            _dataB   = b;
            _data    = d;
            _dataLen = l;
            }

      ~RawMidiEvent() {
            if (_data)
                  delete _data;
            }
      int type() const              { return _type; }
      void setType(int t)           { _type = t; }
      int chan() const              { return _chan; }
      void setChan(int a)           { _chan = a; }
      int dataA() const             { return _dataA; }
      void setDataA(int a)          { _dataA = a;    }
      int dataB() const             { return _dataB; }
      void setDataB(int a)          { _dataB = a;    }
      unsigned char* data() const   { return _data; }
      void setData(unsigned char* p) { _data = p; }
      int dataLen() const           { return _dataLen; }
      void setDataLen(int l)        { _dataLen = l; }
      };

//---------------------------------------------------------
//   MEvent
//    time stamped midi event
//---------------------------------------------------------

class MEvent : public RawMidiEvent {
      int _samplePos;                // time stamp

   public:
      MEvent(unsigned sp, int t, char c, char a, char b, unsigned char* d = 0,
         int l = 0) : RawMidiEvent(t, c, a, b, d, l) {
            _samplePos = sp;
            }
      ~MEvent() {}
      int samplePos() const         { return _samplePos; }
      void adjustSamplePos(int bs)  { _samplePos %= bs; }
      };

typedef std::list<MEvent* > MEventQueue;
typedef MEventQueue::iterator iMEvent;

//---------------------------------------------------------
//  Mess
//    MusE experimental software synth
//    LADSPA instance
//---------------------------------------------------------

class Mess {
      int _sampleRate;

   protected:
      const char* _className;       // synth plugin name
      char* _name;                  // instance name
      snd_seq_t* alsaSeq;
      snd_seq_addr_t _alsaPort;     // registered alsa midi port
      snd_seq_addr_t _musePort;
      struct pollfd* pfd;           // poll file descriptors for midi
      int npfd;
      double startFrame;
      pthread_mutex_t lock;
      MEventQueue events;

      void sendSysex(const unsigned char* p, int n);
      virtual void processEvent(MEvent* ev) = 0;
      virtual void write(int /*n*/, float** /*buffer*/, int /*offset*/) {}
      float** outputPorts;
      int _channels;           // 1 - mono,  2 - stereo

   public:
      uid_t euid, ruid;
      Mess(const char* instName, int channels);
      virtual ~Mess();
      void registerAlsa();

      virtual bool init()            { return false; }
      void setPort(int n, float* p)  { outputPorts[n] = p; }
      void midiRun();
      pthread_t midiThread;
      int channels() const           { return _channels; }
      int sampleRate() const         { return _sampleRate; }
      void setSampleRate(int r)      { _sampleRate = r; }

      int alsaClient() const         { return _alsaPort.client; }
      int alsaPort() const           { return _alsaPort.port; }
      void setName(const char* s);
      const char* name() const       { return _name; }
      const char* className() const  { return _className; }
      virtual const char* getPatchName(int channel, int hbank, int lbank,
         int prog, MType type);
      virtual const MidiPatch* getNextPatch(int /*ch*/, const MidiPatch*) const { return 0;}

      virtual int getMidiInitEvent(int /*id*/, RawMidiEvent*) { return 0; }
      virtual void processreplace(int len);

      static void initMess(unsigned long id, const char* label, const char* maker,
         const char* name, const char* copyright, unsigned long ports,
         Mess* (inst)(const char*));
      };

//---------------------------------------------------------
//   MessMono
//    implements some more functions for monophone
//    synthesizer
//---------------------------------------------------------

struct PitchVelo {
      char channel;
      char pitch;
      char velo;
      PitchVelo(char a, char b, char c)
         : channel(a), pitch(b), velo(c) {}
      };

class MessMono : public Mess {
      std::list<PitchVelo> pitchStack;

      void midiNoteOn(int channel, int pitch, int velo);
      void midiNoteOff(int channel, int pitch);

   protected:
	int ctrlHi;       //	Parametervariables
	int ctrlLo;
	int dataHi;
	int dataLo;
      virtual void processEvent(MEvent* ev);

   public:
      MessMono(const char*, int channels);
      virtual ~MessMono();
      virtual void noteon(int channel, int pitch, int velo) = 0;
      virtual void noteoff(int channel, int pitch) = 0;
      virtual void sysex(const unsigned char* /* data*/, int /*len*/) {}
	virtual void setController(int /*channel*/, int /*ctrl*/, int /*val*/) {}
      };

#endif

