## Copyright (C) 2004  Dragan Tubic
## 
## This program is free software; you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation; either version 2, or (at your option)
## any later version.
## 
## This program is distributed in the hope that it will be useful, but
## WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
## General Public License for more details. 
## 
## You should have received a copy of the GNU General Public License
## along with this file.  If not, write to the Free Software Foundation,
## 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

## -*- texinfo -*-
## @deftypefn {Function File} {} vtk_poly3(@var{x}, @var{y}, @var{z}, [@var{fmt} | @var{prop},@var{val}])
## Plots the specified points as a 3D polygon.  The data point
## coordinates must be given as 3 equal length @var{x}, @var{y}, and
## @var{z} vectors. The optional @var{fmt} takes the form of a "line
## specification"; e.g. "r" would result in a red polygon.  The optional
## @var{prop},@var{val} is a property,value pair argument. Valid
## properties are currently ";Color;Opacity;".
## @end deftypefn
## @seealso{vtk_plot3}

## Author: Dragan Tubic

function vtk_poly3(varargin)
  
  valid_props = ";Color;Opacity;";
  [no_numerical_params, first_prop_index, line_spec_index] = vtk_parse_params(valid_props, all_va_args);
  
  if ( no_numerical_params < 3 )
    error("Syntax is vtk_poly3(x,y,z)");
  end
  
  x = nth (varargin,1);
  y = nth (varargin,2);
  z = nth (varargin,3);

  ## set default color
  color = [0 0 1];   # same as default in vtk_get_line_spec function

  ## use "line specs", if provided
  if ( line_spec_index > 0 )
    line_spec = nth (varargin,line_spec_index);
    [color, marker_type, line_style] = vtk_get_line_spec( line_spec );
  end
  ## use "property specs", if provided
  if ( first_prop_index > 0 )
    properties = struct(varargin{first_prop_index:length(varargin)});
    if ( isfield(properties,"Color") )
      color = properties.Color;
    end
  end
  properties.Color = color;

  [nr nc] = size(x);
  if ( nr < nc )
    x = x';
    y = y';
    z = z';
  end
  
  no_points = length(x);

  
  vtk_init;   

  coords = vtkFloatArray; 
  coords.SetNumberOfTuples( no_points );
  coords.SetNumberOfComponents(3);
  pts = [x y z]';
  
  coords.SetArray( pts(:), 3*no_points, 0 );
  points = vtkPoints;
  points.SetData(coords);	
  
  polygon = vtkPolygon();
  polygon.GetPointIds().SetNumberOfIds(no_points);
  ptids = polygon.GetPointIds();
  for	i = 0:no_points-1
    ptids.SetId( i, i );
  end
  
  poly_grid = vtkUnstructuredGrid;
  poly_grid.Allocate(1,1);
  poly_grid.InsertNextCell( polygon.GetCellType(), polygon.GetPointIds() );
  poly_grid.SetPoints( points );
  
  poly_mapper = vtkDataSetMapper;
  poly_mapper.SetInput( poly_grid );
  
  poly_actor = vtkActor;
  poly_actor.SetMapper( poly_mapper );
  if struct_contains(properties,"Color")
    c = properties.Color;
    poly_actor.GetProperty().SetDiffuseColor( c(1), c(2), c(3) );
  end
  if struct_contains(properties,"Opacity")
    poly_actor.GetProperty().SetOpacity(properties.Opacity);
  end
  
  poly_actor.GetProperty().BackfaceCullingOff();
  
  f = vtk_figure(0);
  
  f.renderer.AddActor(poly_actor);
  vtk_update(f);
  
endfunction
