//                            Package   : omniEvents
// EventChannel.cc            Created   : 2003/12/04
//                            Author    : Alex Tingle
//
//    Copyright (C) 2003 Alex Tingle.
//
//    This file is part of the omniEvents application.
//
//    omniEvents is free software; you can redistribute it and/or
//    modify it under the terms of the GNU Lesser General Public
//    License as published by the Free Software Foundation; either
//    version 2.1 of the License, or (at your option) any later version.
//
//    omniEvents is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//    Lesser General Public License for more details.
//
//    You should have received a copy of the GNU Lesser General Public
//    License along with this library; if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//

#include "EventChannel.h"
#include "ConsumerAdmin.h"
#include "SupplierAdmin.h"
#include "omniEventsLog.h"
#include "Orb.h"

#include <list>

namespace OmniEvents {

// CORBA interface methods
CosEventChannelAdmin::ConsumerAdmin_ptr EventChannel_i::for_consumers()
{
  if(!_consumerAdmin || _shutdownRequested)
      throw CORBA::OBJECT_NOT_EXIST();
  return _consumerAdmin->_this();
}


CosEventChannelAdmin::SupplierAdmin_ptr EventChannel_i::for_suppliers()
{
  if(!_supplierAdmin || _shutdownRequested)
      throw CORBA::OBJECT_NOT_EXIST();
  return _supplierAdmin->_this();
}


void EventChannel_i::destroy()
{
  if(_shutdownRequested)
      throw CORBA::OBJECT_NOT_EXIST();

  // Send disconnect messages to connected clients.
  // (Admins might be NULL if destroy() is somehow called before activate.)
  if(_consumerAdmin)
     _consumerAdmin->disconnect();
  if(_supplierAdmin)
     _supplierAdmin->disconnect();
  if(_mapper)
     _mapper->destroy();

  // Terminate the thread.
  _shutdownRequested=true;
  
  //?? There is some danger that new connections will be established between now
  //?? and channel shutdown, These connections should be rejected.
}


EventChannel_i::EventChannel_i(EventChannelStore* store)
: Servant(PortableServer::POA::_nil()),
  _eventChannelStore(store),
  _consumerAdmin(NULL),
  _supplierAdmin(NULL),
  _poaManager(),
  _shutdownRequested(false),
  _properties(),
  _mapper(NULL)
{}


void EventChannel_i::activate(
  const char*        channelName,
  const PersistNode* node
)
{
  // The order of these various initialization methods is very important.
  // I've documented dependencies as 'REQUIRES' comments.

  createPoa(channelName);

  if(node)
      _properties._attr=node->_attr;

  // REQUIRES: _properties
  _consumerAdmin=new ConsumerAdmin_i(*this,_poa);

  // REQUIRES: _consumerAdmin, _properties
  _supplierAdmin=new SupplierAdmin_i(*this,_poa);

  if(node)
  {
    PersistNode* saNode =node->child("SupplierAdmin");
    if(saNode)
        _supplierAdmin->reincarnate(*saNode);

    PersistNode* caNode =node->child("ConsumerAdmin");
    if(caNode)
        _consumerAdmin->reincarnate(*caNode);
  }

  activateObjectWithId("EventChannel");

  // REQUIRES: activate() ...since it uses _this().
  setInsName(_properties.attrString("InsName"));
}


EventChannel_i::~EventChannel_i()
{
  if(CORBA::is_nil(_poa))
  {
    DB(20,"~EventChannel_i()")
  }
  else
  {
    DB(20,"~EventChannel_i() - destroying POA")
    try {
      _poa->destroy(
        CORBA::Boolean(1) /* etherealize_objects */,
        CORBA::Boolean(0) /* wait_for_completion */
      );
    }
    catch(...) {
      DB(2,"~EventChannel_i() - ERROR destroying POA")
    }
    _poa=PortableServer::POA::_nil();
  }
}


void EventChannel_i::run(void* arg)
{
  // Ensure that activate() is called before start()/run().
  assert(!CORBA::is_nil(_poa));

  try
  {
    // Add this object to the store.
    if(_eventChannelStore)
    {
      _eventChannelStore->insert(this);
      output(WriteLock().os);
    }

    // Process events until the channel is destroyed.
    mainLoop();

    // Remove this object from the store.
    if(_eventChannelStore)
    {
      _eventChannelStore->erase(this);
      CORBA::String_var poaName =_poa->the_name();
      WriteLock log;
      log.os<<"-ecf/"<<poaName.in()<<'\n';
    }

    _poa->destroy(
      CORBA::Boolean(1) /* etherealize_objects */,
      CORBA::Boolean(1) /* wait_for_completion */
    );
    _poaManager->deactivate(
      CORBA::Boolean(1) /* etherealize_objects */,
      CORBA::Boolean(1) /* wait_for_completion */
    );
    _poa=PortableServer::POA::_nil();

  }
  catch(PortableServer::POAManager::AdapterInactive& ex)
  {
    DB(0,"EventChannel_i::run() - POA deactivated from the outside.")
    Orb::inst().reportObjectFailure(HERE,_this(),&ex);
  }
  catch (CORBA::Exception& ex) {
    Orb::inst().reportObjectFailure(HERE,_this(),&ex);
  }
  catch(...)
  {
    Orb::inst().reportObjectFailure(HERE,_this(),NULL);
  }

  // Thread now exits, and this object is deleted.
  // Contents are cleaned up by the destructor.
}


void EventChannel_i::mainLoop()
{
  _poaManager->activate();
  unsigned long localCyclePeriod_ns=cyclePeriod_ns();
  while(!_shutdownRequested)
  {
    //
    // TRANSFER PHASE - transfer events from SupplierAdmin to ConsumerAdmin.
    _poaManager->hold_requests(CORBA::Boolean(1) /* wait_for_completion */);

    list<CORBA::Any*> events;
    _supplierAdmin->collect(events);
    _consumerAdmin->send(events);
    assert(events.empty());

    _poaManager->activate();
    
    //
    // COMMUNICATION PHASE - talk with clients' suppliers & consumers.
    // Note: On Linux the resolution of nanosleep is a huge 10ms.
    omni_thread::sleep(0,localCyclePeriod_ns);
  }
}


void EventChannel_i::output(ostream& os)
{
  CORBA::String_var poaName =_poa->the_name();
  string name =string("ecf/")+poaName.in();
  _properties.output(os,name);
  if(_supplierAdmin)
     _supplierAdmin->output(os);
  if(_consumerAdmin)
     _consumerAdmin->output(os);
}


void EventChannel_i::setInsName(const string v)
{
  Mapper* newMapper =NULL;
  try
  {

    // If _insName is set, then create a mapper object to allow clients to
    // find this object with a `corbaloc' string.
    if(!v.empty())
    {
      // !! Throws when there is already an object named 'v' in the INSPOA.
      newMapper=new Mapper(v.c_str(),_this());
    }
    // Deactivate the old _mapper object.
    if(_mapper)
       _mapper->destroy();
    _mapper=newMapper;

  }
  catch(...)
  {
    // Can't use an auto_ptr, because MS VC++ 6 has no auto_ptr::reset()
    delete newMapper;
    throw;
  }
}


void EventChannel_i::createPoa(const char* channelName)
{
  using namespace PortableServer;
  POA_ptr p=Orb::inst()._RootPOA.in();
  try
  {
    // POLICIES:
    //  Lifespan          =PERSISTENT             // we can persist
    //  Assignment        =USER_ID                // write our own oid
    //  Uniqueness        =[default] UNIQUE_ID    // one servant per object
    //  ImplicitActivation=[default] IMPLICIT_ACTIVATION // auto activation
    //  RequestProcessing =[default] USE_ACTIVE_OBJECT_MAP_ONLY
    //  ServantRetention  =[default] RETAIN       // stateless POA
    //  Thread            =SINGLE_THREAD_MODEL    // keep it simple

    CORBA::PolicyList policies;
    policies.length(3);
    policies[0]=p->create_lifespan_policy(PERSISTENT);
    policies[1]=p->create_id_assignment_policy(USER_ID);
    policies[2]=p->create_thread_policy(SINGLE_THREAD_MODEL);

    // Create a new POA (and new POAManager) for this channel.
    // The POAManager will be used for all of this channel's POAs.
    _poa=p->create_POA(channelName,POAManager::_nil(),policies);
    _poaManager=_poa->the_POAManager();
  }
  catch(POA::AdapterAlreadyExists& ex) // create_POA
  {
    DB(0,"EventChannel_i::createPoa() - POA::AdapterAlreadyExists")
    throw;
  }
  catch(POA::InvalidPolicy& ex) // create_POA
  {
    DB(0,"EventChannel_i::createPoa() - POA::InvalidPolicy: "<<ex.index)
    throw;
  }
}


//
// class EventChannelStore
//


EventChannelStore::EventChannelStore()
:_channels(),_lock()
{}

EventChannelStore::~EventChannelStore()
{
  // ?? IMPLEMENT ME
}

void EventChannelStore::insert(EventChannel_i* channel)
{
  omni_mutex_lock l(_lock);
  bool insertOK =_channels.insert(channel).second;
  if(!insertOK)
      DB(2,"Attempted to store an EventChannel, when it is already stored.");
}

void EventChannelStore::erase(EventChannel_i* channel)
{
  omni_mutex_lock l(_lock);
  set<EventChannel_i*>::iterator pos =_channels.find(channel);
  if(pos==_channels.end())
      DB(2,"Failed to erase unknown EventChannel.")
  else
      _channels.erase(pos);
}

void EventChannelStore::output(ostream &os)
{
  omni_mutex_lock l(_lock);
  for(set<EventChannel_i*>::iterator i=_channels.begin();
      i!=_channels.end();
      ++i)
  {
    (*i)->output(os);
  }
}


}; // end namespace OmniEvents

