///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/scene/objects/SceneObject.h>
#include <core/gui/properties/BooleanPropertyUI.h>
#include <core/gui/properties/FilenamePropertyUI.h>

#include <atomviz/parser/AtomsImportObject.h>
#include "CalcDisplacementsModifier.h"

namespace AtomViz {

IMPLEMENT_SERIALIZABLE_PLUGIN_CLASS(CalcDisplacementsModifier, AtomsObjectModifierBase)
DEFINE_REFERENCE_FIELD(CalcDisplacementsModifier, SceneObject, "Reference Configuration", _referenceObject)
DEFINE_REFERENCE_FIELD(CalcDisplacementsModifier, DisplacementDataChannel, "DisplacementChannelPrototype", _displacementChannelPrototype)
DEFINE_PROPERTY_FIELD(CalcDisplacementsModifier, "ShowReferenceConfiguration", _referenceShown)
SET_PROPERTY_FIELD_LABEL(CalcDisplacementsModifier, _referenceObject, "Reference Configuration")
SET_PROPERTY_FIELD_LABEL(CalcDisplacementsModifier, _displacementChannelPrototype, "Displacement Channel")
SET_PROPERTY_FIELD_LABEL(CalcDisplacementsModifier, _referenceShown, "Show reference configuration")

/******************************************************************************
* Constructs the modifier object.
******************************************************************************/
CalcDisplacementsModifier::CalcDisplacementsModifier(bool isLoading) : AtomsObjectModifierBase(isLoading),
	_referenceShown(false)
{
	INIT_PROPERTY_FIELD(CalcDisplacementsModifier, _referenceObject);
	INIT_PROPERTY_FIELD(CalcDisplacementsModifier, _displacementChannelPrototype);
	INIT_PROPERTY_FIELD(CalcDisplacementsModifier, _referenceShown);
	if(!isLoading) {
		_referenceObject = new AtomsImportObject();
		_displacementChannelPrototype = new DisplacementDataChannel(DataChannel::DisplacementChannel);
		_displacementChannelPrototype->setVisible(true);
	}
}

/******************************************************************************
* Asks the modifier for its validity interval at the given time.
******************************************************************************/
TimeInterval CalcDisplacementsModifier::modifierValidity(TimeTicks time)
{
	TimeInterval interval = TimeForever;
	if(_referenceObject) {
		interval.intersect(_referenceObject->objectValidity(time));
		PipelineFlowState refState = _referenceObject->evalObject(time);
		interval.intersect(refState.stateValidity());
	}
	return interval;
}

/******************************************************************************
* This modifies the input object.
******************************************************************************/
EvaluationStatus CalcDisplacementsModifier::modifyAtomsObject(TimeTicks time, TimeInterval& validityInterval)
{
	// Get the reference positions of the atoms.
	if(!_referenceObject)
		throw Exception(tr("Cannot calculate atomic displacements. No atomic reference configuration has been specified."));

	// Get the reference configuration.
	PipelineFlowState refState = _referenceObject->evalObject(time);
	AtomsObject* refObj = dynamic_object_cast<AtomsObject>(refState.result());
	if(!refObj)
		throw Exception(tr("Please choose an atomic reference configuration."));
	validityInterval.intersect(refState.stateValidity());

	// Deformed and reference configuration must contain the same number of atoms.
	if(refObj->atomsCount() != input()->atomsCount()) {
		throw Exception(tr("Cannot calculate atomic displacements. Mismatch between number of atoms in reference configuration and current configuration."));
	}

	// Get the two position channels.
	DataChannel* posChannel = expectStandardChannel(DataChannel::PositionChannel);
	DataChannel* refPosChannel = refObj->getStandardDataChannel(DataChannel::PositionChannel);
	if(!refPosChannel)
		throw Exception(tr("The reference configuration does not contain atomic positions."));

	// Copy atomic positions in reference configuration into geometry pipeline if enabled to make it visible.
	CloneHelper cloneHelper;
	DataChannel* outputPosChannel = NULL;
	if(referenceShown()) {
		outputPosChannel = outputStandardChannel(DataChannel::PositionChannel);
		output()->simulationCell()->setCellMatrix(refObj->simulationCell()->cellMatrix());
	}

	// Create a copy of the displacement channel prototype that will store the calculated displacement vectors.
	DataChannel::SmartPtr displacementChannel = cloneHelper.cloneObject(_displacementChannelPrototype, true);
	displacementChannel->setSize(output()->atomsCount());

	// Put the new displacement channel into the output object.
	output()->insertDataChannel(displacementChannel);
	OVITO_ASSERT(displacementChannel->size() == refPosChannel->size());
	OVITO_ASSERT(displacementChannel->size() == posChannel->size());

	const array<bool, 3> pbc = input()->simulationCell()->periodicity();

	AffineTransformation simCell;
	if(referenceShown())
		simCell = input()->simulationCell()->cellMatrix();
	else
		simCell = refObj->simulationCell()->cellMatrix();

	const Point3* u0 = refPosChannel->constDataPoint3();
	const Point3* u = posChannel->constDataPoint3();
	Point3* ou = outputPosChannel ? outputPosChannel->dataPoint3() : NULL;
	Vector3* d = displacementChannel->dataVector3();
	for(size_t i = posChannel->size(); i != 0; --i, ++d, ++u, ++u0) {
		*d = *u - *u0;
		for(size_t k=0; k<3; k++) {
			if(!pbc[k]) continue;
			if(LengthSquared(*d + simCell.column(k)) < LengthSquared(*d))
				*d += simCell.column(k);
			else if(LengthSquared(*d - simCell.column(k)) < LengthSquared(*d))
				*d -= simCell.column(k);
		}
		if(referenceShown()) {
			*d = -(*d);
			*ou++ = *u0;
		}
	}

	return EvaluationStatus();
}

/******************************************************************************
* Returns the path to the reference configuration file.
******************************************************************************/
QString CalcDisplacementsModifier::inputFile() const
{
	AtomsImportObject* importObj = dynamic_object_cast<AtomsImportObject>(referenceConfiguration());
	if(importObj)
		return importObj->inputFile();
	else
		return QString();
}

/******************************************************************************
* Displays the file selection dialog and lets the user select the file with the reference configuration.
******************************************************************************/
void CalcDisplacementsModifier::showSelectionDialog(QWidget* parent)
{
	AtomsImportObject* importObj = dynamic_object_cast<AtomsImportObject>(referenceConfiguration());
	if(importObj)
		importObj->showSelectionDialog(parent);
}


IMPLEMENT_PLUGIN_CLASS(CalcDisplacementsModifierEditor, AtomsObjectModifierEditorBase)

/******************************************************************************
* Sets up the UI widgets of the editor.
******************************************************************************/
void CalcDisplacementsModifierEditor::createUI(const RolloutInsertionParameters& rolloutParams)
{
	// Create a rollout.
	QWidget* rollout = createRollout(tr("Calculate Displacements"), rolloutParams, "atomviz.modifiers.calculate_displacements");

    // Create the rollout contents.
	QVBoxLayout* layout = new QVBoxLayout(rollout);
	layout->setContentsMargins(4,4,4,4);
#ifndef Q_WS_MAC
	layout->setSpacing(0);
#endif

	BooleanPropertyUI* showReferenceUI = new BooleanPropertyUI(this, PROPERTY_FIELD_DESCRIPTOR(CalcDisplacementsModifier, _referenceShown));
	layout->addWidget(showReferenceUI->checkBox());

	FilenamePropertyUI* inputFilePUI = new FilenamePropertyUI(this, "inputFile", SLOT(showSelectionDialog(QWidget*)));
	layout->addWidget(inputFilePUI->selectorWidget());
	inputFilePUI->selectorWidget()->setSizePolicy(QSizePolicy(QSizePolicy::Ignored, QSizePolicy::Fixed));

	// Status label.
	layout->addSpacing(12);
	layout->addWidget(statusLabel());

	// Open a sub-editor for the displacement channel protoype.
	new SubObjectParameterUI(this, PROPERTY_FIELD_DESCRIPTOR(CalcDisplacementsModifier, _displacementChannelPrototype), rolloutParams.after(rollout));

	// Open a sub-editor for the internal AtomsImportObject.
	subObjectUI = new SubObjectParameterUI(this, PROPERTY_FIELD_DESCRIPTOR(CalcDisplacementsModifier, _referenceObject), RolloutInsertionParameters().animate());
	subObjectUI->setEnabled(false);

	QPushButton* btn = new QPushButton(tr("Show properties of reference configuration"));
	layout->addSpacing(6);
	layout->addWidget(btn);
	btn->setCheckable(true);
	connect(btn, SIGNAL(toggled(bool)), subObjectUI, SLOT(setEnabled(bool)));
}

};	// End of namespace AtomViz
