/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999, 2000, 2001  Pan Development Team <pan@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>

#include <ctype.h>
#include <stdlib.h>
#include <string.h>

#include <glib.h>

#include <pan/base/acache.h>
#include <pan/base/article.h>
#include <pan/base/debug.h>
#include <pan/base/gnksa.h>
#include <pan/base/group.h>
#include <pan/base/pan-glib-extensions.h>
#include <pan/base/pan-i18n.h>
#include <pan/base/run.h>
#include <pan/base/server.h>
#include <pan/base/group.h>
#include <pan/base/util-mime.h>

const gchar * default_incoming_name_addr;
const gchar * default_incoming_name_real;

static void fire_articles_changed (Group*, Article**, gint article_qty, ArticleChangeType);

/***
****
****  LIFE CYCLE
****
***/

void
article_destructor (Article * a)
{
        g_return_if_fail (a != NULL);

	/* clean up article parts */
	if (a->headers != NULL) {
		g_hash_table_destroy (a->headers);
		a->headers = NULL;
	}
	if (a->threads != NULL) {
		g_slist_free (a->threads);
		a->threads = NULL;
	}
	a->date = 0;
	a->number = 0;
	a->linecount = 0;
	a->message_id = NULL;
	a->xref = NULL;
	a->references = NULL;
	a->subject = NULL;
	a->author_addr = NULL;
	a->author_real = NULL;
	a->group = NULL;
}

static void
article_constructor (Article * article, Group * group)
{
	g_return_if_fail (article!=NULL);
	g_return_if_fail (group!=NULL);

	/* init this class */
	article->group = group;
	article->is_new = FALSE;
	article->self_passes_filter = FALSE;
	article->tree_passes_filter = FALSE;
	article->part = (gint16)0;
	article->parts = (gint16)0;
	article->score = (gint16)0;
	article->crosspost_qty = (gint8)0;
	article->linecount = (guint16)0;
	article->state = (guint16)0;
	article->unread_children = (guint16)0;
	article->new_children = (guint16)0;
	article->date = (time_t)0;
	article->number = (gulong)0;
	article->subject = NULL;
	article->author_real = NULL;
	article->author_addr = NULL;
	article->message_id = NULL;
	article->xref = NULL;
	article->references = NULL;
	article->parent = NULL;
	article->threads = NULL;
	article->headers = NULL;
}

Article*
article_new (Group * group)
{
	Article * article;

	g_return_val_if_fail (group!=NULL, NULL);

	article = group_alloc_new_article (group);
	article_constructor (article, group);
	return article;
}


static void
article_dup_ghfunc (gpointer key, gpointer val, gpointer user_data)
{
	Article * article = ARTICLE(user_data);
	article_init_header (article, key, val, DO_CHUNK_SHARE);
}
Article*
article_dup (Group * group, const Article * article)
{
	Article * retval;

	g_return_val_if_fail (article_is_valid(article), NULL);
	g_return_val_if_fail (group!=NULL, NULL);

#define chunk_if_exists(A, share) \
	is_nonempty_string (A) ? group_chunk_string (group, A, share) : NULL

	retval                      = article_new (group);
	retval->self_passes_filter  = FALSE;
	retval->tree_passes_filter  = FALSE;
	retval->unread_children     = 0;
	retval->new_children        = 0;
	retval->group               = group;
	retval->parts               = article->parts;
	retval->part                = article->part;
	retval->linecount           = article->linecount;
	retval->crosspost_qty       = article->crosspost_qty;
	retval->state               = article->state;
	retval->date                = article->date;
	retval->number              = article->number;
	retval->xref                = chunk_if_exists (article->xref, FALSE);
	retval->subject             = chunk_if_exists (article->subject, TRUE);
	retval->author_real         = chunk_if_exists (article->author_real, TRUE);
	retval->author_addr         = chunk_if_exists (article->author_addr, TRUE);
	retval->message_id          = chunk_if_exists (article->message_id, FALSE);
	retval->references          = chunk_if_exists (article->references, TRUE);
	retval->parent              = NULL;
	retval->threads             = NULL;
	retval->headers             = NULL;

	if (article->headers != NULL)
		g_hash_table_foreach (article->headers, article_dup_ghfunc, retval);

	return retval;
}

/***
****
***/

static void
articles_set_dirty (Article ** articles, gint qty)
{
	Group * group;
	debug_enter ("articles_set_dirty");

	/* sanity check */
	g_return_if_fail (qty >= 1);
	g_return_if_fail (articles_are_valid ((const Article **)articles, qty));

	/* mark the articles' group as read */
	group = articles[0]->group;
	group_set_articles_dirty (group);
	fire_articles_changed (group, articles, qty, ARTICLE_CHANGED_DIRTY);

	debug_exit ("articles_set_dirty");
}


gboolean
article_is_new (const Article * a)
{
	g_return_val_if_fail (article_is_valid(a), FALSE);

	return a->is_new && !article_is_read(a);
}

/***
****  NEW / OLD
***/

static void
articles_set_new_impl (Article ** articles, int article_qty, gboolean is_new, gboolean fire)
{
	int i;
	GPtrArray * changed;
	debug_enter ("articles_set_new_impl");

	/* sanity check */
	g_return_if_fail (article_qty > 0);
	g_return_if_fail (articles_are_valid ((const Article **)articles, article_qty));

	/* mark each article's newness field */
	changed = g_ptr_array_new ();
	for (i=0; i<article_qty; ++i) {
		Article * a = articles[i];
		if (!!is_new != !!a->is_new) {
			a->is_new = is_new;
			g_ptr_array_add (changed, a);
		}
	}

	/* update parent counts of changed articles */
	for (i=0; i<changed->len; ++i) {
		Article * a = ARTICLE(g_ptr_array_index(changed, i));
		for (a=a->parent; a!=NULL; a=a->parent)
			a->new_children += is_new ? 1 : -1;
	}

	/* fire notification of changed articles */
	if (fire && changed->len!=0)
		fire_articles_changed (articles[0]->group, (Article**)changed->pdata, changed->len, ARTICLE_CHANGED_NEW);

	/* cleanup */
	g_ptr_array_free (changed, TRUE);
	debug_exit ("articles_set_new_impl");
}

void
articles_set_new (Article ** articles, int article_qty, gboolean is_new)
{
	articles_set_new_impl (articles, article_qty, is_new, TRUE);
}

/***
****
****  READ / UNREAD
****
***/

gboolean
article_is_read (const Article * a)
{
	g_return_val_if_fail (a!=NULL, FALSE);
	g_return_val_if_fail (a->group!=NULL, FALSE);

	return newsrc_is_article_read (group_get_newsrc(a->group), a->number);
}

static void
articles_set_read_numbers_ghfunc (gpointer key, gpointer value, gpointer user_data)
{
	gint i;
	gint changed_qty;
	Group * g = GROUP(key);
	GArray * numbers = (GArray*) value;
	gboolean read = user_data!=NULL;
	Newsrc * newsrc = group_get_newsrc (g);

	/**
	***  Mark each number as read/unread
	**/

	for (i=changed_qty=0; i<numbers->len; ++i)
	{
		const gulong number = g_array_index (numbers, gulong, i);

		if (read != newsrc_mark_article (newsrc, number, read))
			++changed_qty;
	}

	/**
	***  If any changed, update the group's count
	**/

	if (changed_qty != 0)
	{
		group_inc_article_read_qty (g, read ? changed_qty : -changed_qty);
	}

	g_array_free (numbers, TRUE);
}

static void
articles_set_read_articles_ghfunc (gpointer key, gpointer value, gpointer user_data)
{
	guint i;
	Group * g = GROUP(key);
	GArray * articles = (GArray*) value;
	gboolean read = user_data!=NULL;
	Newsrc * newsrc = group_get_newsrc (g);
	GPtrArray * changed = g_ptr_array_new ();

	/**
	***  Walk through the articles & make a list of those to be changed
	**/

	for (i=0; i!=articles->len; ++i)
	{
		Article * a = ARTICLE(g_array_index(articles,Article*,i));

		if (!!read != !!newsrc_mark_article (newsrc, a->number, read))
			g_ptr_array_add (changed, a);
	}

	/**
	***  If anything to be changed:
	***  (1) update parent counts
	***  (2) update group's counts
	***  (3) fire an event
	**/

	if (changed->len != 0)
	{
		/* update parent counts */
		for (i=0; i!=changed->len; ++i)
		{
			Article * a = ARTICLE(g_ptr_array_index(changed,i));
			for (a=a->parent; a!=NULL; a=a->parent)
				a->unread_children += read ? -1 : 1;
		}
		if (read)
			articles_set_new_impl ((Article**)changed->pdata, changed->len, FALSE, FALSE);

		/* update the group's count */
		group_inc_article_read_qty (g, read ? changed->len : -changed->len);

		/* fire an event */
		fire_articles_changed (g, (Article**)changed->pdata, changed->len, ARTICLE_CHANGED_READ);
	}


	/* cleanup */
	g_array_free (articles, TRUE);
}

/**
 * The hashtable maps from a Group* to a GArray*, which is returned.
 * If no matching GArray* exists, it's created and added to the hash before returning.
 */
static GArray*
get_group_array_from_hash (GHashTable * hash, Group * group, gint size)
{
	GArray * arr = (GArray*) g_hash_table_lookup (hash, group);
	if (arr == NULL)
		g_hash_table_insert (hash, group, arr = g_array_new (FALSE, FALSE, size));
	return arr;
}

/**
 * Builds a GHashTable of Group* -> GArray*, where the GArray is filled with message numbers.
 */
static void
mark_read_xreffunc (Group * g, gulong number, gpointer user_data)
{
	GHashTable ** group_to_numbers = (GHashTable**) user_data;

	if (group_is_subscribed(g))
	{
		GArray * group_numbers;

		if (*group_to_numbers == NULL)
			*group_to_numbers = g_hash_table_new (g_direct_hash, g_direct_equal);

		group_numbers = get_group_array_from_hash (*group_to_numbers, g, sizeof(gulong));
		g_array_append_val (group_numbers, number);
	}
}

void
articles_set_read (Article ** articles, int article_qty, gboolean read)
{
	int i;
	GHashTable * group_to_articles;
	GHashTable * group_to_numbers = NULL;
	debug_enter ("articles_set_read");

	/* sanity check */
	g_return_if_fail (article_qty > 0);
	g_return_if_fail (articles_are_valid ((const Article **)articles, article_qty));

	/* get a big hash, keyed by Group*,  of all the articles that we're
	   marking read.  We have two hashes -- one for the articles passed
	   in, and one for article numbers of the xposts.  */
	group_to_articles = g_hash_table_new (g_direct_hash, g_direct_equal);
	for (i=0; i<article_qty; ++i)
	{
		GArray * arr;
		Article * a = articles[i];

		/* first add this article to the group's array */
	       	arr = get_group_array_from_hash (group_to_articles, a->group, sizeof(Article*));
		g_array_append_val (arr, a);

		/* secondly, add the xrefs to their group's arrays */
		article_xref_foreach (a, mark_read_xreffunc, &group_to_numbers,
		                      SERVER_GROUPS_SUBSCRIBED, TRUE);

		/* lastly, if it's a multipart, process the child parts too */
		if (a->part==1 && a->parts>1 && a->threads!=NULL)
		{
			GSList * l;
			for (l=a->threads; l!=NULL; l=l->next)
			{
				Article * child = ARTICLE(l->data);
				if (child->part>a->part && child->parts==a->parts)
				{
					/* add this article to the group's array */
					g_array_append_val (arr, child);

					/* add the xrefs to their group's arrays */
					article_xref_foreach (child,
						mark_read_xreffunc,
						&group_to_numbers,
						SERVER_GROUPS_SUBSCRIBED, TRUE);
				}
			}
		}
	}

	/* mark the passed-in articles as read/unread */
	g_hash_table_foreach (group_to_articles,
	                      articles_set_read_articles_ghfunc,
	                      GINT_TO_POINTER(read));
	g_hash_table_destroy (group_to_articles);


	/* mark their crosspost counterparts, if any, as read/unread */
	if (group_to_numbers != NULL) {
		g_hash_table_foreach (group_to_numbers,
		                      articles_set_read_numbers_ghfunc,
		                      GINT_TO_POINTER(read));
		g_hash_table_destroy (group_to_numbers);
	}

	debug_exit ("articles_set_read");
}


/**
*** PUBLIC MUTATORS
**/

void
articles_add_flag (Article  ** articles,
                   gint        qty,
                   gushort     flag)
{
	gint i;
	gint changed_qty;
	Article ** changed;

	/* sanity clause */
	g_return_if_fail (qty >= 1);
	g_return_if_fail (articles_are_valid ((const Article **)articles, qty));

	/* mark 'em */
	changed = g_malloc (sizeof(Article*) * qty);
	for (changed_qty=i=0; i<qty; ++i) {
		if (!(articles[i]->state & flag)) {
			articles[i]->state |= flag;
			changed[changed_qty++] = articles[i];
		}
	}

	if (changed_qty > 0)
		articles_set_dirty (changed, changed_qty);

	g_free (changed);
}


void
articles_remove_flag (Article         ** articles,
                      gint               qty,
                      gushort            flag)
{
	gint i;
	gint changed_qty;
	Article ** changed;

	/* sanity clause */
	g_return_if_fail (qty >= 1);
	g_return_if_fail (articles_are_valid ((const Article **)articles, qty));

	/* mark 'em */
       	changed = g_malloc (sizeof(Article*) * qty);
	for (changed_qty=i=0; i<qty; ++i) {
		if (articles[i]->state & flag) {
			articles[i]->state &= ~flag;
			changed[changed_qty++] = articles[i];
		}
	}

	if (changed_qty > 0)
		articles_set_dirty (changed, changed_qty);

	g_free (changed);
}



/***
****
****  THREADS
****
***/

gchar*
article_get_thread_message_id (const Article* article)
{
	char * pch;
	const gchar * refs;

	/* sanity clause */
	g_return_val_if_fail (article_is_valid(article), NULL);

	/* go up as high as we can from our own threading, because sometimes
	 * the References: header is broken by other newreaders. */
	while (article->parent != NULL)
		article = article->parent;

	refs = article->references;
	if (is_nonempty_string(refs))
	{
		pch = get_next_token_str (refs, ' ', NULL);
	}
	else /* top of the thread */
	{
		const gchar * msg_id = article_get_message_id (article);
		g_return_val_if_fail (is_nonempty_string(msg_id), NULL);
		pch = g_strdup (msg_id);
	}

	return pch;
}

static void
article_get_entire_thread_impl (Article* top, GPtrArray* setme)
{
	GSList *l;
	g_ptr_array_add (setme, top);
	for (l=top->threads; l!=NULL; l=l->next)
		article_get_entire_thread_impl (ARTICLE(l->data), setme);
}

static void
article_get_thread_impl (Article * article,
                         GPtrArray * setme,
                         gboolean children_only)
{
	Article* top = article;

	/* sanity clause */
	g_return_if_fail (article_is_valid(article));
	g_return_if_fail (setme!=NULL);

	if (top != NULL)
	{
		if (!children_only)
			while (top->parent != NULL)
				top = top->parent;

		pan_g_ptr_array_reserve (setme, 128);
		article_get_entire_thread_impl (top, setme);
	}
}

void
article_get_subthread (Article* article, GPtrArray* setme)
{
	article_get_thread_impl (article, setme, TRUE);
}

void
article_get_entire_thread (Article* article, GPtrArray* setme)
{
	article_get_thread_impl (article, setme, FALSE);
}

void
article_get_references (Article* article, GPtrArray* setme)
{
	g_return_if_fail (article_is_valid(article));
	g_return_if_fail (setme!=NULL);

	pan_g_ptr_array_reserve (setme, 128);
	while (article != NULL) {
		g_ptr_array_add (setme, article);
		article = article->parent;
	}
}

GPtrArray*
article_get_unique_threads (const GPtrArray    * articles,
                            ThreadGet            thread_get)
{
	int i;
	GPtrArray * thread;
	GPtrArray * retval;
	GHashTable * all;
	debug_enter ("articlelist_get_unique_threads");

	/* sanity clause */
	retval = g_ptr_array_new ();
	g_return_val_if_fail (articles_are_valid ((const Article **)articles->pdata, articles->len), retval);

	/* get the thread for each article passed in */
	thread = g_ptr_array_new ();
	pan_g_ptr_array_reserve (retval, 128);
	pan_g_ptr_array_reserve (thread, 128);
	all = g_hash_table_new (g_str_hash, g_str_equal);
	for (i=0; i<articles->len; ++i)
	{
		int j;
		Article * a = ARTICLE(g_ptr_array_index(articles,i));

		/* if we already have the article, then we have its thread, so skip */
		if (g_hash_table_lookup (all, article_get_message_id(a)) != NULL)
			continue;

		/* get the thread associated with sel... */
		g_ptr_array_set_size (thread, 0);
		if (thread_get == GET_SUBTHREAD)
			article_get_subthread (a, thread);
		else
			article_get_entire_thread (a, thread);

		/* add the articles into "all" hash */
		for (j=0; j<thread->len; ++j)
		{
			Article * b = ARTICLE(g_ptr_array_index(thread,j));
			const gchar * message_id = article_get_message_id (b);

			if (g_hash_table_lookup (all, message_id) == NULL)
			{
				g_hash_table_insert (all, (gpointer)article_get_message_id(b), (gpointer)b);
				g_ptr_array_add (retval, b);
			}
		}
	}

	/* cleanup */
	g_hash_table_destroy (all);
	g_ptr_array_free (thread, TRUE);
	debug_exit ("article_get_unique_threads");
	return retval;
}


/***
****
****  OTHER HEADERS
****
***/

void
article_remove_header (Article * a, const gchar * key)
{
	g_return_if_fail (a!=NULL);
	g_return_if_fail (is_nonempty_string(key));
		 
	if (!strcmp(key, HEADER_MESSAGE_ID))
		a->message_id = NULL;
	else if (!strcmp(key, HEADER_XREF))
		a->xref = NULL;
	else if (!strcmp(key, HEADER_SUBJECT))
		a->subject = NULL;
	else if (!strcmp(key, HEADER_REFERENCES))
		a->references = NULL;
	else if (!strcmp(key, HEADER_FROM))
	{
		a->author_addr = NULL;
		a->author_real = NULL;
	}
	else if (a->headers!=NULL)
	{
		g_hash_table_remove (a->headers, key);
		if (!g_hash_table_size(a->headers))
		{
			g_hash_table_destroy (a->headers);
			a->headers = NULL;
		}
	}
}

const gchar*
article_get_extra_header (const Article * a, const gchar * key)
{
	const gchar * retval = NULL;

	/* sanity clause */
	g_return_val_if_fail (article_is_valid(a), NULL);
	g_return_val_if_fail (is_nonempty_string(key), NULL);

	if (a->headers!=NULL)
		retval = (const gchar*) g_hash_table_lookup (a->headers, key);

	return retval;
}

gboolean
article_header_is_internal (const gchar * key)
{
	return is_nonempty_string(key) && !g_strncasecmp(key,"Pan-",4);
}

gboolean
article_header_is_extra (const gchar * key)
{
	/* sanity check */
	if (!is_nonempty_string(key)) return FALSE;

	/* pan internals aren't user-specified headers */
	if (article_header_is_internal(key)) return FALSE;

	/* other headers that are handled explicitly elsewhere */
	if (!strcmp(key,HEADER_FOLLOWUP_TO)) return FALSE;
	if (!strcmp(key,HEADER_NEWSGROUPS)) return FALSE;
	if (!strcmp(key,HEADER_ORGANIZATION)) return FALSE;
	if (!strcmp(key,HEADER_REPLY_TO)) return FALSE;
	if (!strcmp(key,HEADER_DATE)) return FALSE;

	return TRUE;
}

static void
ensure_extra_headers_exists (Article * a)
{
	if (a->headers == NULL)
		a->headers = g_hash_table_new (g_str_hash, g_str_equal);
}

void
article_xref_foreach (const Article       * a,
                      ArticleXRefFunc       func,
                      gpointer              user_data,
		      ServerGroupsType      set,
                      gboolean              skip_group_a)
{
	/* sanity clause */
	g_return_if_fail (article_is_valid(a));

	/* get the xref header */
	if (is_nonempty_string(a->xref))
	{
		Run run;
		const gchar * march;
		Server * const server = a->group->server;

		/* skip the servername; we've got the server already */
		skip_next_token (a->xref, ' ', &march);

		/* walk through the xrefs, of format "group1:number group2:number" */
		while (get_next_token_run (march, ' ', &march, &run.str, &run.len))
		{
			const gchar * delimit = run_strchr (&run, ':');
			if (delimit != NULL)
			{
				const gulong number = strtoul (delimit+1, NULL, 10);
				Group * g = NULL;
				gchar * group_name;

				pan_strndup_alloca (group_name, run.str, delimit-run.str);

				if (server_has_group_in_type (server, group_name, set))
					g = server_get_named_group (server, group_name);

				if (g!=NULL && (a->group!=g || !skip_group_a))
					(*func)(g, number, user_data);
			}
		}
	}
}

static void
article_set_header_impl (Article         * a,
                         const gchar     * key,
                         const gchar     * val,
                         HeaderAction      action,
                         gboolean          init)
{
	g_return_if_fail (a!=NULL);
	g_return_if_fail (is_nonempty_string(key));

	/* chunk if necessary */
	if (action==DO_CHUNK || action==DO_CHUNK_SHARE)
		val = group_chunk_string (a->group, val, action==DO_CHUNK_SHARE);

	/* if we're possibly erasing a header, remove the old value */
	if (!init && !is_nonempty_string (val))
		article_remove_header (a, key);

	/* add the new header */
	if (is_nonempty_string (val))
	{
		if      (!strcmp (key, HEADER_MESSAGE_ID)) a->message_id = val;
		else if (!strcmp (key, HEADER_SUBJECT)) a->subject = val;
		else if (!strcmp (key, HEADER_REFERENCES)) a->references = val;
		else if (!strcmp (key, HEADER_XREF)) a->xref = val;
		else if (!strcmp (key, HEADER_FROM))
		{
			if (init)
				article_init_author_from_header (a, val);
			else
				article_set_author_from_header (a, val);
		}
		else
		{
			ensure_extra_headers_exists (a);
			key = group_chunk_string (a->group, key, TRUE);
			g_hash_table_insert (a->headers,
			                     (gpointer)key,
			                     (gpointer)val);
			if (!init)
				articles_set_dirty (&a, 1);
		}
	}
}

void
article_init_header (Article         * a,
                     const gchar     * key,
                     const gchar     * val,
                     HeaderAction      action)
{
	article_set_header_impl (a, key, val, action, TRUE);
}

void
article_set_header (Article         * a,
                    const gchar     * key,
                    const gchar     * val,
                    HeaderAction      action)
{
	article_set_header_impl (a, key, val, action, FALSE);
}

static void
article_get_all_headers_ghfunc (gpointer key, gpointer value, gpointer user_data)
{
	GPtrArray * a = (GPtrArray*)user_data;
	g_ptr_array_add (a, key);
	g_ptr_array_add (a, value);
}

GPtrArray*
article_get_all_headers (const Article * a)
{
	GPtrArray * retval = g_ptr_array_new ();

	/* sanity clause */
	g_return_val_if_fail (a!=NULL, retval);

	/* add the headers */
	if (a->headers != NULL)
	{
		pan_g_ptr_array_reserve (retval, g_hash_table_size(a->headers)*2);
		g_hash_table_foreach (a->headers, article_get_all_headers_ghfunc, retval);
	}

	return retval;
}

gboolean
article_has_attachment (const Article *a)
{
	g_return_val_if_fail (article_is_valid(a), FALSE);

	return article_get_extra_header(a,PAN_ATTACH_FILE) != NULL;
}

/***
****
****  BODY
****
***/

gboolean
article_has_body (const Article * a)
{
	gboolean retval = FALSE;

	g_return_val_if_fail (article_is_valid(a), FALSE);

	/* try to get it from acache... */
	if (!retval && is_nonempty_string(a->message_id))
		retval = acache_has_message (a->message_id);

	/* maybe the article's in a folder so the body
	 * is hidden as a header... */
	if (!retval)
		if (article_get_extra_header (a, PAN_BODY) != NULL)
			retval = TRUE;

	return retval;
}

gchar*
article_get_author_str (const Article * a, gchar * buf, gint bufsize)
{
	const gboolean have_addr = a!=NULL && is_nonempty_string (a->author_addr);
	const gboolean have_name = a!=NULL && is_nonempty_string (a->author_real);

	*buf = '\0';

	if (have_addr && have_name) {
		g_snprintf (buf, bufsize, "\"%s\" <%s>", a->author_real, a->author_addr);
	}
	else if (have_addr) {
		strncpy (buf, a->author_addr, bufsize);
		buf[bufsize-1] = '\0';
	}
	else if (have_name) {
		strncpy (buf, a->author_real, bufsize);
		buf[bufsize-1] = '\0';
	}

	g_strstrip (buf);
	return buf;
}

void
article_get_short_author_str (const Article * a, gchar * buf, gint len)
{
	g_return_if_fail (article_is_valid(a));
	g_return_if_fail (buf!=NULL);
	g_return_if_fail (len>0);

	if (is_nonempty_string (a->author_real) && strcmp(a->author_real,default_incoming_name_real))
	{
		strncpy (buf, a->author_real, len);
		buf[len-1] = '\0';
	}
	else if (is_nonempty_string (a->author_addr) && strcmp(a->author_addr,default_incoming_name_addr))
	{
		const gchar * author = a->author_addr;
		const gchar * pch = strchr (author, '@');
		if (pch == NULL)
			pch = author + strlen(author);
		strncpy (buf, author, MIN(len,pch-author));
		buf[len-1] = '\0';
	}
	else
	{
		strncpy (buf, default_incoming_name_real, len);
		buf[len-1] = '\0';
	}
}


gchar*
article_get_message (const Article * a)
{
	gchar * retval = NULL;

	/* sanity clause */
	g_return_val_if_fail (article_is_valid(a), g_strdup(""));

	if (retval==NULL)
	{
		const gchar * message_id = article_get_message_id (a);
		if (acache_has_message (message_id))
			retval = acache_get_message (message_id);
	}

	if (retval==NULL)
	{
		GString * s = g_string_new (NULL);
		gchar * pch;

		pch = article_get_headers (a);
		g_string_append (s, pch);
		g_free (pch);

		g_string_append (s, "\n");

		pch = article_get_body (a);
		g_string_append (s, pch);
		g_free (pch);

		retval = s->str;
		g_string_free (s, FALSE);
	}

	return retval;
}

gchar*
article_get_headers (const Article * a)
{
	gchar * retval = NULL;

	g_return_val_if_fail (article_is_valid(a), g_strdup(""));

	/* do we have it in acache? */
	if (retval==NULL)
	{
		const gchar * message_id = article_get_message_id (a);
		if (acache_has_message (message_id))
		{
			gchar * s = acache_get_message (message_id);
			gchar * pch = pan_strstr (s, "\n\n");
			if (pch != NULL) {
				*pch = '\0';
				retval = s;
			}
		}
	}

	/* can we build it from a? */
	if (retval==NULL)
	{
		gchar * tmp;
		const gchar * c_tmp;
		GString * s = g_string_new (NULL);

		/* subject */
		c_tmp = article_get_subject (a);
		if (is_nonempty_string (c_tmp))
			g_string_sprintfa (s, "%s: %s\n", HEADER_SUBJECT, c_tmp);

		/* author */
		if (1) {
			gchar author[512] = { '\0' };
	       		article_get_author_str (a, author, sizeof(author));
			if (is_nonempty_string (author))
				g_string_sprintfa (s, "%s: %s\n", HEADER_FROM, author);
		}

		/* date */
		if (a->date != 0) {
			tmp = rfc822_date_generate (a->date);
			g_string_sprintfa (s, "%s: %s\n", HEADER_DATE, tmp);
			g_free (tmp);
		}

		/* message-id */
		c_tmp = article_get_message_id (a);
		if (is_nonempty_string (c_tmp))
			g_string_sprintfa (s, "%s: %s\n", HEADER_MESSAGE_ID, c_tmp);

		/* references */
		c_tmp = a->references;
		if (is_nonempty_string (c_tmp))
			g_string_sprintfa (s, "%s: %s\n", HEADER_REFERENCES, c_tmp);

		/* extra headers */
		if (1) {
			gint i;
			GPtrArray * z = article_get_all_headers (a);
			for (i=0; i<z->len; i+=2) {
				const char* key = (const char*) g_ptr_array_index (z, i);
				const char* val = (const char*) g_ptr_array_index (z, i+1);
				if (strncmp (key, PAN_BODY, strlen(PAN_BODY)) != 0)
					g_string_sprintfa (s, "%s: %s\n",  key, val);
			}
			g_ptr_array_free (z, TRUE);
		}

		retval = s->str;
		g_string_free (s, FALSE);
	}

	return retval;
}

gchar*
article_get_body (const Article * a)
{
	gchar * retval = NULL;

	g_return_val_if_fail (article_is_valid(a), g_strdup(""));

	/* see if we've got the body hidden in a header... */
	if (retval==NULL)
	{
		const gchar * body = article_get_extra_header (a, PAN_BODY);
		if (body != NULL)
			retval = g_strdup (body);
	}

	/* see if we've got it in acache... */
	if (retval==NULL)
	{
		const gchar * message_id = article_get_message_id (a);
		if (acache_has_message (message_id))
		{
			gchar * s = acache_get_message (message_id);
			gboolean is_html = FALSE;
			GMimeMessage * m = pan_g_mime_parser_construct_message (s);
			retval = g_mime_message_get_body (m, TRUE, &is_html);
			g_mime_object_unref (GMIME_OBJECT(m));
			g_free (s);
		}
	}

	return retval;
}

gboolean
article_get_header_run_hash_from_text (const gchar    * article_text,
                                       GHashTable    ** makeme_hash,
                                       gpointer       * allocme_buf)
{
	guint line_qty;
	guint line_len;
	const gchar * line; 
	const gchar * march;
	GHashTable * hash;
	Run * buf;
	guint buf_qty;
	guint buf_size;

	/* sanity clause */
	g_return_val_if_fail (article_text, FALSE);
	g_return_val_if_fail (makeme_hash!=NULL, FALSE);
	g_return_val_if_fail (allocme_buf!=NULL, FALSE);

	/* count number of lines */
	line = NULL;
	line_qty = 0;
	for (march=article_text; get_next_token_run(march,'\n',&march,&line,&line_len) && line_len!=0; )
		++line_qty;

	/* alloc a buffer of enough Run structs */
	buf_size = line_qty * 2u;
	buf_qty = 0u;
	buf = g_malloc (sizeof(Run) * buf_size);

	/* alloc the hashtable */
	hash = g_hash_table_new (run_hash, run_equal);

	/* populate the hashtable */
	march = article_text;
	while (get_next_token_run (march, '\n', &march, &line, &line_len))
	{
		guint key_len = 0;
		const gchar * key = NULL;

		if (line_len == 0) /* \n\n -- end of headers */
			break;

		if (get_next_token_run (line, ':', NULL, &key, &key_len))
		{
			gint val_len = 0;
			const gchar * val = NULL;
			Run * key_run = buf + buf_qty++;
			Run * val_run = buf + buf_qty++;

			val = key + key_len + 1; /* skip past ':' */
			val_len = 0;
			while (val[val_len]!='\n' && val[val_len]!='\0')
				++val_len;

			g_hash_table_insert (hash, 
			                     run_strstrip (run_init (key_run, key, key_len)),
			                     run_strstrip (run_init (val_run, val, val_len)));
		}
	}

	*makeme_hash = hash;
	*allocme_buf = buf;
	return TRUE;
}


static void
article_set_author_from_header_impl (Article       * a,
                                     const gchar   * header_from,
                                     gboolean        init)
{
	gchar addr_buf[512];
	gchar real_buf[512];
	gchar * addr = addr_buf;
	gchar * real = real_buf;
	gchar * pch;

	g_return_if_fail (a!=NULL);
	g_return_if_fail (a->group!=NULL);
	g_return_if_fail (is_nonempty_string(header_from));

	/* note that strict is FALSE here because we can't control
	 * how other people have posted to usenet */
	gnksa_do_check_from (header_from,
	                     addr_buf, sizeof(addr_buf),
	                     real_buf, sizeof(real_buf),
	                     FALSE);
	if (pan_header_is_8bit_encoded (addr_buf))
		addr = g_mime_utils_8bit_header_decode ((const guchar *)addr_buf);
	if (pan_header_is_8bit_encoded (real_buf))
		real = g_mime_utils_8bit_header_decode ((const guchar *)real_buf);

	/* use the real mail address, or fill in a default */
	pch = addr;
	if (!is_nonempty_string (pch))
		pch = (gchar*)default_incoming_name_addr;
	a->author_addr = group_chunk_string (a->group, pch, TRUE);

	/* use the real name, or fill in a default. */
	if (is_nonempty_string(real)) {
		gnksa_strip_realname (real);
		a->author_real = group_chunk_string (a->group, real, TRUE);
	}
	else if (addr!=NULL && ((pch=strchr(addr,'@'))!=NULL)) {
		gchar * tmp;
		pan_strndup_alloca (tmp, addr, pch-addr);
		a->author_real = group_chunk_string (a->group, tmp, TRUE);
	}
	else {
		const gchar * cpch = (gchar*)default_incoming_name_real;
		a->author_real = group_chunk_string (a->group, cpch, TRUE);
	}

	if (!init)
		articles_set_dirty (&a, 1);

	/* cleanup */
	if (addr!=addr_buf)
		g_free (addr);
	if (real!=real_buf)
		g_free (real);
}

void
article_init_author_from_header (Article         * a,
                                 const gchar     * header_from)
{
	article_set_author_from_header_impl (a, header_from, TRUE);
}
void
article_set_author_from_header (Article         * a,
                                const gchar     * header_from)
{
	article_set_author_from_header_impl (a, header_from, FALSE);
}


static void
set_from_raw_init_header_gmhfunc (const gchar * name,
                                  const gchar * value,
                                  gpointer data)
{
	Article * a = ARTICLE(data);

	g_return_if_fail (a!=NULL);
	g_return_if_fail (is_nonempty_string(name));
	g_return_if_fail (is_nonempty_string(value));

	article_init_header (a, name, value, DO_CHUNK);
}

void
article_set_from_raw_message   (Article         * a,
                                const gchar     * text)
{
	guint16 lines;
	GMimeMessage * msg;
	gchar * body;
	const gchar * pch;
	gboolean foo;
	debug_enter ("article_set_from_raw_message");

	/* get the message */
	g_return_if_fail (a != NULL);
	g_return_if_fail (is_nonempty_string(text));
	msg = pan_g_mime_parser_construct_message (text);
	g_return_if_fail (msg != NULL);

	/* body */
	body = g_mime_message_get_body (msg, TRUE, &foo);
	article_init_header (a, PAN_BODY, body, DO_CHUNK);
	for (lines=0, pch=body; is_nonempty_string(pch); ++pch)
		if (*pch=='\n')
			++lines;
	a->linecount = lines;
	g_free (body);

	/* headers */
	if (msg->header != NULL) {
		a->date = msg->header->date;
		if (msg->header->from != NULL)
			article_init_header (a, HEADER_FROM, msg->header->from, DO_CHUNK_SHARE);
		if (msg->header->reply_to != NULL)
			article_init_header (a, HEADER_REPLY_TO, msg->header->reply_to, DO_CHUNK_SHARE);
		if (msg->header->subject != NULL)
			article_init_header (a, HEADER_SUBJECT, msg->header->subject, DO_CHUNK_SHARE);
		if (msg->header->message_id != NULL)
			article_init_header (a, HEADER_MESSAGE_ID, msg->header->message_id, DO_CHUNK);
		g_mime_header_foreach (msg->header->headers, set_from_raw_init_header_gmhfunc, a);
	}

	/* fallback if no message-id */
	if (!is_nonempty_string(a->message_id)) {
		gchar * id;
		g_warning (_("Couldn't parse a Message-ID from the raw message!"));
		id = gnksa_generate_message_id_from_email_addr (a->author_addr);
		article_init_header (a, HEADER_MESSAGE_ID, id, DO_CHUNK);
		g_free (id);
		article_init_header (a, HEADER_SUBJECT, _("Unparseable Subject"), DO_CHUNK_SHARE);
	}

	/* state */
	pch = article_get_extra_header (a, "Status");
	if (pch != NULL) {
		const gboolean read = strchr (pch, 'R') != NULL;
		articles_set_read (&a, 1, read);
		article_remove_header (a, "Status");
	}

	/* references */
	pch = article_get_extra_header (a, HEADER_REFERENCES);
	if (pch != NULL)
		a->references = pch;

	/* cleanup */
	g_mime_object_unref (GMIME_OBJECT(msg));
	debug_exit ("article_set_from_raw_message");
}


static gboolean
breakpoint (const Article * a)
{
	if (a!=NULL && a->group!=NULL)
		g_message ("bad article's group: [%s]", a->group->name);
	return FALSE;
}

gboolean
article_is_valid (const Article * a)
{
	g_return_val_if_fail (a!=NULL, breakpoint(a));
	g_return_val_if_fail (a->group!=NULL, breakpoint(a));
	g_return_val_if_fail (group_is_valid(a->group), breakpoint(a));
	g_return_val_if_fail (a->group->_articles_refcount>=0, breakpoint(a));
	g_return_val_if_fail (group_is_folder(a->group) || a->group->_articles_refcount>0, breakpoint(a));
	g_return_val_if_fail (a->message_id!=NULL, breakpoint(a));
	g_return_val_if_fail (*a->message_id=='<', breakpoint(a));
	g_return_val_if_fail (is_nonempty_string(a->subject), breakpoint(a));
	g_return_val_if_fail (a->number!=0, breakpoint(a));
	g_return_val_if_fail (!a->references || *a->references=='<', breakpoint(a));

	return TRUE;
}

gboolean
articles_are_valid (const Article ** a, int qty)
{
	gint i;

	/* sanity clause */
	g_return_val_if_fail (qty>=0, FALSE);
	g_return_val_if_fail (qty==0 || a!=NULL, FALSE);

	/* check each article */
	for (i=0; i<qty; ++i)
		g_return_val_if_fail (article_is_valid (a[i]), FALSE);

	return TRUE;
}

const gchar*
article_get_message_id (const Article * a)
{
	g_return_val_if_fail (article_is_valid(a), "");
	return a->message_id;
}
 
const gchar*
article_get_subject (const Article * a)
{
	g_return_val_if_fail (article_is_valid(a), "");
	return is_nonempty_string(a->subject) ? a->subject : "";
}


gulong
article_get_combined_linecount (const Article * a)
{
	gulong retval = 0;

	g_return_val_if_fail (article_is_valid(a), FALSE);

	retval = a->linecount;
	if (a->part!=0 && a->parts!=0 && a->threads!=NULL) {
		GSList * l;
		gint part = a->part + 1;
		for (l=a->threads; l; l=l->next) {
			Article * child = ARTICLE(l->data);
			if (child->part == part) {
				retval += child->linecount;
				++part;
			}
		}
	}

	return retval;
}


/***
****  EVENTS
***/

PanCallback*
article_get_articles_changed_callback (void)
{
	static PanCallback * cb = NULL;
	if (cb==NULL) cb = pan_callback_new ();
	return cb;
}

static void
fire_articles_changed (Group              * group,
                       Article           ** articles,
                       gint                 article_qty,
                       ArticleChangeType    type)
{
	ArticleChangeEvent e;
	debug_enter ("fire_articles_changed");

	g_return_if_fail (group!=NULL);
	g_return_if_fail (articles!=NULL);
	g_return_if_fail (article_qty>0);
	g_return_if_fail (type==ARTICLE_CHANGED_READ || type==ARTICLE_CHANGED_DIRTY || type==ARTICLE_CHANGED_NEW);

	e.group = group;
	e.articles = articles;
	e.article_qty = article_qty;
	e.type = type;
	pan_callback_call (article_get_articles_changed_callback(), &e, NULL);

	debug_exit ("fire_articles_changed");
}
