/* image.c - image related functionality */
/*
 *  Penguineyes
 *  Copyright (C) 1998/1999 Neil Howie
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.

 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.

 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <gdk_imlib.h>
#include <gtk/gtk.h>

#include <stdlib.h>
#include <stdio.h>
#include <math.h>

#include "penguineyes.h"

static GdkImlibImage *image = NULL;
static GdkImlibImage *mask_image = NULL;
static GdkPixmap *image_mask = NULL;
static GdkPixmap *window_mask = NULL;

GSList *themeBalls = NULL;
int screen_width = -1;
int screen_height = -1;

static void eyeballs_set(GSList *balls, gboolean set_size)
{
    struct eyeBall *ball;

    for (;balls; balls = g_slist_next(balls)) {
	GdkImlibImage *pimage;
	int w, h;

	ball = (struct eyeBall*)(balls->data);

	if (!ball->ins) {
	    ball->ins = (struct eye_temp*)g_malloc(sizeof(struct eye_temp));

	    ball->ins->gc = NULL;
	    ball->ins->pixmap = NULL;
	    ball->ins->mask = NULL;
	    ball->ins->rball = ball->ball;
	}

	if (!ball->pfile)
	    continue;

	if (ball->ins->pixmap) {
	    gdk_imlib_free_pixmap(ball->ins->pixmap);
	    ball->ins->pixmap = NULL;
	}

	pimage = gdk_imlib_load_image(ball->pfile);

	if (!pimage) {
	    gchar *tmp = g_strdup_printf("unable to load file '%s'",
		    ball->pfile);
	    g_warning(tmp);
	    g_free(tmp);
	    g_free(ball->pfile);
	    ball->pfile = NULL;
	    continue;
	}

	if (default_theme->ins_flags & T_FLAG_FLIP_VER)
	    gdk_imlib_flip_image_vertical(pimage);

	if (default_theme->ins_flags & T_FLAG_FLIP_HOR)
	    gdk_imlib_flip_image_horizontal(pimage);

	if (set_size) {
	    w = ball->ball.w;
	    h = ball->ball.h;
	} else {
	    w = ball->ins->rball.w;
	    h = ball->ins->rball.h;
	}

	if (w == -1)
	    ball->ball.w = w = pimage->rgb_width;

	if (h == -1)
	    ball->ball.h = h = pimage->rgb_width;

	if (!set_size) {
	    gdk_imlib_render(pimage, w, h);

	    ball->ins->pixmap = gdk_imlib_move_image(pimage);
	    ball->ins->mask = gdk_imlib_move_mask(pimage);
	}

	gdk_imlib_kill_image(pimage);

	if (ball->ins->gc)
	    gdk_gc_destroy(ball->ins->gc);

	ball->ins->gc = gdk_gc_new(mainWindow->window);
    }
}

void penguin_set_size(gint w, gint h)
{
    GSList *l = themeBalls;
    double xd, yd;

    g_return_if_fail(image);

    screen_width = w;
    screen_height = h;

    gdk_imlib_render(image, w, h);

    if (window_mask != NULL)
	gdk_imlib_free_pixmap(window_mask);

    if (sourcePenguin)
	gdk_imlib_free_pixmap(sourcePenguin);

    if ((sourcePenguin = gdk_imlib_move_image(image)) == NULL) {
	g_error("set_size - Unable to create image!\n");
	exit(1);
    }

    if (image_mask) {
	gdk_imlib_free_pixmap(image_mask);
	image_mask = NULL;
    }

    if (mask_gc != NULL) {
	gdk_gc_destroy(mask_gc);
	mask_gc = NULL;
    }

    if (mask_image != NULL) {
	gdk_imlib_render(mask_image, w, h);

	mask_gc = gdk_gc_new(mainWindow->window);
	image_mask = gdk_imlib_move_mask(mask_image);
	gdk_gc_set_clip_mask(mask_gc, image_mask);
    }

    if ((window_mask = gdk_imlib_move_mask(image)) != NULL)
	gdk_window_shape_combine_mask(mainWindow->window, window_mask, 0, 0);

    xd = (double)w/(double)image->rgb_width;
    yd = (double)h/(double)image->rgb_height;

    do {
	struct eyeBall *ball = (struct eyeBall*)l->data;

	if (!ball->ins) {
	    g_error("eyeball not initialised!!");
	    exit(1);
	}

	if (default_theme->ins_flags & T_FLAG_FLIP_HOR) {
	    ball->ins->rnorm.x
		= image->rgb_width - ball->norm.x - ball->socket.w;
	} else {
	    ball->ins->rnorm.x = ball->norm.x;
	}

	if (default_theme->ins_flags & T_FLAG_FLIP_VER) {
	    ball->ins->rnorm.y
		= image->rgb_height - ball->norm.y - ball->socket.h;
	} else {
	    ball->ins->rnorm.y = ball->norm.y;
	}

	ball->ins->pos.x = (int)(0.5 + (float)ball->ins->rnorm.x * xd);
	ball->ins->pos.y = (int)(0.5 + (float)ball->ins->rnorm.y * yd);

	ball->ins->rsocket.w = (int)(0.5 + (float)ball->socket.w * xd);
	ball->ins->rsocket.h = (int)(0.5 + (float)ball->socket.h * yd);

	ball->ins->rball.w = MAX((int)(0.5 + (float)ball->ball.w * xd), 1);
	ball->ins->rball.h = MAX((int)(0.5 + (float)ball->ball.h * yd), 1);

	ball->ins->screen.x = -1;
	ball->ins->screen.y = -1;
    } while((l = g_slist_next(l)) != NULL); 
    eyeballs_set(themeBalls, FALSE);

    if (default_theme->clock != NULL) {
	struct clock_settings *c = default_theme->clock;
	c->rpos.x = (int)((float)c->pos.x*xd);
	c->rpos.y = (int)((float)c->pos.y*yd);
	c->rdim.w = (int)((float)c->dim.w*xd);
	c->rdim.h = (int)((float)c->dim.h*yd);
    }

  /* redraw explicitly */
    gdk_draw_pixmap(mainWindow->window, penguineyes_gc, sourcePenguin, 0, 0,
	    0, 0, w, h);

    draw_eyes();

    if (default_theme->clock)
	update_clock(TRUE);

    gdk_flush();
}
static void eyeballs_free(GSList *balls)
{
    struct eyeBall *ball;

    for (;balls; balls = g_slist_next(balls)) {
	ball = (struct eyeBall*)(balls->data);

	if (!ball->ins)
	    continue;

	if (ball->ins->pixmap)
	    gdk_imlib_free_pixmap(ball->ins->pixmap);

	if (ball->ins->mask)
	    gdk_imlib_free_bitmap(ball->ins->mask);

	if (ball->ins->gc)
	    gdk_gc_destroy(ball->ins->gc);

	g_free(ball->ins);
	ball->ins = NULL;
    }
}

/* change the current drawing colour */
void setColour(GdkGC *gc, int red, int blue, int green) {
    GdkColor colour;
    colour.red = red * (65535/255);
    colour.green = green * (65535/255);
    colour.blue = blue * (65535/255);

    colour.pixel = (gulong)(red*65536 + green*256 + blue);

    gdk_imlib_best_color_get(&colour);
    gdk_gc_set_foreground(gc, &colour);
}

static struct theme *last_theme = NULL;

static gboolean first_time = TRUE;

static gboolean set_theme(struct theme *th)
{
    g_return_val_if_fail(th != NULL, FALSE);
    g_return_val_if_fail(th->flags & T_FLAG_EXISTS, FALSE);
    g_return_val_if_fail(th->filename != NULL, FALSE);

    if (last_theme != th) {
	if (last_theme) eyeballs_free(last_theme->balls);
	if (!first_time)
	    th->ins_flags = th->flags;
    }
    first_time = FALSE;

    last_theme = default_theme = th;

    themeBalls = th->balls;

    if (th->colour)
	setColour(penguineyes_gc,
		th->colour->red, th->colour->blue, th->colour->green);
    else
	setColour(penguineyes_gc, 0, 0, 0);

    image = gdk_imlib_load_image(th->filename);

    if (image == NULL) {
	gchar *t = g_strdup_printf("Imlib couldn't load file '%s'",
		th->filename);

	g_warning(t);
	g_free(t);
	th->flags &= ~T_FLAG_EXISTS;
	return FALSE;
    }

    if (th->width == -1 && th->height == -1) {
	th->width = image->rgb_width;
	th->height = image->rgb_height;
    } else if (th->width == -1) { // implies th->height != -1
	th->width = (image->rgb_width*th->height)/image->rgb_height;
    } else if (th->height == -1 ) {
	th->height = (image->rgb_height*th->width)/image->rgb_width;
    }

    if (mask_image)
	gdk_imlib_kill_image(mask_image);

    if (th->mask) {
	if ((mask_image = gdk_imlib_load_image(th->mask)) == NULL) {
	    gchar *tmp = g_strdup_printf("Unable to load image '%s'",
		    th->mask);
	    g_warning(tmp);
	    g_free(tmp);
	    g_free(th->mask);
	    th->mask = NULL;
	}
    } else
	mask_image = NULL;

    if (th->ins_flags & T_FLAG_FLIP_VER) {
	gdk_imlib_flip_image_vertical(image);
	if (mask_image)
	    gdk_imlib_flip_image_vertical(mask_image);
    }

    if (th->ins_flags & T_FLAG_FLIP_HOR) {
	gdk_imlib_flip_image_horizontal(image);
	if (mask_image)
	    gdk_imlib_flip_image_horizontal(mask_image);
    }

    eyeballs_set(themeBalls, TRUE);

#ifdef HAVE_GNOME
    if (is_gnome_applet) {
	if (th->say) {
	    applet_widget_set_tooltip(APPLET_WIDGET(appletWidget),
		    th->say);
	} else {
	    applet_widget_set_tooltip(APPLET_WIDGET(appletWidget),
		    "This space for hire");
	}
    }
#endif
    create_right_click_menu();
    if (th->clock)
	update_clock(TRUE);
    return TRUE;
}

static struct theme *get_first_exist(void)
{
    GSList *thm;

    for (thm = theme_list; thm; thm = g_slist_next(thm)) {
	struct theme *a = thm->data;

	if (a->flags & T_FLAG_EXISTS) {
	    return a;
	}
    }
    g_error("No valid themes!!!!!!!");
    exit(1);
    return NULL;
}

void set_theme_auto(struct theme *th)
{
    int w, h;

    g_return_if_fail(th != NULL);

    if (!set_theme(th)) {
	/* recurse until either sets working theme, or dies */
	set_theme_auto(get_first_exist());
	return;
    }

#ifdef HAVE_GNOME
    if (is_gnome_applet) {
	h = 55;
	w = (image->rgb_width*h)/image->rgb_height;
    } else
#endif
    {
	w = th->width;
	h = th->height;
    }

    if (w == 0 || h == 0) {
	g_error("image has zero area!\n");
	exit(1);
    }

#ifdef HAVE_GNOME
    if (is_gnome_applet)
	gtk_widget_set_usize(appletWidget, w, h);
#endif

    gdk_window_resize(mainWindow->window, w, h);

    penguin_set_size(w, h);

    return;
}

void set_theme_size(struct theme *th, int w, int h)
{
    g_return_if_fail(th != NULL);

    if (w <= 0 || h <= 0) {
	set_theme_auto(th);
	return;
    }

    if (!set_theme(th)) {
	set_theme_auto(get_first_exist());
	return;
    }

    gdk_window_resize(mainWindow->window, w, h);

    penguin_set_size(w, h);
}
