/*
 * Copyright © 2017 Miklós Máté
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the next
 * paragraph) shall be included in all copies or substantial portions of the
 * Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
 * IN THE SOFTWARE.
 */

/** Paragraph 6 of the Errors section:
 *
 * The error INVALID_OPERATION is generated by PassTexCoordATI or
 * SampleMapATI if <coord> passed to PassTexCoordATI or <interp> passed
 * to SampleMapATI is a register in the first pass, or a register with
 * SWIZZLE_STQ_ATI or SWIZZLE_STQ_DQ_ATI <swizzle> in the second pass,
 * or if different <swizzle> parameters are specified for the same
 * <coord> or <interp> in the same pass.
 */

/* Note that the spec is sloppy at the last part. Only STR and STQ are
 * incompatible, but e.g. STR and STR_DR are not. This only applies to
 * texture sources, but not to registers. And instead of "same pass" it
 * should read "same shader".
 * See r200_fragshader.c:333
 */

#include "piglit-util-gl.h"

static struct piglit_gl_test_config *piglit_config;

PIGLIT_GL_TEST_CONFIG_BEGIN

	piglit_config = &config;
	config.supports_gl_compat_version = 10;
	config.window_visual = PIGLIT_GL_VISUAL_DOUBLE | PIGLIT_GL_VISUAL_RGBA;

PIGLIT_GL_TEST_CONFIG_END

enum piglit_result
piglit_display(void)
{
	/* UNREACHED */
	return PIGLIT_FAIL;
}

static enum piglit_result
reg_src_in_first_pass(void *data)
{
	bool pass = true;

	glBeginFragmentShaderATI();
	pass &= piglit_check_gl_error(GL_NO_ERROR);
	glPassTexCoordATI(GL_REG_0_ATI, GL_REG_0_ATI, GL_SWIZZLE_STR_ATI);
	pass &= piglit_check_gl_error(GL_INVALID_OPERATION);
	/* note: Mesa requires at least 1 arith instruction per pass,
	 * but this is not in the spec */
	glColorFragmentOp1ATI(GL_MOV_ATI, GL_REG_0_ATI, GL_NONE, GL_NONE,
			GL_REG_1_ATI, GL_NONE, GL_NONE);
	glEndFragmentShaderATI();

	glBeginFragmentShaderATI();
	pass &= piglit_check_gl_error(GL_NO_ERROR);
	glSampleMapATI(GL_REG_0_ATI, GL_REG_0_ATI, GL_SWIZZLE_STR_ATI);
	pass &= piglit_check_gl_error(GL_INVALID_OPERATION);
	/* note: Mesa requires at least 1 arith instruction per pass,
	 * but this is not in the spec */
	glColorFragmentOp1ATI(GL_MOV_ATI, GL_REG_0_ATI, GL_NONE, GL_NONE,
			GL_REG_1_ATI, GL_NONE, GL_NONE);
	glEndFragmentShaderATI();
	pass &= piglit_check_gl_error(GL_NO_ERROR);

	return pass ? PIGLIT_PASS : PIGLIT_FAIL;
}

static enum piglit_result
stq_swizzle_on_reg(void *data)
{
	bool pass = true;

	glBeginFragmentShaderATI();
	glColorFragmentOp1ATI(GL_MOV_ATI, GL_REG_0_ATI, GL_NONE, GL_NONE,
			GL_REG_1_ATI, GL_NONE, GL_NONE);
	pass &= piglit_check_gl_error(GL_NO_ERROR);
	glPassTexCoordATI(GL_REG_0_ATI, GL_REG_0_ATI, GL_SWIZZLE_STQ_ATI);
	pass &= piglit_check_gl_error(GL_INVALID_OPERATION);
	/* note: Mesa requires at least 1 arith instruction per pass,
	 * but this is not in the spec */
	glColorFragmentOp1ATI(GL_MOV_ATI, GL_REG_0_ATI, GL_NONE, GL_NONE,
			GL_REG_1_ATI, GL_NONE, GL_NONE);
	glEndFragmentShaderATI();

	glBeginFragmentShaderATI();
	glColorFragmentOp1ATI(GL_MOV_ATI, GL_REG_0_ATI, GL_NONE, GL_NONE,
			GL_REG_1_ATI, GL_NONE, GL_NONE);
	pass &= piglit_check_gl_error(GL_NO_ERROR);
	glPassTexCoordATI(GL_REG_0_ATI, GL_REG_0_ATI, GL_SWIZZLE_STQ_DQ_ATI);
	pass &= piglit_check_gl_error(GL_INVALID_OPERATION);
	/* note: Mesa requires at least 1 arith instruction per pass,
	 * but this is not in the spec */
	glColorFragmentOp1ATI(GL_MOV_ATI, GL_REG_0_ATI, GL_NONE, GL_NONE,
			GL_REG_1_ATI, GL_NONE, GL_NONE);
	glEndFragmentShaderATI();

	glBeginFragmentShaderATI();
	glColorFragmentOp1ATI(GL_MOV_ATI, GL_REG_0_ATI, GL_NONE, GL_NONE,
			GL_REG_1_ATI, GL_NONE, GL_NONE);
	pass &= piglit_check_gl_error(GL_NO_ERROR);
	glSampleMapATI(GL_REG_0_ATI, GL_REG_0_ATI, GL_SWIZZLE_STQ_ATI);
	pass &= piglit_check_gl_error(GL_INVALID_OPERATION);
	/* note: Mesa requires at least 1 arith instruction per pass,
	 * but this is not in the spec */
	glColorFragmentOp1ATI(GL_MOV_ATI, GL_REG_0_ATI, GL_NONE, GL_NONE,
			GL_REG_1_ATI, GL_NONE, GL_NONE);
	glEndFragmentShaderATI();

	glBeginFragmentShaderATI();
	glColorFragmentOp1ATI(GL_MOV_ATI, GL_REG_0_ATI, GL_NONE, GL_NONE,
			GL_REG_1_ATI, GL_NONE, GL_NONE);
	pass &= piglit_check_gl_error(GL_NO_ERROR);
	glSampleMapATI(GL_REG_0_ATI, GL_REG_0_ATI, GL_SWIZZLE_STQ_DQ_ATI);
	pass &= piglit_check_gl_error(GL_INVALID_OPERATION);
	/* note: Mesa requires at least 1 arith instruction per pass,
	 * but this is not in the spec */
	glColorFragmentOp1ATI(GL_MOV_ATI, GL_REG_0_ATI, GL_NONE, GL_NONE,
			GL_REG_1_ATI, GL_NONE, GL_NONE);
	glEndFragmentShaderATI();
	pass &= piglit_check_gl_error(GL_NO_ERROR);

	return pass ? PIGLIT_PASS : PIGLIT_FAIL;
}

static bool
check_swizzle_passtexcoord(unsigned src, unsigned swizzle1, unsigned swizzle2, unsigned error)
{
	bool pass = true;

	glBeginFragmentShaderATI();
	glColorFragmentOp1ATI(GL_MOV_ATI, GL_REG_0_ATI, GL_NONE, GL_NONE,
			GL_REG_1_ATI, GL_NONE, GL_NONE);
	glPassTexCoordATI(GL_REG_0_ATI, src, swizzle1);
	pass &= piglit_check_gl_error(GL_NO_ERROR);
	glPassTexCoordATI(GL_REG_1_ATI, src, swizzle2);
	pass &= piglit_check_gl_error(error);
	/* note: Mesa requires at least 1 arith instruction per pass,
	 * but this is not in the spec */
	glColorFragmentOp1ATI(GL_MOV_ATI, GL_REG_0_ATI, GL_NONE, GL_NONE,
			GL_REG_1_ATI, GL_NONE, GL_NONE);
	glEndFragmentShaderATI();
	pass &= piglit_check_gl_error(GL_NO_ERROR);

	return pass;
}

static enum piglit_result
different_swizzle_on_same_tex_passtexcoord(void *data)
{
	bool pass = true;

	pass &= check_swizzle_passtexcoord(GL_TEXTURE0_ARB, GL_SWIZZLE_STR_ATI, GL_SWIZZLE_STQ_ATI, GL_INVALID_OPERATION);
	pass &= check_swizzle_passtexcoord(GL_TEXTURE0_ARB, GL_SWIZZLE_STR_ATI, GL_SWIZZLE_STR_DR_ATI, GL_NO_ERROR);
	pass &= check_swizzle_passtexcoord(GL_TEXTURE0_ARB, GL_SWIZZLE_STR_ATI, GL_SWIZZLE_STQ_DQ_ATI, GL_INVALID_OPERATION);
	pass &= check_swizzle_passtexcoord(GL_TEXTURE0_ARB, GL_SWIZZLE_STQ_ATI, GL_SWIZZLE_STR_DR_ATI, GL_INVALID_OPERATION);
	pass &= check_swizzle_passtexcoord(GL_TEXTURE0_ARB, GL_SWIZZLE_STQ_ATI, GL_SWIZZLE_STQ_DQ_ATI, GL_NO_ERROR);
	pass &= check_swizzle_passtexcoord(GL_TEXTURE0_ARB, GL_SWIZZLE_STR_DR_ATI, GL_SWIZZLE_STQ_DQ_ATI, GL_INVALID_OPERATION);

	return pass ? PIGLIT_PASS : PIGLIT_FAIL;
}

static bool
check_swizzle_samplemap(unsigned src, unsigned swizzle1, unsigned swizzle2, unsigned error)
{
	bool pass = true;

	glBeginFragmentShaderATI();
	glColorFragmentOp1ATI(GL_MOV_ATI, GL_REG_0_ATI, GL_NONE, GL_NONE,
			GL_REG_1_ATI, GL_NONE, GL_NONE);
	glPassTexCoordATI(GL_REG_0_ATI, src, swizzle1);
	pass &= piglit_check_gl_error(GL_NO_ERROR);
	glSampleMapATI(GL_REG_1_ATI, src, swizzle2);
	pass &= piglit_check_gl_error(error);
	/* note: Mesa requires at least 1 arith instruction per pass,
	 * but this is not in the spec */
	glColorFragmentOp1ATI(GL_MOV_ATI, GL_REG_0_ATI, GL_NONE, GL_NONE,
			GL_REG_1_ATI, GL_NONE, GL_NONE);
	glEndFragmentShaderATI();
	pass &= piglit_check_gl_error(GL_NO_ERROR);

	return pass;
}

static enum piglit_result
different_swizzle_on_same_tex_samplemap(void *data)
{
	bool pass = true;

	pass &= check_swizzle_samplemap(GL_TEXTURE0_ARB, GL_SWIZZLE_STR_ATI, GL_SWIZZLE_STQ_ATI, GL_INVALID_OPERATION);
	pass &= check_swizzle_samplemap(GL_TEXTURE0_ARB, GL_SWIZZLE_STR_ATI, GL_SWIZZLE_STR_DR_ATI, GL_NO_ERROR);
	pass &= check_swizzle_samplemap(GL_TEXTURE0_ARB, GL_SWIZZLE_STR_ATI, GL_SWIZZLE_STQ_DQ_ATI, GL_INVALID_OPERATION);
	pass &= check_swizzle_samplemap(GL_TEXTURE0_ARB, GL_SWIZZLE_STQ_ATI, GL_SWIZZLE_STR_DR_ATI, GL_INVALID_OPERATION);
	pass &= check_swizzle_samplemap(GL_TEXTURE0_ARB, GL_SWIZZLE_STQ_ATI, GL_SWIZZLE_STQ_DQ_ATI, GL_NO_ERROR);
	pass &= check_swizzle_samplemap(GL_TEXTURE0_ARB, GL_SWIZZLE_STR_DR_ATI, GL_SWIZZLE_STQ_DQ_ATI, GL_INVALID_OPERATION);

	return pass ? PIGLIT_PASS : PIGLIT_FAIL;
}

static enum piglit_result
different_swizzle_on_same_reg(void *data)
{
	bool pass = true;

	pass &= check_swizzle_passtexcoord(GL_REG_0_ATI, GL_SWIZZLE_STR_ATI, GL_SWIZZLE_STR_DR_ATI, GL_NO_ERROR);
	pass &= check_swizzle_samplemap(GL_REG_0_ATI, GL_SWIZZLE_STR_ATI, GL_SWIZZLE_STR_DR_ATI, GL_NO_ERROR);

	return pass ? PIGLIT_PASS : PIGLIT_FAIL;
}

static bool check_swizzle_2pass(bool samplemap, unsigned swizzle1, unsigned swizzle2, unsigned error)
{
	bool pass = true;

	glBeginFragmentShaderATI();
	if (samplemap)
		glSampleMapATI(GL_REG_0_ATI, GL_TEXTURE0_ARB, swizzle1);
	else
		glPassTexCoordATI(GL_REG_0_ATI, GL_TEXTURE0_ARB, swizzle1);
	glColorFragmentOp1ATI(GL_MOV_ATI, GL_REG_0_ATI, GL_NONE, GL_NONE,
			GL_REG_0_ATI, GL_NONE, GL_NONE);
	pass &= piglit_check_gl_error(GL_NO_ERROR);
	if (samplemap)
		glSampleMapATI(GL_REG_0_ATI, GL_TEXTURE0_ARB, swizzle2);
	else
		glPassTexCoordATI(GL_REG_0_ATI, GL_TEXTURE0_ARB, swizzle2);
	pass &= piglit_check_gl_error(error);
	/* note: Mesa requires at least 1 arith instruction per pass,
	 * but this is not in the spec */
	glColorFragmentOp1ATI(GL_MOV_ATI, GL_REG_0_ATI, GL_NONE, GL_NONE,
			GL_REG_0_ATI, GL_NONE, GL_NONE);
	glEndFragmentShaderATI();

	return pass;
}

static enum piglit_result
different_swizzle_in_different_pass(void *data)
{
	bool pass = true;

	pass &= check_swizzle_2pass(false, GL_SWIZZLE_STR_ATI, GL_SWIZZLE_STQ_ATI, GL_INVALID_OPERATION);
	pass &= check_swizzle_2pass(true, GL_SWIZZLE_STR_ATI, GL_SWIZZLE_STQ_ATI, GL_INVALID_OPERATION);

	return pass ? PIGLIT_PASS : PIGLIT_FAIL;
}

static const struct piglit_subtest subtests[] = {
	{
		"REG source in first pass",
		"reg-src-in-first-pass",
		reg_src_in_first_pass,
		NULL
	},
	{
		"STQ swizzle on REG",
		"stq-swizzle-on-reg",
		stq_swizzle_on_reg,
		NULL
	},
	{
		"Different swizzle on the same TEX in PassTexCoord",
		"different_swizzle_on_same_tex_passtexcoord",
		different_swizzle_on_same_tex_passtexcoord,
		NULL
	},
	{
		"Different swizzle on the same TEX in SampleMap",
		"different_swizzle_on_same_tex_samplemap",
		different_swizzle_on_same_tex_samplemap,
		NULL
	},
	{
		"Different swizzle on the same REG",
		"different_swizzle_on_same_reg",
		different_swizzle_on_same_reg,
		NULL
	},
	{
		"Different swizzle in different pass",
		"different_swizzle_in_different_pass",
		different_swizzle_in_different_pass,
		NULL
	},
	{
		NULL,
		NULL,
		NULL,
		NULL
	}
};

void
piglit_init(int argc, char **argv)
{
	piglit_require_extension("GL_ATI_fragment_shader");

	piglit_report_result(piglit_run_selected_subtests(subtests,
				piglit_config->selected_subtests,
				piglit_config->num_selected_subtests,
				PIGLIT_SKIP));
}
