/* -*- Mode: C; c-basic-offset: 2; indent-tabs-mode: nil -*-
 *
 * Pigment OpenGL plugin
 *
 * Copyright © 2006, 2007, 2008 Fluendo Embedded S.L.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Loïc Molinari <loic@fluendo.com>
 */

#ifndef __PGM_CONTEXT_H__
#define __PGM_CONTEXT_H__

/* pgmcontext.h and pgmbackend.h/pgmglviewport.h include eachother */
typedef struct _PgmContext            PgmContext;
typedef struct _PgmContextTask        PgmContextTask;
typedef struct _PgmContextProcAddress PgmContextProcAddress;

#include "pgmtexture.h"
#include "pgmprogram.h"
#include "pgmbackend.h"
#include "pgmglviewport.h"
#include "pgmgldefs.h"

G_BEGIN_DECLS

/* Context locking */
#define PGM_CONTEXT_LOCK(context)   (g_mutex_lock (context->mutex))
#define PGM_CONTEXT_UNLOCK(context) (g_mutex_unlock (context->mutex))

/* Type casting */
#define PGM_CONTEXT(obj)      ((PgmContext *) (obj))
#define PGM_CONTEXT_TASK(obj) ((PgmContextTask *) (obj))

/* Task handler */
typedef void  (*PgmContextTaskFunc) (PgmContext *context, gpointer data);

/* Task types */
typedef enum {
  /* Various */
  PGM_CONTEXT_PROJECTION = 0, /* Update viewport and projection matrix */
  PGM_CONTEXT_SIZE,           /* Update window size */
  PGM_CONTEXT_TITLE,          /* Update window and icon titles */
  PGM_CONTEXT_DECORATION,     /* Update decoration state */
  PGM_CONTEXT_FULLSCREEN,     /* Update fullscreen state */
  PGM_CONTEXT_VISIBILITY,     /* Update visibility state */
  PGM_CONTEXT_ALPHA_BLENDING, /* Update alpha blending state */
  PGM_CONTEXT_RESOLUTION,     /* Update desktop resolution */
  PGM_CONTEXT_OPACITY,        /* Update opacity */
  PGM_CONTEXT_CURSOR,         /* Update system cursor */
  PGM_CONTEXT_ICON,           /* Update window icon */
  PGM_CONTEXT_DRAG_STATUS,    /* Update drag status */
  PGM_CONTEXT_READ_PIXELS,    /* Read frame buffer */
  PGM_CONTEXT_QUIT,           /* Clean up and quit the rendering thread */
  /* Texture related */
  PGM_CONTEXT_GEN_TEXTURE,    /* Generate texture */
  PGM_CONTEXT_CLEAN_TEXTURE,  /* Clean up texture */
  PGM_CONTEXT_UPLOAD_TEXTURE, /* Upload texture buffer */
  PGM_CONTEXT_UPDATE_TEXTURE, /* Update texture parameters */
  PGM_CONTEXT_FREE_TEXTURE,   /* Free texture */
  /* Task types count */
  PGM_CONTEXT_LAST_TASK
} PgmContextTaskType;

/* Context */
struct _PgmContext {
  PgmGlViewport *glviewport;

  /* PgmContext structure access mutex */
  GMutex *mutex;

  /* Rendering thread */
  GThread      *render_thread;
  GMainContext *render_context;
  GMainLoop    *render_loop;

  /* Immediate event source */
  gint        immediate_fd[2];
  GIOChannel *immediate_out;
  GIOChannel *immediate_in;
  guint       immediate_tag;

  /* Task queues */
  GList *immediate_task;
  GList *deferred_task;

  /* Auto-update */
  GMutex   *update_mutex;
  gboolean  auto_updated;
  GTimeVal  update_timestamp;
  guint     update_tag;
  guint     requested_fps;

  /* Initialization lock */
  GMutex   *init_mutex;
  GCond    *init_cond;
  gboolean  initialized;

  /* Physical screen informations */
  gint screen_width_mm;
  gint screen_height_mm;

  /* Limits */
  gint max_texture_2d_size;
  gint max_texture_rect_size;
  gint max_texture_units;

  /* Pixel store modes */
  gint row_length;
  gint skip_rows;
  gint skip_pixels;

  /* Renderer informations */
  const gchar *vendor;
  const gchar *version_string;
  const gchar *renderer;
  const gchar *extensions;
  gfloat       version;

  /* Features */
  gulong feature_mask;

  /* OpenGL backend */
  PgmBackend *backend;

  /* OpenGL function pointers */
  PgmContextProcAddress *gl;

  /* Frame rate per second */
  GTimeVal fps_tick_time;
  guint    fps;

  /* Task handlers */
  PgmContextTaskFunc task_func[PGM_CONTEXT_LAST_TASK];
};

/* Task */
struct _PgmContextTask {
  /* Task type */
  PgmContextTaskType type;
  /* Opaque pointer depending on task type */
  gpointer data;
};

/* OpenGL function pointers, used for extensions loading */
struct _PgmContextProcAddress {
  /* OpenGL core functions */
  pgm_gl_enable               enable;
  pgm_gl_disable              disable;
  pgm_gl_get_error            get_error;
  pgm_gl_get_string           get_string;
  pgm_gl_begin                begin;
  pgm_gl_end                  end;
  pgm_gl_vertex_3f            vertex_3f;
  pgm_gl_tex_coord_2f         tex_coord_2f;
  pgm_gl_enable_client_state  enable_client_state;
  pgm_gl_disable_client_state disable_client_state;
  pgm_gl_vertex_pointer       vertex_pointer;
  pgm_gl_color_pointer        color_pointer;
  pgm_gl_tex_coord_pointer    tex_coord_pointer;
  pgm_gl_draw_arrays          draw_arrays;
  pgm_gl_tex_env_f            tex_env_f;
  pgm_gl_tex_env_fv           tex_env_fv;
  pgm_gl_tex_gen_i            tex_gen_i;
  pgm_gl_tex_gen_fv           tex_gen_fv;
  pgm_gl_color_4f             color_4f;
  pgm_gl_color_4fv            color_4fv;
  pgm_gl_scissor              scissor;
  pgm_gl_depth_func           depth_func;
  pgm_gl_blend_func           blend_func;
  pgm_gl_line_width           line_width;
  pgm_gl_clear                clear;
  pgm_gl_clear_color          clear_color;
  pgm_gl_clear_stencil        clear_stencil;
  pgm_gl_stencil_func         stencil_func;
  pgm_gl_stencil_op           stencil_op;
  pgm_gl_push_attrib          push_attrib;
  pgm_gl_pop_attrib           pop_attrib;
  pgm_gl_matrix_mode          matrix_mode;
  pgm_gl_push_matrix          push_matrix;
  pgm_gl_pop_matrix           pop_matrix;
  pgm_gl_load_identity        load_identity;
  pgm_gl_load_matrix_f        load_matrix_f;
  pgm_gl_depth_range          depth_range;
  pgm_gl_viewport             viewport;
  pgm_gl_raster_pos_2f        raster_pos_2f;
  pgm_gl_bitmap               bitmap;
  pgm_gl_read_buffer          read_buffer;
  pgm_gl_draw_buffer          draw_buffer;
  pgm_gl_copy_pixels          copy_pixels;
  pgm_gl_flush                flush;
  pgm_gl_finish               finish;
  pgm_gl_pixel_store_i        pixel_store_i;
  pgm_gl_frustum              frustum;
  pgm_gl_ortho                ortho;
  pgm_gl_scale_f              scale_f;
  pgm_gl_translate_f          translate_f;
  pgm_gl_rotate_f             rotate_f;
  pgm_gl_is_list              is_list;
  pgm_gl_delete_lists         delete_lists;
  pgm_gl_gen_lists            gen_lists;
  pgm_gl_new_list             new_list;
  pgm_gl_end_list             end_list;
  pgm_gl_call_list            call_list;
  pgm_gl_call_lists           call_lists;
  pgm_gl_list_base            list_base;
  pgm_gl_hint                 hint;
  pgm_gl_depth_mask           depth_mask;
  pgm_gl_polygon_mode         polygon_mode;
  pgm_gl_shade_model          shade_model;
  pgm_gl_color_mask           color_mask;
  pgm_gl_read_pixels          read_pixels;
  pgm_gl_get_tex_image        get_tex_image;
  pgm_gl_tex_sub_image_2d     tex_sub_image_2d;
  pgm_gl_gen_textures         gen_textures;
  pgm_gl_delete_textures      delete_textures;
  pgm_gl_bind_texture         bind_texture;
  pgm_gl_tex_image_2d         tex_image_2d;
  pgm_gl_tex_parameter_i      tex_parameter_i;
  pgm_gl_copy_tex_sub_image_2d copy_tex_sub_image_2d;
  pgm_gl_get_integer_v        get_integer_v;
  pgm_gl_get_float_v          get_float_v;

  /* OpenGL extension functions */
  pgm_gl_blend_func_separate      blend_func_separate;
  pgm_gl_blend_color              blend_color;
  pgm_gl_active_texture           active_texture;
  pgm_gl_client_active_texture    client_active_texture;
  pgm_gl_multi_draw_arrays        multi_draw_arrays;
  pgm_gl_gen_programs             gen_programs;
  pgm_gl_delete_programs          delete_programs;
  pgm_gl_program_string           program_string;
  pgm_gl_bind_program             bind_program;
  pgm_gl_program_local_param_4fv  program_local_param_4fv;
  pgm_gl_get_program_iv           get_program_iv;
  pgm_gl_gen_buffers              gen_buffers;
  pgm_gl_delete_buffers           delete_buffers;
  pgm_gl_bind_buffer              bind_buffer;
  pgm_gl_buffer_data              buffer_data;
  pgm_gl_buffer_sub_data          buffer_sub_data;
  pgm_gl_get_buffer_sub_data      get_buffer_sub_data;
  pgm_gl_map_buffer               map_buffer;
  pgm_gl_unmap_buffer             unmap_buffer;
  pgm_gl_gen_framebuffers         gen_framebuffers;
  pgm_gl_delete_framebuffers      delete_framebuffers;
  pgm_gl_bind_framebuffer         bind_framebuffer;
  pgm_gl_framebuffer_renderbuffer framebuffer_renderbuffer;
  pgm_gl_framebuffer_texture_2d   framebuffer_texture_2d;
  pgm_gl_check_framebuffer_status check_framebuffer_status;
  pgm_gl_gen_renderbuffers        gen_renderbuffers;
  pgm_gl_delete_renderbuffers     delete_renderbuffers;
  pgm_gl_bind_renderbuffer        bind_renderbuffer;
  pgm_gl_renderbuffer_storage     renderbuffer_storage;
  pgm_gl_get_renderbuffer_parameter_iv get_renderbuffer_parameter_iv;
};

/* Create a new context */
PgmContext     *pgm_context_new                    (PgmGlViewport *glviewport);

/* Free a context */
void            pgm_context_free                   (PgmContext *context);

/* Create a new task */
PgmContextTask *pgm_context_task_new               (PgmContextTaskType type,
                                                    gpointer data);

/* Free a task */
void            pgm_context_task_free              (PgmContextTask *task);

/* Request a rendering update */
void            pgm_context_update                 (PgmContext *context);

/* Push a new task on the immediate queue */
void            pgm_context_push_immediate_task    (PgmContext *context,
                                                    PgmContextTask *task);

/* Push a new task on the deferred queue */
void            pgm_context_push_deferred_task     (PgmContext *context,
                                                    PgmContextTask *task);

/* Remove tasks with the given data from the immediate and deferred queues */
void            pgm_context_remove_tasks_with_data (PgmContext *context,
                                                    gconstpointer data);

/* Dump a formatted output on stdout showing informations about the GPU */
void            pgm_context_dump_features          (PgmContext *context);

G_END_DECLS

#endif /* __PGM_CONTEXT_H__ */
