/*	Notify

PIRL CVS ID: Notify.java,v 1.25 2012/04/16 06:04:10 castalia Exp

Copyright (C) 2004-2012  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/

package PIRL.Conductor;

import PIRL.Utilities.Streams;
import PIRL.Strings.String_Buffer;

import java.util.*;
import java.net.*;
import java.io.*;

/*	Provides a more sophisticated alternative to the mailto URL mechanism.
*/
import javax.mail.MessagingException;
import javax.mail.Session;
import javax.mail.Transport;
import javax.mail.internet.InternetAddress;
import javax.mail.internet.MimeMessage;

/**	<i>Notify</i> is a mechanism for delivering messages to
	destinations.
<p>
	Each of Notify's destinations is represented by a {@link
	java.net.URI} address which specifies the recipient of the message
	and the protocol to be used to deliver it. The mailto URI, which
	specifies how to deliver an e-mail message, is typical. If the
	address does not specify a protocol, it is treated as a mailto URI.
	If, for any reason, a destination cannot be used during the delivery
	of the notice, that destination is added to a Vector of
	undeliverable destinations.
<p>
	Notify's message consists of text that is delivered to each
	destination via a {@link java.net.URLConnection}. The message
	contents may be provided by a resource specifier which may be a
	local resource - either a system filename or path within an
	enclosing jar file - or an external resource via a URL. The message
	may also be specified directly on the command line. The size of the
	message may be limited to a maximum number of lines and the number
	of last lines to always be retained in the message body may also be
	controlled. This is particularly useful for preventing unexpectedly
	large message files from causing memory exhaustion while still
	delivering the beginning and end of the message.
<p>
	For mailto deliveries a From and Reply-To address and a Subject
	line may be provided.
<p>
@author		Christian Schaller - UA/PIRL
@version	1.25
*/
public class Notify
{
/*==============================================================================
	Constants
*/
//	Class ID
public static final String
	ID = "PIRL.Conductor.Notify (1.25 2012/04/16 06:04:10)";

/**	Marker for a message source as opposed to a message string.
*/
public static final char
	FILENAME_MARKER				= '@';

/**	Exit status values.
*/
public static final int
	SUCCESS						= 0,
	ILLEGAL_SYNTAX				= 1,
	NO_DESTINATION				= 2,
	CANNOT_READ_MESSAGE_FILE	= 3;

//	The destination(s) of the notification message
private Vector
	Destinations				= new Vector (),
	Undeliverable				= new Vector ();

//	The message to be delivered.
private String
	Message						= "";

private int
	Max_Message_Lines			= 1000,
	Max_Last_Lines				= 50,
	Total_Lines					= 0;

private Vector
	Last_Lines					= new Vector (Max_Last_Lines);

private static final String
	USERNAME					= System.getProperty ("user.name"),
	NL							= System.getProperty ("line.separator");

public static final String
	DEFAULT_SMTP_SERVER					= "localhost";
private static Properties
	Mail_Props;
static
{
	Mail_Props = new Properties ();
	Mail_Props.put ("mail.smtp.host", DEFAULT_SMTP_SERVER);
	Mail_Props.put ("mail.from", USERNAME);
}

private String
	From						= USERNAME,
	Reply_To					= null,
	Subject						= null;


//	Debug control
private static final int
	DEBUG_OFF					= 0,
	DEBUG_SETUP					= 1 << 0,
	DEBUG_PROCESSING			= 1 << 1,
	DEBUG_GET_STREAM			= 1 << 2,
	DEBUG_MESSAGE				= 1 << 3,
	DEBUG_ALL					= -1,

	DEBUG						= DEBUG_OFF;

/*==============================================================================
	Constructors
*/
/**	Constructs a Notify object with the specified destinations.
<p>
	@param	destinations	The Vector of destinations to which the
		notification message will be delivered.
*/
public Notify
	(
	Vector	destinations
	)
{Destinations (destinations);}

/**	Constructs a Notify object with no destinations.
*/
public Notify ()
{}

/*==============================================================================
	Message Header
*/
/** Sets the outgoing SMTP server hostname.
<p>
	@param	smtp_server	The hostname of the outgoing SMTP server. The
		default hostname is {@link #DEFAULT_SMTP_SERVER}.
	@return	This Notify object.
*/
public Notify SMTP_Server 
	(
	String	smtp_server
	)
{
Mail_Props.setProperty ("mail.smtp.host", smtp_server);
return this;
}

/** Gets the current SMTP Server setting.
<p>
	@return	The SMTP hostname String
 */
public String SMTP_Server ()
{return Mail_Props.getProperty ("mail.smtp.host");}


/**	Gets the list of destinations to which the notice will be
	delivered.
<p>
	Notify will attempt to deliver to each of these destinations on the
	next invocation of the {@link #Deliver()} method. A failed delivery
	will add that destination to the list of undeliverable destinations.

	@return	the destinations list as a Vector
	@see #Undeliverable()
*/
public Vector Destinations ()
{return Destinations;}

/**	Sets the list of destinations to which the message will be
	delivered.
<p>
	Redundant destination addresses are removed from the list.
<p>
	Setting the destinations list has the side effect of resetting the
	list of undeliverable destinations.

	@param	destinations	The Vector of destination URIs.
	@return	This Notify object.
	@see #Undeliverable()
*/
public Notify Destinations
	(
	Vector destinations
	)
{
Destinations = Remove_Redundancies (destinations);
Undeliverable.clear ();
return this;
}

/**	Gets the list of destinations to which the notice could not be
	delivered.
<p>
	Whenever a {@link #Deliver() Deliver}y cycle is done destinations
	to which the notice could not be delivered - for whatever reason -
	are added to a list of undeliverable destinations.

	@return	The Vector of undeliverable destinations.
	@see	#Deliver()
*/
public Vector Undeliverable ()
{return Undeliverable;}

/**	Gets the From address for e-mail deliveries of the notice.
<p>
	The From address is not used for other types of deliveries.

	@return	The From address String.
*/
public String From ()
{return From;}

/**	Sets the From address for e-mail deliveries of the notice.
<p>
	The From address is not used for other types of deliveries.
	Furthermore, if the From address is null, it will not be used at
	all. In this case, the underlying Java e-mail delivery mechanism
	(the mailto URL handler) will attempt to use an appropriate From
	address based on the owner of the Notify process.
<p>
	Note that there is no checking provided to ensure that the From
	address is a valid e-mail address. The address should be a regular
	e-mail address and not a mailto URL.

	@param	from_address	The From address String.
	@return	This Notify object.
*/
public Notify From
	(
	String	from_address
	)
{
if ((From = from_address) == null)
	From = USERNAME;

Mail_Props.setProperty ("mail.from", From);

return this;
}

/**	Gets the Reply-To address for e-mail deliveries of the notice.
<p>
	The Reply-To address is not used for other types of deliveries.

	@return	The Reply-To address String (may be null).
*/
public String Reply_To ()
{return Reply_To;}

/**	Sets the Reply-To address for e-mail deliveries of the notice.
<p>
	The Reply-To address is not used for other types of deliveries.
	Furthermore, if the Reply-To address is null, it will not be used
	at all.
<p>
	Note that there is no checking provided to ensure that the Reply-To
	address is a valid e-mail address. The address should be a regular
	e-mail address and not a mailto URL.

	@param	reply_to_address	The Reply-To address String.
	@return	This Notify object.
*/
public Notify Reply_To
	(
	String	reply_to_address
	)
{
Reply_To = reply_to_address;
return this;
}

/**	Gets the Subject line for e-mail deliveries of the notice.
<p>
	The Subject line is not used for other types of deliveries.

	@return	The Subject line String.
*/
public String Subject ()
{return Subject;}

/**	Sets the Subject line for e-mail deliveries of the notice.
<p>
	The Subject line is not used for other types of deliveries.
	Furthermore, if the Subject line is null, it will not be used at
	all.
	
	@param	subject_line	The Subject line String.
	@return	This Notify object.
*/
public Notify Subject
	(
	String	subject_line
	)
{
Subject = subject_line;
return this;
}

/*==============================================================================
	Message Body
*/
/**	Gets the maximum number of message lines to be delivered.
<p>
	@return	The maximum number of message lines to be delivered.
*/
public int Max_Message_Lines ()
{return Max_Message_Lines;}

/**	Sets the maximum number of message lines to be delivered.
<p>
	@param	max_lines	The maximum number of message lines to be
		delivered. If this is less than or equal to 0, the {@link
		Integer#MAX_VALUE maximum integer value} will be used. If it is
		less than {@link #Max_Last_Lines()} then {@link
		#Max_Last_Lines()} will be lowered to the same value and there
		will be no initial message lines.
	@return	This Notify object.
*/
public Notify Max_Message_Lines
	(
	int		max_lines
	)
{
if (max_lines <= 0)
	Max_Message_Lines = Integer.MAX_VALUE;
else
	{
	Max_Message_Lines = max_lines;
	if (Max_Last_Lines > max_lines)
		Max_Last_Lines = max_lines;
	}
return this;
}

/**	Gets the maximum number of message last lines to be delivered.
<p>
	@return	The maximum number of message last lines to be delivered.
*/
public int Max_Last_Lines ()
{return Max_Last_Lines;}

/**	Sets the maximum number of message last lines to be delivered.
<p>
	@param	max_lines	The maximum number of message lines to be
		delivered. If this is less than 0, 0 will be used. If it is
		greater than {@link #Max_Message_Lines()} then {@link
		#Max_Message_Lines()} will be used and there will be no initial
		message lines.
	@return	This Notify object.
*/
public Notify Max_Last_Lines
	(
	int		max_lines
	)
{
if (max_lines < 0)
	max_lines = 0;
else if (max_lines > Max_Message_Lines)
	Max_Last_Lines = Max_Message_Lines;
else
	Max_Last_Lines = max_lines;
return this;
}

/**	Gets the message to be delivered.
<p>
	The message is composed of two parts: The initial lines of the
	message and the last lines of the message. No more than {@link
	#Max_Message_Lines()} user provided lines - where a line is a sequence
	of characters terminated by a line separator - will be included. If
	this limit is exceeded additional lines will be dropped; however,
	the last {@link #Max_Last_Lines()} lines will always be included. In
	this case the initial lines will be separated from the last lines
	by a line of the form:
<p>
	... [<i>number</i>> lines omitted]
<p>
	where <i>number></i> is number of user provided lines that were
	omitted from the message.
<p>
	@return	The message String.
*/
public String Message ()
{
String
	the_message;
if (Last_Lines.size () == 0)
	the_message = Message;
else
	{
	//	Add the last lines.
	StringBuffer
		message = new StringBuffer (Message);
	if (Total_Lines > Max_Message_Lines)
		message.append ("... ["
			+ (Total_Lines - Max_Message_Lines) + " lines omitted]" + NL);
	Iterator
		lines = Last_Lines.iterator ();
	while (lines.hasNext ())
		message.append ((String)lines.next ());
	the_message = message.toString ();
	}
return the_message;
}

/**	Reads a message from a named source.
<p>
	The source can be the name of a local file, the full pathname to
	that file, the name of a file contained within a jar that packages
	Notify, a file URL that points to a local file, or a URL that
	points to an external resource.
<p>
	Only {@link #Max_Message_Lines()} lines - where a line is a sequence
	of characters terminated by a line separator - will be retained. The
	last {@link #Max_Last_Lines()} lines from the source will be saved
	separately so they will be included in the delivered message
	regardless of the total number of lines available from the source
	(@see #Message()).
<p>
	@param	source	The name of the message file. If null an empty
		message is used.
	@param	convert_escapes	If true, any escape sequences
		in the message are converted to their character equivalents;
		otherwise the string is unchanged.
	@param	append	If true, the message is appended to any existing
		message; otherwise the any existing message is replaced.
	@return	This Notify object.
	@throws	IOException	If an input stream cannot be obtained
		from the message source or if there is a problem reading
		from the input stream.
	@see	Streams#Get_Stream(String)
	@see	String_Buffer#escape_to_special()
*/
public Notify Message_Source
	(
	String	source,
	boolean	convert_escapes,
	boolean	append
	)
	throws IOException
{
if (source == null)
	return this;

if ((DEBUG & DEBUG_MESSAGE) != 0)
	System.out.println
		(">>> Message_Source: " + source + NL
		+"    Append - " + append
		+"    Max message lines - " + Max_Message_Lines + NL
		+"    Max last lines - " + Max_Last_Lines);
InputStream
	message_stream = Streams.Get_Stream (source);
if (message_stream == null)
	throw new IOException
		(ID + NL
		+"Unable to access the message input source - " + source);
BufferedReader
	input = new BufferedReader (new InputStreamReader (message_stream));

if (! append)
	{
	//	Clear the existing message.
	Total_Lines = 0;
	Message = "";
	Last_Lines.clear ();
	}

if ((DEBUG & DEBUG_MESSAGE) != 0)
	System.out.println
		("--- Source lines -" + NL);
String
	line;
int
	lines_read = 0,
	last_line_threshold = Max_Message_Lines - Max_Last_Lines;
while (input.ready ())
	{
	//	Get the next message line.
	line = input.readLine () + NL;
	if ((DEBUG & DEBUG_MESSAGE) != 0)
		++lines_read;
	if (convert_escapes)
		//	Expand any escape sequences.
		line = new String_Buffer (line).escape_to_special ().toString ();
	if (++Total_Lines > last_line_threshold)
		{
		//	Add to last lines list.
		if (Last_Lines.size () == Max_Last_Lines)
			//	Scroll down.
			Last_Lines.remove (0);
		Last_Lines.add (line);
		}
	else
		{
		if ((DEBUG & DEBUG_MESSAGE) != 0)
			System.out.print (line);
		Message += line;
		}
	}
if ((DEBUG & DEBUG_MESSAGE) != 0)
	{
	if (Last_Lines.size () != 0)
		{
		System.out.println
			("--- Last lines -");
		Iterator
			lines = Last_Lines.iterator ();
		while (lines.hasNext ())
			System.out.print ((String)lines.next ());
		}
	System.out.println
		("--- Total lines:" + NL
		+"    " + lines_read + " source lines read." + NL
		+"    " + Total_Lines + " message lines, first." + NL
		+"    " + Last_Lines.size () + " message lines, last." + NL
		+"<<< Message_Source");
	}
return this;
}

/**	Reads a message from a named source.
<p>
	Any escape sequences are converted and any existing message
	appended.
<p>
	@param	source	The name of the message file. If null the
		message is set to the empty String.
	@return	This Notify object.
	@throws	IOException	If an input stream cannot be obtained
		from the message source or if there is a problem reading
		from the input stream.
	@see	#Message_Source(String, boolean, boolean)
*/
public Notify Message_Source
	(
	String	source
	)
	throws IOException
{return Message_Source (source, true, true);}

/**	Sets the message to be delivered.
<p>
	@param	message	The message String. A null message is empty.
	@param	convert_escapes	If true, any escape sequences in the message
		are converted to their character equivalents; otherwise the
		string is unchanged.
	@param	append	If true, the message is appended to any existing
		message; otherwise an existing message is replaced.
	@return	This Notify object.
	@see	String_Buffer#escape_to_special()
*/
public Notify Message
	(
	String	message,
	boolean	convert_escapes,
	boolean	append
	)
{
if ((DEBUG & DEBUG_MESSAGE) != 0)
	System.out.println (">>> Message:"
		+" convert escapes " + convert_escapes
		+", append " + append);
if (! append)
	{
	//	Clear the existing message.
	Total_Lines = 0;
	Message = "";
	Last_Lines.clear ();
	}
if (message != null &&
	message.length () != 0)
	{
	if (convert_escapes)
		message = new String_Buffer (message).escape_to_special ().toString ();
	Append_Message (message);
	}
if ((DEBUG & DEBUG_MESSAGE) != 0)
	System.out.println ("<<< Message");
return this;
}

/**	Sets the message to be delivered.
<p>
	Any escape sequences are expanded. Any existing message is cleared.
<p>
	@param	message	The message String to be delivered. A null message
		is empty.
	@return	This Notify object.
	@see	#Message(String, boolean, boolean)
*/
public Notify Message
	(
	String	message
	)
{return Message (message, true, false);}

/**	Append lines from a String to the Message.
<p>
	@param	string	The String from which to extract lines.
	@return	The string index of the last character (exclusive) used.
		This will be 0 if the Message is already full (Total_Lines >=
		Max_Message_Lines). It will be -1 if the entire string was used.
*/
private void Append_Message
	(
	String	string
	)
{
if ((DEBUG & DEBUG_MESSAGE) != 0)
	System.out.println
		(">>> Append_Message: " + Total_Lines + " total lines so far" + NL
		+"    Max message lines - " + Max_Message_Lines + NL
		+"    Max last lines - " + Max_Last_Lines);
int
	last = 0,
	last_line_threshold = Max_Message_Lines - Max_Last_Lines,
	NL_length = NL.length ();
if (Total_Lines < last_line_threshold)
	{
	if ((DEBUG & DEBUG_MESSAGE) != 0)
		System.out.println
			("--- MESSAGE");
	//	Count message lines.
	for (last = string.indexOf (NL);
		 last >= 0 &&
			++Total_Lines < last_line_threshold;
		 last = string.indexOf (NL, last + NL_length));
	if (last < 0)
		//	Take the entire message.
		Message += string;
	else
		{
		//	Include the final NL.
		last += NL_length;
		if (last == string.length ())
			{
			Message += string;
			last = -1;
			}
		else
			Message += string.substring (0, last);
		}
	if ((DEBUG & DEBUG_MESSAGE) != 0)
		System.out.println
			("--- MESSAGE: " + Total_Lines + " total lines");
	}

if (last >= 0)
	{
	if ((DEBUG & DEBUG_MESSAGE) != 0)
		System.out.println
			("--- LAST LINES: " + Last_Lines.size ());
	int
		start = last;
	for (last = string.indexOf (NL, start);
		 last >= 0;
		 last = string.indexOf (NL, start))
		{
		//	Add to last lines list.
		if (Last_Lines.size () == Max_Last_Lines)
			//	Scroll down.
			Last_Lines.remove (0);
		Last_Lines.add (string.substring (start, last += NL_length));
		++Total_Lines;
		start = last;
		}
	if (start != string.length ())
		{
		//	Add unterminated line to last lines list.
		if (Last_Lines.size () == Max_Last_Lines)
			//	Scroll down.
			Last_Lines.remove (0);
		Last_Lines.add (string.substring (start));
		++Total_Lines;
		}
	if ((DEBUG & DEBUG_MESSAGE) != 0)
		System.out.println
			("--- LAST LINES: " + Last_Lines.size ());
	}
if ((DEBUG & DEBUG_MESSAGE) != 0)
	System.out.println
		("<<< Append_Message: " + Total_Lines + " total lines so far");
}

/*==============================================================================
	Delivery methods
*/
/**	Delivers the notice to each destination.
<p>
	The list of undeliverable destinations (destinations for which, for
	whatever reason, the delivery could not be made) is cleared prior
	to delivery.
	
	@return	The number of successful destinations.
	@see	#Undeliverable()
*/
public int Deliver ()
{
if ((DEBUG & DEBUG_PROCESSING) != 0)
	System.out.println
		(">>> Deliver:" + NL
		+"    To: " + Destinations + NL
		+"    From: " + From + NL
		+"    Reply_To: " + Reply_To + NL
		+"    Subject: " + Subject + NL
		+"    Size of Destinations: " + Destinations.size ());
		
Undeliverable.clear ();
Iterator
	list = Destinations.iterator ();
while (list.hasNext ())
	{
	String
		destination = (String)list.next ();
	URI
		uri = null;
	try {uri = new URI (destination);}
	catch (URISyntaxException e)
		{
		Undeliverable.add (destination);
		continue;
		}
	URL
		url = null;
	try {url = uri.toURL ();}
	catch (Exception e)
		{
		//	Attempt it as a mailto: URL
		try {url = new URL ("mailto:" + destination);}
		catch (Exception e_2)
			{
			Undeliverable.add (destination);
			continue;
			}
		}
	Deliver_To (url);
	}
if ((DEBUG & DEBUG_PROCESSING) != 0)
	System.out.println
		("<<< Deliver");
return Destinations.size () - Undeliverable.size ();
}

/**	Delivers the notice to each destination.
<p>
	@param	destinations	The Vector of desinations.
	@return	The number of successful destinations.
	@see	#Deliver()
*/
public int Deliver
	(
	Vector	destinations
	)
{
Destinations (destinations);
return Deliver ();
}

/**	Delivers the message to a single destination.
<p>
	The destination email address is obtained from the URL. A {@link
	javax.mail.MimeMessage} is created from the obtained address. The
	email is then sent via a {@link javax.mail.Session} object created
	from the set of Mail_Props Properties. If the mail can not be sent
	for any reason, the destination is added to the list of undeliverable
	destinations along with the reason for the failure.
<p>
	Depending on the type of URL, additional information may be
	written. For a mailto URL, for example, a To address, a From
	address, a Reply-To address, and a Subject line may be added.

	@param	url	The destination URL.
	@see	#Deliver()
	@see	#Undeliverable()
*/
private void Deliver_To
	(
	URL		url
	)
{
if ((DEBUG & (DEBUG_PROCESSING | DEBUG_MESSAGE)) != 0)
	System.out.println
		(">>> Deliver_To: " + url + NL);
if ((DEBUG & DEBUG_MESSAGE) != 0)
	{
	System.out.println
		("    To: " + url.getPath () + NL
		+"    From: " + From + NL
		+((Reply_To != null) ?
			("    Reply-To: " + Reply_To + NL) : "")
		+((Subject != null) ?
			("    Subject: " + Subject + NL) : "")
		+"    Message -" + NL
		+ Message () + NL
		+"<<< Deliver_To");
	return;
	}

Session
	mail_session = Session.getInstance (Mail_Props, null);
MimeMessage
	mail_message = new MimeMessage (mail_session);
try 
	{
	mail_message.setRecipients (MimeMessage.RecipientType.TO, url.getPath ());
	mail_message.setFrom ();
	if (Reply_To != null)
		mail_message.setReplyTo
			(new InternetAddress [] {new InternetAddress (Reply_To)});
	if (Subject != null)
		mail_message.setSubject (Subject);
	mail_message.setSentDate (new Date ());
	mail_message.setContent (Message (), "text/plain");
	
	Transport.send (mail_message);
	} 
catch (MessagingException exception) 
	{
	String
		error_message =
			"Mail sent to " + url.getPath () + " failed." + NL
			+ "Reason: " + exception.getMessage ();
	Undeliverable.add (error_message);
	}

if ((DEBUG & (DEBUG_PROCESSING | DEBUG_MESSAGE)) != 0)
	System.out.println
		("<<< Deliver_To");
}

/*==============================================================================
	Helpers
*/
/**	Removes redundant entries from a Vector.

	@param	vector	The Vector to be vetted.
	@return	A copy of the vector without redundant entries.
*/
private Vector Remove_Redundancies
	(
	Vector	vector
	)
{
if (vector == null ||
	vector.isEmpty () ||
	vector.size () == 1)
	return vector;
Vector
	copy = new Vector (vector.size ());
Iterator
	entries = vector.iterator ();
while (entries.hasNext ())
	{
	Object
		object = entries.next ();
	if (! copy.contains (object))
		copy.add (object);
	}
return copy;
}

/*==============================================================================
	Application main
*/
/**	Processes the command line arguments.
<p>
	All error messages are written to stderr. If any destinations
	are undeliverable, they are listed.
<p>
	The exit status will be one of:
<ul>
	<li>0 - SUCCESS
	<li>1 - ILLEGAL_SYNTAX
	<li>2 - NO_DESTINATION
	<li>3 - CANNOT_READ_MESSAGE_FILE
</ul>
<p>
	@param	args	The array of command line arguments.
	@see	#Usage(int)
*/
public static void main
	(
	String[]	args
	)
{
if ((DEBUG & DEBUG_SETUP) != 0)
	System.out.println
		("*** " + ID + " ***");
Notify
	notifier			= new Notify ();
Vector
	destinations		= new Vector ();
String
	smtp_server			= null,
	message				= null,
	from_address		= null,
	reply_to_address	= null,
	subject_line		= null;

for (int index = 0;
		 index < args.length;
		 index++)
	{
	if (args[index].length () > 1 &&
		args[index].charAt (0) == '-')
		{
		switch (args[index].charAt (1))
			{
			case 'T':	//	To
			case 't':
			case 'D':	//	Destination
			case 'd':
				if (++index == args.length ||
					args[index].charAt (0) == '-')
					{
					System.err.println
						("Missing To address.");
					Usage (ILLEGAL_SYNTAX);
					}
				//	Break down comma separated list.
				for (int
						from = 0,
						to = args[index].indexOf (',', from);
						from < args[index].length ();
						to = args[index].indexOf (',', from))
					{
					if (to < 0)
						to = args[index].length ();
					if (from < to)
						destinations.add (args[index].substring (from, to));
					from = ++to;
					}
				break;
			case 'F':	//	From
			case 'f':
				if (++index == args.length ||
					args[index].charAt (0) == '-')
					{
					System.err.println
						("Missing From address.");
					Usage (ILLEGAL_SYNTAX);
					}
				notifier.From (args[index]);
				break;
			case 'O':
			case 'o':
				if (++index == args.length || 
					args[index].charAt(0) == '-')
					{
					System.err.println ("Missing Outgoing hostname.");
					Usage (ILLEGAL_SYNTAX);
					}
				notifier.SMTP_Server (args[index]);
				break;				
			case 'R':	//	Reply-to
			case 'r':
				if (++index == args.length ||
					args[index].charAt (0) == '-')
					{
					System.err.println
						("Missing Reply-To address.");
					Usage (ILLEGAL_SYNTAX);
					}
				notifier.Reply_To (args[index]);
				break;
			case 'S':	//	Subject
			case 's':
				if (++index == args.length ||
					args[index].charAt (0) == '-')
					{
					System.err.println
						("Missing Subject line.");
					Usage (ILLEGAL_SYNTAX);
					}
				notifier.Subject (args[index]);
				break;
			case 'M':	//	Message
			case 'm':
				if (++index == args.length ||
					args[index].charAt (0) == '-')
					{
					System.err.println
						("Missing message.");
					Usage (ILLEGAL_SYNTAX);
					}
				if (args[index].length () > 1 &&
					args[index].charAt (0) == FILENAME_MARKER)
					{
					try {notifier.Message_Source (args[index].substring (1));}
					catch (IOException exception)
						{
						System.err.println (exception.getMessage ());
						System.exit (CANNOT_READ_MESSAGE_FILE);
						}
					}
				else
					notifier.Message (args[index]);
				break;
			case 'L':	//	Limit
			case 'l':
				if (++index == args.length ||
					args[index].charAt (0) == '-')
					{
					System.err.println
						("Missing message limit value.");
					Usage (ILLEGAL_SYNTAX);
					}
				int
					limit = 0;
				try {limit = Integer.parseInt (args[index]);}
				catch (NumberFormatException exception)
					{
					System.err.println
						("Invalid limit value: " + args[index]);
					Usage (ILLEGAL_SYNTAX);
					}
				if (args[index - 1].indexOf ('L', 2) > 1 ||
					args[index - 1].indexOf ('l', 2) > 1)
					notifier.Max_Last_Lines (limit);
				else
					notifier.Max_Message_Lines (limit);
				break;
			case 'V':	//	Version
			case 'v':
				System.out.println (ID);
				System.exit (SUCCESS);
			case 'H':	//	Help
			case 'h':
				Usage (SUCCESS);
			default:
				System.err.println
					("Unrecognized option: " + args[index]);
				Usage (ILLEGAL_SYNTAX);
			}
		}
	else
		destinations.add (args[index]);
	}
if (destinations.isEmpty ())
	{
	System.err.println
		("No To address provided.");
	Usage (NO_DESTINATION);
	}
notifier.Destinations (destinations);

if ((DEBUG & DEBUG_SETUP) != 0)
	System.out.println ("Delivering -" + NL
		+ "To " + destinations.size () + " Destinations: " + destinations + NL
		+ "From: " + notifier.From () + NL
		+ "Reply-To: " + notifier.Reply_To () + NL
		+ "Subject: " + notifier.Subject () + NL
		+ "Message:" + NL
		+ notifier.Message ());
notifier.Deliver ();

Vector
	undeliverable = notifier.Undeliverable ();
if (! undeliverable.isEmpty ())
	{
	System.err.println
		("Undeliverable destination"
			+ (undeliverable.size () > 1 ? "s" : "") + " -");
	Iterator
		list = undeliverable.iterator ();
	while (list.hasNext ())
		System.err.println (list.next ());
	}
System.exit (SUCCESS);
}

/**	Prints the command line usage.
<p>
<blockquote><pre>
Usage: <b>Notify</b> &lt;<i>Switches</i>&gt;
&nbsp;&nbsp;Switches -
&nbsp;&nbsp;&nbsp;&nbsp;[<b>-<u>T</u>o</b>] &lt;<i>address | URI</i>&gt;[<b>,</b>...]
&nbsp;&nbsp;&nbsp;&nbsp;[<b>-<u>F</u>rom</b> &lt;<i>address</i>&gt;] (default: username)
&nbsp;&nbsp;&nbsp;&nbsp;[<b>-<u>R</u>eply-to</b> &lt;<i>address</i>&gt;]
&nbsp;&nbsp;&nbsp;&nbsp;[<b>-<u>S</u>ubject</b> &lt;<i>subject line</i>&gt;]
&nbsp;&nbsp;&nbsp;&nbsp;[<b>-<u>M</u>essage</b> &lt;<i>message</i>&gt; | <b>@</b>&lt;<i>source</i>&gt;]
&nbsp;&nbsp;&nbsp;&nbsp;[<b>-<u>L</u>imit</b>[<b>_<u>L</u>ast</b>] &lt;<i>lines</i>&gt; (default: message 1000, last 50)
&nbsp;&nbsp;&nbsp;&nbsp;[<b>-<u>O</u>utgoing_server</b>] &lt;<i>hostname</i>&gt; (default: localhost)
&nbsp;&nbsp;&nbsp;&nbsp;[<b>-<u>V</u>ersion</b>
&nbsp;&nbsp;&nbsp;&nbsp;[<b>-<u>H</u>elp</b>
</pre></blockquote>
<p>
	At least one destination (To) address must be provided. Each
	destination is treated as a URI. If the address does not specify a
	protocol, it is treated as a mailto URI. A comma separated list of
	addresses may be provided.
<p>
	The From and Reply-to addresses are optional and only used in
	association with mailto destinations. By default the user's
	address, if available, will be used for the From address. If no
	Reply-to address is provided, none will be used.
<p>
	The Subject line is optional and only used in association with
	mailto destinations. If none is provided, none will be used.
<p>
	To message may be any text. If the message text begins with the
	{@link #FILENAME_MARKER} ('@') the text without the marker is
	interpreted as a file or URL resource specifiction. This may be a
	local resource, either a system filename or path within an enclosing
	jar file, or it may refer to an external resource via a URL.
<p>
	Multiple messages may be specified. They will be concatenated in
	the order they occur on the command line.
<p>
	The size of the message may be limited to a maximum number of lines.
	Excess lines will be dropped but the last lines of the message - the
	number may be specified - will be retained.
<p>
	The hostname of an SMTP server to which the message will be transmitted
	may be specified. The default hostname is "localhost".
<p>
	<b>N.B.</b>: Command line options are applied in the order in which
	they are encountered. Thus, for example, message limit values should
	be set before the message body is specified.
<p>
	The -Version option causes the class {@link #ID} to be written to
	stdout followed by a System.exit.
<p>
	<b>N.B.</b>: This method always results in a System.exit.

	@param	exit_status	The exit status code. The usage statement is
		written to the stdout if the exit_status is {@link #SUCCESS},
		otherwise to stderr.
*/
public static void Usage
	(
	int		exit_status
	)
{
String
	usage =
		 "Usage: Notify" + NL
		+"    [-To] <address> [...]" + NL
		+"    [-From <address>] (default: username)" + NL
		+"    [-Reply-to <address>]" + NL
		+"    [-Subject <subject line>]" + NL
		+"    [-Message <text> | @<source>]" + NL
		+"    [-Limit[_Last] <lines>] (default: message 1000, last 50)" + NL
		+"    [-Outgoing_server <hostname>] (default: localhost)" + NL
		+"    [-Version]" + NL
		+"    [-Help]";
if (exit_status == SUCCESS)
	System.out.println (usage);
else
	System.err.println (usage);

System.exit (exit_status);
}

}
