/* Copyright (c) 2001-2010, David A. Clunie DBA Pixelmed Publishing. All rights reserved. */

package com.pixelmed.dose;

import com.pixelmed.anatproc.CTAnatomy;
import com.pixelmed.anatproc.DisplayableAnatomicConcept;

import com.pixelmed.dicom.*;

import com.pixelmed.doseocr.ExposureDoseSequence;
import com.pixelmed.doseocr.OCR;

import com.pixelmed.utils.FileUtilities;

import java.io.File;
import java.io.IOException;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.TreeSet;
import java.util.Vector;

public class CTIrradiationEventDataFromImages {

	protected ArrayList<String> doseScreenFilenames = new ArrayList<String>();
	
	public ArrayList<String> getDoseScreenFilenames() { return doseScreenFilenames; }
	
	protected Set<String> irradiationEventUIDs = new TreeSet<String>();
	
	protected Map<String,String> imageTypeByEvent = new TreeMap<String,String>();
	boolean imageTypeByEventIsClean = true;
	
	protected Map<String,String> seriesNumberByEvent = new TreeMap<String,String>();
	boolean seriesNumberByEventIsClean = true;
	
	protected Map<String,String> seriesDescriptionByEvent = new TreeMap<String,String>();
	boolean seriesDescriptionByEventIsClean = true;
		
	protected Map<String,String> exposureTimeByEvent = new TreeMap<String,String>();
	boolean exposureTimeByEventIsClean = true;
	
	protected Map<String,String> kvpByEvent = new TreeMap<String,String>();
	boolean kvpByEventIsClean = true;
	
	protected Map<String,String> tubeCurrentByEvent = new TreeMap<String,String>();
	boolean tubeCurrentByEventIsClean = true;
	
	protected Map<String,String> exposureTimePerRotationByEvent = new TreeMap<String,String>();
	boolean exposureTimePerRotationByEventIsClean = true;
	
	protected Map<String,String> nominalSingleCollimationWidthInMMByEvent = new TreeMap<String,String>();
	boolean nominalSingleCollimationWidthInMMByEventIsClean = true;
	
	protected Map<String,String> nominalTotalCollimationWidthInMMByEvent = new TreeMap<String,String>();
	boolean nominalTotalCollimationWidthInMMByEventIsClean = true;
	
	protected Map<String,String> pitchFactorByEvent = new TreeMap<String,String>();
	boolean pitchFactorByEventIsClean = true;
	
	protected Map<String,CodedSequenceItem> anatomyByEvent = new TreeMap<String,CodedSequenceItem>();
	boolean anatomyByEventIsClean = true;
	
	protected Map<String,String> startAcquisitionDateTimeByEvent = new TreeMap<String,String>();
	protected Map<String,String> endAcquisitionDateTimeByEvent = new TreeMap<String,String>();
	
	protected Map<String,Double> lowestSliceLocationByEvent = new TreeMap<String,Double>();
	protected Map<String,Double> highestSliceLocationByEvent = new TreeMap<String,Double>();
	
	protected Map<String,CTAcquisitionParameters> acquisitionParametersBySeriesNumberScanRangeKey = null;

	protected String overallEarliestAcquisitionDateTime;
	protected String overallLatestAcquisitionDateTime;
	
	public String getOverallEarliestAcquisitionDateTime() { return overallEarliestAcquisitionDateTime; }
	public String getOverallLatestAcquisitionDateTime()   { return overallLatestAcquisitionDateTime; }
	
	protected static boolean putCodedSequenceItemByStringIndexIfNotDifferentElseFlagAsUnclean(Map<String,CodedSequenceItem> map,String key,CodedSequenceItem newValue) {
		boolean clean = true;
		if (newValue != null) {
			CodedSequenceItem existingValue = map.get(key);
			if (existingValue == null) {
				map.put(key,newValue);
			}
			else {
				// already there
				if (!existingValue.equals(newValue)) {
					clean=false;
				}
				// else do nothing ... is same so OK
			}
		}
		return clean;
	}
	
	protected static void putNumericStringValueByStringIndexIfNumericSortIsEarlier(Map<String,Double> map,String key,String newValueString) {
		if (newValueString != null && !newValueString.equals("")) {
			try {
				Double newValue = new Double(newValueString);
				Double existingValue = map.get(key);
				if (existingValue == null) {
					map.put(key,newValue);
				}
				else {
					if (newValue.compareTo(existingValue) < 0) {
						map.put(key,newValue);
					}
				}
			}
			catch (NumberFormatException e) {
				// do nothing
			}
		}
	}
	
	protected static void putNumericStringValueByStringIndexIfNumericSortIsLater(Map<String,Double> map,String key,String newValueString) {
		if (newValueString != null && !newValueString.equals("")) {
			try {
				Double newValue = new Double(newValueString);
				Double existingValue = map.get(key);
				if (existingValue == null) {
					map.put(key,newValue);
				}
				else {
					if (newValue.compareTo(existingValue) > 0) {
						map.put(key,newValue);
					}
				}
			}
			catch (NumberFormatException e) {
				// do nothing
			}
		}
	}
	
	protected static void putStringValueByStringIndexIfLexicographicSortIsEarlier(Map<String,String> map,String key,String newValue) {
		if (newValue != null && !newValue.equals("")) {
			String existingValue = map.get(key);
			if (existingValue == null || existingValue.equals("")) {
				map.put(key,newValue);
			}
			else {
				if (newValue.compareTo(existingValue) < 0) {
					map.put(key,newValue);
				}
			}
		}
	}
	
	protected static void putStringValueByStringIndexIfLexicographicSortIsLater(Map<String,String> map,String key,String newValue) {
		if (newValue != null && !newValue.equals("")) {
			String existingValue = map.get(key);
			if (existingValue == null || existingValue.equals("")) {
				map.put(key,newValue);
			}
			else {
				if (newValue.compareTo(existingValue) > 0) {
					map.put(key,newValue);
				}
			}
		}
	}
	
	protected static boolean putStringValueByStringIndexIfNotDifferentElseFlagAsUnclean(Map<String,String> map,String key,String newValue) {
		boolean clean = true;
		if (newValue != null && !newValue.equals("")) {
			String existingValue = map.get(key);
			if (existingValue == null || existingValue.equals("")) {
				map.put(key,newValue);
			}
			else {
				// already there
				if (!existingValue.equals(newValue)) {
					clean=false;
				}
				// else do nothing ... is same so OK
			}
		}
		return clean;
	}
	
	public CTAcquisitionParameters getAcquisitionParameters(String seriesNumberScanRangeKey) {
		if (acquisitionParametersBySeriesNumberScanRangeKey == null) {
			acquisitionParametersBySeriesNumberScanRangeKey = new TreeMap<String,CTAcquisitionParameters>();
			for (String uid : irradiationEventUIDs) {
				String useSeriesNumber = seriesNumberByEventIsClean ? seriesNumberByEvent.get(uid) : "";
				String useExposureTime = exposureTimeByEventIsClean ? exposureTimeByEvent.get(uid) : "";
				String useKVP = kvpByEventIsClean ? kvpByEvent.get(uid) : "";
				String useTubeCurrent = tubeCurrentByEventIsClean ? tubeCurrentByEvent.get(uid) : "";
				String useExposureTimePerRotation = exposureTimePerRotationByEventIsClean ? exposureTimePerRotationByEvent.get(uid) : "";
				CodedSequenceItem useAnatomy = anatomyByEventIsClean ? anatomyByEvent.get(uid) : null;
				
				String useNominalSingleCollimationWidthInMM = nominalSingleCollimationWidthInMMByEventIsClean ?  nominalSingleCollimationWidthInMMByEvent.get(uid) : "";
				String useNominalTotalCollimationWidthInMM = nominalTotalCollimationWidthInMMByEventIsClean ? nominalTotalCollimationWidthInMMByEvent.get(uid) : "";
				String usePitchFactor = pitchFactorByEventIsClean ? pitchFactorByEvent.get(uid) : "";

				String useStartLocation = getLocationAsString(highestSliceLocationByEvent.get(uid));
				String useEndLocation   = getLocationAsString(lowestSliceLocationByEvent.get(uid));
				String key = useSeriesNumber+"+"+useStartLocation+"+"+useEndLocation;
				if (!key.equals("++")) {
					CTAcquisitionParameters ap = new CTAcquisitionParameters(uid,useAnatomy,useExposureTime,null/*scanningLength is filled in later from DLP/CTDIvol*/,
						useNominalSingleCollimationWidthInMM,useNominalTotalCollimationWidthInMM,usePitchFactor,
						useKVP,useTubeCurrent,useExposureTimePerRotation);
//System.err.println("CTIrradiationEventDataFromImages.getAcquisitionParameters(): adding key="+key+" with parameters="+ap);
					acquisitionParametersBySeriesNumberScanRangeKey.put(key,ap);
				}
			}
		}
		CTAcquisitionParameters ap = acquisitionParametersBySeriesNumberScanRangeKey.get(seriesNumberScanRangeKey);
//System.err.println("CTIrradiationEventDataFromImages.getAcquisitionParameters(): looking for key="+seriesNumberScanRangeKey+" found parameters="+ap);
		return ap;
	}
	
	public CTIrradiationEventDataFromImages(String path) {
		add(path);
	}
	
	public CTIrradiationEventDataFromImages(Vector<String> paths) {
		for (int j=0; j< paths.size(); ++j) {
			add(paths.get(j));
		}
	}
	
	public void add(String path) {
		add(new File(path));
	}
		
	public void add(File file) {
		if (file.exists()) {
			if (file.isDirectory()) {
				ArrayList<File> files = FileUtilities.listFilesRecursively(file);
				for (File f : files) {
					add(f);
				}
			}
			else if (file.isFile() && file.getName().toUpperCase().equals("DICOMDIR")) {
//System.err.println("Doing DICOMDIR from "+file);
				try {
					AttributeList list = new AttributeList();
					list.read(file.getCanonicalPath());
					DicomDirectory dicomDirectory = new DicomDirectory(list);
					HashMap allDicomFiles = dicomDirectory.findAllContainedReferencedFileNamesAndTheirRecords(file.getParentFile().getCanonicalPath());
//System.err.println("Referenced files: "+allDicomFiles);
					Iterator it = allDicomFiles.keySet().iterator();
					while (it.hasNext()) {
						String doFileName = (String)it.next();
						if (doFileName != null) {
							add(doFileName);
						}
					}
				}
				catch (IOException e) {
					e.printStackTrace(System.err);
				}
				catch (DicomException e) {
					e.printStackTrace(System.err);
				}
			}
			else if (file.isFile() && DicomFileUtilities.isDicomOrAcrNemaFile(file)) {
				try {
					AttributeList list = new AttributeList();
					list.read(file.getCanonicalPath(),null,true,true,TagFromName.PixelData);
//System.err.println("Doing file "+file);
					String irradiationEventUID = "";
					if (OCR.isGEDoseScreenInstance(list)
					 || ExposureDoseSequence.isPhilipsDoseScreenInstance(list)
					) {
//System.err.println("Found dose screen in file "+file);
						doseScreenFilenames.add(file.getCanonicalPath());
					}
					else {
						irradiationEventUID = Attribute.getSingleStringValueOrEmptyString(list,TagFromName.IrradiationEventUID);
						if (irradiationEventUID.equals("")) {
							irradiationEventUID = Attribute.getSingleStringValueOrEmptyString(list,TagFromName.SeriesInstanceUID);
//System.err.println("Found irradiationEventUID from SeriesInstanceUID "+irradiationEventUID);
						}
					}
					
					if (!irradiationEventUID.equals("")) {
						irradiationEventUIDs.add(irradiationEventUID);
						
						imageTypeByEventIsClean = putStringValueByStringIndexIfNotDifferentElseFlagAsUnclean(imageTypeByEvent,irradiationEventUID,Attribute.getDelimitedStringValuesOrEmptyString(list,TagFromName.ImageType));
						seriesNumberByEventIsClean = putStringValueByStringIndexIfNotDifferentElseFlagAsUnclean(seriesNumberByEvent,irradiationEventUID,Attribute.getDelimitedStringValuesOrEmptyString(list,TagFromName.SeriesNumber));
						seriesDescriptionByEventIsClean = putStringValueByStringIndexIfNotDifferentElseFlagAsUnclean(seriesDescriptionByEvent,irradiationEventUID,Attribute.getDelimitedStringValuesOrEmptyString(list,TagFromName.SeriesDescription));
						
						{
							String exposureTimeInMilliSeconds = Attribute.getSingleStringValueOrEmptyString(list,TagFromName.ExposureTime);
							if (!exposureTimeInMilliSeconds.equals("")) {
								String exposureTimeSeconds = "";
								try {
									exposureTimeSeconds = new Double(new Double(exposureTimeInMilliSeconds).doubleValue()/1000).toString();
								}
								catch (NumberFormatException e) {
									// do nothing
								}
								exposureTimeByEventIsClean = putStringValueByStringIndexIfNotDifferentElseFlagAsUnclean(exposureTimeByEvent,irradiationEventUID,exposureTimeSeconds);
							}
						}
						
						kvpByEventIsClean = putStringValueByStringIndexIfNotDifferentElseFlagAsUnclean(kvpByEvent,irradiationEventUID,Attribute.getDelimitedStringValuesOrEmptyString(list,TagFromName.KVP));
						tubeCurrentByEventIsClean = putStringValueByStringIndexIfNotDifferentElseFlagAsUnclean(tubeCurrentByEvent,irradiationEventUID,Attribute.getDelimitedStringValuesOrEmptyString(list,TagFromName.XRayTubeCurrent));

						{
							String exposureTimePerRotation = Attribute.getDelimitedStringValuesOrEmptyString(list,TagFromName.RevolutionTime);
							if (exposureTimePerRotation.equals("")) {
								if (Attribute.getDelimitedStringValuesOrEmptyString(list,new AttributeTag(0x0019,0x0010)).equals("GEMS_ACQU_01")) {
									exposureTimePerRotation = Attribute.getDelimitedStringValuesOrEmptyString(list,new AttributeTag(0x0019,0x1027));	//  Rotation Speed (Gantry Period)
								}
							}
							exposureTimePerRotationByEventIsClean = putStringValueByStringIndexIfNotDifferentElseFlagAsUnclean(exposureTimePerRotationByEvent,irradiationEventUID,exposureTimePerRotation);
						}

						{
							String nominalSingleCollimationWidth = Attribute.getDelimitedStringValuesOrEmptyString(list,TagFromName.SingleCollimationWidth);
							if (nominalSingleCollimationWidth.equals("")) {
								if (Attribute.getDelimitedStringValuesOrEmptyString(list,new AttributeTag(0x0045,0x0010)).equals("GEMS_HELIOS_01")) {
									nominalSingleCollimationWidth = Attribute.getDelimitedStringValuesOrEmptyString(list,new AttributeTag(0x0045,0x1002));	//   Macro width at ISO Center
								}
							}
							nominalSingleCollimationWidthInMMByEventIsClean = putStringValueByStringIndexIfNotDifferentElseFlagAsUnclean(nominalSingleCollimationWidthInMMByEvent,irradiationEventUID,nominalSingleCollimationWidth);

							String nominalTotalCollimationWidth = Attribute.getDelimitedStringValuesOrEmptyString(list,TagFromName.TotalCollimationWidth);
							if (nominalTotalCollimationWidth.equals("") && !nominalSingleCollimationWidth.equals("")) {
								if (Attribute.getDelimitedStringValuesOrEmptyString(list,new AttributeTag(0x0045,0x0010)).equals("GEMS_HELIOS_01")) {
									try {
										double dNumberOfMacroRowsInDetector = Attribute.getSingleDoubleValueOrDefault(list,new AttributeTag(0x0045,0x1001),0d);	//   Number of Macro Rows in Detector
										double dNominalSingleCollimationWidth = Double.valueOf(nominalSingleCollimationWidth).doubleValue();
										if (dNumberOfMacroRowsInDetector > 0 && dNominalSingleCollimationWidth > 0) {
											double dnominalTotalCollimationWidth = dNumberOfMacroRowsInDetector * dNominalSingleCollimationWidth;
											nominalTotalCollimationWidth = Double.toString(dnominalTotalCollimationWidth);
										}
									}
									catch (NumberFormatException e) {
										e.printStackTrace(System.err);
									}
								}
							}
							nominalTotalCollimationWidthInMMByEventIsClean = putStringValueByStringIndexIfNotDifferentElseFlagAsUnclean(nominalTotalCollimationWidthInMMByEvent,irradiationEventUID,nominalTotalCollimationWidth);

							String pitchFactor = Attribute.getDelimitedStringValuesOrEmptyString(list,TagFromName.SpiralPitchFactor);
							if (pitchFactor.equals("")) {
								if (Attribute.getDelimitedStringValuesOrEmptyString(list,new AttributeTag(0x0043,0x0010)).equals("GEMS_PARM_01")) {
									pitchFactor = Attribute.getDelimitedStringValuesOrEmptyString(list,new AttributeTag(0x0043,0x1027));	//   Scan Pitch Ratio in the form "n.nnn:1"
									pitchFactor = pitchFactor.trim().replace(":1","");
								}
								if (pitchFactor.equals("") && !nominalTotalCollimationWidth.equals("")) {
									// Pitch Factor: For Spiral Acquisition, the Pitch Factor is the ratio of the Table Feed per Rotation
									// to the Nominal Total Collimation Width. For Sequenced Acquisition, the Pitch Factor is the ratio
									// of the Table Feed per single sequenced scan to the Nominal Total Collimation Width.
									try {
										double dTableFeedPerRotation = Attribute.getSingleDoubleValueOrDefault(list,TagFromName.TableFeedPerRotation,0d);
										if (dTableFeedPerRotation == 0) {
											if (Attribute.getDelimitedStringValuesOrEmptyString(list,new AttributeTag(0x0045,0x0010)).equals("GEMS_ACQU_01")) {
												dTableFeedPerRotation = Attribute.getSingleDoubleValueOrDefault(list,new AttributeTag(0x0019,0x1023),0d);	// Table Speed [mm/rotation]
											}
										}
										if (dTableFeedPerRotation > 0) {
											double dNominalTotalCollimationWidth = Double.valueOf(nominalTotalCollimationWidth).doubleValue();
											if (dNominalTotalCollimationWidth > 0) {
												double dPitchFactor = dTableFeedPerRotation / dNominalTotalCollimationWidth;
												pitchFactor = Double.toString(dPitchFactor);
											}
										}
									}
									catch (NumberFormatException e) {
										e.printStackTrace(System.err);
									}
								}
							}
							pitchFactorByEventIsClean = putStringValueByStringIndexIfNotDifferentElseFlagAsUnclean(pitchFactorByEvent,irradiationEventUID,pitchFactor);
						}

						//(0x0018,0x9305) FD Revolution Time 	 VR=<FD>   VL=<0x0008>  {0.6} 
						//(0x0018,0x9306) FD Single Collimation Width 	 VR=<FD>   VL=<0x0008>  {0.625} 
						//(0x0018,0x9307) FD Total Collimation Width 	 VR=<FD>   VL=<0x0008>  {40} 
						//(0x0018,0x9309) FD Table Speed 	 VR=<FD>   VL=<0x0008>  {65.625} 
						//(0x0018,0x9310) FD Table Feed per Rotation 	 VR=<FD>   VL=<0x0008>  {39.375} 
						//(0x0018,0x9311) FD Spiral Pitch Factor 	 VR=<FD>   VL=<0x0008>  {0.984375} 

						//(0x0019,0x0010) LO PrivateCreator 	 VR=<LO>   VL=<0x000c>  <GEMS_ACQU_01> 
						//(0x0019,0x1023) DS Table Speed [mm/rotation] 	 VR=<DS>   VL=<0x000a>  <39.375000 > 
						//(0x0019,0x1027) DS Rotation Speed (Gantry Period) 	 VR=<DS>   VL=<0x0008>  <0.600000> 
						
						//(0x0043,0x0010) LO PrivateCreator 	 VR=<LO>   VL=<0x000c>  <GEMS_PARM_01>
						//(0x0043,0x1027) SH Scan Pitch Ratio 	 VR=<SH>   VL=<0x0008>  <0.984:1 > 
						
						//(0x0045,0x0010) LO PrivateCreator 	 VR=<LO>   VL=<0x000e>  <GEMS_HELIOS_01> 
						//(0x0045,0x1001) SS Number of Macro Rows in Detector 	 VR=<SS>   VL=<0x0002>  [0x0040] 
						//(0x0045,0x1002) FL Macro width at ISO Center 	 VR=<FL>   VL=<0x0004>  {0.625} 

						
						//10 >> CONTAINS NUM EV (113826, DCM, “Nominal Single Collimation Width”) 1 M  Units = EV (mm, UCUM, “mm”)
						//11 >> CONTAINS NUM EV (113827, DCM, “Nominal Total Collimation Width”) 1 M  Units = EV (mm, UCUM, “mm”)
						//12 >> CONTAINS NUM EV (113828, DCM, “Pitch Factor”) 
						
						{
							// handles midnight crossing, but not robust if one or the other is sometimes missing in the data set
							String acquisitionDateTime = Attribute.getSingleStringValueOrEmptyString(list,TagFromName.AcquisitionDateTime);
							if (acquisitionDateTime.equals("")) {
								acquisitionDateTime = Attribute.getSingleStringValueOrEmptyString(list,TagFromName.AcquisitionDate) + Attribute.getSingleStringValueOrEmptyString(list,TagFromName.AcquisitionTime);
							}
							putStringValueByStringIndexIfLexicographicSortIsEarlier(startAcquisitionDateTimeByEvent,irradiationEventUID,acquisitionDateTime);
							putStringValueByStringIndexIfLexicographicSortIsLater(endAcquisitionDateTimeByEvent,irradiationEventUID,acquisitionDateTime);
							if (overallEarliestAcquisitionDateTime == null || acquisitionDateTime.compareTo(overallEarliestAcquisitionDateTime) < 0) {
								overallEarliestAcquisitionDateTime = acquisitionDateTime;
							}
							if (overallLatestAcquisitionDateTime == null || acquisitionDateTime.compareTo(overallLatestAcquisitionDateTime) > 0) {
								overallLatestAcquisitionDateTime = acquisitionDateTime;
							}
						}
						{
							String sliceLocation = Attribute.getSingleStringValueOrEmptyString(list,TagFromName.SliceLocation).trim();
							putNumericStringValueByStringIndexIfNumericSortIsEarlier(lowestSliceLocationByEvent,irradiationEventUID,sliceLocation);
							putNumericStringValueByStringIndexIfNumericSortIsLater(highestSliceLocationByEvent,irradiationEventUID,sliceLocation);
						}
						{
							DisplayableAnatomicConcept anatomy = CTAnatomy.findAnatomicConcept(list);
							if (anatomy != null) {
								CodedSequenceItem item = anatomy.getCodedSequenceItem();
								if (item != null) {
									anatomyByEventIsClean = putCodedSequenceItemByStringIndexIfNotDifferentElseFlagAsUnclean(anatomyByEvent,irradiationEventUID,item);
								}
							}
						}
					}
				}
				catch (Exception e) {
				// probably wasn't a DICOM file after all, so don't sweat it
				}
			}
		}
	}
	
	protected static String toStringCodedSequenceItem(String name,Map<String,CodedSequenceItem> map,boolean isClean,String uid) {
		String value;
		if (isClean) {
			CodedSequenceItem item = map.get(uid);
			if (item == null) {
				value = "-- not found --";
			}
			else {
				value = item.toString();
			}
		}
		else {
			value = "-- inconsistent values for event --";
		}
		return "\t\t"+name+" = "+value+"\n";
	}
	
	protected static String toString(String name,Map<String,String> map,boolean isClean,String uid) {
		String value;
		if (isClean) {
			value = map.get(uid);
			if (value == null) {
				value = "-- not found --";
			}
		}
		else {
			value = "-- inconsistent values for event --";
		}
		return "\t\t"+name+" = "+value+"\n";
	}
	
	protected static String getLocationAsString(Double dValue) {
		String value;
		if (dValue == null) {
			value = "";
		}
		else {
			//value = dValue.toString();
			java.text.DecimalFormat formatter = (java.text.DecimalFormat)(java.text.NumberFormat.getInstance());
			formatter.setGroupingUsed(false);
			formatter.setMinimumFractionDigits(3);
			formatter.setMaximumFractionDigits(3);
			value=formatter.format(dValue.doubleValue());
			if (value.startsWith("-")) {
				value = "I"+value.substring(1);
			}
			else {
				value = "S"+value;
			}
		}
		return value;
	}
	
	protected static String toStringLocation(String name,Map<String,Double> map,String uid) {
		String value = getLocationAsString(map.get(uid));
		if (value.equals("")) {
			value = "-- not found --";
		}
		return "\t\t"+name+" = "+value+"\n";
	}
	
	public String toString() {
		StringBuffer buf = new StringBuffer();
		for (String uid : irradiationEventUIDs) {
			buf.append("\tIrradiationEventUID = "+uid+"\n");
			buf.append(toString("ImageType",imageTypeByEvent,imageTypeByEventIsClean,uid));
			buf.append(toString("SeriesNumber",seriesNumberByEvent,seriesNumberByEventIsClean,uid));
			buf.append(toString("SeriesDescription",seriesDescriptionByEvent,seriesDescriptionByEventIsClean,uid));
			buf.append(toStringCodedSequenceItem("Anatomy",anatomyByEvent,anatomyByEventIsClean,uid));
			
			buf.append(toString("StartAcquisitionDateTime",startAcquisitionDateTimeByEvent,true,uid));
			buf.append(toString("EndAcquisitionDateTime",endAcquisitionDateTimeByEvent,true,uid));
			buf.append(toStringLocation("LowestSliceLocation",lowestSliceLocationByEvent,uid));
			buf.append(toStringLocation("HighestSliceLocation",highestSliceLocationByEvent,uid));
			
			buf.append(toString("ExposureTime",exposureTimeByEvent,exposureTimeByEventIsClean,uid));
			buf.append(toString("KVP",kvpByEvent,kvpByEventIsClean,uid));
			buf.append(toString("TubeCurrent",tubeCurrentByEvent,tubeCurrentByEventIsClean,uid));
			buf.append(toString("ExposureTimePerRotation",exposureTimePerRotationByEvent,exposureTimePerRotationByEventIsClean,uid));
			
			buf.append(toString("NominalSingleCollimationWidthInMM",nominalSingleCollimationWidthInMMByEvent,nominalSingleCollimationWidthInMMByEventIsClean,uid));
			buf.append(toString("NominalTotalCollimationWidthInMM",nominalTotalCollimationWidthInMMByEvent,nominalTotalCollimationWidthInMMByEventIsClean,uid));
			buf.append(toString("PitchFactor",pitchFactorByEvent,pitchFactorByEventIsClean,uid));
		}
		return buf.toString();
	}
	
	public static final void main(String arg[]) {
		try {
			CTIrradiationEventDataFromImages eventDataFromImages = new CTIrradiationEventDataFromImages(arg[0]);
System.err.print(eventDataFromImages);
		}
		catch (Exception e) {
			e.printStackTrace(System.err);
		}
	}
	
}

