/*
 *  plex86: run multiple x86 operating systems concurrently
 *  Copyright (C) 1999-2001 Kevin P. Lawton
 *
 *  dt-mon.c: Dynamic Translation engine.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */


#include "plex86.h"
#define IN_MONITOR_SPACE
#include "monitor.h"





static unsigned dtTranslateLPAToMI(vm_t *vm, Bit32u lpi, Bit32u ppi);
static Bit32u   dtMetaLookupTcode(vm_t *vm, unsigned metaI, Bit32u gla);
static void     dtSetG2TPair(vm_t *vm, unsigned hashRow, Bit32u guestOff,
                             Bit32u tOff);
static unsigned dtGetFreeMetaEntry(vm_t *);
static void     dtFreeMetaEntry(vm_t *vm, unsigned metaI);
static void     dtSetLPAToMIPair(vm_t *vm, Bit32u lpi, unsigned metaI);


/* A function to validate that addresses within the tcode buffer
 * area are within bounds of the tcode buffer.  Since the guest has
 * read/write access to this area, we need to do this before we
 * can take for granted, that the address is OK for an access.
 */
  static inline void
dtValidateTcodeAddr(vm_t *vm, void *tcodePtr, unsigned len)
{
  Bit32u tcodeAddr = (Bit32u) tcodePtr;
  Bit32u buffAddr = (Bit32u) vm->guest.addr.tcodeChunk;

  if ( (tcodeAddr < buffAddr) ||
       ((tcodeAddr+len) > (buffAddr+SizeOfTcodeBuffer)) )
    monpanic(vm, "dtValidateTcodeAddr: fail 0x%x\n", (Bit32u) tcodeAddr);
}

  static inline void
dtValidateMetaI(vm_t *vm, unsigned metaI)
{
  if (metaI >= DTPageMetaTableN)
    monpanic(vm, "dtValidateMetaI: fail metaI(%u) >= DTPageMetaTableN.\n",
      metaI);
}

  Bit32u
dtTranslateG2T(vm_t *vm, Bit32u guestOff, Bit32u guestLinAddr,
               Bit32u guestPhyAddr)
{
  /* Translate from a guest offset to tcode offset */

  unsigned hashRow, hashCol, metaI;
  Bit32u tcodeAddr, monLinAddr;
  dtG2THash_t  *dtG2THash       = vm->guest.addr.dtG2THash;
  dtPageMeta_t *dtPageMetaTable = vm->guest.addr.dtPageMetaTable;
  Bit32u lpi, ppi;

  /*
   *  Search the G2T table first, ideally the instruction will have
   *  been translated already, and the translation address in there.
   */

  hashRow = DT_G2THashSelect(guestOff);

  if ((*dtG2THash)[hashRow][0].tOff == TcodeOffsetNone) {
    /* First entry in Row was marked empty.  This is our signal to mark the
     * rest of the entries in the row as empty.
     */
    for (hashCol = 0; hashCol < DT_G2THashWidth; hashCol++) {
      (*dtG2THash)[hashRow][hashCol].tOff = TcodeOffsetNone;
      }
    hashCol = 0; /* Signal that 0th entry is available */
    }
  else {
    for (hashCol = 0; hashCol < DT_G2THashWidth; hashCol++) {
      /* If the guest offset matches, and the tcode address is valid, then
       * return the tcode address.
       */
      if ( ((*dtG2THash)[hashRow][hashCol].gOff == guestOff) &&
           ((*dtG2THash)[hashRow][hashCol].tOff != TcodeOffsetNone) ) {
        /* xxx It might be smart to reorder this entry to the beginning
         * of the list for performance.
         */
        tcodeAddr = (*dtG2THash)[hashRow][hashCol].tOff;
dtValidateTcodeAddr(vm, (Bit8u *)tcodeAddr, 1); // xxx Could remove this
        return( tcodeAddr );
        }
      /* If we find a free entry, that's the end of the row.  Stop here.
       * The value of hashCol is useful for allocating a new pair.
       */
      if ( (*dtG2THash)[hashRow][hashCol].tOff == TcodeOffsetNone ) {
        break;
        }
      }
    }

  /* Guest offset not in hash tables.  Find the corresponding meta
   * index (if any) and look it up in the tcode for that page.
   */

  lpi = guestLinAddr >> 12;
  ppi = guestPhyAddr >> 12;
  metaI = dtTranslateLPAToMI(vm, lpi, ppi);
  if (metaI == MetaIndexNone) {
    /* LPA not in either hash table or meta table.  We need to create
     * a new meta table entry for this page.
     */

    metaI = dtGetFreeMetaEntry(vm);

    dtPageMetaTable[metaI].lpi = lpi;
    dtPageMetaTable[metaI].ppi = ppi;
    dtPageMetaTable[metaI].cs = vm->guest_cpu.desc_cache[SRegCS];
    dtPageMetaTable[metaI].cpl = G_GetCPL(vm);
    dtSetLPAToMIPair(vm, lpi, metaI);
    }
  tcodeAddr = dtMetaLookupTcode(vm, metaI, guestLinAddr);
  if (tcodeAddr == TcodeOffsetNone) {
    /* Instruction does not have associated tcode; we must translate. */
    monLinAddr = Guest2Monitor(vm, guestLinAddr);
    tcodeAddr = dtTranslateSequence(vm, metaI, guestOff, guestLinAddr,
                                    monLinAddr);
    if (tcodeAddr == TcodeOffsetNone) {
      return(tcodeAddr);
      }
    }
dtValidateTcodeAddr(vm, (Bit8u *)tcodeAddr, 1); // xxx Could remove this

  /* Return the tcode address */
  return( tcodeAddr );
}

  unsigned
dtTranslateLPAToMI(vm_t *vm, Bit32u lpi, Bit32u ppi)
{
  /* Translate from a Linear Page Address to a Meta Index */

  unsigned metaI, hashRow, hashCol, tag;
  dtL2MHash_t  *dtL2MHash            = vm->guest.addr.dtL2MHash;
  dtPageMeta_t *dtPageMetaTable      = vm->guest.addr.dtPageMetaTable;
  Bit8u        *dtPageMetaTableUsage = vm->guest.addr.dtPageMetaTableUsage;

  /* Note: lpa is the upper 20 bits; the 32bit linear address >> 12 */
  hashRow = DT_LPAToMIHash(lpi);
  tag = DT_LPAToMITag(lpi);

  if ((*dtL2MHash)[hashRow][0].metai == MetaIndexNone) {
    /* First entry in Row was marked empty.  This is our signal to mark the
     * rest of the entries in the row as empty.
     */
    for (hashCol = 1; hashCol < DT_L2MHashWidth; hashCol++) {
      (*dtL2MHash)[hashRow][hashCol].metai = MetaIndexNone;
      }
    }
  else {
    /* There are some entries in this row.  Look for a match. */
    for (hashCol = 0; hashCol < DT_L2MHashWidth; hashCol++) {
      /* If the tag matches and the meta index is valid, then return
       * the meta index.
       */
      if ( ((*dtL2MHash)[hashRow][hashCol].tag == tag) &&
           ((*dtL2MHash)[hashRow][hashCol].metai != MetaIndexNone) )
        return((*dtL2MHash)[hashRow][hashCol].metai);
      /* If we find a free entry, that's the end of the row.  Stop here.
       * The value of hashCol is useful for allocating a new pair.
       */
      if ( (*dtL2MHash)[hashRow][hashCol].metai == MetaIndexNone )
        break;
      }
    }

  /* LPA not in LPAToMI hash table.  Search meta entries for a match
   * of this LPA and processor context.  If we find a match, add it to the
   * LPAToMI hash table.  Several contexts can have mappings for a given
   * LPA, so it's important that the processor context matches also.
   */

  /* Perform a brute-force lookup in the Meta table. */
  for (metaI = 0; metaI < DTPageMetaTableN; metaI++) {
    if ( (dtPageMetaTable[metaI].ppi == ppi) &&
         (dtPageMetaTable[metaI].lpi == lpi) &&
         (dtPageMetaTableUsage[metaI>>3] & (1<<(metaI & 0x7))) &&
         (dtPageMetaTable[metaI].cpl == G_GetCPL(vm)) ) {
      Bit32u *descPtrMeta, *descPtrGuest;

      /* Compare the current CS descriptor values to those stored in
       * the page meta entry.  Ignore bits 21/20 (0 and AVL).
       */
      descPtrMeta   = (Bit32u *) &dtPageMetaTable[metaI].cs.desc;
      descPtrGuest  = (Bit32u *) &vm->guest_cpu.desc_cache[SRegCS].desc;
      if ( *descPtrMeta++ == *descPtrGuest++ ) {
        if ( (*descPtrMeta++ & 0xffcfffff) == (*descPtrGuest++ & 0xffcfffff) ) {
          /* Found a matching meta entry which is in-use.  Cache this
           * pairing in the LPA to Mi hash table so next time an efficient
           * lookup will occur.
           */
          dtSetLPAToMIPair(vm, lpi, metaI);
          return( metaI );
          }
        }
      }
    }

  return( MetaIndexNone );
}


  void
dtSetLPAToMIPair(vm_t *vm, Bit32u lpi, unsigned metaI)
{
  unsigned hashRow, hashCol, tag;
  dtL2MHash_t  *dtL2MHash = vm->guest.addr.dtL2MHash;

  hashRow = DT_LPAToMIHash(lpi);
  tag = DT_LPAToMITag(lpi);

  if ((*dtL2MHash)[hashRow][0].metai == MetaIndexNone) {
    /* First entry in Row was marked empty.  This is our signal to mark the
     * rest of the entries in the row as empty.
     */
    for (hashCol = 1; hashCol < DT_L2MHashWidth; hashCol++) {
      (*dtL2MHash)[hashRow][hashCol].metai = MetaIndexNone;
      }
    hashCol = 0; /* Signal that 0th entry is available */
    }
  else {
    /* There are some entries in this row.  Look for a match. */
    for (hashCol = 0; hashCol < DT_L2MHashWidth; hashCol++) {
      if ( (*dtL2MHash)[hashRow][hashCol].metai == MetaIndexNone )
        break;
// xxx Sanity check.
      if ( (*dtL2MHash)[hashRow][hashCol].tag == tag )
        monpanic(vm, "dtSetLPAToMIPair: duplicate tag found.\n");
      }
    }

  if (hashCol < DT_L2MHashWidth) {
    /* Row is not full; just add at next available slot */
    (*dtL2MHash)[hashRow][hashCol].tag = tag;
    (*dtL2MHash)[hashRow][hashCol].metai = metaI;
    }
  else {
    /* Row is full; xxx For now, just bump 0th entry.  Could
     * shift all to right one, or use other replacement strategy.
     */
    (*dtL2MHash)[hashRow][0].tag = tag;
    (*dtL2MHash)[hashRow][0].metai = metaI;
    }
}

  Bit32u
dtMetaLookupTcode(vm_t *vm, unsigned metaI, Bit32u gla)
{
  /* Lookup a tcode offset associated with the guest linear address,
   *   in a particular meta page.
   */

  unsigned l0Bits, l1Bits, l2Bits;
  stForwardL1Frame_t   *l1Frame;
  stForwardL2Cluster_t *l2Cluster;
  unsigned clusterI;
  dtPageMeta_t *dtPageMetaTable      = vm->guest.addr.dtPageMetaTable;

// xxx Sanity check
if (dtPageMetaTable[metaI].lpi != (gla >> 12))
  monpanic(vm, "dtMetaLookupTcode: lpi sanity check failed.\n");

  l0Bits = (gla >> 8) & 0xf; /* 4 bits */
  l1Bits = (gla >> 5) & 0x7; /* 3 bits */
  l2Bits = gla & 0x01f; /* 5 bits */

  /* The L0 frame is embedded in the PageMeta entry.  We can index it to
   * get a pointer to the L1 frame.
   */
  l1Frame = dtPageMetaTable[metaI].i2tL0[l0Bits];

  if (l1Frame) {
    /* There is an allocated L1 frame for this part of the address range.
     * Index it to get a pointer to the L2 cluster.
     */
dtValidateTcodeAddr(vm, l1Frame, sizeof(stForwardL1Frame_t));
    l2Cluster = (*l1Frame)[l1Bits];
    while (l2Cluster) {
dtValidateTcodeAddr(vm, l2Cluster, sizeof(stForwardL2Cluster_t));
      /* One or more L2 clusters have been allocated for this part of the
       * address range.  Search them for a match of the final address bits.
       */
      for (clusterI=0; clusterI<STForwardL2N; clusterI++) {
        if ( l2Cluster->element[clusterI].raw &&
            (l2Cluster->element[clusterI].fields.addr4_0 == l2Bits) ) {
          /* Found address match in sparse table for this page */
          Bit32u tcodeAddr;
          /* Address is start of tcode buffer plus offset */
          tcodeAddr = ((Bit32u) vm->guest.addr.tcodeChunk) +
            l2Cluster->element[clusterI].fields.tcodeOffset;
dtValidateTcodeAddr(vm, (Bit8u *) tcodeAddr, 1);
          return( tcodeAddr );
          }
        }
      /* No match found in this cluster, search next one in linked list */
      l2Cluster = l2Cluster->next;
      }
    }

  return(TcodeOffsetNone); /* not found */
}


  void
dtSetG2TPair(vm_t *vm, unsigned hashRow, Bit32u guestOff, Bit32u tOff)
{
  /* Add a {guest offset, tcode offset} pairing to the G2T hash table 
   * at the hash table row provided.
   */

  unsigned hashCol;
  dtG2THash_t  *dtG2THash = vm->guest.addr.dtG2THash;

dtValidateTcodeAddr(vm, (Bit8u *) tOff, 1);

  if ((*dtG2THash)[hashRow][0].tOff == TcodeOffsetNone) {
    /* First entry in Row was marked empty.  This is our signal to mark the
     * rest of the entries in the row as empty.
     */
    for (hashCol = 1; hashCol < DT_G2THashWidth; hashCol++) {
      (*dtG2THash)[hashRow][hashCol].tOff = TcodeOffsetNone;
      }
    hashCol = 0; /* Signal that 0th entry is available */
    }
  else {
    for (hashCol = 0; hashCol < DT_G2THashWidth; hashCol++) {
      if ((*dtG2THash)[hashRow][hashCol].tOff == TcodeOffsetNone) {
        break;
        }
      }
    }

  /* xxx Perhaps its smart to reorder the requested pair to the front
   * of the list, for performance.
   */
  if (hashCol < DT_G2THashWidth) {
    /* Row is not full; just add at next available slot for now. */
    (*dtG2THash)[hashRow][hashCol].gOff = guestOff;
    (*dtG2THash)[hashRow][hashCol].tOff = tOff;
    }
  else {
    /* Row is full; xxx for now, bump 0th entry. Could shift all to
     * the right one, or use other replacement strategy here.
     */
    (*dtG2THash)[hashRow][0].gOff = guestOff;
    (*dtG2THash)[hashRow][0].tOff = tOff;
    }
}


  Bit8u *
dtAddTcode(vm_t *vm, unsigned metaI, Bit8u *tcode, unsigned tcodeLen,
           tcodeSnippet_t *tcodeSnippets, unsigned tcodeSnippetsN,
           Bit32u guestOff)
{
  unsigned o, l0Bits, l1Bits, l2Bits;
  stForwardL1Frame_t   *l1Frame;
  stForwardL2Cluster_t *forwardL2Cluster;
  stReverseL2Cluster_t *reverseL2Cluster;
  unsigned clusterI;
  Bit8u *tcodePtr, *snippetPtr;
  tcodeChunk_t *tcodeChunkUsed;
  dtPageMeta_t *dtPageMetaTable = vm->guest.addr.dtPageMetaTable;

  /* Allocate space for tcode, no need to zero because we are going
   * to memcpy() it to the allocated space.
   */
  tcodePtr = dtAllocTcodeSpace(vm, metaI, tcodeLen,
                               DontZero | AtHead, &tcodeChunkUsed);
dtValidateTcodeAddr(vm, tcodePtr, tcodeLen);
  /* Copy tcode sequence to tcode buffer (chunk) area. */
  mon_memcpy(tcodePtr, tcode, tcodeLen);

  for (o=0; o<tcodeSnippetsN; o++) {
    Bit32u tcodeOffset, gOff;
    unsigned hashRow;

    /* ===============================================================
     * FORWARD mapping: create a mapping from guest instruction address
     * (page offset) to tcode address.
     */
    l0Bits = (tcodeSnippets[o].pOff >> 8) & 0xf; /* L0: 4 bits */
    l1Bits = (tcodeSnippets[o].pOff >> 5) & 0x7; /* L1: 3 bits */
    l2Bits = tcodeSnippets[o].pOff & 0x01f;      /* L2: 5 bits */
    snippetPtr = tcodePtr + tcodeSnippets[o].tcodeBuffOff;
    /* Calculate guest offset for this snippet. */
    gOff = guestOff + (tcodeSnippets[o].pOff - tcodeSnippets[0].pOff);
    /* Calculate hashRow for G2T hash table. */
    hashRow = DT_G2THashSelect(gOff);
    dtSetG2TPair(vm, hashRow, gOff, (Bit32u) snippetPtr);
  
    /* The L0 frame is embedded in the PageMeta entry.  We can index it to
     * get a pointer to the L1 frame.
     */
    l1Frame = dtPageMetaTable[metaI].i2tL0[l0Bits];
  
    if (!l1Frame) {
      /* An L1 frame does not exist.  Allocate one now. */
      l1Frame =
        dtPageMetaTable[metaI].i2tL0[l0Bits] =
          dtAllocTcodeSpace(vm, metaI, sizeof(stForwardL1Frame_t),
                            DoZero | AtTail, 0);
      }
dtValidateTcodeAddr(vm, l1Frame, sizeof(stForwardL1Frame_t));
  
    /* Index the L1 frame to get a pointer to the L2 cluster. */
    forwardL2Cluster = (*l1Frame)[l1Bits];
    if ( !forwardL2Cluster ) {
      /* An L2 cluster has not yet been allocated.  Allocate now, and
       * use the 0th entry.
       */
      forwardL2Cluster =
        (*l1Frame)[l1Bits] =
          dtAllocTcodeSpace(vm, metaI, sizeof(stForwardL2Cluster_t),
                            DoZero | AtTail, 0);
      clusterI = 0; /* Use the 0th entry in the cluster */
dtValidateTcodeAddr(vm, forwardL2Cluster, sizeof(stForwardL2Cluster_t));
      }
    else {
      /* An L2 cluster list is already allocated.  Use convenience
       * function to add an entry to it.  This function will either find
       * an available slot, or create a new cluster in the list.  It
       * returns a pointer to the cluster, and the element index.
       */
  
      /* Search through the L2 cluster linked list for a matching address. */
dtValidateTcodeAddr(vm, forwardL2Cluster, sizeof(stForwardL2Cluster_t));
      while (1) {
        for (clusterI=0; clusterI<STForwardL2N; clusterI++) {
          if ( forwardL2Cluster->element[clusterI].raw == 0 ) {
            /* Found empty slot in cluster */
            goto foundForwardL2;
            }
#if STExtraSanityChecks
          if ( forwardL2Cluster->element[clusterI].fields.addr4_0 == l2Bits ) {
            /* If we find existing tcode which maps to the 1st instruction,
             * then we have a problem, because it should have been found
             * in a search before adding.  Otherwise, we are encountering
             * a situation where we are scanning a piece of starting before
             * a previously translated sequence, and we are seeing the
             * duplicate code now.  No need to register any more forward
             * lookup offsets.
             */
            if (o==0)
              monpanic(vm, "dtAddTcode: found duplicate forward address.\n");
            goto doReverseMapping;
            }
#endif
          }
        /* A match was not found in cluster, try next in linked list. */
        if (forwardL2Cluster->next) {
          forwardL2Cluster = forwardL2Cluster->next;
dtValidateTcodeAddr(vm, forwardL2Cluster, sizeof(stForwardL2Cluster_t));
          }
        else {
          forwardL2Cluster =
            forwardL2Cluster->next =
              dtAllocTcodeSpace(vm, metaI, sizeof(stForwardL2Cluster_t),
                                DoZero | AtTail, 0);
          clusterI = 0; /* Use the 0th entry in the cluster */
dtValidateTcodeAddr(vm, forwardL2Cluster, sizeof(stForwardL2Cluster_t));
          goto foundForwardL2;
          }
        }
  
foundForwardL2:
      }
  
    tcodeOffset = ((Bit32u) snippetPtr) - ((Bit32u) vm->guest.addr.tcodeChunk);
#if STExtraSanityChecks
    if (tcodeOffset >= SizeOfTcodeBuffer) {
      monpanic(vm, "dtAddTcode: sanity check fails: tcodeOffset too big.\n");
      }
#endif
  
    /* Fill in L2 values */
    forwardL2Cluster->element[clusterI].fields.addr4_0 = l2Bits;
    forwardL2Cluster->element[clusterI].fields.attributes = 0; /* xxx for now */
    forwardL2Cluster->element[clusterI].fields.tcodeOffset = tcodeOffset;
  
doReverseMapping:
  
    /* ===============================================================
     * REVERSE mapping: create a mapping from tcode address to the
     * corresponding guest instruction address (page offset).
     * Note that sparse table to cover the FORWARD mapping (i2t)
     * spans 12-bits (4096-bytes).  It is broken up into
     * L0,L1,L2={4,3,5}-bits.  But each tcode chunk is only 8-bits (256-bytes).
     * An exception can easily yield the tcode chunk ID, and each chunk
     * lists the ownerMetaIndex in it's header.  So the sparse table for
     * the tcode chunk only needs to span 8-bits.  We can use the same
     * constructs as from the FOWARD mapping, if we view the REVERSE
     * address space as the L1 & L2 ranges: L1,L2={3,5}-bits.
     */
  
    /* Get L1 and L2 address bits */
    l1Bits = (((Bit32u) snippetPtr) >> 6) & 0x3; /* L1: 2 bits */
    l2Bits = ((Bit32u) snippetPtr) & 0x03f;      /* L2: 6 bits */
  
    /* Index the L1 frame to get a pointer to the L2 cluster. */
    reverseL2Cluster = tcodeChunkUsed->header.t2iL1[l1Bits];
    if ( !reverseL2Cluster ) {
      reverseL2Cluster =
        tcodeChunkUsed->header.t2iL1[l1Bits] =
          dtAllocTcodeSpace(vm, metaI, sizeof(stReverseL2Cluster_t),
                            DoZero | AtTail, 0);
      clusterI = 0; /* Use the 0th entry in the cluster. */
dtValidateTcodeAddr(vm, reverseL2Cluster, sizeof(stReverseL2Cluster_t));
      }
    else {
dtValidateTcodeAddr(vm, reverseL2Cluster, sizeof(stReverseL2Cluster_t));
      /* Search through the L2 cluster linked list for a matching address. */
      while (1) {
        for (clusterI=0; clusterI<STReverseL2N; clusterI++) {
          if ( reverseL2Cluster->element[clusterI].raw == 0 ) {
            /* Found empty slot in cluster */
            goto foundReverseL2;
            }
          }
        /* A match was not found in cluster, try next in linked list. */
        if (reverseL2Cluster->next) {
          reverseL2Cluster = reverseL2Cluster->next;
dtValidateTcodeAddr(vm, reverseL2Cluster, sizeof(stReverseL2Cluster_t));
          }
        else {
          reverseL2Cluster =
            reverseL2Cluster->next =
              dtAllocTcodeSpace(vm, metaI, sizeof(stReverseL2Cluster_t),
                                DoZero | AtTail, 0);
          clusterI = 0; /* Use the 0th entry in the cluster */
dtValidateTcodeAddr(vm, reverseL2Cluster, sizeof(stReverseL2Cluster_t));
          goto foundReverseL2;
          }
        }
  
  foundReverseL2:
      }
  
    /* Fill in the reverse mapping L2 values */
    reverseL2Cluster->element[clusterI].fields.addr5_0 = l2Bits;
    reverseL2Cluster->element[clusterI].fields.tcodeLen =
        tcodeSnippets[o].tcodeLen;
    reverseL2Cluster->element[clusterI].fields.pageOffset =
        tcodeSnippets[o].pOff;
    reverseL2Cluster->element[clusterI].fields.notUsed = 0;
    }
  
dtValidateTcodeAddr(vm, tcodePtr, 1);
  return( tcodePtr );
}



  void *
dtAllocTcodeSpace(vm_t *vm, unsigned metaI, unsigned size, unsigned requests,
                tcodeChunk_t **chunk)
{
  tcodeChunk_t *chunkCurrent;
  void         *dataPtr;
  dtPageMeta_t *dtPageMetaTable = vm->guest.addr.dtPageMetaTable;

  /* Bounds sanity check on size; check against biggest possible space
   * which can be allocated.
   */
  if (size > (sizeof(tcodeChunk_t) -
              sizeof(vm->guest.addr.tcodeChunk[0].header)) ) {
    monpanic(vm, "dtAllocTcodeSpace: requested space of %u bytes too big\n",
             size);
    }

  /* Find a tcode chunk with room for given data item. */
  chunkCurrent = dtPageMetaTable[metaI].tcodeChunkCurrent;
  if ( !chunkCurrent ) {
    /* No chunks allocated yet for this page; allocate one now,
     * and point both head and current pointers to it.
     */
    chunkCurrent =
      dtPageMetaTable[metaI].tcodeChunkHead =
        dtPageMetaTable[metaI].tcodeChunkCurrent =
          dtAllocTcodeChunk(vm, metaI);
dtValidateTcodeAddr(vm, chunkCurrent, sizeof(tcodeChunk_t));
    }
  else {
    int room;

dtValidateTcodeAddr(vm, chunkCurrent, sizeof(tcodeChunk_t));
    room = ( ((int)chunkCurrent->header.tail) -
             ((int)chunkCurrent->header.head) ) - 1;
    if ( room < (int) size ) {
      /* There is no room in this chunk for the tcode sequence.
       * Allocate another chunk and chain it with the current one.
       */
      chunkCurrent =
        chunkCurrent->header.next =
          dtPageMetaTable[metaI].tcodeChunkCurrent =
            dtAllocTcodeChunk(vm, metaI);
dtValidateTcodeAddr(vm, chunkCurrent, sizeof(tcodeChunk_t));
      }
    }

  if ( (chunkCurrent->header.head >= chunkCurrent->header.tail) ||
       (chunkCurrent->header.tail >= sizeof(tcodeChunk_t)) ) {
    monpanic(vm, "dtAllocTcodeSpace: head/tail OOB.\n");
    }
  /* Get pointer to storage area */
  if (requests & AtHead) {
    /* Request to place data at head - towards beginning of chunk */
    dataPtr = (void *) &chunkCurrent->raw[chunkCurrent->header.head];
    /* Advance availability index */
    chunkCurrent->header.head += size;
    }
  else {
    /* Request to place data at tail - towards end of chunk */
    dataPtr = (void *) &chunkCurrent->raw[(chunkCurrent->header.tail-size)+1];
    /* Advance availability index */
    chunkCurrent->header.tail -= size;
    }

  /* Return a pointer to the chunk used, if requested */
  if (chunk)
    *chunk = chunkCurrent;

  /* Zero structure if requested. */
  if (requests & DoZero)
    mon_memset(dataPtr, 0, size);

dtValidateTcodeAddr(vm, dataPtr, size);
  return( dataPtr );
}

  tcodeChunk_t *
dtAllocTcodeChunk(vm_t *vm, unsigned metaI)
{
  unsigned cluster, bit, usage, bitmask, chunk;
  tcodeChunk_t *tcodeChunk      = vm->guest.addr.tcodeChunk;
  dtPageMeta_t *dtPageMetaTable = vm->guest.addr.dtPageMetaTable;
  Bit8u *tcodeChunkUsage        = vm->guest.addr.tcodeChunkUsage;
  Bit8u *dtPageMetaTableUsage   = vm->guest.addr.dtPageMetaTableUsage;
  Bit64u oldestPrescanTS, oldestPrescanMetaI;
  unsigned try = 0;

  /* First pass, see if there are any free chunks.
   * Second pass, free a meta entry, then look for free chunks.
   */
loop:
  for (cluster=0; cluster<(TCodeChunkN>>3); cluster++) {
    usage = tcodeChunkUsage[cluster];
    if (usage == 0xff) {
      /* All bits are set means all chunks are in-use */
      continue;
      }
    for (bit=0; bit<8; bit++) {
      bitmask = 1<<bit;
      if ( !(usage & bitmask) ) {
        /* Found unused chunk */
        chunk = (cluster<<3) | bit;
        tcodeChunkUsage[cluster] |= bitmask; /* Mark chunk in-use */
        /* Zero allocated chunk.  xxx We could zero only the header here */
        mon_memset(&tcodeChunk[chunk], 0, sizeof(tcodeChunk_t));
        /* Mark the header as used space */
        tcodeChunk[chunk].header.head = sizeof(tcodeChunk[0].header);
        /* The free-space tail starts at the end of the chunk */
        tcodeChunk[chunk].header.tail = sizeof(tcodeChunk[0]) - 1;
        /* Set owner of chunk to corresponding meta page */
        tcodeChunk[chunk].header.ownerMetaIndex = metaI;
        return( &tcodeChunk[chunk] ); /* Return addr of allocated chunk */
        }
      }
    }

  if (try>0)
    monpanic(vm, "dtAllocTcodeChunk: dtFreeMetaEntry did not work.\n");
  /* No more free chunks.  Free an entire meta page entry and all it's
   * tcode chunks to make room.  Pass metaI as a hint, so we do not
   * free any tcode chunks related to the current page.
   */

  /* Start with 0th entry. */
  oldestPrescanTS = 0;
  oldestPrescanTS--; /* Make max value */
  oldestPrescanMetaI = MetaIndexNone;

  /* See if there are any older entries. */
  for (cluster=0; cluster<(DTPageMetaTableN>>3); cluster++) {
    for (bit=0; bit<8; bit++) {
      unsigned tempMetaI;
      tempMetaI = (cluster<<3) | bit;
      /* Don't free the current metaI. */
      if (tempMetaI == metaI)
        continue;
      bitmask = 1<<bit;
      if ( !(dtPageMetaTableUsage[cluster] & bitmask) )
        continue; /* Skip non-used meta entries. */
      if (dtPageMetaTable[tempMetaI].ts.mon_prescan < oldestPrescanTS) {
        /* Found meta entry with old timestamp. */
        oldestPrescanTS = dtPageMetaTable[tempMetaI].ts.mon_prescan;
        oldestPrescanMetaI = tempMetaI;
        }
      }
    }
  if (oldestPrescanMetaI == MetaIndexNone)
    monpanic(vm, "dtAllocTcodeChunk: no metaI found.\n");
  /* Replace the page with the oldest prescan activity. */
  dtFreeMetaEntry(vm, oldestPrescanMetaI);
  try++;
  goto loop;
}


  void
dtDumpCodePage(vm_t *vm, Bit32u ppi, phy_page_usage_t *pusage)
// xxx Passing of pusage?
{
  unsigned cluster, bit, metaI, bitmask, usage;
  dtPageMeta_t *dtPageMetaTable = vm->guest.addr.dtPageMetaTable;
  Bit8u *dtPageMetaTableUsage   = vm->guest.addr.dtPageMetaTableUsage;

  for (cluster=0; cluster<(DTPageMetaTableN>>3); cluster++) {
    usage = dtPageMetaTableUsage[cluster];
    for (bit=0; bit<8; bit++) {
      bitmask = 1<<bit;
      if ( usage & bitmask ) {
        metaI = (cluster<<3) | bit;
        if (dtPageMetaTable[metaI].ppi == ppi) {
          dtFreeMetaEntry(vm, metaI);
          /* Continue searching.  Conceivably multiple pages could be
           * maintained for a given physical page.  For example, shared
           * code between r0/r3 - each having their own CS segment and
           * other processor state signature.
           */
          }
        }
      }
    }
}

  void
dtFixContext(vm_t *vm, guest_context_t *context)
{
  unsigned chunk;
  tcodeChunk_t *tcodeBuffer = vm->guest.addr.tcodeChunk;
  stReverseL2Cluster_t *reverseL2Cluster;
  Bit32u l1Bits, l2Bits;

  if ( (context->eip >= (Bit32u) tcodeBuffer) &&
       (context->eip <  (((Bit32u)tcodeBuffer) + SizeOfTcodeBuffer)) ) {
    /* Exception from tcode buffer.  We can use the reverse lookup
     * tables to determine the offending guest instruction offset.
     * Here the eip is actually a monitor value, and the tcode buffer
     * chunks are page aligned (and thus aligned with their chunk size),
     * so we can use eip to find the L1/L2 bits.
     */
    /* xxx This / could be a >> */
    chunk = (context->eip - ((Bit32u)tcodeBuffer)) / sizeof(tcodeChunk_t);
    /* Get L1 and L2 address bits */
    l1Bits = (context->eip >> 6) & 0x3; /* L1: 2 bits */
    l2Bits = context->eip & 0x03f;      /* L2: 6 bits */
    reverseL2Cluster = tcodeBuffer[chunk].header.t2iL1[l1Bits];
    if ( !reverseL2Cluster ) {
      // xxx Since code can be expanded, might have to search backwards
      // xxx in tcode buffer space to find the beginning of this instruction.
      monpanic(vm, "dtFixContext: no reverse L2 Cluster found.\n");
      }
    else {
      unsigned clusterI, metaI;
      Bit32u guestEIP;
      Bit32u guestLinAddr;

      while (reverseL2Cluster) {
dtValidateTcodeAddr(vm, reverseL2Cluster, sizeof(stReverseL2Cluster_t));
        for (clusterI=0; clusterI<STReverseL2N; clusterI++) {
          if ( (reverseL2Cluster->element[clusterI].raw != 0) &&
               (reverseL2Cluster->element[clusterI].fields.addr5_0==l2Bits) ) {
            /* Get Page Meta index that owns this chunk, so we can find the
             * linear page address.  The offset is obtained from the L2 cluster
             * information.
             */
            metaI = tcodeBuffer[chunk].header.ownerMetaIndex;
dtValidateMetaI(vm, metaI);
            guestLinAddr = (vm->guest.addr.dtPageMetaTable[metaI].lpi << 12) |
                reverseL2Cluster->element[clusterI].fields.pageOffset;
            guestEIP = guestLinAddr - vm->guest_cpu.desc_cache[SRegCS].base;
            context->eip = guestEIP;
            return; /* OK: reverse lookup successful. */
            }
          }
        reverseL2Cluster = reverseL2Cluster->next;
        }
      monpanic(vm, "dtFixContext: no L2 match found (0x%x).\n", context->eip);
      }
    }
  else {
    monpanic(vm, "dtFixContext: exception not from tcode buffer.\n");
    }
}

  unsigned
dtGetFreeMetaEntry(vm_t *vm)
{
  unsigned metaI, cluster, bit, bitmask;
  Bit64u oldestPrescanTS, oldestPrescanMetaI;
  dtPageMeta_t *dtPageMetaTable = vm->guest.addr.dtPageMetaTable;
  Bit8u        *dtPageMetaTableUsage = vm->guest.addr.dtPageMetaTableUsage;

  /* xxx Need smarter replacement strategy code here. */

  /* Start with 0th entry. */
  oldestPrescanTS = 0;
  oldestPrescanTS--; /* Make max value */
  oldestPrescanMetaI = MetaIndexNone;

  /* See if there are any older entries. */
  for (cluster=0; cluster<(DTPageMetaTableN>>3); cluster++) {
    for (bit=0; bit<8; bit++) {
      metaI = (cluster<<3) | bit;
      bitmask = 1<<bit;
      if ( !(dtPageMetaTableUsage[cluster] & bitmask) ) {
        /* Found unused meta page, allocate and mark used. */
        dtPageMetaTableUsage[cluster] |= bitmask;
        /* Zero allocated meta page. xxx Might be able to avoid this */
        goto allocated;
        }
      else if (dtPageMetaTable[metaI].ts.mon_prescan < oldestPrescanTS) {
        /* Found meta entry with old timestamp. */
        oldestPrescanTS = dtPageMetaTable[metaI].ts.mon_prescan;
        oldestPrescanMetaI = metaI;
        }
      }
    }
  /* Replace the page with the oldest prescan activity. */
  metaI = oldestPrescanMetaI;
  if (metaI == MetaIndexNone)
    monpanic(vm, "dtGetFreeMetaEntry: no metaI found.\n");
  dtFreeMetaEntry(vm, metaI);
  /* Mark meta entry as being used again. */
  dtPageMetaTableUsage[metaI>>3] |= (1 << (metaI & 0x7));

allocated:
  /* Zero allocated meta page. xxx Might be able to avoid this */
  mon_memset(&dtPageMetaTable[metaI], 0, sizeof(dtPageMeta_t));
  dtPageMetaTable[metaI].ts.mon_created = vm_rdtsc();
  return( metaI );
}

  void
dtFreeMetaEntry(vm_t *vm, unsigned metaI)
{
  unsigned chunkID, cluster, bit, bitmask;
  tcodeChunk_t *chunkPtr, *lastChunkPtr=0;
  dtPageMeta_t *dtPageMetaTable = vm->guest.addr.dtPageMetaTable;
  tcodeChunk_t *tcodeChunk      = vm->guest.addr.tcodeChunk;
  Bit8u        *tcodeChunkUsage = vm->guest.addr.tcodeChunkUsage;
  Bit8u *dtPageMetaTableUsage   = vm->guest.addr.dtPageMetaTableUsage;
  dtL2MHash_t  *dtL2MHash       = vm->guest.addr.dtL2MHash;
  unsigned hashRow, hashCol, tag;

  /* Free entry in LPA2MI hash table. */
  hashRow = DT_LPAToMIHash(dtPageMetaTable[metaI].lpi);
  tag = DT_LPAToMITag(dtPageMetaTable[metaI].lpi);

  for (hashCol = 0; hashCol < DT_L2MHashWidth; hashCol++) {
    if ( (*dtL2MHash)[hashRow][hashCol].metai == MetaIndexNone ) {
      break; /* End of list.  No entry found. */
      }
    if ( (*dtL2MHash)[hashRow][hashCol].tag == tag ) {
      /* Found entry for this metaI.  Delete it by moving everything
       * to the left by one position, starting at this hashCol.
       */
      for ( ; hashCol < (DT_L2MHashWidth-1); hashCol++) {
        (*dtL2MHash)[hashRow][hashCol] =
          (*dtL2MHash)[hashRow][hashCol+1];
        }
      (*dtL2MHash)[hashRow][DT_L2MHashWidth-1].metai = MetaIndexNone;
      break;
      }
    }

  /* Free entries in G2T hash table, increment r3hData->hashID. */
  dtInitG2THashTable(vm);

#if 0
  /* NOTE: if we use this method, must increment r3hData->hashID. */
  {
  unsigned l0Bits, l1Bits;
  stForwardL1Frame_t   *l1Frame;
  stForwardL2Cluster_t *l2Cluster;
  unsigned clusterI;
  Bit32u linPageAddr, guestLinAddr;

  linPageAddr = dtPageMetaTable[metaI].lpi << 12;
  for (l0Bits=0; l0Bits<16; l0Bits++) {
    l1Frame = dtPageMetaTable[metaI].i2tL0[l0Bits];
    if (!l1Frame)
      continue; /* Skip empty L0 slots */
    for (l1Bits=0; l1Bits<8; l1Bits++) {
      l2Cluster = (*l1Frame)[l1Bits];
      if (!l2Cluster)
        continue; /* Skip empty L1 slots */
      while (l2Cluster) {
        for (clusterI=0; clusterI<STForwardL2N; clusterI++) {
          if ( l2Cluster->element[clusterI].raw ) {
            guestLinAddr = linPageAddr | (l0Bits<<8) | (l1Bits<<5) |
              l2Cluster->element[clusterI].fields.addr4_0;
            }
          }
        l2Cluster = l2Cluster->next;
        }
      }
    }
  }
#endif

  /* Mark meta page free. */
  dtPageMetaTableUsage[metaI>>3] &= ~(1<<(metaI&0x7));

  /* Free the tcode chunks used for this page. */
  chunkPtr = dtPageMetaTable[metaI].tcodeChunkHead;
  while (chunkPtr) {
dtValidateTcodeAddr(vm, chunkPtr, sizeof(tcodeChunk_t));
     // xxx Could be a shift rather than divide.
     chunkID  = (((Bit32u) chunkPtr) - ((Bit32u) tcodeChunk)) /
                sizeof(tcodeChunk_t);
     /* xxx Sanity checks */
     if (chunkID >= TCodeChunkN)
       monpanic(vm, "dtFreeMetaEntry: chunkID too big.\n");
     if (chunkPtr->header.ownerMetaIndex != metaI)
       monpanic(vm, "dtFreeMetaEntry: owner metaI incorrect.\n");
     cluster = chunkID >> 3;
     bit     = chunkID & 0x7;
     bitmask = (1<<bit);
     tcodeChunkUsage[cluster] &= ~bitmask;
     lastChunkPtr = chunkPtr;
     chunkPtr = chunkPtr->header.next;
     }
  /* xxx Sanity check. */
  if (lastChunkPtr != dtPageMetaTable[metaI].tcodeChunkCurrent)
    monpanic(vm, "dtFreeMetaEntry: failed sanity check.\n");
}
