/*
 *  plex86: run multiple x86 operating systems concurrently
 *  Copyright (C) 1999-2001 Kevin P. Lawton
 *
 *  tasking.c:  x86 hardware tasking
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */


#include "plex86.h"
#include "monitor.h"


/* TODO:
 *   Make loop to handle loading of data segments, rather
 *     than handle individually.
 *
 *
 *
 * Notes:
 * ======
 * Step 2: TSS descriptor is not busy TS (for IRET); GP (for JMP, CALL, INT)
 *   returns error code (Task's backlink TSS)???
 *
 * TSS selector must map to GDT
 * TSS is stored in linear address space
 * what to do with I/O Map Base
 * what to do with T flag
 * where to set CR3 and flush paging cache
 * what happens when fault occurs, with some seg regs having valid
 *   bit cleared?
 * should check validity of current TR(TSS) before writing into it
 *
 *
 *   ======================
 *   286 Task State Segment
 *   ======================
 *   dynamic item                      | hex  dec  offset
 *   0       task LDT selector         | 2a   42
 *   1       DS selector               | 28   40
 *   1       SS selector               | 26   38
 *   1       CS selector               | 24   36
 *   1       ES selector               | 22   34
 *   1       DI                        | 20   32
 *   1       SI                        | 1e   30
 *   1       BP                        | 1c   28
 *   1       SP                        | 1a   26
 *   1       BX                        | 18   24
 *   1       DX                        | 16   22
 *   1       CX                        | 14   20
 *   1       AX                        | 12   18
 *   1       flag word                 | 10   16
 *   1       IP (entry point)          | 0e   14
 *   0       SS for CPL 2              | 0c   12
 *   0       SP for CPL 2              | 0a   10
 *   0       SS for CPL 1              | 08   08
 *   0       SP for CPL 1              | 06   06
 *   0       SS for CPL 0              | 04   04
 *   0       SP for CPL 0              | 02   02
 *           back link selector to TSS | 00   00
 *
 *
 *   ======================
 *   386 Task State Segment
 *   ======================
 *   |31            16|15                    0|
 *   |I/O Map Base    |000000000000000000000|T| 64  static
 *   |0000000000000000| LDT                   | 60  static
 *   |0000000000000000| GS selector           | 5c  dynamic
 *   |0000000000000000| FS selector           | 58  dynamic
 *   |0000000000000000| DS selector           | 54  dynamic
 *   |0000000000000000| SS selector           | 50  dynamic
 *   |0000000000000000| CS selector           | 4c  dynamic
 *   |0000000000000000| ES selector           | 48  dynamic
 *   |                EDI                     | 44  dynamic
 *   |                ESI                     | 40  dynamic
 *   |                EBP                     | 3c  dynamic
 *   |                ESP                     | 38  dynamic
 *   |                EBX                     | 34  dynamic
 *   |                EDX                     | 30  dynamic
 *   |                ECX                     | 2c  dynamic
 *   |                EAX                     | 28  dynamic
 *   |                EFLAGS                  | 24  dynamic
 *   |                EIP (entry point)       | 20  dynamic
 *   |           CR3 (PDPR)                   | 1c  static
 *   |000000000000000 | SS for CPL 2          | 18  static
 *   |           ESP for CPL 2                | 14  static
 *   |000000000000000 | SS for CPL 1          | 10  static
 *   |           ESP for CPL 1                | 0c  static
 *   |000000000000000 | SS for CPL 0          | 08  static
 *   |           ESP for CPL 0                | 04  static
 *   |000000000000000 | back link to prev TSS | 00  dynamic (updated
 *                                        only when return expected)
 *
 *
 *   ==================================================
 *   Effect of task switch on Busy, NT, and Link Fields
 *   ==================================================
 *
 *   Field         jump        call/interrupt     iret
 *   ------------------------------------------------------
 *   new busy bit  Set         Set                No change
 *   old busy bit  Cleared     No change          Cleared
 *   new NT flag   No change   Set                No change
 *   old NT flag   No change   No change          Cleared
 *   new link      No change   old TSS selector   No change
 *   old link      No change   No change          No change
 *   CR0.TS        Set         Set                Set
 *
 *   Note: I checked 386, 486, and Pentium, and they all exhibited
 *         exactly the same behaviour as above.  There seems to
 *         be some misprints in the Intel docs.
 */



  void
task_switch(vm_t *vm, selector_t tss_selector,
            descriptor_cache_t *tss_cache, unsigned source)
{
  Bit32u obase32; /* base address of old TSS */
  Bit32u nbase32; /* base address of new TSS */
  Bit32u temp32, newCR3;
  Bit16u temp16, trap_word;
  selector_t cs_selector, ss_selector, ds_selector, es_selector,
             fs_selector, gs_selector, ldt_selector;
  descriptor_cache_t cs_cache, ss_cache, ds_cache, es_cache,
                     fs_cache, gs_cache, ldt_cache;
  Bit32u old_TSS_max, new_TSS_max, old_TSS_limit, new_TSS_limit;
  Bit32u newEAX, newECX, newEDX, newEBX;
  Bit32u newESP, newEBP, newESI, newEDI;
  Bit32u newEFLAGS, oldEFLAGS, newEIP;
  unsigned exception_no;
  Bit16u error_code;


  invalidate_prefetch_q();

  if (tss_cache->valid==0)
    monpanic(vm, "task_switch: tss_cache.valid=0.\n");
  cache_selector(vm, SRegCS);
  cache_selector(vm, SRegSS);
  cache_selector(vm, SRegDS);
  cache_selector(vm, SRegES);
  cache_selector(vm, SRegFS);
  cache_selector(vm, SRegGS);

  /* Discard any traps and inhibits for new context; traps will */
  /* resume upon return. */
  vm->guest_cpu.debug_trap = 0;
  vm->guest_cpu.inhibit_mask = 0;


  /* The following checks are made before calling task_switch(), for
   * JMP & CALL only.  These checks are NOT made for exceptions,
   * interrupts, & IRET
   *
   *   1) TSS DPL must be >= CPL
   *   2) TSS DPL must be >= TSS selector RPL
   *   3) TSS descriptor is not busy.  TS(for IRET); GP(for JMP, CALL, INT)
   *
   * Privilege and busy checks done in CALL, JUMP, INT, IRET
   */

  exception_no = 256; /* no exception */
  error_code   = 0;
  oldEFLAGS = read_eflags(vm);

/* +++ check for tr_cache.valid? */
  /* Gather info about old TSS */
  if (vm->guest_cpu.tr_cache.desc.type <= 3) {
    obase32 = vm->guest_cpu.tr_cache.base;
    old_TSS_max   = 43;
    old_TSS_limit = vm->guest_cpu.tr_cache.limit_scaled;
    }
  else {
    obase32 = vm->guest_cpu.tr_cache.base;
    old_TSS_max   = 103;
    old_TSS_limit = vm->guest_cpu.tr_cache.limit_scaled;
    }

  /* Gather info about new TSS */
  if (tss_cache->desc.type <= 3) { /* {1,3} */
    nbase32 = tss_cache->base;
    new_TSS_max   = 43;
    new_TSS_limit = tss_cache->limit_scaled;
    }
  else { /* tss_cache->type = {9,11} */
    nbase32 = tss_cache->base;
    new_TSS_max   = 103;
    new_TSS_limit = tss_cache->limit_scaled;
    }

  /* Task State Seg must be present, else #NP(TSS selector) */
  if (tss_cache->desc.p==0) {
    monpanic(vm, "task_switch: TSS.p == 0\n");
    exception(vm, ExceptionNP, tss_selector.raw & 0xfffc);
    }

  /* TSS must have valid limit, else #TS(TSS selector) */
  if ( tss_selector.fields.ti ||
       (tss_cache->valid==0) ||
       (new_TSS_limit < new_TSS_max) ) {
    monpanic(vm, "task_switch: TR not valid\n");
    exception(vm, ExceptionTS, tss_selector.raw & 0xfffc);
    }

  /* Check that old TSS, new TSS, and all segment descriptors */
  /* used in the task switch are paged in. */
  if (vm->guest_cpu.cr0.fields.pg) {
    /* Old TSS */
    (void) dtranslate_linear(vm, nbase32, 0, OP_WRITE);
    (void) dtranslate_linear(vm, nbase32+old_TSS_max, 0, OP_WRITE);

    /* New TSS */
    (void) dtranslate_linear(vm, nbase32, 0, OP_READ);
    (void) dtranslate_linear(vm, nbase32+new_TSS_max, 0, OP_READ);

    /* ??? fix RW above */
    /* ??? touch old/new TSS descriptors here when necessary. */
    }


  /* Need to fetch all new registers and temporarily store them. */

  if (tss_cache->desc.type <= 3) {
    access_linear(vm, nbase32 + 14, 2, 0, OP_READ, &temp16);
      newEIP = temp16; /* zero out upper word */
    access_linear(vm, nbase32 + 16, 2, 0, OP_READ, &temp16);
      newEFLAGS = temp16;

    /* incoming TSS is 16bit:
     *   - upper word of general registers is set to 0xFFFF
     *   - upper word of eflags is zero'd
     *   - FS, GS are zero'd
     *   - upper word of eIP is zero'd
     */
    access_linear(vm, nbase32 + 18, 2, 0, OP_READ, &temp16);
      newEAX = 0xffff0000 | temp16;
    access_linear(vm, nbase32 + 20, 2, 0, OP_READ, &temp16);
      newECX = 0xffff0000 | temp16;
    access_linear(vm, nbase32 + 22, 2, 0, OP_READ, &temp16);
      newEDX = 0xffff0000 | temp16;
    access_linear(vm, nbase32 + 24, 2, 0, OP_READ, &temp16);
      newEBX = 0xffff0000 | temp16;
    access_linear(vm, nbase32 + 26, 2, 0, OP_READ, &temp16);
      newESP = 0xffff0000 | temp16;
    access_linear(vm, nbase32 + 28, 2, 0, OP_READ, &temp16);
      newEBP = 0xffff0000 | temp16;
    access_linear(vm, nbase32 + 30, 2, 0, OP_READ, &temp16);
      newESI = 0xffff0000 | temp16;
    access_linear(vm, nbase32 + 32, 2, 0, OP_READ, &temp16);
      newEDI = 0xffff0000 | temp16;

    access_linear(vm, nbase32 + 34, 2, 0, OP_READ, &es_selector.raw);
    access_linear(vm, nbase32 + 36, 2, 0, OP_READ, &cs_selector.raw);
    access_linear(vm, nbase32 + 38, 2, 0, OP_READ, &ss_selector.raw);
    access_linear(vm, nbase32 + 40, 2, 0, OP_READ, &ds_selector.raw);
    access_linear(vm, nbase32 + 42, 2, 0, OP_READ, &ldt_selector.raw);

    fs_selector.raw = 0; /* use a NULL selector */
    gs_selector.raw = 0; /* use a NULL selector */
    /* No CR3 change for 286 task switch */
    newCR3 = 0;   /* keep compiler happy (not used) */
    trap_word = 0; /* keep compiler happy (not used) */
    }
  else {
    access_linear(vm, nbase32 + 0x1c, 4, 0, OP_READ, &newCR3);
    access_linear(vm, nbase32 + 0x20, 4, 0, OP_READ, &newEIP);
    access_linear(vm, nbase32 + 0x24, 4, 0, OP_READ, &newEFLAGS);
    access_linear(vm, nbase32 + 0x28, 4, 0, OP_READ, &newEAX);
    access_linear(vm, nbase32 + 0x2c, 4, 0, OP_READ, &newECX);
    access_linear(vm, nbase32 + 0x30, 4, 0, OP_READ, &newEDX);
    access_linear(vm, nbase32 + 0x34, 4, 0, OP_READ, &newEBX);
    access_linear(vm, nbase32 + 0x38, 4, 0, OP_READ, &newESP);
    access_linear(vm, nbase32 + 0x3c, 4, 0, OP_READ, &newEBP);
    access_linear(vm, nbase32 + 0x40, 4, 0, OP_READ, &newESI);
    access_linear(vm, nbase32 + 0x44, 4, 0, OP_READ, &newEDI);
    access_linear(vm, nbase32 + 0x48, 2, 0, OP_READ, &es_selector.raw);
    access_linear(vm, nbase32 + 0x4c, 2, 0, OP_READ, &cs_selector.raw);
    access_linear(vm, nbase32 + 0x50, 2, 0, OP_READ, &ss_selector.raw);
    access_linear(vm, nbase32 + 0x54, 2, 0, OP_READ, &ds_selector.raw);
    access_linear(vm, nbase32 + 0x58, 2, 0, OP_READ, &fs_selector.raw);
    access_linear(vm, nbase32 + 0x5c, 2, 0, OP_READ, &gs_selector.raw);
    access_linear(vm, nbase32 + 0x60, 2, 0, OP_READ, &ldt_selector.raw);
    access_linear(vm, nbase32 + 0x64, 2, 0, OP_READ, &trap_word);
    /* I/O Map Base Address ??? */
    }


#if 0
if (ss_descriptor.u.segment.d_b && (tss_descriptor->type<9)) {
  fprintf(stderr, "++++++++++++++++++++++++++\n");
  BX_CPU_THIS_PTR sregs[BX_SEG_REG_SS].cache.valid = 0;
invalidate_sreg_pro(vm, SRegSS, ss_selector);
  exception(ExceptionSS, ss_selector.raw & 0xfffc, 0);
  /* exception(ExceptionTS, tss_selector->value & 0xfffc, 0); */
  }
#endif


  /*
   * Step 6: If JMP or IRET, clear busy bit in old task TSS descriptor,
   *         otherwise leave set.
   */

  /* effect on Busy bit of old task */
  if ( (source==TASK_FROM_JUMP) || (source==TASK_FROM_IRET) ) {
    /* Bit is cleared */
    access_linear(vm, vm->guest_cpu.gdtr.base +
                  vm->guest_cpu.tr_selector.fields.index*8 + 4,
                  4, 0, OP_READ, &temp32);
    temp32 &= ~0x00000200;
    access_linear(vm, vm->guest_cpu.gdtr.base +
                  vm->guest_cpu.tr_selector.fields.index*8 + 4,
                  4, 0, OP_WRITE, &temp32);
    }


  /*
   * Step 7: If IRET, clear NT flag in temp image of EFLAGS, otherwise
   *         leave alone.
   */

  if (source == TASK_FROM_IRET) {
    /* NT flags in old task is cleared with an IRET */
    oldEFLAGS &= ~0x00004000;
    }


  /*
   * Step 8: Save dynamic state of old task.
   */

  if (vm->guest_cpu.tr_cache.desc.type <= 3) {
    temp16 = G_IP(vm);
        access_linear(vm, obase32 + 14, 2, 0, OP_WRITE, &temp16);
    temp16 = oldEFLAGS;
        access_linear(vm, obase32 + 16, 2, 0, OP_WRITE, &temp16);
    temp16 = G_AX(vm);
        access_linear(vm, obase32 + 18, 2, 0, OP_WRITE, &temp16);
    temp16 = G_CX(vm);
        access_linear(vm, obase32 + 20, 2, 0, OP_WRITE, &temp16);
    temp16 = G_DX(vm);
        access_linear(vm, obase32 + 22, 2, 0, OP_WRITE, &temp16);
    temp16 = G_BX(vm);
        access_linear(vm, obase32 + 24, 2, 0, OP_WRITE, &temp16);
    temp16 = G_SP(vm);
        access_linear(vm, obase32 + 26, 2, 0, OP_WRITE, &temp16);
    temp16 = G_BP(vm);
        access_linear(vm, obase32 + 28, 2, 0, OP_WRITE, &temp16);
    temp16 = G_SI(vm);
        access_linear(vm, obase32 + 30, 2, 0, OP_WRITE, &temp16);
    temp16 = G_DI(vm);
        access_linear(vm, obase32 + 32, 2, 0, OP_WRITE, &temp16);
    temp16 = vm->guest_cpu.selector[SRegES].raw;
        access_linear(vm, obase32 + 34, 2, 0, OP_WRITE, &temp16);
    temp16 = vm->guest_cpu.selector[SRegCS].raw;
        access_linear(vm, obase32 + 36, 2, 0, OP_WRITE, &temp16);
    temp16 = vm->guest_cpu.selector[SRegSS].raw;
        access_linear(vm, obase32 + 38, 2, 0, OP_WRITE, &temp16);
    temp16 = vm->guest_cpu.selector[SRegDS].raw;
        access_linear(vm, obase32 + 40, 2, 0, OP_WRITE, &temp16);
    }
  else {
    temp32 = G_EIP(vm);
        access_linear(vm, obase32 + 0x20, 4, 0, OP_WRITE, &temp32);
    temp32 = oldEFLAGS;
        access_linear(vm, obase32 + 0x24, 4, 0, OP_WRITE, &temp32);
    temp32 = G_EAX(vm);
        access_linear(vm, obase32 + 0x28, 4, 0, OP_WRITE, &temp32);
    temp32 = G_ECX(vm);
        access_linear(vm, obase32 + 0x2c, 4, 0, OP_WRITE, &temp32);
    temp32 = G_EDX(vm);
        access_linear(vm, obase32 + 0x30, 4, 0, OP_WRITE, &temp32);
    temp32 = G_EBX(vm);
        access_linear(vm, obase32 + 0x34, 4, 0, OP_WRITE, &temp32);
    temp32 = G_ESP(vm);
        access_linear(vm, obase32 + 0x38, 4, 0, OP_WRITE, &temp32);
    temp32 = G_EBP(vm);
        access_linear(vm, obase32 + 0x3c, 4, 0, OP_WRITE, &temp32);
    temp32 = G_ESI(vm);
        access_linear(vm, obase32 + 0x40, 4, 0, OP_WRITE, &temp32);
    temp32 = G_EDI(vm);
        access_linear(vm, obase32 + 0x44, 4, 0, OP_WRITE, &temp32);
    temp16 = vm->guest_cpu.selector[SRegES].raw;
        access_linear(vm, obase32 + 0x48, 2, 0, OP_WRITE, &temp16);
    temp16 = vm->guest_cpu.selector[SRegCS].raw;
        access_linear(vm, obase32 + 0x4c, 2, 0, OP_WRITE, &temp16);
    temp16 = vm->guest_cpu.selector[SRegSS].raw;
        access_linear(vm, obase32 + 0x50, 2, 0, OP_WRITE, &temp16);
    temp16 = vm->guest_cpu.selector[SRegDS].raw;
        access_linear(vm, obase32 + 0x54, 2, 0, OP_WRITE, &temp16);
    temp16 = vm->guest_cpu.selector[SRegFS].raw;
        access_linear(vm, obase32 + 0x58, 2, 0, OP_WRITE, &temp16);
    temp16 = vm->guest_cpu.selector[SRegGS].raw;
        access_linear(vm, obase32 + 0x5c, 2, 0, OP_WRITE, &temp16);
    }



  /*
   * Commit point.  At this point, we commit to the new
   * context.  If an unrecoverable error occurs in further
   * processing, we complete the task switch without performing
   * additional access and segment availablility checks and
   * generate the appropriate exception prior to beginning
   * execution of the new task.
   */


  /* Task switch clears LE/L3/L2/L1/L0 in DR7 */
  vm->guest_cpu.dr7 &= ~0x00000155;

/* effect on link field of new task */
if ( source==TASK_FROM_CALL_OR_INT ) {
  /* set to selector of old task's TSS */
  temp16 = vm->guest_cpu.tr_selector.raw;
  access_linear(vm, nbase32 + 0, 2, 0, OP_WRITE, &temp16);
  }



  /*
   * Step 9: If call or interrupt, set the NT flag in the eflags
   *         image stored in new task's TSS.  If IRET or JMP,
   *         NT is restored from new TSS eflags image. (no change)
   */

  /* effect on NT flag of new task */
  if ( source==TASK_FROM_CALL_OR_INT ) {
    newEFLAGS |= 0x4000; /* flag is set */
    }


  /*
   * Step 10: If CALL, interrupt, or JMP, set busy flag in new task's
   *          TSS descriptor.  If IRET, leave set.
   */

  if ( (source==TASK_FROM_JUMP) || (source==TASK_FROM_CALL_OR_INT) ) {
    Bit32u dword2;
    /* set the new task's busy bit */
    access_linear(vm, vm->guest_cpu.gdtr.base +
                  tss_selector.fields.index*8 + 4,
                  4, 0, OP_READ, &dword2);
    dword2 |= 0x00000200;
    access_linear(vm, vm->guest_cpu.gdtr.base +
                  tss_selector.fields.index*8 + 4,
                  4, 0, OP_WRITE, &dword2);
    }


  /*
   * Step 11: Set TS flag in the CR0 image stored in the new task TSS.
   */

  /* set TS bit in CR0 register */
  vm->guest_cpu.cr0.fields.ts = 1;


  /*
   * Step 12: Load the task register with the segment selector and
   *          descriptor for the new task TSS.
   */

  vm->guest_cpu.tr_selector = tss_selector;
  vm->guest_cpu.tr_cache    = *tss_cache;


  /*
   * Step 13: Load the new task (dynamic) state from new TSS.
   *          Any errors associated with loading and qualification of
   *          segment descriptors in this step occur in the new task's
   *          context.  State loaded here includes LDTR, CR3,
   *          EFLAGS, EIP, general purpose registers, and segment
   *          descriptor parts of the segment registers.
   */

  if (tss_cache->desc.type >= 9) {
    CR3_change(vm, newCR3); /* Tell paging unit about new cr3 value */
    }

  vm->guest_cpu.prev_eip = newEIP;
  G_EIP(vm) = newEIP;
  /* modify: ID,AC,VM,RF,NT,IOPL,OF,DF,IF,TF,SF,ZF,AF,PF,CF */
  /* +++ ignore: (VIP,VIF) */
  write_eflags(vm, newEFLAGS, 0x00277fd5);
  G_EAX(vm) = newEAX;
  G_ECX(vm) = newECX;
  G_EDX(vm) = newEDX;
  G_EBX(vm) = newEBX;
  G_ESP(vm) = newESP;
  G_EBP(vm) = newEBP;
  G_ESI(vm) = newESI;
  G_EDI(vm) = newEDI;

  /* Fill in selectors for all segment registers.  If errors
   * occur later, the selectors will at least be loaded.
   */
  vm->guest_cpu.ldtr_selector    = ldt_selector;

  /* Start out with invalid descriptor caches, fill in
   * with values only as they are validated.
   */
  vm->guest_cpu.ldtr_cache.valid = 0;
  invalidate_sreg_pro(vm, SRegES, es_selector);
  invalidate_sreg_pro(vm, SRegCS, cs_selector);
  invalidate_sreg_pro(vm, SRegSS, ss_selector);
  invalidate_sreg_pro(vm, SRegDS, ds_selector);
  invalidate_sreg_pro(vm, SRegFS, fs_selector);
  invalidate_sreg_pro(vm, SRegGS, gs_selector);


/* +++ need to test valid bit in fetch_raw_descriptor?()
 * +++ or set limit to 0 instead when LDT is loaded with
 * +++ null. ???
 */
vm->guest_cpu.ldtr_cache.limit_scaled = 0;

  /* LDTR */
  if (ldt_selector.fields.ti) {
    /* LDT selector must be in GDT */
    monprint(vm, "task_switch: bad LDT selector TI=1\n");
    exception_no = ExceptionTS;
    error_code   = ldt_selector.raw & 0xfffc;
    goto post_exception;
    }


/* ??? is LDT loaded in v8086 mode */
  if ( !IsNullSelector(ldt_selector) ) {
    Boolean good;
    good = fetch_descriptor2(vm, ldt_selector, &ldt_cache.desc);
    if (!good) {
      monprint(vm, "task_switch: bad LDT fetch\n");
      exception_no = ExceptionTS;
      error_code   = ldt_selector.raw & 0xfffc;
      goto post_exception;
      }

    descriptor2cache(vm, &ldt_cache);
    /* LDT selector of new task is valid, else #TS(new task's LDT) */
    if ( ldt_cache.valid==0 ||
         (ldt_cache.desc.type!=2) ||
         (ldt_cache.desc.type & D_S) ||
         (ldt_cache.limit_scaled<7) ) {
      monprint(vm, "task_switch: bad LDT segment\n");
      exception_no = ExceptionTS;
      error_code   = ldt_selector.raw & 0xfffc;
      goto post_exception;
      }

    /* LDT of new task is present in memory, else #TS(new tasks's LDT) */
    else if (ldt_cache.desc.p==0) {
      exception_no = ExceptionTS;
      error_code   = ldt_selector.raw & 0xfffc;
      goto post_exception;
      }
    /* All checks pass, fill in LDTR shadow cache */
    vm->guest_cpu.ldtr_cache = ldt_cache;
    }
  else {
    /* NULL LDT selector is OK, leave cache invalid */
    }

  if (V8086Mode(vm)) {
    /* load seg regs as 8086 registers */
    load_seg_reg(vm, SRegCS, cs_selector.raw);
    load_seg_reg(vm, SRegSS, ss_selector.raw);
    load_seg_reg(vm, SRegDS, ds_selector.raw);
    load_seg_reg(vm, SRegES, es_selector.raw);
    load_seg_reg(vm, SRegFS, fs_selector.raw);
    load_seg_reg(vm, SRegGS, gs_selector.raw);
    }
  else {
    /* CS */
    if ( !IsNullSelector(cs_selector) ) {
      Boolean good;
      good = fetch_descriptor2(vm, cs_selector, &cs_cache.desc);
      if (!good) {
        monprint(vm, "task_switch: bad CS fetch\n");
        exception_no = ExceptionTS;
        error_code   = cs_selector.raw & 0xfffc;
        goto post_exception;
        }
  
      descriptor2cache(vm, &cs_cache);
      /* CS descriptor AR byte must indicate code segment else #TS(CS) */
      if ( (cs_cache.valid==0) || !(cs_cache.desc.type & D_S) ||
           !(cs_cache.desc.type & D_EXECUTE) ) {
        monpanic(vm, "task_switch: CS not valid executable seg\n");
        exception_no = ExceptionTS;
        error_code   = cs_selector.raw & 0xfffc;
        goto post_exception;
        }
      /* if non-conforming then DPL must equal selector RPL else #TS(CS) */
      else if ( !(cs_cache.desc.type & D_CONFORM) &&
                (cs_cache.desc.dpl!=cs_selector.fields.rpl) ) {
        monprint(vm, "task_switch: non-conforming: CS.dpl!=CS.RPL\n");
        exception_no = ExceptionTS;
        error_code   = cs_selector.raw & 0xfffc;
        goto post_exception;
        }
      /* if conforming then DPL must be <= selector RPL else #TS(CS) */
      else if ( (cs_cache.desc.type & D_CONFORM) &&
                (cs_cache.desc.dpl>cs_selector.fields.rpl) ) {
        monprint(vm, "task_switch: conforming: CS.dpl>RPL\n");
        exception_no = ExceptionTS;
        error_code   = cs_selector.raw & 0xfffc;
        goto post_exception;
        }
      /* Code segment is present in memory, else #NP(new code segment) */
      else if (cs_cache.desc.p==0) {
        monpanic(vm, "task_switch: CS.p==0\n");
        exception_no = ExceptionNP;
        error_code   = cs_selector.raw & 0xfffc;
        goto post_exception;
        }
      /* All checks pass, fill in shadow cache */
      load_cs_pro(vm, cs_selector, &cs_cache, cs_cache.desc.dpl);
      }
    else {
      /* If new cs selector is null #TS(CS) */
      monpanic(vm, "task_switch: CS NULL\n");
      exception_no = ExceptionTS;
      error_code   = cs_selector.raw & 0xfffc;
      goto post_exception;
      }
  
  
    /* SS */
    if ( !IsNullSelector(ss_selector) ) {
      Boolean good;
      good = fetch_descriptor2(vm, ss_selector, &ss_cache.desc);
      if (!good) {
        monprint(vm, "task_switch: bad SS fetch\n");
        exception_no = ExceptionTS;
        error_code   = ss_selector.raw & 0xfffc;
        goto post_exception;
        }
  
      descriptor2cache(vm, &ss_cache);
      /* SS selector must be within its descriptor table limits else #TS(SS)
       * SS descriptor AR byte must must indicate writable data segment,
       * else #TS(SS)
       */
      if ( (ss_cache.valid==0) ||
           !(ss_cache.desc.type & D_S) ||
           (ss_cache.desc.type & D_EXECUTE) ||
           !(ss_cache.desc.type & D_WRITE) ) {
        monprint(vm, "task_switch: SS not valid\n");
        exception_no = ExceptionTS;
        error_code   = ss_selector.raw & 0xfffc;
        goto post_exception;
        }
  
      /*
       * Stack segment is present in memory, else #SF(new stack segment)
       */
      else if (ss_cache.desc.p==0) {
        monpanic(vm, "task_switch: SS not present\n");
        exception_no = ExceptionSS;
        error_code   = ss_selector.raw & 0xfffc;
        goto post_exception;
        }
  
      /* Stack segment DPL matches CS.RPL, else #TS(new stack segment) */
      else if (ss_cache.desc.dpl != cs_selector.fields.rpl) {
        monpanic(vm, "task_switch: SS.rpl != CS.RPL\n");
        exception_no = ExceptionTS;
        error_code   = ss_selector.raw & 0xfffc;
        goto post_exception;
        }
  
      /* Stack segment DPL matches selector RPL, else #TS(new stack segment) */
      else if (ss_cache.desc.dpl != ss_selector.fields.rpl) {
        monpanic(vm, "task_switch: SS.dpl != SS.rpl\n");
        exception_no = ExceptionTS;
        error_code   = ss_selector.raw & 0xfffc;
        goto post_exception;
        }
  
  #if 0
      /* +++ */
      else if (ss_cache.u.segment.d_b && (tss_cache->type<9)) {
        fprintf(stderr, "++++++++++++++++++++++++++\n");
        exception_no = ExceptionTS;
        error_code   = ss_selector.raw & 0xfffc;
        goto post_exception;
        }
  #endif
      /* All checks pass, fill in shadow cache */
      load_ss_pro(vm, ss_selector, &ss_cache, ss_cache.desc.dpl);
      }
    else {
      /* SS selector is valid, else #TS(new stack segment) */
      monpanic(vm, "task_switch: SS NULL\n");
      exception_no = ExceptionTS;
      error_code   = ss_selector.raw & 0xfffc;
      goto post_exception;
      }
  
  
    /*   if new selector is not null then perform following checks:
     *     index must be within its descriptor table limits else #TS(selector)
     *     AR byte must indicate data or readable code else #TS(selector)
     *     if data or non-conforming code then:
     *       DPL must be >= CPL else #TS(selector)
     *       DPL must be >= RPL else #TS(selector)
     *     AR byte must indicate PRESENT else #NP(selector)
     *     load cache with new segment descriptor and set valid bit
     */
  
  
  
    /* DS */
    if ( !IsNullSelector(ds_selector) ) {
      Boolean good;
      good = fetch_descriptor2(vm, ds_selector, &ds_cache.desc);
      if (!good) {
        monprint(vm, "task_switch: bad DS fetch\n");
        exception_no = ExceptionTS;
        error_code   = ds_selector.raw & 0xfffc;
        goto post_exception;
        }
  
      descriptor2cache(vm, &ds_cache);
      if ( (ds_cache.valid==0) ||
           !(ds_cache.desc.type & D_S) ||
           ((ds_cache.desc.type & D_EXECUTE) &&
            !(ds_cache.desc.type & D_WRITE)) ) {
        monpanic(vm, "task_switch: DS not valid\n");
        exception_no = ExceptionTS;
        error_code   = ds_selector.raw & 0xfffc;
        goto post_exception;
        }
      /* if data or non-conforming code */
      else if ( (ds_cache.desc.type<12) &&
                ((ds_cache.desc.dpl<cs_selector.fields.rpl) ||
                 (ds_cache.desc.dpl<ds_selector.fields.rpl)) ) {
        monpanic(vm, "task_switch: DS.dpl not valid\n");
        exception_no = ExceptionTS;
        error_code   = ds_selector.raw & 0xfffc;
        goto post_exception;
        }
      else if (ds_cache.desc.p==0) {
        monpanic(vm, "task_switch: DS.p==0\n");
        exception_no = ExceptionNP;
        error_code   = ds_selector.raw & 0xfffc;
        goto post_exception;
        }
      /* All checks pass, fill in shadow cache */
      load_sreg_pro(vm, SRegDS, ds_selector, &ds_cache);
      }
    else {
      /* NULL DS selector is OK, leave cache invalid */
      }
  
    /* ES */
    if ( !IsNullSelector(es_selector) ) {
      Boolean good;
      good = fetch_descriptor2(vm, es_selector, &es_cache.desc);
      if (!good) {
        monprint(vm, "task_switch: bad ES fetch\n");
        exception_no = ExceptionTS;
        error_code   = es_selector.raw & 0xfffc;
        goto post_exception;
        }
  
      descriptor2cache(vm, &es_cache);
      if ( (es_cache.valid==0) ||
           !(es_cache.desc.type & D_S) ||
           ((es_cache.desc.type & D_EXECUTE) &&
            !(es_cache.desc.type & D_WRITE)) ) {
        monpanic(vm, "task_switch: ES not valid\n");
        exception_no = ExceptionTS;
        error_code   = es_selector.raw & 0xfffc;
        goto post_exception;
        }
      /* if data or non-conforming code */
      else if ( (es_cache.desc.type<12) &&
                ((es_cache.desc.dpl<cs_selector.fields.rpl) ||
                 (es_cache.desc.dpl<es_selector.fields.rpl)) ) {
        monpanic(vm, "task_switch: ES.dpl not valid\n");
        exception_no = ExceptionTS;
        error_code   = es_selector.raw & 0xfffc;
        goto post_exception;
        }
      else if (es_cache.desc.p==0) {
        monpanic(vm, "task_switch: ES.p==0\n");
        exception_no = ExceptionNP;
        error_code   = es_selector.raw & 0xfffc;
        goto post_exception;
        }
      /* All checks pass, fill in shadow cache */
      load_sreg_pro(vm, SRegES, es_selector, &es_cache);
      }
    else {
      /* NULL ES selector is OK, leave cache invalid */
      }
  
  
    /* FS */
    if ( !IsNullSelector(fs_selector) ) {
      Boolean good;
      good = fetch_descriptor2(vm, fs_selector, &fs_cache.desc);
      if (!good) {
        monprint(vm, "task_switch: bad FS fetch\n");
        exception_no = ExceptionTS;
        error_code   = fs_selector.raw & 0xfffc;
        goto post_exception;
        }
  
      descriptor2cache(vm, &fs_cache);
      if ( (fs_cache.valid==0) ||
           !(fs_cache.desc.type & D_S) ||
           ((fs_cache.desc.type & D_EXECUTE) &&
            !(fs_cache.desc.type & D_WRITE)) ) {
        monpanic(vm, "task_switch: FS not valid\n");
        exception_no = ExceptionTS;
        error_code   = fs_selector.raw & 0xfffc;
        goto post_exception;
        }
      /* if data or non-conforming code */
      else if ( (fs_cache.desc.type<12) &&
                ((fs_cache.desc.dpl<cs_selector.fields.rpl) ||
                 (fs_cache.desc.dpl<fs_selector.fields.rpl)) ) {
        monpanic(vm, "task_switch: FS.dpl not valid\n");
        exception_no = ExceptionTS;
        error_code   = fs_selector.raw & 0xfffc;
        goto post_exception;
        }
      else if (fs_cache.desc.p==0) {
        monpanic(vm, "task_switch: FS.p==0\n");
        exception_no = ExceptionNP;
        error_code   = fs_selector.raw & 0xfffc;
        goto post_exception;
        }
      /* All checks pass, fill in shadow cache */
      load_sreg_pro(vm, SRegFS, fs_selector, &fs_cache);
      }
    else {
      /* NULL FS selector is OK, leave cache invalid */
      }
  
    /* GS */
    if ( !IsNullSelector(gs_selector) ) {
      Boolean good;
      good = fetch_descriptor2(vm, gs_selector, &gs_cache.desc);
      if (!good) {
        monprint(vm, "task_switch: bad GS fetch\n");
        exception_no = ExceptionTS;
        error_code   = gs_selector.raw & 0xfffc;
        goto post_exception;
        }
  
      descriptor2cache(vm, &gs_cache);
      if ( (gs_cache.valid==0) ||
           !(gs_cache.desc.type & D_S) ||
           ((gs_cache.desc.type & D_EXECUTE) &&
            !(gs_cache.desc.type & D_WRITE)) ) {
        monpanic(vm, "task_switch: GS not valid\n");
        exception_no = ExceptionTS;
        error_code   = gs_selector.raw & 0xfffc;
        goto post_exception;
        }
      /* if data or non-conforming code */
      else if ( (gs_cache.desc.type<12) &&
                ((gs_cache.desc.dpl<cs_selector.fields.rpl) ||
                 (gs_cache.desc.dpl<gs_selector.fields.rpl)) ) {
        monpanic(vm, "task_switch: GS.dpl not valid\n");
        exception_no = ExceptionTS;
        error_code   = gs_selector.raw & 0xfffc;
        goto post_exception;
        }
      else if (gs_cache.desc.p==0) {
        monpanic(vm, "task_switch: GS.p==0\n");
        exception_no = ExceptionNP;
        error_code   = gs_selector.raw & 0xfffc;
        goto post_exception;
        }
      /* All checks pass, fill in shadow cache */
      load_sreg_pro(vm, SRegGS, gs_selector, &gs_cache);
      }
    else {
      /* NULL GS selector is OK, leave cache invalid */
      }
    }


  if ( (tss_cache->desc.type>=9) && (trap_word & 0x0001) ) {
    vm->guest_cpu.debug_trap |= 0x00008000; /* BT flag in DR6 */
    vm->guest_cpu.async_event = 1; /* so processor knows to check */
    monprint(vm, "task_switch: T bit set in new TSS.\n");
    }

  /*
   * Step 14: Begin execution of new task.
   */
  return;

post_exception:
  vm->guest_cpu.debug_trap = 0;
  vm->guest_cpu.inhibit_mask = 0;
  monprint(vm, "task switch: posting exception %u after commit point\n",
    exception_no);
  exception(vm, exception_no, error_code);
  return;
}

  void
get_SS_ESP_from_TSS(vm_t *vm, unsigned pl, selector_t *ss, Bit32u *esp)
{
  if (vm->guest_cpu.tr_cache.valid==0)
    monpanic(vm, "get_SS_ESP_from_TSS: TR.cache invalid\n");

  if (vm->guest_cpu.tr_cache.desc.type==9) {
    /* 32-bit TSS */
    Bit32u TSSstackaddr;

    TSSstackaddr = 8*pl + 4;
    if ( (TSSstackaddr+7) > vm->guest_cpu.tr_cache.limit_scaled )
      exception(vm, ExceptionTS, vm->guest_cpu.tr_selector.raw & 0xfffc);

    access_linear(vm, vm->guest_cpu.tr_cache.base +
      TSSstackaddr+4, 2, 0, OP_READ, ss);
    access_linear(vm, vm->guest_cpu.tr_cache.base +
      TSSstackaddr,   4, 0, OP_READ, esp);
    }
  else if (vm->guest_cpu.tr_cache.desc.type==1) {
    /* 16-bit TSS */
    Bit16u temp16;
    Bit32u TSSstackaddr;

    TSSstackaddr = 4*pl + 2;
    if ( (TSSstackaddr+4) > vm->guest_cpu.tr_cache.limit_scaled )
      exception(vm, ExceptionTS, vm->guest_cpu.tr_selector.raw & 0xfffc);

    access_linear(vm, vm->guest_cpu.tr_cache.base +
      TSSstackaddr+2, 2, 0, OP_READ, ss);
    access_linear(vm, vm->guest_cpu.tr_cache.base +
      TSSstackaddr,   2, 0, OP_READ, &temp16);
    *esp = temp16; /* truncate */
    }
  else {
    monpanic(vm, "get_SS_ESP_from_TSS: TR is bogus type (%u)\n",
             (unsigned) vm->guest_cpu.tr_cache.desc.type);
    }
}
