/*
 * $Id: emailform.c,v 1.21 2001/09/16 15:07:38 nordstrom Exp $
 *
 * Viewer - a part of Plucker, the free off-line HTML viewer for PalmOS
 * Copyright (c) 1998-2001, Mark Ian Lillywhite and Michael Nordstrm
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */

#include "const.h"
#include "debug.h"
#include "document.h"
#include "documentdata.h"
#include "emailform.h"
#include "prefsdata.h"
#include "resourceids.h"
#include "screen.h"
#include "util.h"


void DoMail( void ) PLKRDB_SECTION;
void SetMailto( Int16 index ) PLKRDB_SECTION;

static void CheckMem( void ) PLKRDB_SECTION;
static void ScrollMessage( Int16 lines ) PLKRDB_SECTION;
static void SetScrollLabel( FieldPtr field, Int16 id ) PLKRDB_SECTION;
static void UpdateScrollers( void ) PLKRDB_SECTION;
static void InsertText( FieldType* field, Char* text ) PLKRDB_SECTION;
static void GetMailto( void ) PLKRDB_SECTION;
static void SetObjectPosition( UInt16 id, Boolean firstColumn ) PLKRDB_SECTION;
static void EmailFormInit( void ) PLKRDB_SECTION;


/***********************************************************************
 *
 *      Internal Constants
 *
 ***********************************************************************/
static Char toAddress[] = "plucker-dev@rubberchicken.org";


/***********************************************************************
 *
 *      Internal Types
 *
 ***********************************************************************/
typedef struct {
    Int16 to_off;
    Int16 cc_off;
    Int16 subject_off;
    Int16 message_off;
} MailtoData;


/***********************************************************************
 *
 *      Private variables
 *
 ***********************************************************************/
static FieldType*       toField         = NULL;
static FieldType*       ccField         = NULL;
static FieldType*       subjectField    = NULL;
static FieldType*       messageField    = NULL;
static FieldType*       currentField    = NULL;
static ScrollBarType*   scrollBar       = NULL;
static Int16            mailIdx         = 0;



/* Quick and dirty check for enough memory to send a message ( 5000 bytes for body 150 for headers and some more for palm ) */
static void CheckMem( void )
{
    MemHandle test;

    test = MemHandleNew( 7000 );
    if ( test == NULL )
        FrmAlert( warnLowEmailMem );
    else
        MemHandleFree( test );
}



/* Scrolls the message field */
static void ScrollMessage
    (
    Int16 lines /* lines to scroll */
    )
{
    if ( lines < 0 )
        FldScrollField( messageField, -lines, winUp );
    else if ( 0 < lines )
        FldScrollField( messageField, lines, winDown );
}



/* Set label/image for scroll arrows */
static void SetScrollLabel
    (
    FieldPtr    field,  /* field pointer */
    Int16       id      /* object ID */
    )
{
    Char direction[ 2 ];

    if ( FldScrollable( field, winUp ) )
        StrCopy( direction, "\010" );   /* UP */
    else if ( FldScrollable( field, winDown ) )
        StrCopy( direction, "\007" );   /* DOWN */
    else
        StrCopy( direction, "\022" );   /* NONE */

    CtlSetLabel( (ControlType*) GetObjectPtr( id ), direction );
}



/* Updates the scrollbar and scroll arrows */
static void UpdateScrollers( void )
{
    UInt16 scrollPos;
    UInt16 textHeight;
    UInt16 fieldHeight;
    UInt16 maxValue;

    SetScrollLabel( toField, frmEmailToArrow );
    SetScrollLabel( ccField, frmEmailCcArrow );
    SetScrollLabel( subjectField, frmEmailSubjectArrow );

    FldGetScrollValues( messageField, &scrollPos, &textHeight, &fieldHeight );
    if ( fieldHeight < textHeight )
        maxValue = textHeight - fieldHeight;
    else if ( scrollPos )
        maxValue = scrollPos;
    else
        maxValue = 0;
    SclSetScrollBar( scrollBar, scrollPos, 0, maxValue, 6 );
}



/* Insert text in to a textfield and scroll to the top */
static void InsertText
    (
    FieldType*  field,  /* field to insert into */
    Char*       text    /* text to insert */
    )
{
    FldInsert( field, text, StrLen( text ) );
    while ( FldScrollable( field, winUp ) )
        FldScrollField( field, 1, winUp );
}

/* Get pointer into the Mailto record */
static void GetMailto( void )
{
    MemHandle   handle;
    MailtoData* mail;
    Char*       pointer;

    /* Special case for dev-list */
    if ( mailIdx == 0 ) {
        InsertText( toField, toAddress );
        return;
    }

    handle = NULL;
    if ( ReturnRecordHandle( mailIdx, &handle ) ) {
        InsertText( toField, toAddress );
        FrmAlert( errBadMailto );
        return;
    }

    pointer = (Char*) MemHandleLock( handle );
    ErrFatalDisplayIf( pointer == NULL, "GetMailto: MemHandleLock failed" );
    pointer += sizeof( Header );

    mail = (MailtoData*) pointer;
    if ( mail->to_off != 0 )
        InsertText( toField, pointer + mail->to_off );
    if ( mail->cc_off != 0 )
        InsertText( ccField, pointer + mail->cc_off );
    if ( mail->subject_off != 0 )
        InsertText( subjectField, pointer + mail->subject_off );
    if ( mail->message_off != 0 )
        InsertText( messageField, pointer + mail->message_off );

    MemHandleUnlock( handle );
}



/* Set position of object in form */
static void SetObjectPosition
    (
    UInt16  id,         /* object ID */
    Boolean firstColumn /* object should be placed in first column */
    )
{
    FormType*   form;
    UInt16      index;
    Coord       x;
    Coord       y;

    form    = FrmGetFormPtr( frmEmail );
    index   = FrmGetObjectIndex( form, id );
    FrmGetObjectPosition( form, index, &x, &y );

    if ( firstColumn )
        FrmSetObjectPosition( form, index, 0, y );
    else
        FrmSetObjectPosition( form, index, x + 7, y );
}



/* Initialize the email form */
static void EmailFormInit( void )
{
    FormType* form;

    toField         = (FieldType*) GetObjectPtr( frmEmailTo );
    ccField         = (FieldType*) GetObjectPtr( frmEmailCc );
    subjectField    = (FieldType*) GetObjectPtr( frmEmailSubject );
    messageField    = (FieldType*) GetObjectPtr( frmEmailMessage );
    scrollBar       = (ScrollBarType*) GetObjectPtr( frmEmailScrollBar );

    GetMailto();

    if ( Prefs()->scrollbar == SCROLLBAR_LEFT ) {
        SetObjectPosition( frmEmailToLabel, false );
        SetObjectPosition( frmEmailTo, false );
        SetObjectPosition( frmEmailToArrow, true );
        SetObjectPosition( frmEmailCcLabel, false );
        SetObjectPosition( frmEmailCc, false );
        SetObjectPosition( frmEmailCcArrow, true );
        SetObjectPosition( frmEmailSubjectLabel, false );
        SetObjectPosition( frmEmailSubject, false );
        SetObjectPosition( frmEmailSubjectArrow, true );
        SetObjectPosition( frmEmailMessage, false );
        SetObjectPosition( frmEmailScrollBar, true );
    }
    form = FrmGetFormPtr( frmEmail );
    FrmDrawForm( form );
    UpdateScrollers();
}



/* Set current address in the Mailto record */
void SetMailto
    (
    Int16 index /* index of record */
    )
{
    mailIdx = index;
}



/* Event handler for the email form. */
Boolean EmailFormHandleEvent
    (
    const EventType* event  /* pointer to an EventType structure */
    )
{
    Boolean handled;

    SET_A4_FROM_A5 

    handled = false;

    switch ( event->eType ) {
        case ctlSelectEvent:
            switch ( event->data.ctlEnter.controlID ) {
                case frmEmailSend:
                    if ( StrLen( FldGetTextPtr( toField ) ) == 0 ) {
                        FrmAlert( warnNoTo );
                        handled = true;
                        break;
                    }
                    DoMail();
                    FrmGotoForm( Prefs()->lastForm );
                    handled = true;
                    break;

                case frmEmailCancel:
                    FrmGotoForm( Prefs()->lastForm );
                    handled = true;
                    break;

                case frmEmailToArrow:
                    if ( FldScrollable( toField, winUp ) )
                        FldScrollField( toField, 1, winUp );
                    else if ( FldScrollable( toField, winDown ) )
                        FldScrollField( toField, 1, winDown );
                    UpdateScrollers();
                    break;

                case frmEmailCcArrow:
                    if ( FldScrollable( ccField, winUp ) )
                        FldScrollField( ccField, 1, winUp );
                    else if ( FldScrollable( ccField, winDown ) )
                        FldScrollField( ccField, 1, winDown );
                    UpdateScrollers();
                    break;

                case frmEmailSubjectArrow:
                    if ( FldScrollable( subjectField, winUp ) )
                        FldScrollField( subjectField, 1, winUp );
                    else if ( FldScrollable( subjectField, winDown ) )
                        FldScrollField( subjectField, 1, winDown );
                    UpdateScrollers();
                    break;
            }
            break;

        case fldEnterEvent:
            currentField = event->data.fldEnter.pField;
            break;

        case fldChangedEvent:
            UpdateScrollers();
            break;

        case keyDownEvent:
            switch ( event->data.keyDown.chr ) {
                case pageUpChr:
                    if ( FldScrollable( currentField, winUp ) )
                        FldScrollField( currentField, 1, winUp );
                    UpdateScrollers();
                    break;

                case pageDownChr:
                    if ( FldScrollable( currentField, winDown ) )
                        FldScrollField( currentField, 1, winDown );
                    UpdateScrollers();
                    break;

                case chrLineFeed:
                    handled = ( currentField != messageField ); /* no <CR> in other fields */
                    break;
            }
            break;

        case sclRepeatEvent:
            ScrollMessage( event->data.sclRepeat.newValue - event->data.sclRepeat.value );
            break;

        case frmOpenEvent:
            CheckMem();
            EmailFormInit();
            handled = true;
            break;

        case frmCloseEvent:
            handled = false;
            break;

        default:
            handled = false;
    }

    RESTORE_A4 

    return handled;
}



/* Send mail specified in the email form */
void DoMail()
{
    Err                     err;
    LocalID                 dbID;
    UInt32                  result;
    UInt16                  cardNo;
    DmSearchStateType       searchState;
    MailAddRecordParamsType message;

    MemSet( &message, sizeof( message ), 0 );

    message.secret          = false;
    message.signature       = true;
    message.confirmRead     = false;
    message.confirmDelivery = false;
    message.priority        = mailPriorityNormal;
    message.to              = FldGetTextPtr( toField );
    message.cc              = FldGetTextPtr( ccField );
    message.subject         = FldGetTextPtr( subjectField );
    message.body            = FldGetTextPtr( messageField );

    dbID = 0;
    DmGetNextDatabaseByTypeCreator( true, &searchState, sysFileTApplication, 
        'mail', true, &cardNo, &dbID );
    ErrNonFatalDisplayIf( dbID == NULL, "DoMail: Could not find Mail Application" );

    /* Mail app not found. Must be m100! */
    if ( dbID == NULL )
        return;

    err = SysAppLaunch( cardNo, dbID, 0, sysAppLaunchCmdAddRecord, 
            (void*) &message, &result );
    ErrNonFatalDisplayIf( err != errNone, "DoMail: Could not launch Mail Application" );
}
