/*
 * $Id: paragraph.h,v 1.23 2001/09/26 18:33:45 nordstrom Exp $
 *
 * Viewer - a part of Plucker, the free off-line HTML viewer for PalmOS
 * Copyright (c) 1998-2001, Mark Ian Lillywhite and Michael Nordstrm
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */

#ifndef __PLUCKER_PARAGRAPH_H
#define __PLUCKER_PARAGRAPH_H

#include "viewer.h"
#include "document.h"
#include "util.h"


typedef enum {
    ANCHOR_START        = 0x0A,
    NAMED_ANCHOR_START  = 0x0C,
    ANCHOR_END          = 0x08,
    SET_STYLE           = 0x11,
    INLINE_IMAGE        = 0x1A,
    SET_MARGIN          = 0x22,
    ALIGNMENT           = 0x29,
    HRULE               = 0x33,
    NEWLINE             = 0x38,
    ITALIC_START        = 0x40,
    ITALIC_END          = 0x48,
    MULTI_IMAGE         = 0x5C,
    UNDERLINE_START     = 0x60,
    UNDERLINE_END       = 0x68,
    STRIKE_START        = 0x70,
    STRIKE_END          = 0x78
} ParagraphFunction;


typedef enum {
    TOKEN_CHARACTER     = -2, 
    TOKEN_PARAGRAPH_END = -1,
    TOKEN_FUNCTION      = 0
} TokenType;


typedef enum {
    ALIGNMENT_LEFT      = 0, 
    ALIGNMENT_RIGHT     = 1,
    ALIGNMENT_CENTER    = 2, 
    ALIGNMENT_JUSTIFY   = 3
} AlignmentType;


/*
    A paragraph as it appears in the input data stream. The height of the
    paragraph is calculated on the device.

    The "previous" is a relative offset to the previous paragraph; normally
    it is the same as the previous paragraph's length. (The offset is always
    positive; subtract the offset from the current paragraph's pointer to get
    the previous paragraph's pointer).

    Notes...
    A paragraph contains either characters or "functions". A function is
    introduced by a NUL (\0), followed by a function code, followed by up
    to 7 bytes of data (at the moment). The function code's least significant
    3 bits represent the remaining function code length; the most significant
    5 bits represent the function code. Unlike most other places in this
    program, function code length does NOT include the introductory bytes
    (as opposed to objectLength which includes the length of the objectLength
    field). So to get the next character/function, you add the length of
    the function operators + 2 to the pointer to the start of the function.

    Defined function codes:

    00000xxx    0x00  Dummy code (used internally)
    00001010    0x0A  Anchor begins (record ID)
    00001100    0x0C  Named anchor begins (record ID and paragraph offset)
    00001000    0x08  Anchor ends. (==anchor begin with no data)
    00010001    0x11  Set style. (Style is always #0 at start of paragraph)
    00011010    0x1A  Include image (record ID)
    00100010    0x22  Set left and/or right margin (margins in pixels)
    00101001    0x29  Alignment of text (left = 0, right = 1, center = 2)
    00110011    0x33  Horizontal rule
    00111000    0x38  New line
    01000000    0x40  Italic text begins
    01001000    0x48  Italic text ends
    01011100    0x5C  Include scaled and full-size image
    01100000    0x60  Underline begins
    01101000    0x68  Underline ends
    01110000    0x70  Strike-through begins
    01111000    0x78  Strike-through ends


    Note about paragraph attributes:

    The 3 LSB indicates the extra paragraph spacing required
    above the paragraph (value * DEFAULT_PARAGRAPH_SPACING (
    defined in const.h) pixels of spacing). The other bits 
    in the attributes are currently unused.
*/

typedef struct {
    Int16 size;         /* Size of text */
    Int16 attributes;   /* Paragraph info (see above) */
} Paragraph;


typedef struct {
    Int16 height;       /* Height of the paragraph in pixels (see above) */
} MetaParagraph;


/*
   ParagraphContext is used to help parse paragraphs.
 */
typedef struct {
    Char*           last;       /* Points to last character in paragraph + 1 */
    Char*           position;   /* Points to next character in paragraph */
    UInt8*          function;   /* Points to function if last token was a 
                                   function */
    Int16           fontHeight; /* Height of line in current font */
    Int16           left;       /* Left margin */
    Int16           right;      /* Right margin */
    Int16           maxPixels;  /* Max number of pixels for line */
    Int16           linePixels; /* Number of pixels for current line */
    AlignmentType   type;       /* Alignment type */
} ParagraphContext;


void DrawParagraph( TextContext* tContext, Paragraph* paragraph,
        Header* record );
void SetFindPatternData( const Int16 pos, const Int16 len );
void ClearFindPatternData( void );


/* Return pointer to paragraph with given index */
#define GET_PARAGRAPH( RECORD, INDEX )      ( (Paragraph*) ( (UInt8*) ( ( RECORD ) + 1 ) + ( INDEX ) * sizeof( Paragraph ) ) )

/* Return pointer to meta paragraph with given index */
#define GET_METAPARAGRAPH( RECORD, INDEX )  ( (MetaParagraph*) ( (UInt8*) ( ( RECORD ) + 1 ) + ( INDEX ) * sizeof( MetaParagraph ) ) )

/* Return pointer to data in given record */
#define GET_DATA( RECORD )                  ( (UInt8*) ( ( RECORD ) + 1 ) + ( RECORD )->paragraphs * sizeof( Paragraph ) )

#endif
