# This file is released to the Public Domain by Moshe Zadka
# Author: Moshe Zadka <pms@zadka.site.co.il>
import Server,  Folder, MailMessage, Log
import os, httplib, urllib, posixpath, string

def read_netstring(fp):
	len = 0
	while 1:
		c = fp.read(1)
		if c == ':':
			break
		if c == '':
			raise EOFError
		len = len*10 + (ord(c)-ord('0'))
	return fp.read(len)

class MailHTTPServer(Server.Server):

	def __init__(self, name, host, port, secret=None):
		Log.log("initializing PMEP server from", host, port)
		self.name = name
		self.host, self.port = host, port
		if secret is None:
			Log.log("secret was None, reading .secret")
			self.secret = open(os.path.join(os.environ['HOME'], 
			                                '.secret')).read()
		else:
			Log.log("secret was given in constructor")
			self.secret = secret

	def get_message(self, folder, message):
		Log.log("getting message", message, "from", folder)
		return self._get_message(folder, message, 'GET')

	def get_header(self, folder, message):
		Log.log("getting head", message, "from", folder)
		return self._get_message(folder, message, 'HEAD')

	def _get_message(self, folder, message, command):
		Log.log("issuing", command, "for", message, "in", folder)
		url = posixpath.join(folder, message)
		Log.log("url is", url)
		http = httplib.HTTP(self.host, self.port)
		Log.log("initialized HTTP connection");
		http.putrequest(command, url)
		Log.log("issued request", command, url)
		self.endheaders(self, http)
		file = http.sock.makefile('rb')
		try:
			ver, code, msg = string.split(file.readline(), None, 2)
			Log.log("response is", ver, code, msg)
		except ValueError:
			Log.log("bad repsonse!")
			raise IOError("bad response from server")
		if ver[:5] != 'HTTP/':
			raise IOError("bad response from server")
 		if int(code) != 200:
			Log.log("got bad code", code)
			raise IOError(msg)
		Log.log("got message from server")
		return MailMessage.parse(file)

	def endheaders(self, http):
		http.putheader('Host', self.host)
		Log.log("host header", host)
		http.putheader('x-secret', self.secret)
		Log.log("secret header")
		http.endheaders()

	def _get_messages(self, folder, if_messages):
		Log.log("getting list", if_messages, "from", folder)
		if folder[-1] != '/':
			folder = folder + '/'
		http = httplib.HTTP(self.host, self.port)
		http.putrequest('GET', folder)
		self.endheaders(http)
		errcode, errmsg, headers = http.getreply()
		Log.log("got reply", errcode, errmsg, headers)
		file = http.getfile()
		messages = []
		try:
			Log.log("starting to read messages")
			while 1:
				message = read_netstring(file)
				Log.log("read", message)
				if (message[-1] == '/') != if_messages:
					Log.log("appended", message)
					messages.append(message)
		except EOFError:
			pass
		return messages

	def get_list(self, folder):
		Log.log("getting list from", folder)
		return self._get_messages(folder, 1)

	def get_subfolders(self, folder):
		Log.log("getting subfolders from", folder)
		return self._get_messages(folder, 0)

	def subfolder(self, folder, sub):
		Log.log("getting subfolder", sub, "of", folder)
		return Folder.Folder(self.name, posixpath.join(folder, sub))

	def save_message(self, folder, message, uid=None):
		Log.log("saving message to", folder, uid)
		if uid is not None:
			Log.log("specific uid", uid)
			folder = posixpath.join(folder, uid)
		http = httplib.HTTP(self.host, self.port)
		http.putrequest('PUT', folder)
		for (header, value) in message.items():
			Log.log("wrote header", header, value)
			http.putheader(header, value)
		content = message.fp.read()
		http.putheader('content-length', str(len(content)))
		Log.log("content length", len(content))
		self.endheaders(http)
		http.send(content)
		Log.log("sent content")
		errcode, errmsg, headers = http.getreply()
		Log.log("got reply", errcode, errmsg, headers)
		if errcode != 201:
			raise IOError(errmsg)
		return posixpath.basename(headers['location'])

	def create(self, folder):
		Log.log("creating folder", folder)
		http = httplib.HTTP(self.host, self.port)
		http.putrequest('MKCOL', folder)
		self.endheaders(http)
		errcode, errmsg, headers = http.getreply()
		Log.log("got reply", errcode, errmsg, headers)
		if errcode != 201:
			raise IOError(errmsg)
		
	def remove_message(self, folder, uid):
		Log.log("removing", uid, "from", folder)
		url = posixpath.join(folder, uid)
		http = httplib.HTTP(self.host, self.port)
		http.putrequest('DELETE', url)
		self.endheaders(http)
		errcode, errmsg, headers = http.getreply()
		Log.log("got reply", errcode, errmsg, headers)
		if errcode != 206:
			raise IOError(errmsg)

def test():
	import sys
	server = MailHTTPServer('localhost', int(sys.argv[1]))
	messages = server.get_messages('Mail/inbox/')
	print messages
	message = server.get_message('Mail/inbox/', messages[0])
	print message
	uid = server.save_message('Mail/inbox/', message)
	print 'saved at', uid
	messages = server.get_messages('Mail/inbox/')
	print messages
	server.remove_message('Mail/inbox/', uid)
	messages = server.get_messages('Mail/inbox/')
	print messages
	folders = server.get_folders('Mail/inbox/')
	print folders


if __name__ == '__main__':
	test()
