//
// MailStream.cpp
//
// $Id: //poco/1.3/Net/src/MailStream.cpp#1 $
//
// Library: Net
// Package: Mail
// Module:  MailStream
//
// Copyright (c) 2005-2006, Applied Informatics Software Engineering GmbH.
// and Contributors.
//
// Permission is hereby granted, free of charge, to any person or organization
// obtaining a copy of the software and accompanying documentation covered by
// this license (the "Software") to use, reproduce, display, distribute,
// execute, and transmit the Software, and to prepare derivative works of the
// Software, and to permit third-parties to whom the Software is furnished to
// do so, all subject to the following:
// 
// The copyright notices in the Software and this entire statement, including
// the above license grant, this restriction and the following disclaimer,
// must be included in all copies of the Software, in whole or in part, and
// all derivative works of the Software, unless such copies or derivative
// works are solely in the form of machine-executable object code generated by
// a source language processor.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
// SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
// FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.
//


#include "Poco/Net/MailStream.h"


namespace Poco {
namespace Net {


MailStreamBuf::MailStreamBuf(std::istream& istr):
	_pIstr(&istr), 
	_pOstr(0),
	_state(ST_CR_LF)
{
}


MailStreamBuf::MailStreamBuf(std::ostream& ostr):
	_pIstr(0), 
	_pOstr(&ostr),
	_state(ST_CR_LF)
{
}


MailStreamBuf::~MailStreamBuf()
{
}


void MailStreamBuf::close()
{
	if (_pOstr && _state != ST_CR_LF_DOT_CR_LF)
	{
		if (!_buffer.empty())
			_pOstr->write(_buffer.data(), (std::streamsize) _buffer.length());
		if (_state != ST_CR_LF)
			_pOstr->write("\r\n", 2);
		_pOstr->write(".\r\n", 3);
		_state = ST_CR_LF_DOT_CR_LF;
	}
}

		
int MailStreamBuf::readFromDevice()
{
	int c = std::char_traits<char>::eof();
	if (!_buffer.empty())
	{
		c = _buffer[0];
		_buffer.erase(0, 1);
	}
	else
	{
		c = readOne();
		while (c != std::char_traits<char>::eof() && _state != ST_DATA && _state != ST_CR_LF_DOT_CR_LF)
			c = readOne();
		if (!_buffer.empty())
		{
			c = _buffer[0];
			_buffer.erase(0, 1);
		}
	}
	return c;
}


int MailStreamBuf::readOne()
{
	int c = std::char_traits<char>::eof();
	if (_state != ST_CR_LF_DOT_CR_LF)
	{
		c = _pIstr->get();
		switch (c)
		{
		case '\r':
			if (_state == ST_CR_LF_DOT)
				_state = ST_CR_LF_DOT_CR;
			else
				_state = ST_CR;
			break;
		case '\n':
			if (_state == ST_CR)
				_state = ST_CR_LF;
			else if (_state == ST_CR_LF_DOT_CR)
				_state = ST_CR_LF_DOT_CR_LF;
			else
				_state = ST_DATA;
			break;
		case '.':
			if (_state == ST_CR_LF)
				_state = ST_CR_LF_DOT;
			else if (_state == ST_CR_LF_DOT)
				_state = ST_CR_LF_DOT_DOT;
			else
				_state = ST_DATA;
			break;
		default:
			_state = ST_DATA;
		}
		if (_state == ST_CR_LF_DOT_DOT)
			_state = ST_DATA;
		else if (_state == ST_CR_LF_DOT_CR_LF)
			_buffer.resize(_buffer.size() - 2);
		else if (c != std::char_traits<char>::eof())
			_buffer += (char) c;
	}
	return c;
}


int MailStreamBuf::writeToDevice(char c)
{
	switch (c)
	{
	case '\r':
		_state = ST_CR;
		break;
	case '\n':
		if (_state == ST_CR)
			_state = ST_CR_LF;
		else
			_state = ST_DATA;
		break;
	case '.':
		if (_state == ST_CR_LF)
			_state = ST_CR_LF_DOT;
		else
			_state = ST_DATA;
		break;
	default:
		_state = ST_DATA;
	}
	if (_state == ST_DATA)
	{
		if (!_buffer.empty())
		{
			_pOstr->write(_buffer.data(), (std::streamsize) _buffer.length());
			_buffer.clear();
		}
		_pOstr->put(c);
	}
	else if (_state == ST_CR_LF_DOT)
	{
		_pOstr->write("\r\n..", 4);
		_state = ST_DATA;
		_buffer.clear();
	}
	else _buffer += c;
	return charToInt(c);
}


MailIOS::MailIOS(std::istream& istr): _buf(istr)
{
	poco_ios_init(&_buf);
}


MailIOS::MailIOS(std::ostream& ostr): _buf(ostr)
{
	poco_ios_init(&_buf);
}


MailIOS::~MailIOS()
{
}


void MailIOS::close()
{
	_buf.close();
}


MailStreamBuf* MailIOS::rdbuf()
{
	return &_buf;
}


MailInputStream::MailInputStream(std::istream& istr): 
	MailIOS(istr), 
	std::istream(&_buf)
{
}


MailInputStream::~MailInputStream()
{
}


MailOutputStream::MailOutputStream(std::ostream& ostr): 
	MailIOS(ostr), 
	std::ostream(&_buf)
{
}


MailOutputStream::~MailOutputStream()
{
}


} } // namespace Poco::Net
