%{ /* Emacs, use -*- mode:text -*- */
/*
 * Lexically analyizes strace output.
 * Copyright (C) 2004 The MITRE Corporation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdio.h>
#include <unistd.h>
#include <ctype.h>
#include "scanner.h"
#include "parser.h"

/* For error reporting. */
static int lineno = 1;

%}

W			[ \t]+
D			[0-9]+

%%

\n			{ lineno++; ECHO; return '\n'; }

\t			{ yyerror("Stray tab char found in input"); exit(1); }

			/* Recognize and emit a C style string. */
\"([^\"\\\n]|\\.)*\"	{ ECHO; }

			/* Recognize possible start of an argument list. */
"("			{ yylval.str = yytext; return LPAREN; }

			/* Recognize possible separator in an */
			/* argument list. */
","{W}			{ yylval.str = yytext; return COMMA; }

			/* Recognize possible end of an argument list. */
")"{W}"="{W}		{ yylval.str = yytext; return RPAREN; }

			/* Recognize the start of a signal description. */
^("---"|"+++")		{ ECHO; return HERALD; }

			/* Recognize the start of a signal description*/
			/* after printing the leading pid as a */
			/* separate field. */
^{D}{W}("---"|"+++")	{ int i;
			  for (i = 0; i < yyleng; i++) { /* print pid */
			    int ch = yytext[i];
			    if (isspace(ch))
			      break;
			    putchar(ch);
			  }
			  putchar('\t');
			  for (; i < yyleng; i++)	/* skip spaces */
			    if (!isspace(yytext[i]))
			      break;
			  if (i < yyleng)
			    fwrite(yytext + i, yyleng - i, 1, stdout);
			  return HERALD;
			}

			/* Print leading pid as a separate field. */
^{D}{W}			{ int i;
                          for (i = 0; i < yyleng; i++) {
			    int ch = yytext[i];
			    if (isspace(ch))
			      break;
			    putchar(ch);
			  }
			  putchar('\t');
			}

			/* Recognize possible start of a struct. */
"{"			{ ECHO; return LBRACE; }

			/* Recognize possible end of a struct. */
"}"			{ ECHO; return RBRACE; }

			/* Recognize possible start of an array. */
"["			{ ECHO; return LBRACKET; }

			/* Recognize possible end of an array. */
"]"			{ ECHO; return RBRACKET; }

%%

static const char *filename = "-";

void
setfile(const char *arg)
{
  if (arg)
    filename = arg;
}

int
yyerror(const char *msg)
{
  return fprintf(stderr, "%s:%d: %s at token %s\n",
                 filename, lineno, msg, yytext);
}
